/* Copyright (c) 2009 Nordic Semiconductor. All Rights Reserved.
 *
 * The information contained herein is confidential property of Nordic 
 * Semiconductor ASA.Terms and conditions of usage are described in detail 
 * in NORDIC SEMICONDUCTOR STANDARD SOFTWARE LICENSE AGREEMENT. 
 *
 * Licensees are granted free, non-transferable use of the information. NO
 * WARRENTY of ANY KIND is provided. This heading must NOT be removed from
 * the file.
 *              
 * $LastChangedRevision: 5717 $
 */

/** @file
 * @brief Implementation of hal_w2
 */

#include <stdint.h>
#include <stdbool.h>

#include <Nordic\reg24le1.h>
#include "nordic_common.h"
#include "hal_w2_mod.h"

#define BROADCAST_ENABLE      7     // W2CON0 register bit 7
#define CLOCK_STOP            6     // W2CON0 register bit 6
#define X_STOP                5     // W2CON0 register bit 5
#define X_START               4     // W2CON0 register bit 4
#define CLOCK_FREQUENCY_1     3     // W2CON0 register bit 3
#define CLOCK_FREQUENCY_0     2     // W2CON0 register bit 2
#define MASTER_SELECT         1     // W2CON0 register bit 1
#define WIRE_2_ENABLE         0     // W2CON0 register bit 0

void hal_w2_soft_reset();

/* Slave specific functions */

void hal_w2_respond_to_gen_adr(bool resp_gen)
{ 
  if(resp_gen)
  {                                                                   
    W2CON0 = W2CON0 | (1 << BROADCAST_ENABLE);  // Set "broadcastEnable" bit
  }
  else
  {
    W2CON0 = W2CON0 & ~(1 << BROADCAST_ENABLE); // Clear "broadcastEnable" bit
  }
}

void hal_w2_alter_clock(bool alt_clk)
{
  if(alt_clk)                         
  {
    W2CON0 = W2CON0 | (1 << CLOCK_STOP);      // Set "clockStop" bit
  }
  else
  {
    W2CON0 = W2CON0 & ~(1 << CLOCK_STOP);     // Clear "clockStop" bit
  }
}

void hal_w2_irq_stop_cond_enable(bool stop_cond)
{ 
  if(stop_cond)
  {                                                                  
    W2CON0 = W2CON0 & ~(1 << X_STOP);         // Clear "xStop" bit
  }
  else
  {
    W2CON0 = W2CON0 | (1 << X_STOP);          // Set "xStop" bit
  }
}

void hal_w2_irq_adr_match_enable(bool addr_match)
{
  if(addr_match)
  {
    W2CON0 = W2CON0 & ~(1 << X_START);        // Clear "xStart" bit
  }
  else
  {
    W2CON0 = W2CON0 | (1 << X_START);         // Set "xStart" bit
  }
}

void hal_w2_set_slave_address(uint8_t address)
{
  W2SADR = (address & 0x7F);                  // Set 7 bit adress of the slave
}

/* General functions */

void hal_w2_set_clk_freq(hal_w2_clk_freq_t freq)
{                                             
  W2CON0 = (W2CON0 & 0xF3) | (((uint8_t)freq) << CLOCK_FREQUENCY_0);       
}                                             // Update "clockFrequency" bits

void hal_w2_set_op_mode(hal_w2_op_mode_t mode)
{
  if(mode == HAL_W2_MASTER)                   // Check for master mode
  {                                                                  
    W2CON0 = W2CON0 | (1 << MASTER_SELECT);   // Set "masterSelect" bit
  }
  else
  {
    W2CON0 = W2CON0 & ~(1 << MASTER_SELECT);  // Clear "masterSelect" bit
  }
}

void hal_w2_enable(bool en)
{ 
  if(en)
  {
    W2CON0 = W2CON0 | (1 << WIRE_2_ENABLE);   // Set "wire2Enable" bit
  }
  else
  {
    W2CON0 = W2CON0 & ~(1 << WIRE_2_ENABLE);  // Clear "wire2Enable" bit
  }
}

void hal_w2_all_irq_enable(bool irq)
{ /* In this function the standard "read-modify-write" is not used because
     bit 4:0 (the status bits) in W2CON1 are cleared when read. These bits
     are read only so they can not be modified. */
  if(irq)
  {
    W2CON1 = ~(BIT_5);                        // Clear "maskIrq" bit
  }
  else
  {
    W2CON1 = BIT_5;                           // Set "maskIrq" bit
  }
}

void hal_w2_configure_master(hal_w2_clk_freq_t mode)
{
  hal_w2_enable(1);
  hal_w2_set_clk_freq(mode);
  hal_w2_set_op_mode(HAL_W2_MASTER);

  INTEXP |= 0x04;                         // Enable 2 wire interrupts
  hal_w2_all_irq_enable(true);             // Enable interrupts in the 2-wire  
}

uint8_t hal_w2_test_address(uint8_t address)
{
  uint8_t ret_val;
  HAL_W2_ISSUE_START_COND;
  HAL_W2_WRITE(address << 1);
  HAL_W2_WAIT_FOR_INTERRUPT;
  if( W2CON1 & W2CON1_FLAG_NACK ) ret_val = 0; //NACK
  else ret_val = 1; //ACK
  HAL_W2_ISSUE_STOP_COND;
  return ret_val;
}

uint8_t hal_w2_write_bytes(uint8_t address, uint8_t *data_ptr, uint8_t data_len)
{
  uint8_t nack_received = 0;
  HAL_W2_ISSUE_START_COND;
  HAL_W2_WRITE(address << 1);
  HAL_W2_WAIT_FOR_INTERRUPT;
  if( W2CON1 & W2CON1_FLAG_NACK ) nack_received = 1; //NACK
  while(data_len-- && !nack_received)
  {
    HAL_W2_WRITE(*data_ptr++);
    HAL_W2_WAIT_FOR_INTERRUPT;
    if( W2CON1 & W2CON1_FLAG_NACK ) nack_received = 1; 
  }
  HAL_W2_ISSUE_STOP_COND;
  return !nack_received;
}
uint8_t hal_w2_read_bytes(uint8_t address, uint8_t *data_ptr, uint8_t data_len)
{
  uint8_t nack_received = 0, status;
  HAL_W2_ISSUE_START_COND;
  HAL_W2_WRITE(address << 1 | 0x01);
  HAL_W2_WAIT_FOR_INTERRUPT;
  if( W2CON1 & W2CON1_FLAG_NACK ) 
  {
    nack_received = 1;  
    // This situation (NACK received on bus while trying to read from a slave) leads to a deadlock in the 2-wire interface. 
    hal_w2_soft_reset();
  }
  while(data_len-- && !nack_received)
  {
    if( data_len == 0 ) HAL_W2_ISSUE_STOP_COND;
    do
    {
      HAL_W2_WAIT_FOR_INTERRUPT;
      status = W2CON1;
    }while( (status & 0x01) == 0 );
    *data_ptr++ = HAL_W2_READ();
    if(status & W2CON1_FLAG_NACK) nack_received = 1;  
  }
  return !nack_received;  
}
uint8_t hal_w2_write_and_read_bytes(uint8_t address, uint8_t *wr_data_ptr, uint8_t wr_data_len, uint8_t *rd_data_ptr, uint8_t rd_data_len)
{
  uint8_t nack_received = 0, status;
  HAL_W2_ISSUE_START_COND;
  HAL_W2_WRITE(address << 1);
  HAL_W2_WAIT_FOR_INTERRUPT;
  if( W2CON1 & W2CON1_FLAG_NACK ) nack_received = 1; //NACK
  while(wr_data_len-- && !nack_received)
  {
    HAL_W2_WRITE(*wr_data_ptr++);
    HAL_W2_WAIT_FOR_INTERRUPT;
    if( W2CON1 & W2CON1_FLAG_NACK ) nack_received = 1; 
  }
  // Abort if NACK received
  if( nack_received )
  {
    HAL_W2_ISSUE_STOP_COND;
    return 0;
  }
  // If NACK not received, issue repeated start and read command
  HAL_W2_ISSUE_START_COND;
  HAL_W2_WRITE(address << 1 | 0x01);
  HAL_W2_WAIT_FOR_INTERRUPT;
  if( W2CON1 & W2CON1_FLAG_NACK ) 
  {
    nack_received = 1;  
    // This situation (NACK received on bus while trying to read from a slave) leads to a deadlock in the 2-wire interface. 
    hal_w2_soft_reset();
  }
  while(rd_data_len-- && !nack_received)
  {
    if( rd_data_len == 0 ) HAL_W2_ISSUE_STOP_COND;
    do
    {
      HAL_W2_WAIT_FOR_INTERRUPT;
      status = W2CON1;
    }while( (status & 0x01) == 0 );
    *rd_data_ptr++ = HAL_W2_READ();
    if(status & W2CON1_FLAG_NACK) nack_received = 1;  
  }
  return !nack_received;  
}

void delay_5us()
{
  xdata uint8_t delvar = 3;
  while(delvar--);
}
void hal_w2_soft_reset()
{
#ifndef W2_SOFT_RESET_NOT_AVAILABLE
  uint8_t pulsecount, w2_freq;

  // Store the selected 2-wire frequency 
  w2_freq = W2CON0 & 0x0C;
  // Prepare the GPIO's to take over SDA & SCL
  HAL_W2_CLEAR_SDA_SCL;
  HAL_W2_OVERRIDE_SDA_SCL(1, 1);
  //P0DIR = 0xFF;
  
  // Reset 2-wire. SCL goes high.
  W2CON0 = 0x03;
  W2CON0 = 0x07;
  
  // Disable 2-wire.
  W2CON0 = 0x06;
  
  // SDA and SCL are now under software control, and both are high. 
  // Complete first SCL pulse.
  //P0DIR = 0xEF;
  HAL_W2_OVERRIDE_SDA_SCL(1, 0);
  
  // SCL low
  delay_5us();
  //P0DIR = 0xCF;
  HAL_W2_OVERRIDE_SDA_SCL(0, 0);
  
  // SDA low
  // Create SCL pulses for 7 more data bits and ACK/NACK
  delay_5us();
  for( pulsecount = 0; pulsecount < 8; pulsecount++ )
  {
    //P0DIR = 0xDF;
    HAL_W2_OVERRIDE_SDA_SCL(0, 1);
    delay_5us();
    //P0DIR = 0xCF;
    HAL_W2_OVERRIDE_SDA_SCL(0, 0);
    delay_5us();
  }
  
  // Generating stop condition by driving SCL high
  delay_5us();
  //P0DIR = 0xDF;
  HAL_W2_OVERRIDE_SDA_SCL(0, 1);
  
  // Drive SDA high
  delay_5us();
  //P0DIR = 0xFF;
  HAL_W2_OVERRIDE_SDA_SCL(1, 1);
  
  // Work-around done. Return control to 2-wire.
  W2CON0 = 0x07;
  
  // Reset 2-wire and return to master mode at the frequency selected before calling this function
  W2CON0 = 0x03;
  W2CON0 = 0x03 | w2_freq;
#endif
}