/**************** (c) 2004  STMicroelectronics ********************************
     
PROJECT  : 3-phase PMAC induction motor drive Software Library
COMPILER : COSMIC / METROWERKS

MODULE  :  acmotor.c
VERSION :  Release 1.0.0

CREATION DATE : 

AUTHOR :	Microcontroller Division Applications
			Consumer & Micro Group

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

DESCRIPTION :   3-phase PMAC motor control routines, include Start/Stop 
					management and PI regulation
              
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

MODIFICATIONS :

*******************************************************************************
 THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. ST MICROELECTRONICS
 SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
 WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
******************************************************************************/

#include "lib.h"		/*General purpose typedefs and macros */
#include "mtc.h"		/*Public motor control peripheral function prototypes*/ 
#include "PMACparam.h"	/*AC Motor Control parameters */
#include "acmotor.h"	/*Public AC motor control function prototypes */
#include "pwmart.h"		/*Public general purpose Timebase function prototypes*/    
#include "config.h"

#ifdef PI_PARAM_TUNING
#include "adc.h"
	u8	Kp, Ki;  
#else
	static u8	Kp, Ki;
#endif

static s32 VoltageIntegralTerm;                      
static s16 Error;
static s32 DeltaVoltage;
static s32 Voltage_s32;         
static BOOL MinPiOut;
static BOOL MaxPiOut;

#ifdef __HIWARE__				/* Metrowerks Compiler */
	extern	volatile u16 	Phase;
#elif defined(__CSMC__)		/* Cosmic Compiler */
	extern	@tiny volatile u16 	Phase;	
#endif

/*-----------------------------------------------------------------------------
ROUTINE Name : ACM_Init

Description: This function performs the initialisation of ACM module.
             Conditions for proper initialisation of ACM module:
             _ MTC functions called in below described order,
             _ Init Sinewave generation with Voltage = 0 to be sure that duty 
               cycle on each phase is 50% when starting.
-----------------------------------------------------------------------------*/
void ACM_Init(void)
{
	MTC_InitPeripheral();			// Initialize peripheral for three phase PWM generation
	MTC_InitSineGen();				// Configure SW variables for sinewave generation
        MTC_Set_ClockWise_Direction();
	ART_SetSpeedRegPeriod(REG_LOOP_TIME);
        VoltageIntegralTerm=0;
}

/*-----------------------------------------------------------------------------
ROUTINE Name : ACM_InitSoftStart

Description: This function performs:
         _ Init of Tacho acquisition 
         _ Set the initial value of Stator Voltage & Freq 
           (Stator Voltage to zero, Stator Freq to LOWEST_FREQ)
         _ Set the time interval in ART equal to VOLT_SLEWRATE
           (so each ART interrupt the SoftStart increase by 1 Stator Voltage)
         _ If we are in closed loop is set the timeout
-----------------------------------------------------------------------------*/
void ACM_InitSoftStart()
{
        u16 indice=0;
        
	MTC_InitTachoMeasure( );
	MTC_StartTachoFiltering();

	MTC_UpdateSine(INIT_START_VOLTAGE,STARTUP_STATOR_FREQ);
	MTC_EnableMCOutputs();			// PWM is present on MCO outputs  

	#ifdef ALIGNMENT_PHASE 
	
		// Alignment Phase
		ART_SetMainTimeBase(1);    
	
		MIMR &= 0x7F;
        	while(indice<ALIGNMENT_DUTY)
    		{
    			ToCMPxL(MCPUL,indice);	// 6% duty cycle on U
			ToCMPxH(MCPUH,indice);	
			ToCMPxL(MCPVL,0);	// 0% duty cycle on V                 
			ToCMPxH(MCPVH,0);                                          
			ToCMPxL(MCPWL,0);	// 0% duty cycle on W
			ToCMPxH(MCPWH,0);    		
			indice+=1;
    			while (!ART_IsMainTimeElapsed());
    		} 
    		MIMR |= MIMR_PUM_OR;		// Re-enable the PWM U interrupts
    		Phase = (u16) Shift_Phase + 5461; // + 30 Degree
    		//MTC_UpdateSine(INIT_START_VOLTAGE,STARTUP_STATOR_FREQ);
    		//MTC_UpdateSine((u8)(ADC_GetRV1()),STARTUP_STATOR_FREQ);
    	#endif
    	
	if (VOLT_SLEWRATE == 0)
	{
		ART_Set_TimeInMs(1);	// Max slope: avoid applying full voltage directly
	}
	else
	{
		ART_Set_TimeInMs(VOLT_SLEWRATE);		// Set soft start slope
	}

	//#if (CONTROL == 1)	
	ART_SetSequenceDuration(START_TIMEOUT);   
	//#endif

}

#if (CONTROL == 1)
/*-----------------------------------------------------------------------------
ROUTINE Name : ACM_SoftStart

Description: This function performs:  Close Loop Startup
         _ Soft Start by increasing Voltage from zero to predefined value
         _ Init of Tacho acquisition for closed loop mode
         _ Monitoring of tacho feedback to limit starting voltage
           (and therefore limit starting torque).
Note: if this function is called with MinRotorFreq = 0, no tests are made on
      rotor speed feedback (open loop mode).
End of Soft start. Additional time given for high inertia load 
-----------------------------------------------------------------------------*/
StartStatus_t ACM_SoftStart()
{
	u8 CurrentVoltage, VoltageMax;
	u16 StatorFreq, CurrentFrequency;

	StatorFreq  = MTC_GetStatorFreq();
	CurrentVoltage = MTC_GetVoltage();
	CurrentFrequency = MTC_GetRotorFreq();

	if (MTC_ValidSpeedInfo(CLOSEDLOOP_MIN_FREQ))
	{
		MTC_StartTachoFiltering();
		return (START_OK);
	}
	else
	{
		if (ART_IsSequenceCompleted())
		{
			MTC_DisableMCOutputs();			// PWM is disabled on MCO outputs
			return (START_FAIL);
		}
		else
		{
			if ( (CurrentVoltage < STATOR_VOLTAGE) && (ART_Is_TimeInMsElapsed()) )
			{
				CurrentVoltage++;                        
				if (CurrentFrequency < STARTUP_STATOR_FREQ)
					CurrentFrequency = STARTUP_STATOR_FREQ;
				MTC_UpdateSine(CurrentVoltage,CurrentFrequency);
			}
			return (START_ONGOING);
		}
	}
}
#endif

#if (CONTROL == 0)
/*-----------------------------------------------------------------------------
ROUTINE Name : ACM_SoftStartOL

Description: This function performs:
         _ Increasing the Stator voltage by 1 each VOLT_SLEWRATE ms until 
           TargetVoltage is reached 
         _ If Rotor Freq is below LOWEST_FREQ the Stator freq is set to LOWEST_FREQ
           else the Synchronism is imposed setting Stator freq equal to Rotor Freq
-----------------------------------------------------------------------------*/
BOOL ACM_SoftStartOL(u8 TargetVoltage)
{
	u8 CurrentVoltage;
	u16 CurrentFrequency;

	CurrentVoltage = MTC_GetVoltage();
	CurrentFrequency = MTC_GetRotorFreq();
	
	if ( ART_Is_TimeInMsElapsed() )
	{
		if (CurrentVoltage < TargetVoltage)   // increase voltage
		{
			CurrentVoltage++;
		}
		if (CurrentFrequency < STARTUP_STATOR_FREQ)
			CurrentFrequency = STARTUP_STATOR_FREQ;
		MTC_UpdateSine(CurrentVoltage, CurrentFrequency);
	}	
        
	if (CurrentVoltage >= TargetVoltage)
	{
		return (TRUE);
	}
	else 
		return (FALSE);
}
#endif

#if (CONTROL == 1)
void ACM_InitSpeedReg(u16 RifFreq)
{
	u8 CurrentVoltage = (u8)MTC_GetVoltage();

	{
		// PI error calculation
		Error = RifFreq - MTC_GetRotorFreq();
	}

	#if !defined PI_PARAM_TUNING
		ACM_GetPIParam( MTC_GetStatorFreq() );
	#else   
		Kp=(u8)(ADC_GetRV2());
		Ki=(u8)(ADC_GetRV3());  
	#endif 
	  
	Voltage_s32 = ((s32) Kp * (s32)Error) >> 8;
	VoltageIntegralTerm =  ( (s32)CurrentVoltage - Voltage_s32 );

	MinPiOut = FALSE;
	MaxPiOut = FALSE;

}
                                     
u8 ACM_SpeedRegulation(u16 RifFreq,u16 currFreq)                                   
{                          
	s16 NewVoltage;
	u8  Voltage_out;      
	
	// PI error calculation
	Error = RifFreq - currFreq;
	
	#if !defined PI_PARAM_TUNING
		ACM_GetPIParam( MTC_GetStatorFreq() );		// Ki and Kp coeff setup
	#else
		Kp=(u8)(ADC_GetRV2());
		Ki=(u8)(ADC_GetRV3());   
	#endif	
	/* Proportional term */ 
	Voltage_s32 = (s32) 256 * ((s32) Kp * (s32) Error);
	NewVoltage = (s16)(Voltage_s32 >> 16);
	
	// Integral term 
	// If modulation is maximum, integral term must be "frozen" 
	{  
		s32 PreviousVoltageIntegral = VoltageIntegralTerm;   

		DeltaVoltage = (s32)Ki * (s32)Error; 
		if( ((Error>0) && !MaxPiOut) || ((Error<0) && !MinPiOut) )
	   	{
			VoltageIntegralTerm += DeltaVoltage; // "integral" output 
		}

		if ( ( DeltaVoltage >= 0 ) && ( VoltageIntegralTerm < PreviousVoltageIntegral ) )
		{
			VoltageIntegralTerm = S32_MAX;	// Avoid IntTerm Overflow
		}

		if ( ( DeltaVoltage < 0 ) &&  ( VoltageIntegralTerm > PreviousVoltageIntegral ) )
		{
			VoltageIntegralTerm = S32_MIN;	// Avoid IntTerm Underflow
		}    
	}         
	
	
	NewVoltage += (s16)( VoltageIntegralTerm / 256 ); // Sum "proportional" and "integral" terms */
        
        if ( NewVoltage < 0 )
	{
		Voltage_out = 0;
		MinPiOut = TRUE;
	}
	else
	{
		u8 MaxVoltage;
		#ifdef V_F_LIM
		MaxVoltage = ACM_VoltageMaxAllowed( MTC_GetStatorFreq() ); // V/F Limitation
		#else
		MaxVoltage = 255; // Don't use V/F limitation
		#endif
		if ( NewVoltage > (s16)MaxVoltage )
		{
			Voltage_out = MaxVoltage;
			MaxPiOut = TRUE;  /* Set ClampFlag if modulation reaches maximum value */
		}
		else
		{
			Voltage_out = (u8)NewVoltage;
			MinPiOut = FALSE;
			MaxPiOut = FALSE;
		}
	}
        
	return (Voltage_out);
}

/*-----------------------------------------------------------------------------
ROUTINE Name : ACM_GetPIParam

Description: This function returns the proportional and integral parameters of
			 PI slip regulation.
             It performs a linear interpolation between the 2 setpoints.
-----------------------------------------------------------------------------*/
void ACM_GetPIParam(u16 StatorFrequency)
{

	if ( StatorFrequency <= PI_LOWFREQ_LIMIT )
	{
	     Kp = PI_PROP_LOWSPD;
	     Ki = PI_INT_LOWSPD;	
	} 
	
	else if ( StatorFrequency >= PI_HIGHFREQ_LIMIT )
	{
	     Kp = PI_PROP_HIGHSPD;
	     Ki = PI_INT_HIGHSPD;	
	}
	
	else
	{
		u16 Buffer;
		Buffer = StatorFrequency * PI_PROP_COEFF - PI_PROP_OFFSET;
		Kp = (u8)(PI_PROP_LOWSPD - (u8) (Buffer / 256));
		Buffer = StatorFrequency * PI_INT_COEFF - PI_INT_OFFSET;
		Ki = (u8)(PI_INT_LOWSPD + (u8) (Buffer / 256));
	}

}
#endif  

/*-----------------------------------------------------------------------------
ROUTINE Name : ACM_VoltageMaxAllowed

Description: This function limits the voltage on motor stator according to
             setpoints given in MTCParam.h.
             It performs a linear interpolation.
             No error codes are returned if StatorFrequency is outside absolute
             domain [VF_LOWFREQ_LIMIT .. VF_HIGHFREQ_LIMIT].

Duration:

Important caution: when modifying setpoints, user must keep in mind that
            VF_COEFF, VF_OFFSET and local variable Buffer are declared as u16.
            They must therefore be declared as u32 if overflows can occur.
-----------------------------------------------------------------------------*/
u8 ACM_VoltageMaxAllowed(u16 StatorFrequency)
{
         u8  VoltageMax;

  if ( StatorFrequency <= VF_LOWFREQ_LIMIT )
  {
       VoltageMax = V_MIN;

  } else if ( StatorFrequency >= VF_HIGHFREQ_LIMIT )
         {
              VoltageMax = V_MAX;

         } else {
                       u16 Buffer;
                   Buffer = (u16)(StatorFrequency * VF_COEFF) - VF_OFFSET;
                   VoltageMax = (u8)(V_MIN + (u8)((u16)Buffer / (u16)256));
            /* 256 factor is included in VF COEFF and VF_OFFSET calculations */
                        /* Aim is to keep calculation accuracy */
                }

  return ( VoltageMax );
}


/*** (c) 2004  STMicroelectronics **************************** END OF FILE ***/