/**************** (c) 2004  STMicroelectronics **********************
     
PROJECT  : 3-phase PMAC induction motor drive Software Library
COMPILER : COSMIC / METROWERKS

MODULE  :  main.c
VERSION :  Release 1.0.0

CREATION DATE : 

AUTHOR :	Microcontroller Division Applications
			Consumer & Micro Group

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

DESCRIPTION :   Main Routine
              
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

MODIFICATIONS :

 ******************************************************************************
 THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. ST MICROELECTRONICS
 SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
 WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
*******************************************************************************

******************************************************************************/

// Standard types definitions and ST7 specific macros
#include "lib.h"
// General purpose peripheral function prototypes
#include "ports.h"
#include "adc.h"
#include "pwmart.h"
#include "sci.h"
// ST7 specifics related functions prototypes
#include "ST7_Misc.h"
// Motor control specific functions prototypes
#include "mtc.h"		/* Peripheral specifics */
#include "acmotor.h"	/* AC motor related */
#include "config.h"		/* AC motor library configuration keys */
#include "MainParam.h"	/* Motor control parameters used in functions called from main.c */
// ST7FMC peripherals Hardware Registers declaration
#define __DEFINE_REGISTERS_STVD7_INCLUDE__   
#include "PMACparam.h" 
#if defined (ENABLE_RS232) && defined (__CSMC__)
	#include <string.h>
#endif 

///////////////////////////////////////////////////////////////////////////////

/* Private typedefs (for demo purposes only) */
typedef enum
{
IDLE, START, RUN, STOP, BRAKE, WAIT, FAULT
} SystStatus_t;

/* Private variable (for demo purposes only) */
static SystStatus_t State; 

#ifdef ENABLE_RS232  
	// WARNING: If RS232_SPEED and/or RS232_VOLTAGE are defined 
	// the UartBuffer array must be enlarged accordingly
	static u8 UartBuffer[32]; // Only Phase angle or Param Tuning trasmit is allowed with 16 byte of buffer
#endif

/* Private functions (for demo purposes only) */
static void DoMotorControl( void );
static SystStatus_t StartMotor( void );
static void	CheckPowerStage(void);
static void	SendSystemDataRS232(void);  

//////////////////* Private constants (for demo purposes only) *///////////////
// Set LED blinking rate (x10ms), refer to ART_SetMainTimeBase prototype
#define LED_BLINKING_RATE	((u8)10)		/* 10 x 10ms = 100ms */

#define IDLE_TIME		((u16)1000)	/* 1000 ms idle time (before allowing 
						motor re-start */   
#if (CONTROL == 0)
	static u8 Target_Curr;
#endif

void main(void)
{        
	               
	u16 Shift_Phase_tmp; // Temporary variable to calculate phase shift
	
	ST7_IntPrioritySetUp();
	PORTS_Init();
	ADC_Init();
	ART_Init();						// PWM ART provided general purpose time bases
	ACM_Init();
	#ifdef ENABLE_RS232
	//	Init SCI for hyperterminal: 38400bps/8 data bit/no parity/1 stop bit
	SCI_Init(SCI_DEFAULT_PARAM1,SCI_DEFAULT_PARAM2); 
	SCI_Select_Baudrate(SCI_PR_13 + SCI_TR_1 + SCI_RR_1);
	SCI_Mode(SCI_TX_ENABLE);
	#endif

	ART_SetMainTimeBase(LED_BLINKING_RATE);		// 100ms timebase for LED Toggle

	EnableInterrupts();

	ART_Start();					// Time bases are now available
	State = IDLE;
    
	// Flash LEDs to indicate proper MCU start-up
	PORTS_GreenLedOn();
	while (!ART_IsMainTimeElapsed());
	PORTS_RedLedOn();
	while (!ART_IsMainTimeElapsed());
	PORTS_GreenLedOn(); 
	
	// Main Loop
	while(1)
	{       
	
		// Setting of Phase Angle
		#ifdef PHASE_ANGLE_RV3
			Shift_Phase_tmp = ((u16)(ADC_GetRV3()) << 8 ); // Selected By RV3     
		#else
		   	if (MTC_GetRotorFreq()<PHI_LOWFREQ)	
				Shift_Phase_tmp = (u16) PHI_MIN << 8;
                	else if (MTC_GetRotorFreq()<PHI_HIGHFREQ)	
                	{
                		Shift_Phase_tmp =  (s16)((coeff_phase*(MTC_GetRotorFreq()-PHI_LOWFREQ))/256 + PHI_MIN); 
                		Shift_Phase_tmp = Shift_Phase_tmp << 8;
			}                	
                	else Shift_Phase_tmp = (u16) PHI_MAX << 8;
		#endif
		if (State == BRAKE)
			Shift_Phase = (u16)(Shift_Phase_tmp - 32768);
		else
			Shift_Phase = Shift_Phase_tmp;
	
		switch ( State )
		{
		case IDLE:	
			if ( PORTS_KeyScan() == TRUE )	// Start motor
			{
				ACM_InitSoftStart();
				State = START;  
			}
			break;

		case START:
			State = StartMotor(); 
			if (State == RUN)
			{             
				#if (CONTROL == 0)
				Target_Curr = (u8)ADC_GetRV1();
				ART_Set_TimeInMs(SLEW_LIMIT); 
				#endif
			}
			break;

		case RUN:	
			if ( PORTS_KeyScan() == TRUE )	State = STOP;   // Stop motor
			else	DoMotorControl();
			break;

		case STOP:
		{
			if (BRAKE_VOLTAGE == 0)
			{
				ART_SetTimeOutDuration(IDLE_TIME);
				MTC_DisableMCOutputs();			// PWM is disabled on MCO outputs
				State = WAIT;
			}
			else
			{   
				MTC_UpdateSine(0, MTC_GetRotorFreq());
				State = BRAKE;
			}
		}
		break;

		case BRAKE: 
			if (MTC_GetRotorFreq()>BRAKE_MIN_SPEED)
			{       
				MTC_Brake();
			}
			else
			{
				MTC_StopBraking();
				ART_SetTimeOutDuration(IDLE_TIME);
				State = WAIT;
			}
			break;

		case WAIT:	
			if (ART_IsTimeOutElapsed())
			{        
				//MTC_Toggle_Direction(); /* Uncomment if you want to restart the motor in opposite direction */
				State = IDLE; 
			}
			break;

		case FAULT:	
		default:	
			MTC_DisableMCOutputs();			// PWM is disabled on MCO outputs
			State = FAULT;
			break;
		}

		// Verify that inverter is operating properly
		CheckPowerStage();
                
		// LED management
		if (ART_IsMainTimeElapsed())
		{
        	switch ( State )
			{
				case IDLE:	PORTS_GreenLedOn();
							break;
				case START:
				case RUN:	
				case STOP:
				case BRAKE:
				case WAIT:  PORTS_RedLedOn();
							break;
				case FAULT:	
				default:	PORTS_RedLedToggle();
							break;
			}
		}

		#ifdef ENABLE_RS232     
			SendSystemDataRS232();
		#endif 

	}	// End of while(1)
} // main loop


/*-----------------------------------------------------------------------------
ROUTINE Name : CheckPowerStage

Description : Verify that inverter power stage is running fine:
					heatsink temperature
					voltage bus value
					current limitation not triggerred
-----------------------------------------------------------------------------*/
void CheckPowerStage(void)
{

if (ADC_CheckOverVoltage() || ADC_CheckOverTemp() || MTC_CheckEmergencyStop())
	{
		State = FAULT;
	}

}

/*-----------------------------------------------------------------------------
ROUTINE Name : SendSystemDataRS232

Description : upload internal data to the PC via RS232 and hyperterminal for 
debug purposes
-----------------------------------------------------------------------------*/
#ifdef ENABLE_RS232
void SendSystemDataRS232(void)
{
	if(SCI_IsTransmitCompleted())
	{		
		strcpy(UartBuffer, "");
		#ifdef PHASE_ANGLE_RV3
			strcat(UartBuffer, "PhAng=");
			strcatu16(UartBuffer, Shift_Phase >> 8);
			strcat(UartBuffer, " ");			
		#endif

		#ifdef PI_PARAM_TUNING
			strcat(UartBuffer, "Kp=");
                        strcatu16(UartBuffer, (u8)(ADC_GetRV2()));
		        
		        strcat(UartBuffer, " Ki=");
                        strcatu16(UartBuffer, (u8)(ADC_GetRV3()));
                        strcat(UartBuffer, " ");			
		#endif
                
                #ifdef RS232_SPEED  
                	strcat(UartBuffer, "ElFrq=");
                        strcatu16(UartBuffer, MTC_GetRotorFreq());
		        strcat(UartBuffer, " ");
		#endif
		
		#ifdef RS232_VOLTAGE            
			strcat(UartBuffer, "StVlt=");
                        strcatu16(UartBuffer, MTC_GetVoltage());
		        strcat(UartBuffer, " ");       
		#endif
		
		strcat(UartBuffer, "\n\r"); // Go to next line  
		
		SCI_PutString(UartBuffer);
	}
}
#endif	/* ENABLE_RS232 */

/*-----------------------------------------------------------------------------
ROUTINE Name : DoMotorControl

Description : Performs control of the AC motor: the stator frequency 
and voltage and updated according to trimmer reading.
-----------------------------------------------------------------------------*/
void DoMotorControl( void )
{
	u8 NewVoltage, TargetVolt;
	u16 NewFreq, TargetFreq;  
	
	if ( ART_IsRegPeriodElapsed() )
	{
		#if (CONTROL == 0)
        		// Open loop regulation
			TargetVolt = (u8)ADC_GetRV1();		
			if (Target_Curr > TargetVolt)
				Target_Curr--;
			else
		   	if (Target_Curr < TargetVolt)
		   		Target_Curr++;
	        	NewVoltage = Target_Curr;                     
		#endif
                                
		NewFreq = MTC_GetRotorFreq();
	        
		#if (CONTROL == 1)		
        		// Closed loop regulation
			NewVoltage = ACM_SpeedRegulation(TARGET_FREQ_CL,NewFreq);      
		#endif
		
		
		
	        // Frequency Synchronization
		if (NewFreq == 0)	
        		NewFreq = LOWEST_FREQ; // Minimum rotor freq
                               
        	MTC_UpdateSine(NewVoltage, NewFreq);
        }
}

#if (CONTROL == 0)
/*-----------------------------------------------------------------------------
ROUTINE Name : StartMotor

Description : Starts the motor in open loop
-----------------------------------------------------------------------------*/
SystStatus_t StartMotor( void )
{
	u8 NewVoltage;
	// NewVoltage = STATOR_VOLTAGE; // End startup when STATOR_VOLTAGE is reached
	NewVoltage = (u8)(ADC_GetRV1()); // End startup when RV1 Value is reached
        
	if ( ACM_SoftStartOL(NewVoltage) )
	{
		return(RUN);
	}
	else
	{
		return(START);
	}

}
#endif

#if (CONTROL == 1)
/*-----------------------------------------------------------------------------
ROUTINE Name : StartMotor

Description : Starts the motor in closed loop
-----------------------------------------------------------------------------*/
SystStatus_t StartMotor( void )
{
	StartStatus_t startStatus;        
        startStatus = ACM_SoftStart();
        if (startStatus == START_ONGOING)
        	return (START);
	if (startStatus == START_FAIL)
        	return (FAULT);
        if (startStatus == START_OK)
        {
       		//u16 NewFrequency; 
        	//NewFrequency = LOWEST_FREQ+((u16)(ADC_GetRV1())*((HIGHEST_FREQ-LOWEST_FREQ)/255));
        	ACM_InitSpeedReg(TARGET_FREQ_CL);
        	return (RUN);
        }
        return (FAULT);        
}
#endif



/*** (c) 2004  STMicroelectronics **************************** END OF FILE ***/