;;; Copyright ARM Ltd 2001. All rights reserved.
;
; This module performs ROM/RAM remapping (if required), initializes stack 
; pointers and interrupts for each mode, and finally branches to __main in 
; the C library (which eventually calls main()).
;
; On reset, the ARM core starts up in Supervisor (SVC) mode, in ARM state, 
; with IRQ and FIQ disabled.


        AREA    Init, CODE, READONLY
        PRESERVE8        
; --- ensure no functions that use semihosting SWIs are linked in from the C library
                IMPORT __use_no_semihosting_swi
      
; --- Standard definitions of mode bits and interrupt (I & F) flags in PSRs

Mode_USR        EQU     0x10
Mode_FIQ        EQU     0x11
Mode_IRQ        EQU     0x12
Mode_SVC        EQU     0x13
Mode_ABT        EQU     0x17
Mode_UND        EQU     0x1B
Mode_SYS        EQU     0x1F ; available on ARM Arch 4 and later

I_Bit           EQU     0x80 ; when I bit is set, IRQ is disabled
F_Bit           EQU     0x40 ; when F bit is set, FIQ is disabled

; --- Amount of memory (in bytes) allocated for stacks

Len_FIQ_Stack    EQU     0
Len_IRQ_Stack    EQU     256
Len_ABT_Stack    EQU     0
Len_UND_Stack    EQU     0
Len_SVC_Stack    EQU     1024
Len_USR_Stack    EQU     0

; Add lengths >0 for FIQ_Stack, ABT_Stack, UND_Stack if you need them.
; Offsets will be loaded as immediate values.
; Offsets must be 8 byte aligned.

Offset_FIQ_Stack         EQU     0
Offset_IRQ_Stack         EQU     Offset_FIQ_Stack + Len_FIQ_Stack
Offset_ABT_Stack         EQU     Offset_IRQ_Stack + Len_IRQ_Stack
Offset_UND_Stack         EQU     Offset_ABT_Stack + Len_ABT_Stack
Offset_SVC_Stack         EQU     Offset_UND_Stack + Len_UND_Stack
Offset_USR_Stack         EQU     Offset_SVC_Stack + Len_SVC_Stack



        ENTRY

        EXPORT  Reset_Handler

Reset_Handler

; --- Initialize stack pointer registers

; Enter each mode in turn and set up the stack pointer

        IMPORT  top_of_stacks       ; defined in stack.s and located by scatter file
        LDR     r0, =top_of_stacks

        MSR     CPSR_c, #Mode_FIQ:OR:I_Bit:OR:F_Bit ; No interrupts
        SUB     sp, r0, #Offset_FIQ_Stack

;        MSR     CPSR_c, #Mode_ABT:OR:I_Bit:OR:F_Bit ; No interrupts
;        SUB     sp, r0, #Offset_ABT_Stack

;        MSR     CPSR_c, #Mode_UND:OR:I_Bit:OR:F_Bit ; No interrupts
;        SUB     sp, r0, #Offset_UND_Stack

        MSR     CPSR_c, #Mode_IRQ:OR:I_Bit:OR:F_Bit ; No interrupts
        SUB     sp, r0, #Offset_IRQ_Stack

        MSR     CPSR_c, #Mode_SVC:OR:I_Bit:OR:F_Bit ; No interrupts
        SUB     sp, r0, #Offset_SVC_Stack

; --- Initialize critical IO devices
        IMPORT OS_InitPLL
; Init PLL
        LDR     r0,=OS_InitPLL
        MOV     lr, pc
        BX      r0

;******************************************************************************
;EIC initialization
;Description  : Initialize the EIC as following :
;              - IRQ disabled
;              - FIQ disabled
;              - IVR contain the load PC opcode (0xF59FF00)
;              - Current priority level equal to 0
;              - All channels are disabled
;              - All channels priority equal to 0
;              - All SIR registers contain offset to the related IRQ table entry
;******************************************************************************


EIC_Base_addr       EQU    0xFFFFF800         ; EIC base address
ICR_off_addr        EQU    0x00               ; Interrupt Control register offset
CIPR_off_addr       EQU    0x08               ; Current Interrupt Priority Register offset
IVR_off_addr        EQU    0x18               ; Interrupt Vector Register offset
FIR_off_addr        EQU    0x1C               ; Fast Interrupt Register offset
IER_off_addr        EQU    0x20               ; Interrupt Enable Register offset
IPR_off_addr        EQU    0x40               ; Interrupt Pending Bit Register offset
SIR0_off_addr       EQU    0x60               ; Source Interrupt Register 0

EIC_INIT
        LDR     r3, =EIC_Base_addr
        LDR     r4, =0x00000000
        STR     r4, [r3, #ICR_off_addr]   ; Disable FIQ and IRQ
        STR     r4, [r3, #IER_off_addr]   ; Disable all channels interrupts
        LDR     r4, =0xFFFFFFFF
        STR     r4, [r3, #IPR_off_addr]   ; Clear all IRQ pending bits
        LDR     r4, =0x0C
        STR     r4, [r3, #FIR_off_addr]   ; Disable FIQ channels and clear FIQ pending bits
        LDR     r4, =0x00000000
        STR     r4, [r3, #CIPR_off_addr]  ; Reset the current priority register
        LDR     r4, =0x0000
        STR     r4, [r3, #IVR_off_addr]   ; Write 0 in IVR[31:16]
        LDR     r2, =32                   ; 32 Channel to initialize
        LDR     r0, =0                    ; Read the address of the IRQs address table
        LDR     r5, =SIR0_off_addr        ; Read SIR0 address
EIC_INI                                  ; 
        STR     r0, [r3, r5]              ; Store 0 in SIRx register
        ADD     r1, r1, #4                ; Next IRQ address
        ADD     r5, r5, #4                ; Next SIR
        SUBS    r2, r2, #1                ; Decrement the number of SIR registers to initialize
        BNE     EIC_INI                   ; If more then continue



        ; ...

        IMPORT  __main

; --- Now enter the C code
        B       __main   ; note use B not BL, because an application will never return this way


;*********************************************************************
;*
;*      Stack info for embOS
;*
;*********************************************************************


        EXPORT OS_GetStackInfo
OS_GetStackInfo
        IMPORT SVC_Stack_Base
        IMPORT SVC_Stack_Limit
        IMPORT IRQ_Stack_Base
        IMPORT IRQ_Stack_Limit

        IMPORT  top_of_stacks       ; defined in stack.s and located by scatter file
        LDR     r0, =top_of_stacks
; Get the IRQ Stack information
        LDR     r1, =IRQ_Stack_Limit
        sub     r2, r0, #Offset_IRQ_Stack
        STR     r2, [r1]
        LDR     r1, =IRQ_Stack_Base
        SUB     r2, r2, #Len_IRQ_Stack
        STR     r2, [r1]
; Get the Supervisor Stack information
        LDR     r1, =SVC_Stack_Limit
        sub     r2, r0, #Offset_SVC_Stack
        STR     r2, [r1]
        LDR     r1, =SVC_Stack_Base
        SUB     r2, r2, #Len_SVC_Stack
        STR     r2, [r1]
        BX      lr

        END


