/////////////////////////////////////////////////////////////////////////////
// DK3200_IAP.cpp
//
// IAP implementation for RS232 example application.
//
// Author: Marian Ilecko
//
// Revision History:
//
// 11/03/03 (MI) V1.1.1 - Derived from USB IAP Demo made by Jon Moore
// 11/04/03 (MI) V1.1.1 - IAP RS232 routines imported from NewPSDload project by M.I.
// 11/05/03 (MI) V1.1.1 - Code rearranged, adjusted and cleaned
// 11/06/03 (MI) V1.1.1 - Some bugfixes and improvements
// 11/07/03 (MI) V1.1.1 - Rearrangements, commenting
//
// Copyright (c)2003 ST Microelecronics
// All rights reserved.
//
//---------------------------------------------------------------------------
// This example demo code is provided as is and has no warranty,
// implied or otherwise.  You are free to use/modify any of the provided
// code at your own risk in your applications with the expressed limitation
// of liability (see below) so long as your product using the code contains
// at least one uPSD products (device).
//
// LIMITATION OF LIABILITY:   NEITHER STMicroelectronics NOR ITS VENDORS OR
// AGENTS SHALL BE LIABLE FOR ANY LOSS OF PROFITS, LOSS OF USE, LOSS OF DATA,
// INTERRUPTION OF BUSINESS, NOR FOR INDIRECT, SPECIAL, INCIDENTAL OR
// CONSEQUENTIAL DAMAGES OF ANY KIND WHETHER UNDER THIS AGREEMENT OR
// OTHERWISE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
//---------------------------------------------------------------------------

#include <vcl.h>
#include <DateUtils.hpp>

#pragma hdrstop

#include "DK3200_IAP.h"
#include "RS232_demo.h"

#pragma package(smart_init)

AnsiString com_port;    // name of COM port
int com_baudrate;       // baud rate parameter
int comm_port_open = 0; // flag to indicate whether the port is open or not
RS232 *com;             // object for handling low-level port operations

int esc_pressed; // used for cancelling of some time-consuming operations

#define SHORT_BUFFER_SIZE 100  // size of short command buffer
#define LONG_BUFFER_SIZE 1000  // size of long command buffer
#define BIN_BUFFER_SIZE 100000 // size of binary data buffer

#define READ_FLASH_SLICE_SIZE 241  // size of block used in READ FLASH command
#define WRITE_FLASH_SLICE_SIZE 100 // size of block used in WRITE FLASH command

//---------------------------------------------------------------------------
//              MISCELLANEOUS IAP-RELATIVE FUNCTIONS
//---------------------------------------------------------------------------

/////////////////// OnError()
//
// Generic error handler.
//
void OnError(char *pszMsg)
{
   if(!test_runs)
      Application->MessageBox(pszMsg, "Error", MB_OK|MB_ICONEXCLAMATION);
}
//---------------------------------------------------------------------------

/////////////////// sprintf_2hex()
//
// Renders byte value into 2-characters null-terminated string.
// This function should not be replaced by sprintf, because under some
// operational systems the sprintf renders certain values in unicode.
//
int sprintf_2hex(char *dest, BYTE c)
{
   if((c/16)<10)dest[0]='0'+(c/16);
   else dest[0]='A'+(c/16)-10;
   if((c%16)<10)dest[1]='0'+(c%16);
   else dest[1]='A'+(c%16)-10;
   dest[2]=' ';
   dest[3]=0;
   return 1;
}
//---------------------------------------------------------------------------

/////////////////// HexToDec()
//
// Decodes single hexadecimal char into decimal value.
//
int HexToDec(char hex)
{
   if((hex>='0')&&(hex<='9'))return hex-'0';
   if((hex>='a')&&(hex<='f'))return 10+hex-'a';
   if((hex>='A')&&(hex<='F'))return 10+hex-'A';
   else return -1;
}
//---------------------------------------------------------------------------

/////////////////// HexToDec()
//
// Decodes pair of hexadecimal characters into decimal value.
// This function should not be replaced by sscanf, because under some
// operational systems the sscanf recognizes some sequences
// as the unicode characters.
//
int HexToDec(char *hexa)
{
   int v1,v2;
   while(hexa[0]==' ')hexa++;
   if(hexa[0]=='\0')return -1;
   v1=HexToDec(hexa[0]);
   v2=HexToDec(hexa[1]);
   if((v2==-1)&&(hexa[1]!=0)&&(hexa[1]!=' '))return -1;
   if(v1==-1)return -1;
   if((v2!=-1)&&(HexToDec(hexa[2])!=-1))
      return -2; //Invalid data value: value is more than two characters long.
   if(v2==-1)return v1;
   return 16*v1+v2;
}
//---------------------------------------------------------------------------

/////////////////// RenderBinaryDataToHexaString()
//
// Renders the binary data block into string of hexadecimal values.
// The digit pairs are separated by spaces.
//
void RenderBinaryDataToHexaString(char *bin_buffer, int bin_buffer_length, char *hexastring)
{
   for(int i=0; i<bin_buffer_length; i++)
   {
      sprintf_2hex(hexastring+3*i, bin_buffer[i]);
      hexastring[3*i+2]=' ';
   }
   hexastring[3*bin_buffer_length-1]=0;
}
//---------------------------------------------------------------------------

/////////////////// ParseHexaDataToBin()
//
// Decodes string containing hexadecimal characters into binary values.
// The number of correctly parsed bytes is stored in variable.
//
int ParseHexaDataToBin(char *hexastring, char *bin_buffer, int *bytes_parsed)
{
   int nBytes = 0;
   int value;
   for(int i=0; (hexastring!=NULL)&&(hexastring[i]!=0); i++)
   {
      if(!(i%2)){
         if(-1==(bin_buffer[nBytes]=HexToDec(hexastring[i])))
            break;
      }
      else{
         if(-1==(value=HexToDec(hexastring[i])))
            break;
         bin_buffer[nBytes]*=16;
         bin_buffer[nBytes++]+=value;
      }
   }
   bytes_parsed[0]=nBytes;
   return 1;
}
//---------------------------------------------------------------------------

/////////////////// ParseHexaStringToBinAndAscii()
//
// Decodes string containing hexadecimal characters into binary values
// and its ASCII representation (only printable characters, of course).
// The number of correctly parsed bytes is stored in variable.
//
int ParseHexaStringToBinAndAscii(char *hexastring, char *bin_buffer, char *ascii_buffer, int *bytes_parsed)
{
   int nBytes = 0;
   char *p;
   p=hexastring;
   int value;
   for(p; p;)
   {
      while(p[0]==' ')p++;
      value=HexToDec(p);
      if(value==-1){// Invalid data value: not a valid numeric value.
         return -1;
      }
      else if(value==-2){// Invalid data value: not a valid numeric value.
         return -2;
      }
      bin_buffer[nBytes++] = (BYTE)(value);
      while((p[0]!=0)&&(p[0]!=' '))p++;
      while(p[0]==' ')p++;
      if(p[0] == 0){
         break;
      }
   }
   bytes_parsed[0]=nBytes;
   if(ascii_buffer==NULL)
      return 1;
   else {
      memcpy(ascii_buffer,bin_buffer,nBytes);
      for(int i=0;i<nBytes;i++)
         if(!isprint(ascii_buffer[i]))ascii_buffer[i]='.';
      ascii_buffer[nBytes]=0;
   }
   return 1;
}
//---------------------------------------------------------------------------

/////////////////// LoadFileToBuffer()
//
// Loads the data from given file into binary buffer.
// The file can contain data in binary or hexadecimal format.
//
int LoadFileToBuffer(char *filename, char *buffer, int buffer_size, int *bytes_loaded)
{
   FILE *f;
   int single_warning_message=1;
   if(0==(f=fopen(filename,"rb"))){
      Application->MessageBox("Error opening file !","Data load error",MB_OK);
      return 0;
   }
   else if(ExtractFileExt(AnsiString(filename)).LowerCase()==".bin"){
      bytes_loaded[0] = fread(buffer,1,buffer_size,f);
      fclose(f);
   }
   else if(ExtractFileExt(AnsiString(filename)).LowerCase()==".hex"){
      // Read the hex file into a large memory buffer
      memset(buffer, 0xFF, buffer_size);
      BOOL bRet = FALSE;
      int leave_cycle = 0;
      char lineBuf[1000];
      while((!leave_cycle) && (fgets(lineBuf, sizeof(lineBuf) - 1, f))){
         BYTE   cbData;
         USHORT addr;
         BYTE   type;
         int temp, temp2, temp3; // temporary storage, function scanf requires data type of int
         addr = -1;
         BYTE   sum;
         if(sscanf(lineBuf, ":%02x%04x%02x", &temp, &temp2, &temp3) != 3)
         {
            bRet = false;
            leave_cycle = 1;
         }
         cbData = temp;
         addr = temp2;
         type = temp3;
         sum = cbData + ((addr & 0xFF00) >> 8) + (addr & 0xFF) + type;
         if(type == 1) // End of file
         {
            bRet = true;
            leave_cycle = 1;
         }
         if(type == 0) // Data record
         {
            char* p = lineBuf + 9;
            for(; cbData; p+=2, cbData--, addr++)
            {
               sscanf(p, "%02x", &temp);
               *(buffer + addr) = temp;
               sum += temp;
            }
            // Read and validate checksum
            if(!sscanf(p, "%02x", &temp))
            {
               bRet = false;
               leave_cycle = 1;
            }
            if(((0x100 - sum) & 0xFF) != temp)
            {
               bRet = false;
               leave_cycle = 1;
            }
            if((addr + cbData) > buffer_size)
            {
               Application->MessageBox("Warning: hex file address exceeds selected sector size.\nData past the last sector address will be ignored.",
                  "Warning", MB_ICONEXCLAMATION|MB_OK);
               bRet = true;
               single_warning_message = 0;
               leave_cycle = 1;
            }
         }
      }
      fclose(f);
      bytes_loaded[0] = 0;
      if(!bRet){
         Application->MessageBox("Invalid syntax in hex file","Data load error",MB_OK);
         single_warning_message = 0;
         return 0;
      }
      else
      {
         // memset(bin_buffer, 0xFF, sizeof(bin_buffer));
         for(int i=buffer_size-1;i>0;i--)
         if(buffer[i]!=-1)
         {
            bytes_loaded[0]=i+1;
            break;
         }
      }
   }
   if(single_warning_message)
   if(bytes_loaded[0]==0)
      OnError("File contains no data");
   return 1;
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
//              COM PORT MANAGING ROUTINES
//---------------------------------------------------------------------------

/////////////////// initCommPort()
//
// Initializes and opens the COM port.
//
int initCommPort()
{
   if(comm_port_open)
      return -1;
   com_port = "COM1";
   com_baudrate = 19200;
   com = new RS232;
   if(com->Open(com_port.c_str(),1600,1600,com_baudrate))
   {
      comm_port_open = 1;
   } else
   {
      OnError(AnsiString("Can't open "+com_port+", port does not exist or is already allocated.").c_str());
      comm_port_open = 0;
   }
   return(comm_port_open);
}
//---------------------------------------------------------------------------

/////////////////// deinitCommPort()
//
// Closes the COM port.
//
int deinitCommPort()
{
   if(comm_port_open)
   {
      comm_port_open = 0;
      com->Close();
   }
   return 1;
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
//              PACKET PREPARAION AND SENDING ROUTINES
//---------------------------------------------------------------------------

#define PACKET_HEADER 0x21

/////////////////// encapsulate_packet()
//
// Prepare the packet for sending via RS232 line.
// This function sets the header and computes the checksum.
//
// The function returns the size of encapsulated packet (number of bytes).
//
int encapsulate_packet(char *buffer, int data_length)
{
   buffer[0] = PACKET_HEADER;
   buffer[1] = (data_length + 6) / 256;
   buffer[2] = (data_length + 6) % 256;
   buffer[3] = -buffer[1] - buffer[2];
   unsigned char csum1=0,csum2=0;
   for(int i=0; i<data_length; i++){
      csum1 += buffer[i+4];
      csum2 ^= buffer[i+4];
   }
   buffer[data_length+4] = csum1;
   buffer[data_length+4+1] = csum2;
   return(data_length + 6);
}
//---------------------------------------------------------------------------

/////////////////// encapsulate_packet()
//
// Send the packet via RS232 line and receive the answer from the board.
// Function returns 1 on success, 0 if failed.
// The retry attempts can be adjusted by parameter, default is 10.
// The length of answer received is stored in variable.
//
int SendCommand_DK3200(char *command, int command_length, char *result_buffer, int result_buffer_size, int *bytes_received, int TX_RETRIES)
{
   char tx_buf[LONG_BUFFER_SIZE],rx_buf[LONG_BUFFER_SIZE];
   int packet_length;
   int result_success;
   int retry = 0;
   char tmp_print[100];
   int read_count,buff_ptr,k;

   if(!command_length)command_length=strlen(command);
   memcpy(tx_buf+4,command,command_length);
   packet_length = encapsulate_packet(tx_buf,command_length);

   TDateTime t1,t2;
   t1 = Now();

   do {
      com->Write(tx_buf,packet_length);
      buff_ptr = 0;
      do {
         Sleep(30);
         buff_ptr+=(read_count = com->Read(rx_buf+buff_ptr, LONG_BUFFER_SIZE-buff_ptr));
      }
      while(read_count);

      read_count=buff_ptr;
      if(result_buffer_size&&(read_count>result_buffer_size))read_count=result_buffer_size;
      if(read_count>0)
      {
         rx_buf[read_count] = '\0';  // for null terminated string operations
         if(
         (!strncmp(rx_buf,"WFT ",4))|| // Write Flash toggle
         (!strncmp(rx_buf,"WFP ",4))|| // Write Flash poll
         (!strncmp(rx_buf,"WRC ",4))|| // Write RAM code
         (!strncmp(rx_buf,"WRD ",4))|| // Write RAM data
         (!strncmp(rx_buf,"RRC ",4))|| // Read RAM code
         (!strncmp(rx_buf,"RRD ",4))|| // Read RAM data
//       (!strncmp(rx_buf,"RFC ",4))|| // Read Flash code (unused, identical to RRC)
//       (!strncmp(rx_buf,"RFD ",4))|| // Read Flash data (unused, identical to RRD)
         (!strncmp(rx_buf,"RST ",4))|| // Board Reset
         (!strncmp(rx_buf,"BRD ",4))|| // Board name request
         (!strncmp(rx_buf,"MIR ",4))|| // LCD content request
         (!strncmp(rx_buf,"PG ",3))||  // PAGE Register set
         (!strncmp(rx_buf,"VM ",3))||  // VM Register set
         (!strncmp(rx_buf,"LCD ",4))|| // Write text to LCD
         (!strncmp(rx_buf,"EM ",3))||  // Erase Main Flash sector
         (!strncmp(rx_buf,"EB ",3))    // Erase Boot Flash sector
         ){
            if(bytes_received!=NULL)bytes_received[0] = read_count;
            result_success=1;
         }
         else {
            result_success=0;
            retry++;
         }
      }
   else {
      result_success=0;
      retry++;
      }
   }
   while((retry<=TX_RETRIES)&&(result_success!=1));

   t2 = Now();
//   int time_consumed = MilliSecondsBetween(t1, t2);
   if(result_buffer!=NULL)
   {
      memcpy(result_buffer,rx_buf,result_buffer_size);
   }
   if(retry>TX_RETRIES)
   {
      if(result_buffer!=NULL)
      {
         if(bytes_received!=NULL) bytes_received[0] = 0;
         strcpy(result_buffer,"RETRY Timeout");
      }
      return 0;
   }
   return 1;
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
//              MAIN IAP DEMO COMMANDS IMPLEMENTATION
//---------------------------------------------------------------------------

/////////////////// ResetBoard()
//
// Sends a command to reset the chip.
//
// Function returns 1 on success, 0 on error.
//
int ResetBoard()
{
   // Send command
   if(!SendCommand_DK3200("RST", NULL, 0, NULL, NULL))
   {
      OnError("Error sending RESET command.");
      return 0;
   }
   return 1;
}
//---------------------------------------------------------------------------

/////////////////// SetPage()
//
// Sets the page register.
//
// The function takes one argument - value which will be set
// to PAGE register of uPSD.
//
// Function returns 1 if successful, 0 on error.
//
int SetPage(BYTE page)
{
   char command[SHORT_BUFFER_SIZE],reply[SHORT_BUFFER_SIZE];
   strcpy(command,"PG ");
   sprintf_2hex(command+3,page);
   command[5]=0;
   if(!SendCommand_DK3200(command, 0, reply, SHORT_BUFFER_SIZE, NULL))
   {
      OnError("Error sending SET PAGE command.");
      return 0;
    }
    return 1;
}
//---------------------------------------------------------------------------

/////////////////// SetVM()
//
// Sets the VM register.
//
// The function takes one argument - value which will be set
// to VM register of uPSD.
//
// Function returns 1 if successful, 0 on error.
//
int SetVM(BYTE vm)
{
   char command[SHORT_BUFFER_SIZE],reply[SHORT_BUFFER_SIZE];
   strcpy(command,"VM ");
   sprintf_2hex(command+3,vm);
   command[5]=0;
   if(!SendCommand_DK3200(command, 0, reply, SHORT_BUFFER_SIZE, NULL))
   {
      OnError("Error sending SET VM command.");
      return 0;
   }
   return 1;
}
//---------------------------------------------------------------------------

/////////////////// SelectFlash()
//
// Selects primary or secondary flash for manipulation in data space.
//
// list and meaning of parameters:
// BYTE flash    - flash selector, 0 = Main Flash (primary), 1 = Boot Flash (secondary)
// BYTE sector   - sector selector, 0..7 for Main Flash, 0..3 for Boot flash
//
// Function returns 1 if successful, 0 on error.
//
int SelectFlash(BYTE flash, BYTE sector)
{
   if(flash == PRIMARY_FLASH)
   {
      // Put primary in data space; secondary/boot in code space
      if(!SetVM(PRIMARY_FLASH_VM))return 0;
      if(!SetPage(sector))return 0;
   }
   else if(flash == SECONDARY_FLASH)
   {
      if(sector >= 4)
      {
         OnError("Secondary/boot flash sector can not be > 3.");
         return 0;
      }
      // Put secondary/boot flash in data space
      if(!SetVM(SECONDARY_FLASH_VM))return 0;
      if(!SetPage(0))return 0;
   }
   return 1;
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
//              FLASH READ & WRITE ROUTINES IMPLEMENTATION
//---------------------------------------------------------------------------

/////////////////// ReadFlash()
//
// Reads and returns a number of bytes from the
// actual flash sector starting at a specified offset.
//
// list and meaning of parameters:
// int offset   - offset address where to write the data, valid values are
//                0x0000-0x7fff for Main Flash, 0x0000-0x1fff for Boot Flash
// BYTE *buffer - destination buffer where the read data will be stored
// int nBytes   - amount of data to read
//
// Function returns 1 on success, 0 on error.
//
int ReadFlash(int offset, BYTE *buffer, int nBytes)
{
   char command[SHORT_BUFFER_SIZE],reply[LONG_BUFFER_SIZE];
   int bytes_read;
   strcpy(command, "RRD");
   command[3]=offset/256;
   command[4]=offset%256;
   command[5]=nBytes;
   if(!SendCommand_DK3200(command, 6, reply, LONG_BUFFER_SIZE, &bytes_read))
   {
      OnError("Error sending READ FLASH command.");
      return 0;
   }
   memcpy(buffer,reply,bytes_read);
   return 1;
}
//---------------------------------------------------------------------------

/////////////////// WriteFlash()
//
// Writes a number of bytes to the actual flash sector,
// starting at a specified offset.
//
// list and meaning of parameters:
// int offset   - offset address where to write the data, valid values are
//                0x0000-0x7fff for Main Flash, 0x0000-0x1fff for Boot Flash
// BYTE *buffer - source buffer containing data
// int nBytes   - amount of data to write
//
// Function returns 1 on success, 0 on error.
//
int WriteFlash(int offset, BYTE *buffer, int nBytes)
{
   char command[LONG_BUFFER_SIZE],reply[SHORT_BUFFER_SIZE];
   strcpy(command, "WFP");
   command[3]=offset/256;
   command[4]=offset%256;
   command[5]=nBytes;
   memcpy(command+6, buffer, nBytes);
   if(!SendCommand_DK3200(command, 6+nBytes, reply, SHORT_BUFFER_SIZE, NULL))
   {
      OnError("Error sending WRITE FLASH command.");
      return 0;
   }
   int bytes_written = 0;
   sscanf(reply+15,"%02x",&bytes_written);
   if(strncmp(reply+18,"B WRITTEN",9)||(bytes_written!=nBytes)){
      OnError("Error writing data, target memory space probably not blank.");
      return 0;
   }
   else return 1;
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
//              FLASH ERASE & BLANK CHECK ROUTINES IMPLEMENTATION
//---------------------------------------------------------------------------

/////////////////// SectorErase()
//
// Sends a command to erase the selected flash sector.
//
// list and meaning of parameters:
// int flash   - flash selector, 0 = Main Flash (primary), 1 = Boot Flash (secondary)
// int sector  - sector selector, 0..7 for Main Flash, 0..3 for Boot flash
// int offset  - offset parameter, specifies the memory block to erase
//
// The offset parameter is needed because in some memory map configurations
// the more physical flash sectors can be mapped in one logical block sequentially.
//
// Function returns 1 on success, 0 on error.
//
int SectorErase(int flash, int sector, int offset)
{
   char command[SHORT_BUFFER_SIZE],reply[SHORT_BUFFER_SIZE];
   int bytes_received;
   if(!SelectFlash(flash, sector))return 0;
   if(flash==PRIMARY_FLASH)strcpy(command,"EM");
   else strcpy(command,"EB");
   command[2]=(unsigned char)(offset/256);
   command[3]=(unsigned char)(offset%256);
   if(!SendCommand_DK3200(command, 4, reply, SHORT_BUFFER_SIZE, &bytes_received))
   {
      OnError("Error sending SECTOR ERASE command.");
      return 0;
   }
   return 1;
}
//---------------------------------------------------------------------------

/////////////////// BlankCheck()
//
// Checks to see if a flash sector is blank.
//
// list and meaning of parameters:
// int flash   - flash selector, 0 = Main Flash (primary), 1 = Boot Flash (secondary)
// int sector  - sector selector, 0..7 for Main Flash, 0..3 for Boot flash
// int offset  - offset parameter, specifies the memory block to check
// int sector_size - the size of the source sector
//
// Function returns 1 on success, 0 on error.
//
// The result of the test is displayed using MessageBoxes.
// "Sector is blank" means that every byte in the sector has value of 0xFF
// "Sector is not blank" means that at least one of the bytes in the sector
// has value different from 0xFF
//
int BlankCheck(int flash, int sector, int offset, int sector_size)
{
   char bin_buffer[BIN_BUFFER_SIZE],command[SHORT_BUFFER_SIZE],reply[LONG_BUFFER_SIZE];
   int bytes_received;
   int slice_size;
   int bin_buffer_cnt = 0;
   int data_match = 1;
   esc_pressed = 0;
   if(!SelectFlash(flash, sector))return 0;
   while(data_match && !esc_pressed && (bin_buffer_cnt < sector_size)){
      strcpy(command, "RRD");
      command[3]=(offset+bin_buffer_cnt)/256;
      command[4]=(offset+bin_buffer_cnt)%256;
      slice_size = ((sector_size-bin_buffer_cnt)>READ_FLASH_SLICE_SIZE)?READ_FLASH_SLICE_SIZE:(sector_size-bin_buffer_cnt);
      command[5]= slice_size;
      if(!SendCommand_DK3200(command, 6, reply, LONG_BUFFER_SIZE, &bytes_received))
      {
         OnError("Error sending READ FLASH command.");
         return 0;
      }
      int mismatch_position;
      for(mismatch_position=0;mismatch_position<slice_size;mismatch_position++){
         if(-1!=(reply+15)[mismatch_position]){
            char str_m[100];
            sprintf(str_m,"Sector is not blank.\nSome data found at address 0x%04X",offset+bin_buffer_cnt+mismatch_position);
            Application->MessageBox(str_m,"Blank check result",MB_OK);
            data_match = 0;
            break;
         }
      }
      bin_buffer_cnt += slice_size;
      // update text in status bar
      mainForm->sbStatus->Panels->Items[0]->Text = "Reading flash, "+AnsiString(sector_size - bin_buffer_cnt)+" bytes remaining.";
      mainForm->Repaint();
      Application->ProcessMessages();
   }
   if(!esc_pressed && data_match){
      Application->MessageBox("The sector is blank","Blank check result",MB_OK);
   }
   if(esc_pressed)return -1; // cancelled by user
   return 1;
}
//---------------------------------------------------------------------------


//---------------------------------------------------------------------------
//              FLASH PROGRAM, VERIFY and UPLOAD ROUTINES IMPLEMENTATION
//---------------------------------------------------------------------------

/////////////////// ProgramSector()
//
// Programs the selected flash from a hex file.
//
// list and meaning of parameters:
// int flash   - flash selector, 0 = Main Flash (primary), 1 = Boot Flash (secondary)
// int sector  - sector selector, 0..7 for Main Flash, 0..3 for Boot flash
// int offset  - offset parameter, specifies the memory block to program
// int sector_size - the size of the target sector
// char *filename - name of the file to load data from
//
// The offset parameter is needed because in some memory map configurations
// the more physical flash sectors can be mapped in one logical block sequentially.
//
// Function returns 1 on success, 0 on error.
//
int ProgramSector(int flash, int sector, int offset, int sector_size, char *filename)
{
   char bin_buffer[BIN_BUFFER_SIZE],command[SHORT_BUFFER_SIZE],reply[LONG_BUFFER_SIZE];
   int bytes_to_program;
   if(!SelectFlash(flash, sector))return 0;
   if(LoadFileToBuffer(filename, bin_buffer, sector_size, &bytes_to_program))
   // If the hex file was valid, write the data to flash
   if(bytes_to_program)
   {
      int bin_buffer_cnt = 0;
      int slice_size,reply_size;
      bin_buffer_cnt = 0;
      esc_pressed = 0;
      while(!esc_pressed && (bin_buffer_cnt < bytes_to_program)){
         strcpy(command, "WFP");
         command[3]=(offset+bin_buffer_cnt)/256;
         command[4]=(offset+bin_buffer_cnt)%256;
         slice_size = ((bytes_to_program-bin_buffer_cnt)>WRITE_FLASH_SLICE_SIZE)?WRITE_FLASH_SLICE_SIZE:(bytes_to_program-bin_buffer_cnt);
         command[5]= slice_size;
         memcpy(command+6,bin_buffer+bin_buffer_cnt,slice_size);
         if(!SendCommand_DK3200(command, 6+slice_size, reply, LONG_BUFFER_SIZE, &reply_size))
         {
            OnError("Error sending WRITE FLASH command.");
            return 0;
         }
         bin_buffer_cnt += slice_size;
         int bytes_written = 0;
         sscanf(reply+15,"%02x",&bytes_written);
         if(strncmp(reply+18,"B WRITTEN",9)||(bytes_written!=slice_size))
         {
            Application->MessageBox("Error writing data. Probably the target sector was not blank.","Program Sector error",MB_OK);
            return 0;
         }
         // update text in status bar
         mainForm->sbStatus->Panels->Items[0]->Text = "Programming flash, "+AnsiString(bytes_to_program - bin_buffer_cnt)+" bytes remaining.";
         mainForm->Repaint();
         Application->ProcessMessages();
      }
      if(esc_pressed)return -1; // cancelled by user
   }
   else return 0;
   return 1;
}
//---------------------------------------------------------------------------

/////////////////// VerifySector()
//
// Verifies the selected flash sector content with a hex file.
//
// list and meaning of parameters:
// int flash   - flash selector, 0 = Main Flash (primary), 1 = Boot Flash (secondary)
// int sector  - sector selector, 0..7 for Main Flash, 0..3 for Boot flash
// int offset  - offset parameter, specifies the memory block to verify data with
// int sector_size - the size of the source sector
// char *filename - name of the file to load data from
//
// Function returns 1 on success, 0 on error.
//
// The result of the verification is displayed using MessageBoxes.
//
int VerifySector(int flash, int sector, int offset, int sector_size, char *filename)
{
   char bin_buffer[BIN_BUFFER_SIZE],command[SHORT_BUFFER_SIZE],reply[LONG_BUFFER_SIZE];
   int bytes_to_verify;
   if(!SelectFlash(flash, sector))return 0;
   if(LoadFileToBuffer(filename, bin_buffer, sector_size, &bytes_to_verify))
   // If the hex file was valid, verify the data with flash
   if(bytes_to_verify)
   {
      int bytes_received;
      int slice_size;
      int bin_buffer_cnt = 0;
      int data_match = 1;
      esc_pressed = 0;
      while(data_match && !esc_pressed && (bin_buffer_cnt < bytes_to_verify)){
         strcpy(command, "RRD");
         command[3]=(offset+bin_buffer_cnt)/256;
         command[4]=(offset+bin_buffer_cnt)%256;
         slice_size = ((bytes_to_verify-bin_buffer_cnt)>READ_FLASH_SLICE_SIZE)?READ_FLASH_SLICE_SIZE:(bytes_to_verify-bin_buffer_cnt);
         command[5]= slice_size;

         if(!SendCommand_DK3200(command, 6, reply, LONG_BUFFER_SIZE, &bytes_received))
         {
            OnError("Error sending READ FLASH command.");
            return 0;
         }
         if(memcmp(bin_buffer+bin_buffer_cnt, reply+15, slice_size)){
            int mismatch_position;
            for(mismatch_position=0;mismatch_position<slice_size;mismatch_position++){
               if((bin_buffer+bin_buffer_cnt)[mismatch_position]!=(reply+15)[mismatch_position])
                  break;
            }
            char str_m[100];
            sprintf(str_m,"Data do not match at address 0x%04X",offset+bin_buffer_cnt+mismatch_position);
            Application->MessageBox(str_m,"Verify Error",MB_OK);
            data_match = 0;
         }
         bin_buffer_cnt += slice_size;
         // update text in status bar
         mainForm->sbStatus->Panels->Items[0]->Text = "Reading flash, "+AnsiString(bytes_to_verify - bin_buffer_cnt)+" bytes remaining.";
         mainForm->Repaint();
         Application->ProcessMessages();
      }
      if(!esc_pressed && data_match){
         Application->MessageBox("Data Verify OK","Verify result",MB_OK);
      }
   }
   if(esc_pressed)return -1; // cancelled by user
   else return 1;
}
//---------------------------------------------------------------------------

/////////////////// DownloadSector()
//
// Stores flash contents to a hex file.
//
// list and meaning of parameters:
// BYTE flash  - flash selector, 0 = Main Flash (primary), 1 = Boot Flash (secondary)
// BYTE sector - sector selector, 0..7 for Main Flash, 0..3 for Boot flash
// int offset  - offset parameter, specifies the memory block to read data from
// int sector_size - the size of the source sector
// AnsiString seg_name - the name of the sector (used for naming of destination file)
//
// Function returns 1 on success, 0 on error.
//
int DownloadSector(int flash, int sector, int offset, int sector_size, AnsiString seg_name)
{
   char command[SHORT_BUFFER_SIZE],reply[LONG_BUFFER_SIZE],bin_buffer[100000];
   int bin_buffer_cnt;
   int bytes_received;
   int slice_size;
   bin_buffer_cnt = 0;
   esc_pressed = 0;
   if(!SelectFlash(flash, sector))return 0;
   while(!esc_pressed && (bin_buffer_cnt < sector_size)){
      strcpy(command, "RRD");
      command[3]=(offset+bin_buffer_cnt)/256;
      command[4]=(offset+bin_buffer_cnt)%256;
      slice_size = ((sector_size-bin_buffer_cnt)>READ_FLASH_SLICE_SIZE)?READ_FLASH_SLICE_SIZE:(sector_size-bin_buffer_cnt);
      command[5]= slice_size;
      if(!SendCommand_DK3200(command, 6, reply, LONG_BUFFER_SIZE, &bytes_received))
      {
         OnError("Error sending READ FLASH command.");
         return 0;
      }
      memcpy(bin_buffer+bin_buffer_cnt, reply+15, slice_size);
      bin_buffer_cnt += slice_size;
      // update text in status bar
      mainForm->sbStatus->Panels->Items[0]->Text = "Reading flash, "+AnsiString(sector_size - bin_buffer_cnt)+" bytes remaining.";
      mainForm->Repaint();
      Application->ProcessMessages();
   }
   mainForm->SaveDialog1->FileName="DK3200_seg("+seg_name+")";
   if(!esc_pressed)
   if(mainForm->SaveDialog1->Execute()){
      FILE *f;
      if(0==(f=fopen(mainForm->SaveDialog1->FileName.c_str(),"wb"))){
         Application->MessageBox("Error creating file !","Uploaded data save error",MB_OK);
         return 0;
      }
      if(ExtractFileExt(mainForm->SaveDialog1->FileName).LowerCase()==".bin"){
         fwrite(bin_buffer,1,bin_buffer_cnt,f);
      }
      else if(ExtractFileExt(mainForm->SaveDialog1->FileName).LowerCase()==".txt"){
         fputs("\n Memory dump saved by RS232 IAP Demo (c)2003 ST Microelectronics\n\n",f);
         char line[256],ascii_line[100];
         char temp[8];
         for(int i=0; i<bin_buffer_cnt; i++)
         {
            if((i%16)==0){
               line[0]=0;
               ascii_line[0]=0;
               sprintf_2hex(temp, i/256);
               sprintf_2hex(temp+2, i%256);
               temp[4]=0;
               strcat(line, " ");
               strcat(line, temp);
               strcat(line, ": ");
            }
            sprintf_2hex(temp, bin_buffer[i]);
            strcat(line, temp);
            if(bin_buffer[i]==-1)strcat(ascii_line," ");
            else if(!isprint(bin_buffer[i]))strcat(ascii_line,".");
            else strcat(ascii_line,AnsiString(bin_buffer[i]).c_str());
            if((i%16) ==15){
               strcat(line," ");
               strcat(line,ascii_line);
               strcat(line,"\n");
               fputs(line,f);
            }
         }
      }
      else if(ExtractFileExt(mainForm->SaveDialog1->FileName).LowerCase()==".hex"){
         // Write data in hex file format
         int cbBuffer = bin_buffer_cnt;
         unsigned char* pbuf = bin_buffer;
         unsigned int addr = 0;
         while(cbBuffer)
         {
            int    cbWrite;
            char   line[256];
            char   temp[8];
            unsigned char  sum;
            cbWrite = ((cbBuffer > 16) ? 16 : cbBuffer);
            sprintf(line, ":%02X%04X00", cbWrite, addr);
            sum = cbWrite + ((addr & 0xFF00) >> 8) + (addr & 0xFF);
            for(int i = 0; i < cbWrite; i++, pbuf++)
            {
               sprintf(temp, "%02x", *pbuf);
               strcat(line, temp);
               sum += *pbuf;
            }
            sprintf(temp, "%02x\n", ((BYTE)(0x100 - sum))&0xFF);
            strcat(line, temp);
            fputs(line, f);
            addr += cbWrite;
            cbBuffer -= cbWrite;
         }
         fputs(":00000001FF\n", f);
      }
      fclose(f);
      return 1;
   }
   return 0;
}
//---------------------------------------------------------------------------

