/////////////////////////////////////////////////////////////////////////////
// RS232_Demo.cpp
//
// Main source for IAP via RS232 example application.
//
// Author: Marian Ilecko
//
// Revision History:
//
// 11/03/03 (MI) V1.1.1 - Derived from USB IAP Demo made by Jon Moore
// 11/04/03 (MI) V1.1.1 - IAP RS232 routines imported from NewPSDload project by M.I.
// 11/05/03 (MI) V1.1.1 - Code rearranged, adjusted and cleaned
// 11/06/03 (MI) V1.1.1 - Some bugfixes and improvements
// 11/07/03 (MI) V1.1.1 - Rearrangements, commenting
// 11/19/03 (MI) V1.1.2 - Bugfix in ReadFlash() (bug appeared if Count>0x59) 
//
// Copyright (c)2003 ST Microelecronics
// All rights reserved.
//
//---------------------------------------------------------------------------
// This example demo code is provided as is and has no warranty,
// implied or otherwise.  You are free to use/modify any of the provided
// code at your own risk in your applications with the expressed limitation
// of liability (see below) so long as your product using the code contains
// at least one uPSD products (device).
//
// LIMITATION OF LIABILITY:   NEITHER STMicroelectronics NOR ITS VENDORS OR
// AGENTS SHALL BE LIABLE FOR ANY LOSS OF PROFITS, LOSS OF USE, LOSS OF DATA,
// INTERRUPTION OF BUSINESS, NOR FOR INDIRECT, SPECIAL, INCIDENTAL OR
// CONSEQUENTIAL DAMAGES OF ANY KIND WHETHER UNDER THIS AGREEMENT OR
// OTHERWISE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
//---------------------------------------------------------------------------

#include <DateUtils.hpp>
#pragma hdrstop

#include "RS232_demo.h"
#include "test_form.h"
#include "dk3200_iap.h"
//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma resource "*.dfm"
TmainForm *mainForm;
//---------------------------------------------------------------------------

/////////////////// Global variables declaration
//
int test_runs=0,stop_test,time_consumed_for_ReadFlash; // used in link testing


//---------------------------------------------------------------------------
//              MAIN FORM CONSTRUCTOR AND MISCELLANEOUS FUNCTIONS
//---------------------------------------------------------------------------


/////////////////// TmainForm()
//
// Bring the program's main window to the front,
// to prevent it from disappearing among other windows.
//
__fastcall TmainForm::TmainForm(TComponent* Owner)
        : TForm(Owner)
{
   mainForm->BringToFront();
}
//---------------------------------------------------------------------------


/////////////////// FormClose()
//
// Before closing the form, we must check whether the test is in progress.
//
void __fastcall TmainForm::FormClose(TObject *Sender, TCloseAction &Action)
{
   if(test_runs) {
      Application->MessageBox("Please stop the running test first","Closing of the program",MB_OK);
      Action=caNone; // the form is not allowed to close, so nothing happens
   }
   else{
      stop_test = 1; // if the USB link test is running, stop it
      esc_pressed = 1; // stop the upload if it is in progress
      mirrorTimer->Enabled = 0; // stop the mirroring refresh timer
      deinitCommPort(); // close connection to the board
      Action=caFree; // the form is closed and all allocated memory for the form is freed
   }
}
//---------------------------------------------------------------------------


/////////////////// FormKeyPress()
//
// If the key was pressed, check if it is Escape, and if yes,
// set the flag variable to cancel all operations in progress.
//
void __fastcall TmainForm::FormKeyPress(TObject *Sender, char &Key)
{
   if(Key==0x1B)
   esc_pressed = 1;
}
//---------------------------------------------------------------------------



//---------------------------------------------------------------------------
//              LCD CONTENT MIRRORING ROUTINES
//---------------------------------------------------------------------------


/////////////////// UpdateDisplay()
//
// Displays mirrored data to panel at main form.
//
void __fastcall UpdateDisplay(char *src)
{
   mainForm->lbDisplayMirror->Caption = AnsiString(src);
}
//---------------------------------------------------------------------------


/////////////////// mirrorTimerTimer()
//
// Periodically requests LCD content via IAP command,
// and displays received data in main form.
//
void __fastcall TmainForm::mirrorTimerTimer(TObject *Sender)
{
   char reply[100];
   int recv_bytes;
   SendCommand_DK3200("MIR",3,reply,100,&recv_bytes,3);
   if(recv_bytes){ // Update LCD display mirror
      UpdateDisplay(reply+4); // display received text in panel at main form
   }
   else UpdateDisplay(""); // if no data received, clear the panel
}
//---------------------------------------------------------------------------



//---------------------------------------------------------------------------
//              PORT CONNECTION ROUTINES
//---------------------------------------------------------------------------

// In this section there is code for managing of RS232 link connection.


/////////////////// IsConnected()
//
// Checks if connection to the board exists.
//
// Function returns 1 if the board is connected, or 0 if it isn't.
//
int IsConnected()
{
   return comm_port_open;
}
//---------------------------------------------------------------------------


/////////////////// ConnectDevice()
//
// Establishes the connection to the board, and checks
// if the proper type of board is present.
//
// Function returns 1 if connection attempt succeeded or 0 if failed.
//
int ConnectDevice()
{
   if(IsConnected())
      return 2;
   else
   {
      // Open COM port
      if(initCommPort())
      {
         char board_name[100];
         SendCommand_DK3200("BRD",3,board_name,100,NULL);
         board_name[7]='X';
         if(!strcmp(board_name,"BRD DK3X00"))
            return 1;
         // else
         OnError("Failed to open connection to ST demo board.\n\n"
         "Please check if the RS232 link cable is attached properly,\n"
         "and if the demo board is powered on.");
         deinitCommPort();
      }
   }
   return 0;
}
//---------------------------------------------------------------------------



//---------------------------------------------------------------------------
//              BUTTON FUNCTIONALITY IMPLEMENTATION
//---------------------------------------------------------------------------

// In this section there is code which is executed after
// user pressed some of the controll buttons.


/////////////////// btConnectClick()
//
// Creates or cancels the connection to the board
//
void __fastcall TmainForm::btConnectClick(TObject *Sender)
{
   if(btConnect->Caption == "Connect") // Check if the board is already connected...
   { // if not, we will try to establish a connection.
      if(ConnectDevice()) // If connection process resulted in success...
      {
         mirrorTimer->Enabled = 1; // enable the mirroring refresh timer
         btConnect->Caption = "Disconnect";
         btRead->Enabled = 1; // enable all IAP buttons and other controls
         btWrite->Enabled = 1;
         btErase->Enabled = 1;
         btVerify->Enabled = 1;
         btProgram->Enabled = 1;
         btBlankCheck->Enabled = 1;
         btUpload->Enabled = 1;
         btReset->Enabled = 1;
         btStopMirror->Enabled = 1;
         btTest->Enabled = 1;
         rbMain->Enabled = 1;
         rbBoot->Enabled = 1;
         cbSector->Enabled = 1;
         edOffset->Enabled = 1;
         edCount->Enabled = 1;
         edData->Enabled = 1;
         sbStatus->Panels->Items[0]->Text = "Connected to the board, ready...";
      }
   }
   else
   { // if the board is connected, we will disconnect it.
      if(Sender != NULL)
      {
         mirrorTimer->Enabled = 0; // stop the mirroring refresh timer
         sbStatus->Panels->Items[0]->Text = "Connection to the board has been closed.";
      }
//      else
//      {
//         sbStatus->Panels->Items[0]->Text = "Connection to the board lost...";
//      }
      btConnect->Caption = "Connect";
      btRead->Enabled = 0; // disable all IAP buttons and other controls
      btWrite->Enabled = 0;
      btErase->Enabled = 0;
      btVerify->Enabled = 0;
      btProgram->Enabled = 0;
      btBlankCheck->Enabled = 0;
      btUpload->Enabled = 0;
      btReset->Enabled = 0;
      btStopMirror->Enabled = 0;
      btTest->Enabled = 0;
      rbMain->Enabled = 0;
      rbBoot->Enabled = 0;
      cbSector->Enabled = 0;
      edOffset->Enabled = 0;
      edCount->Enabled = 0;
      edData->Enabled = 0;
      lbDisplayMirror->Caption = "";
      mirrorTimer->Enabled = 0; // stop the mirroring refresh timer
      deinitCommPort(); // deinitialize the port
   }
   btStopMirror->Caption = "Stop Mirror";
}
//---------------------------------------------------------------------------


/////////////////// btStopMirrorClick()
//
// Stops or restarts the mirroring of LCD display content.
// The mirroring is time-consuming operation, so it's better to stop it
// before executing of commands like Upload, Program, Verify or Blank Check.
//
void __fastcall TmainForm::btStopMirrorClick(TObject *Sender)
{
   if(btStopMirror->Caption == "Stop Mirror")// Check if mirroring runs...
   { // If yes, we will stop it.
      UpdateDisplay("mirroring stopped\n=================");
      btStopMirror->Caption = "Start Mirror";
      mirrorTimer->Enabled = 0; // disable the mirroring refresh timer
   }
   else {  // If no, we will start it again.
      btStopMirror->Caption = "Stop Mirror";
      mirrorTimer->Enabled = 1; // enable the mirroring refresh timer
   }
}
//---------------------------------------------------------------------------


/////////////////// btReadClick()
//
// Reads data of given length from specified offset of selected
// flash sector and displays it in edit controls.
//
void __fastcall TmainForm::btReadClick(TObject *Sender)
{
   int nBytes,offset;
   BYTE buffer[256];
   char text[1000];
   // Prepare the error message, it will appear if command sequence not successful
   sbStatus->Panels->Items[0]->Text = "Flash reading unsuccessful.";
   // Get specified count of how many bytes to read
   strcpy(text,edCount->Text.c_str());
   if(!sscanf(text, "%x", &nBytes) || (nBytes > 0xf0))
      OnError("Invalid byte count field value (maximum count allowed is F0 (240 in decimal), input is HEXADECIMAL number).");
   else {
      // Get specified address (offset within flash sector)
      strcpy(text,edOffset->Text.c_str());
      if(!sscanf(text, "%x", &offset))
         OnError("Invalid address field value: not a numeric value.");
      else {
         // Select the target sector
         if(SelectFlash(SELECTED_FLASH, SELECTED_SECTOR))
         {
            // Read flash and display the result message
            if(!ReadFlash(offset, buffer, nBytes))
               sbStatus->Panels->Items[0]->Text = "Error while reading data from Flash.";
            else {
               // Display the return values
               RenderBinaryDataToHexaString(buffer+15, nBytes, text);
               edData->Text = AnsiString(text);
               sbStatus->Panels->Items[0]->Text = "Read Flash done.";
            }
         }
      }
   }
}
//---------------------------------------------------------------------------


/////////////////// btWriteClick()
//
// Reads data and length from edit controls and writes it
// to selected flash sector at specified offset.
//
void __fastcall TmainForm::btWriteClick(TObject *Sender)
{
   int offset,parsed;
   char bin_data[256],text[1000];
   // Prepare the error message, it will appear if command sequence not successful
   sbStatus->Panels->Items[0]->Text = "Flash writing unsuccessful.";
   // Get offset value
   if(!sscanf(edOffset->Text.c_str(),"%04x",&offset))
      OnError("Offset value is incorrect. Please enter hexadecimal number in format 0x1234.");
   else {
      // Parse the data to write
      if(1!=ParseHexaStringToBinAndAscii(edData->Text.c_str(), bin_data, NULL, &parsed))
         OnError("Data to write is invalid.");
      else {
         // Set byte count text field in dialog
         sprintf_2hex(text,parsed);
         text[2]=0;
         edCount->Text = AnsiString(text);
         if(!parsed)
            OnError("No data given to write.");
         else {
            // Get specified offset within flash sector
            if(!sscanf(edOffset->Text.c_str(), "%x", &offset))
               OnError("Invalid address field value: not a numeric value.");
            else {
               // Select the target sector
               if(SelectFlash(SELECTED_FLASH, SELECTED_SECTOR))
               {
                  // Write flash and display the result message
                  if(!WriteFlash(offset, bin_data, parsed))
                     sbStatus->Panels->Items[0]->Text = "Error while writing data to Flash.";
                  else
                     sbStatus->Panels->Items[0]->Text = "Write Flash done.";
               }
            }
         }
      }
   }
}
//---------------------------------------------------------------------------


/////////////////// btEraseClick()
//
// Erases the selected flash sector.
//
void __fastcall TmainForm::btEraseClick(TObject *Sender)
{
   if(SectorErase(SELECTED_FLASH, SELECTED_SECTOR, SEGMENT_BASIC_OFFSET))
      sbStatus->Panels->Items[0]->Text = "Erase Sector done.";
}
//---------------------------------------------------------------------------


/////////////////// btBlankCheckClick()
//
// Checks the selected flash sector whether it is blank or not.
// Being blank means to have all bytes set to 0xFF.
//
void __fastcall TmainForm::btBlankCheckClick(TObject *Sender)
{
   int result;
   if(1==(result=BlankCheck(SELECTED_FLASH, SELECTED_SECTOR, SEGMENT_BASIC_OFFSET, SEGMENT_SIZE)))
      sbStatus->Panels->Items[0]->Text = "Blank Check done.";
   else if(result==-1)sbStatus->Panels->Items[0]->Text = "Blank Check cancelled.";
   else sbStatus->Panels->Items[0]->Text = "Blank Check operation unsuccessful.";
}
//---------------------------------------------------------------------------


/////////////////// btUploadClick()
//
// Reads data from the selected flash sector and saves it to file.
//
void __fastcall TmainForm::btUploadClick(TObject *Sender)
{
   int result;
   if(1==(result=DownloadSector(SELECTED_FLASH, SELECTED_SECTOR, SEGMENT_BASIC_OFFSET, SEGMENT_SIZE, SEGMENT_NAME)))
      sbStatus->Panels->Items[0]->Text = "Upload done.";
   else if(result==-1)sbStatus->Panels->Items[0]->Text = "Upload cancelled.";
   else sbStatus->Panels->Items[0]->Text = "Upload operation unsuccessful.";
}
//---------------------------------------------------------------------------


/////////////////// btProgramClick()
//
// Loads data from file and writes it to selected flash sector.
//
void __fastcall TmainForm::btProgramClick(TObject *Sender)
{
   TDateTime t1,t2; // used for measurement of operation's time consumption
   sbStatus->Panels->Items[0]->Text = "Program Sector...";
   if(OpenDialog1->Execute()) // Prompt user for file
   {  int result;
      t1 = Now();
      if(1==(result=ProgramSector(SELECTED_FLASH, SELECTED_SECTOR, SEGMENT_BASIC_OFFSET, SEGMENT_SIZE, OpenDialog1->FileName.c_str()))){
         sbStatus->Panels->Items[0]->Text = "Program Sector done.";
         t2 = Now();
         sbStatus->Panels->Items[0]->Text += " Time consumed is " + AnsiString(MilliSecondsBetween(t1, t2)) +" ms.";
      }
      else if(result==-1)
         sbStatus->Panels->Items[0]->Text = "Program Sector cancelled.";
      else if(result==0)
         sbStatus->Panels->Items[0]->Text = "Error while programming of the sector.";
   }
   else sbStatus->Panels->Items[0]->Text = "File open aborted.";
}
//---------------------------------------------------------------------------


/////////////////// btVerifyClick()
//
// Verifies the content of selected flash sector with data from given file.
//
void __fastcall TmainForm::btVerifyClick(TObject *Sender)
{
   sbStatus->Panels->Items[0]->Text = "Verify Sector...";
   if(OpenDialog1->Execute()) // Prompt user for file
   {  int result;
      if(1==(result=VerifySector(SELECTED_FLASH, SELECTED_SECTOR, SEGMENT_BASIC_OFFSET, SEGMENT_SIZE, OpenDialog1->FileName.c_str())))
         sbStatus->Panels->Items[0]->Text = "Verify Sector done.";
      else if(result==-1)
         sbStatus->Panels->Items[0]->Text = "Verify Sector cancelled.";
      else if(result==0)
         sbStatus->Panels->Items[0]->Text = "Error while verifying of the sector.";
   }
   else sbStatus->Panels->Items[0]->Text = "File open aborted.";
}
//---------------------------------------------------------------------------


/////////////////// btResetClick()
//
// Resets the board.
//
void __fastcall TmainForm::btResetClick(TObject *Sender)
{
   sbStatus->Panels->Items[0]->Text = "Board reset...";
   mainForm->Repaint(); // refresh the texts displayed on the main form
   ResetBoard();
   Sleep(1000);
   sbStatus->Panels->Items[0]->Text = "Board reset command sent.";
}
//---------------------------------------------------------------------------


/////////////////// rbMainClick()
//
// This control changes execution page between primary and secondary flash.
// While the program is executed from primary, it can access secondary flash
// mapped in data space. On the contrary, when executing from secondary, the
// primary flash can be accessed for reading and writing.
// This method is also attached to ComboBox control, which selects among
// flash sectors. After changing these options, the command to the board is
// sent immediately. You can see the result on the LCD display.
//
void __fastcall TmainForm::rbMainClick(TObject *Sender)
{
   if(rbMain->Checked)
   {
      if(cbSector->Items->Count==4){
         cbSector->Items->Add("4");
         cbSector->Items->Add("5");
         cbSector->Items->Add("6");
         cbSector->Items->Add("7");
      }
      edOffset->Text="8000";
   }
   else {
      if(cbSector->ItemIndex>3)
         cbSector->ItemIndex=3;
      if(cbSector->Items->Count==8)
         for(int i=0;i<4;i++)
            cbSector->Items->Delete(4);
      if(cbSector->ItemIndex==0)edOffset->Text="8000";
      if(cbSector->ItemIndex==1)edOffset->Text="A000";
      if(cbSector->ItemIndex==2)edOffset->Text="C000";
      if(cbSector->ItemIndex==3)edOffset->Text="E000";
   }
   // Select the target sector
   if(SelectFlash(SELECTED_FLASH, SELECTED_SECTOR))
      sbStatus->Panels->Items[0]->Text = "Execution page changed...";
   else
      sbStatus->Panels->Items[0]->Text = "Error changing the execution page...";
}
//---------------------------------------------------------------------------


/////////////////// btTestClick()
//
// Runs the RS232 link data transfer test.
// This test reads blocks of data, of variable size,
// from randomly choosen flash sector and offset.
//
void __fastcall TmainForm::btTestClick(TObject *Sender)
{
   if(btStopMirror->Caption == "Stop Mirror")
      btStopMirror->Click();
   TDateTime time1,time2;
   stop_test = 0;
   testForm->logMemo->Clear();
   testForm->btStartAgain->Enabled = 0;
   testForm->btEndTest->Enabled = 1;
   testForm->lbDescription->Caption="This test reads blocks of\ndata, of variable size,\nfrom randomly choosen \nFlash sector and offset.";
   time1 = Now();
   testForm->logMemo->Lines->Add("The format is: SelectFlash(flash,sector)[OK/failed],");
   testForm->logMemo->Lines->Add("ReadFlash(flash,sector,offset,block_size)[OK/failed],");
   testForm->logMemo->Lines->Add("time consumed[value in miliseconds].");
   testForm->logMemo->Lines->Add("All parameters are generated randomly in whole respective range.");
   testForm->logMemo->Lines->Add("");
   testForm->logMemo->Lines->Add("Testing started "+time1.FormatString("d-mmm-yyyy, hh:mm:ss"));
#define MIN_TEST_BLOCK_SIZE 10
#define MAX_TEST_BLOCK_SIZE 100
   AnsiString str;
   char str_c[100];
   int test_block_size,total_bytes_read,SelectFlash_OK,ReadFlash_OK;
   int cycles_total,total_time,flash,sector,offset;
   BYTE buffer[256];
   testForm->Show();
   test_runs = 1; // this is to avoid the error message box to being displayed
   cycles_total = 0;
   SelectFlash_OK = 0;
   ReadFlash_OK = 0;
   total_bytes_read = 0;
   total_time = 0;
   while(!stop_test){
      Application->ProcessMessages();
      flash = random(2);
      sector = random(flash ? 4 : 8);
      test_block_size = MIN_TEST_BLOCK_SIZE + random(MAX_TEST_BLOCK_SIZE - MIN_TEST_BLOCK_SIZE);
      offset = 0x8000 + random(flash ? 0x2000-test_block_size : 0x8000-test_block_size);
      str = "SelectFlash("+AnsiString(flash)+","+AnsiString(sector)+") ";
      // Select the target sector
      if(SelectFlash(flash, sector)){
         str += "OK, ";
         SelectFlash_OK++;
         }
      else str += "failed, ";
      // Read flash
      sprintf(str_c, "ReadFlash(%d,%d,0x%04x,%d) ", flash, sector, offset, test_block_size);
      str += AnsiString(str_c);
      if(ReadFlash(offset, buffer, test_block_size)){
         str += "OK.";
         ReadFlash_OK++;
         total_bytes_read += test_block_size;
         total_time += time_consumed_for_ReadFlash;
         }
      else str += "failed.";
      str += "t="+AnsiString(time_consumed_for_ReadFlash);
      testForm->logMemo->Lines->Add(str);
      cycles_total++;
      testForm->lbStatistics->Caption =
      "Total cycles: "+
      AnsiString(cycles_total)+
      "\nSelectFlash failed: "+
      AnsiString(cycles_total-SelectFlash_OK)+
      " times\nReadFlash failed: "+
      AnsiString(cycles_total-ReadFlash_OK)+
      " times\nTotal of "+
      AnsiString(total_bytes_read)+
      " bytes read\n";//Total time is "+AnsiString(total_time)+" ms";
   }
   time2 = Now();
   testForm->logMemo->Lines->Add("Testing stopped "+time2.FormatString("d-mmm-yyyy, hh:mm:ss"));
   testForm->logMemo->Lines->Add("Total cycles: "+AnsiString(cycles_total));
   testForm->logMemo->Lines->Add("SelectFlash failed: "+AnsiString(cycles_total-SelectFlash_OK)+" times");
   testForm->logMemo->Lines->Add("ReadFlash failed: "+AnsiString(cycles_total-ReadFlash_OK)+" times");
   testForm->logMemo->Lines->Add("Total of "+AnsiString(total_bytes_read)+" bytes read successfully");
   test_runs = 0;
}
//---------------------------------------------------------------------------


/////////////////// btAboutClick()
//
// Displays the About Program info.
//
void __fastcall TmainForm::btAboutClick(TObject *Sender)
{
   Application->MessageBox("RS232 In-Application-Programming Demo v.1.1.2\n\n(c)2003 ST Microelectronics\n\nMemory Products Group, Design & Application Center, Prague\n\nMembers of the team are:\nMarian Ilecko (responsible for Windows SW)\nPetr Pfeifer (responsible for uPSD FW)\n\nIf you have any questions, contact us on our addresses:\nmarian.ilecko@st.com\npetr.pfeifer@st.com","About the program",MB_OK);
}
//---------------------------------------------------------------------------


/////////////////// btCloseClick()
//
// Closes the program.
//
void __fastcall TmainForm::btCloseClick(TObject *Sender)
{
   // Close the main form
     mainForm->Close();
}
//---------------------------------------------------------------------------

