/*

FLASHCODE.C:  Flash Memory management routines

--------------------------------------------------------------------------
Copyright (c)2002-2003 ST Microelectronics
This example demo code is provided as is and has no warranty,
implied or otherwise.  You are free to use/modify any of the provided
code at your own risk in your applications with the expressed limitation
of liability (see below) so long as your product using the code contains
at least one uPSD products (device).

LIMITATION OF LIABILITY:   NEITHER STMicroelectronics NOR ITS VENDORS OR 
AGENTS SHALL BE LIABLE FOR ANY LOSS OF PROFITS, LOSS OF USE, LOSS OF DATA,
INTERRUPTION OF BUSINESS, NOR FOR INDIRECT, SPECIAL, INCIDENTAL OR
CONSEQUENTIAL DAMAGES OF ANY KIND WHETHER UNDER THIS AGREEMENT OR
OTHERWISE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
--------------------------------------------------------------------------

These functions are provided to help you develop your initial code. 
They are optimized for speed rather that size. As a result, you will 
see very few nested function calls. If speed is not critical, you 
can use function calls for common tasks (like dat polling after 
writing a byte to Flash or EEPROM) The penalty is the extra processor 
time to make the nested calls.

These files have been compiled using a C cross compiler from KEIL 
Software Inc. You may have to implement some syntax changes to be 
compatible with other compilers. The intent of this generated C code 
is to provide you with a core of useful broadbased functions that are 
adaptable to many vendor's compilers and microcontrollers.


Note:

Some of the routines provided may not have been thoroughly tested.  
Please check them in your system. If you find a bug, or a place 
where the code could be improved, PLEASE forward your comments by 
emailing to apps.psd@st.com. Any comments and feedback are 
appreciated. Please tell us what you like and or what you think 
can be improved.


Revision History:

08/21/2002 (Jon Moore)
Uncommented psd834f2.h include. Added #defines to enable flash routines 
used by firmware. Added xdata modifier to flash_write_with_poll pointer 
variables. Changed all variables named 'data' to 'dat' to avoid conflict 
with Keil compiler reserved word.

11/07/2003 (Marian Ilecko)
All headers and library files rearranged and cleaned up.


Be sure to uncomment the associated #define statements inside the 
header file, flashcode.h, to use individual desired functions 
contained in this file, flashcode.c. Your compiler will ignore the functions
in this file unless you do so.
*/

#include "flashcode.h"
#include "upsd3300.h"

typedef unsigned char uchar;

/* Function prototypes */

/*
Group: Main Flash Memory
Coverage: Program, Erase, Reset, Read ID, Read Protection
*/

/*
Module: flash_write_with_poll
Programs a single byte, checks status using polling method.
You'll need to include the header files generated by PSDsoft
Express. Important: if memory paging is used, the correct 
page value must be set in the PSD page register prior to 
calling this function.
*/

#ifdef _F_W_W_P

unsigned char flash_write_with_poll(volatile uchar xdata* addr, uchar dat)
{
	unsigned char done;
	unsigned char error;
	unsigned char err;
	unsigned char poll;

	done = FALSE;
   	err = FALSE;

	//  Note:  the following constants (FLASH_COMMON_XXXX)
	//     are declared type volatile in the header file 
	//	   so they are not optimized away by the compiler

	
	*(FLASH_COMMON_X555) = 0xAA;     // unlock main flash, write 0xAA to addess 0xX555
	*(FLASH_COMMON_XAAA) = 0x55;     // unlock main flash, write 0x55 to addess 0xXAAA
	*(FLASH_COMMON_X555) = 0xA0;     // write 0xA0 command to program

	*(addr) = dat;                  // write byte to flash   
	
	dat = dat & NVM_DATA_POLL;     // get bit DQ7 of original dat   
	
	do                               // now use dat polling method to verify successful write
    {  

   		poll = *(addr);          // Read the location that was just programmed

		error = poll & NVM_ERROR;   // save timeout error bit at DQ5

		poll = poll & NVM_DATA_POLL;  // get DQ7 of poll byte read from flash  

		if (dat == poll)        // compare DQ7 

			done = TRUE;     // dat byte programmed into flash OK,
                                         // indicate successful exit criteria

		else if (error ==  NVM_ERROR )	 // check for timeout error   
			err = TRUE;      // indicate timeout error occurred

    } while((done == FALSE) && (err == FALSE)); 


	if (err == TRUE)                 // make sure timeout error and dat poll didn't 
                                         // occur simultaneously
	{
		poll = *(addr);          // Read location in flash again

		poll = poll & NVM_DATA_POLL;   // get DQ7 of poll byte read from flash  

		if (dat == poll)        // compare DQ7 

			done = TRUE;     // dat byte programmed into flash OK at the same
                                         // time timout error occured, indicate successful 
		                        // exit criteria

		*(FLASH_COMMON_X555) = 0xF0;  // reset the flash array (short reset instruction) 
		        // now delay 3 msec per dat sheet
    }

   	return(done);         // a successful flash write returns 1, timeout error returns 0
}
#endif


#ifdef _F_B_W_W_P

unsigned char flash_boot_write_with_poll(volatile uchar xdata* addr, uchar dat)
{
	unsigned char done;
	unsigned char error;
	unsigned char err;
	unsigned char poll;

	done = FALSE;
   	err = FALSE;

	//  Note:  the following constants (FLASH_BOOT_XXXX)
	//     are declared type volatile in the header file 
	//	   so they are not optimized away by the compiler

	
	*(FLASH_BOOT_X555) = 0xAA;     // unlock main flash, write 0xAA to addess 0xX555
	*(FLASH_BOOT_XAAA) = 0x55;     // unlock main flash, write 0x55 to addess 0xXAAA
	*(FLASH_BOOT_X555) = 0xA0;     // write 0xA0 command to program

	*(addr) = dat;                  // write byte to flash   
	
	dat = dat & NVM_DATA_POLL;     // get bit DQ7 of original dat   
	
	do                               // now use dat polling method to verify successful write
    {  

   		poll = *(addr);          // Read the location that was just programmed

		error = poll & NVM_ERROR;   // save timeout error bit at DQ5

		poll = poll & NVM_DATA_POLL;  // get DQ7 of poll byte read from flash  

		if (dat == poll)        // compare DQ7 

			done = TRUE;     // dat byte programmed into flash OK,
                                         // indicate successful exit criteria

		else if (error ==  NVM_ERROR )	 // check for timeout error   
			err = TRUE;      // indicate timeout error occurred

    } while((done == FALSE) && (err == FALSE)); 


	if (err == TRUE)                 // make sure timeout error and dat poll didn't 
                                         // occur simultaneously
	{
		poll = *(addr);          // Read location in flash again

		poll = poll & NVM_DATA_POLL;   // get DQ7 of poll byte read from flash  

		if (dat == poll)        // compare DQ7 

			done = TRUE;     // dat byte programmed into flash OK at the same
                                         // time timout error occured, indicate successful 
		                        // exit criteria

		*(FLASH_BOOT_X555) = 0xF0;  // reset the flash array (short reset instruction) 
		        // now delay 3 msec per dat sheet
    }

   	return(done);         // a successful flash write returns 1, timeout error returns 0
}


#endif



/*
Module: flash_write_with_toggle
Programs a single byte, checks status using toggle method.
You'll need to include the header files generated by PSDsoft
Express. Important: if memory paging is used, the correct page 
value must be set in the PSD page register prior to calling this 
function.
*/

#ifdef _F_W_W_T

unsigned char flash_write_with_toggle(addr,dat)	
volatile unsigned char *addr;
unsigned char dat;

	{
	unsigned char done;
	unsigned char error;
	unsigned char err;
	volatile unsigned char toggle_A;
	volatile unsigned char toggle_B;

	done = FALSE;
   	err = FALSE;

	//  Note:  the following constants (FLASH_BOOT_XXXX)
	//     are declared type volatile in the header file 
	//	   so they are not optimized away by the compiler

	*(FLASH_BOOT_X555) = 0xAA;		// unlock main flash, write 0xAA to addess 0xX555
	*(FLASH_BOOT_XAAA) = 0x55;		// unlock main flash, write 0x55 to addess 0xXAAA
	*(FLASH_BOOT_X555) = 0xA0;		// write 0xA0 command to program

	*(addr) = dat;		// write byte to flash   

		// now use toggling method to verify successful write

	toggle_A = *(addr);			// Read the location that was just programmed

	toggle_A = toggle_A & NVM_DATA_TOGGLE;	// mask toggle bit at DQ6
   		  								
	do
		{  
   		toggle_B = *(addr);		// Again read the location that was just programmed

		error = toggle_B & NVM_ERROR;	// save timeout error flag at DQ5

		toggle_B = toggle_B & NVM_DATA_TOGGLE;	// mask toggle bit at DQ6
   		  								
		if (toggle_A == toggle_B)	// compare toggle bit DQ6

			done = TRUE;		// bit did not toggle, dat byte programmed into 
						// flash OK, indicate successful exit criteria
		else
			{
	 		 if (error ==  NVM_ERROR )	// check for timeout error   
				err = TRUE; 	// indicate timeout error occurred

			toggle_A = toggle_B;  	// save most recent sample of toggle bit 
						// to compare with next sample
			}

		} while((done == FALSE) && (err == FALSE)); 


	if (err == TRUE)			// make sure timeout error and dat toggle didn't 
						// occur simultaneously
		{
		toggle_B = *(addr);		// Read location in flash again

		toggle_B = toggle_B & NVM_DATA_TOGGLE;	// mask toggle bit at DQ6

		if (toggle_A == toggle_B)		// compare toggle bit DQ6

			done = TRUE;			// dat byte programmed into flash OK at the same
							// time timout error occured, indicate successful 
							// exit criteria

		*(FLASH_BOOT_X555) = 0xF0;  // reset the flash array (short reset instruction) 
		        // now delay 3 msec per dat sheet
		}

   	return(done);		// a successful flash write returns 1, timeout error returns 0
	}

#endif



/*
Module: flash_erase_sector
Erases the entire main Flash memory (all sectors).
You'll need to include the header files generated by PSDsoft
Express. Important: The address passed to this function should 
be independent of memory paging or else the PSD page 
register value should be set to the correct page prior to calling this 
function. 

Note: The address that is passed in this function can be an address that 
      resides in any Flash segment that has a chip select. For example, if 
      fs0 and fs5 are used in the design, passing an address in this function
      that resides in either fs0 or fs5 will invoke the bulk erase operation.
*/

#ifdef _F_E_B

unsigned char flash_erase_sector(
    volatile unsigned char xdata* flash_bulk_erase_address)
{
	unsigned char done;
 	unsigned char poll;
 	unsigned char error;
 	unsigned char err;

	done = FALSE;
   	err = FALSE;

	//  Note:  the following constants (FLASH_COMMON_XXXX)
	//     are declared type volatile in the header file 
	//	   so they are not optimized away by the compiler

	*(FLASH_COMMON_X555) = 0xAA;		// unlock main flash, write 0xAA to addess 0xX555
	*(FLASH_COMMON_XAAA) = 0x55;		// unlock main flash, write 0x55 to addess 0xXAAA
	*(FLASH_COMMON_X555) = 0x80;		// write 0x80 command to erase entire chip
	*(FLASH_COMMON_X555) = 0xAA;        	// continue unlock sequence
	*(FLASH_COMMON_XAAA) = 0x55;		// continue unlock sequence
	//*(FLASH_COMMON_X555) = 0x10;		// write 0x10 command to complete erase command
    *(flash_bulk_erase_address) = 0x30; // write 0x30 to sector address to erase

	do		  			// now use dat polling method to verify successful erase
    {  
		poll = *(flash_bulk_erase_address); 	// read flash status from any address
		                                // within the defined flash address space

		error = poll & NVM_ERROR;	// save timeout error bit at DQ5

		poll = poll & NVM_DATA_POLL;	// look at D7   

		if (poll == NVM_DATA_POLL)	// compare DQ7 

			done = TRUE;		// bulk erase OK,
						// indicate successful exit criteria

		else if (error == NVM_ERROR)	// check for timeout error   
			err = TRUE;		// indicate timeout error occurred

    } while((done == FALSE) && (err == FALSE)); 


	if (err == TRUE)			// make sure timeout error and dat poll didn't 
						// occur simultaneously
    {
		poll = *(flash_bulk_erase_address); 	// Read flash status again

		poll = poll & NVM_DATA_POLL;	// get DQ7 of poll byte read from flash  

		if (poll == NVM_DATA_POLL)	// compare DQ7 

			done = TRUE;		// the flash erased OK at the same
						// time timout error occured, indicate successful 
						// exit criteria

		*(FLASH_COMMON_X555) = 0xF0;  // reset the flash array (short reset instruction) 
		        // now delay 3 msec per dat sheet
    }

   	return(done);		// a successful flash erase returns 1, timeout error returns 0

}
#endif

#ifdef _F_B_E_B

unsigned char flash_boot_erase_sector(
    volatile unsigned char xdata* flash_bulk_erase_address)
{
	unsigned char done;
 	unsigned char poll;
 	unsigned char error;
 	unsigned char err;

	done = FALSE;
   	err = FALSE;

	//  Note:  the following constants (FLASH_COMMON_XXXX)
	//     are declared type volatile in the header file 
	//	   so they are not optimized away by the compiler

	*(FLASH_BOOT_X555) = 0xAA;		// unlock main flash, write 0xAA to addess 0xX555
	*(FLASH_BOOT_XAAA) = 0x55;		// unlock main flash, write 0x55 to addess 0xXAAA
	*(FLASH_BOOT_X555) = 0x80;		// write 0x80 command to erase entire chip
	*(FLASH_BOOT_X555) = 0xAA;        	// continue unlock sequence
	*(FLASH_BOOT_XAAA) = 0x55;		// continue unlock sequence
	// *(FLASH_BOOT_X555) = 0x10;		// write 0x10 command to complete erase command
    *(flash_bulk_erase_address) = 0x30; // write 0x30 to sector address to erase

	do		  			// now use dat polling method to verify successful erase
    {  
		poll = *(flash_bulk_erase_address); 	// read flash status from any address
		                                // within the defined flash address space

		error = poll & NVM_ERROR;	// save timeout error bit at DQ5

		poll = poll & NVM_DATA_POLL;	// look at D7   

		if (poll == NVM_DATA_POLL)	// compare DQ7 

			done = TRUE;		// bulk erase OK,
						// indicate successful exit criteria

		else if (error == NVM_ERROR)	// check for timeout error   
			err = TRUE;		// indicate timeout error occurred

    } while((done == FALSE) && (err == FALSE)); 


	if (err == TRUE)			// make sure timeout error and dat poll didn't 
						// occur simultaneously
    {
		poll = *(flash_bulk_erase_address); 	// Read flash status again

		poll = poll & NVM_DATA_POLL;	// get DQ7 of poll byte read from flash  

		if (poll == NVM_DATA_POLL)	// compare DQ7 

			done = TRUE;		// the flash erased OK at the same
						// time timout error occured, indicate successful 
						// exit criteria

		*(FLASH_BOOT_X555) = 0xF0;  // reset the flash array (short reset instruction) 
		        // now delay 3 msec per dat sheet
    }

   	return(done);		// a successful flash erase returns 1, timeout error returns 0

}

#endif


/*
Module: flash_reset
Resets the main Flash memory to Read Array mode.
After this reset, the Flash memory may be read like a ROM device. 
You'll need to include the header files generated by PSDsoft
Express.
*/

#ifdef _F_R

void flash_reset()	// reset flash, read array mode

	{

	//  Note:  the following constants (FLASH_COMMON_XXXX)
	//     are declared type volatile in the header file 
	//	   so they are not optimized away by the compiler

	*(FLASH_COMMON_X555) = 0xAA;		// unlock main flash, write 0xAA to addess 0xX555
	*(FLASH_COMMON_XAAA) = 0x55;		// unlock main flash, write 0x55 to addess 0xXAAA
	*(FLASH_COMMON_X555) = 0xF0;		// write 0xF0 command to reset 
						// Flash memory to Read Array Mode
                      // now delay 3 msec per dat sheet
	}

void flash_boot_reset()	// reset boot flash, read array mode

	{

	//  Note:  the following constants (FLASH_BOOT_XXXX)
	//     are declared type volatile in the header file 
	//	   so they are not optimized away by the compiler

	*(FLASH_BOOT_X555) = 0xAA;		// unlock main flash, write 0xAA to addess 0xX555
	*(FLASH_BOOT_XAAA) = 0x55;		// unlock main flash, write 0x55 to addess 0xXAAA
	*(FLASH_BOOT_X555) = 0xF0;		// write 0xF0 command to reset 
						// Flash memory to Read Array Mode
                      // now delay 3 msec per dat sheet
	}

#endif


/* 
Module: flash_read_id
Reads the Flash Identifier byte from main Flash memory. You'll need 
to include the header files generated by PSDsoft Express.
Important: The address passed to this function should be independent 
of memory paging or else the PSD page register value should be set to 
the correct page prior to calling this function.

Note: The flash memory in the boot area of a PSD813F2 or PSD813F4 does 
not provide an ID. Only the main Flash memory in all PSD813FX provides 
an ID.
*/

#ifdef _F_R_I

unsigned char flash_read_id(flash_id_address)	// read flash identifier
volatile unsigned char *flash_id_address;  

  	{
	unsigned char id;

	//  Note:  the following constants (FLASH_COMMON_XXXX)
	//     are declared type volatile in the header file 
	//	   so they are not optimized away by the compiler

	*(FLASH_COMMON_X555) = 0xAA;		// unlock main flash, write 0xAA to addess 0xX555
	*(FLASH_COMMON_XAAA) = 0x55;		// unlock main flash, write 0x55 to addess 0xXAAA
	*(FLASH_COMMON_X555) = 0x90;		// write 0x90 command to get ID

	id = *(flash_id_address);		// read flash status, address bit A6 = 0
						//				  A1 = 0
						//				  A0 = 1
	*(FLASH_COMMON_X555) = 0xF0;  // reset the flash array (short reset instruction) 
		        // now delay 3 msec per dat sheet

	return (id);				// return byte ID value
	}

#endif


/* 
Module: flash_read_sector_protect
Reads main Flash sector protection status.
An unsigned char byte value is returned that contains the status 
for all eight sectors of main Flash memory. This byte value can 
be decoded as follows:

BitD7   BitD6   BitD5   BitD4   BitD3   BitD2   BitD1   BitD0
 fs7	 fs6     fs5     fs4     fs3     fs2     fs1     fs0

  fsx >> 1 = Flash sector is write protected
  fsx >> 0 = Flash sector is not write protected

The protection bits may only be read by the microcontroller. They
can be set or cleared only through the JTAG channel or a device
programmer, not the microcontroller.


Note:  You'll need to include the header files generated by PSDsoft
Express. 
*/

#ifdef _F_R_S_P

extern xdata PSD_REGS PSD8xx_reg;// _at_ PSD_REG_ADDR;	// Define PSD registers at address "csiop" space

unsigned char flash_read_sector_protect()	

	{
 
 	return (PSD8xx_reg.MAINPROTECT);   //This is a register inside the PSD
					// For reference, see the header file 
					// upsd3200.h for the location of 
					// the PSD register, MAINPROTECT.
	}

#endif

/*
End of Group Main Flash Memory
*/
