/*

SIO.C:  Serial Communication Routines.

--------------------------------------------------------------------------
Copyright (c)2003 ST Microelectronics
This example demo code is provided as is and has no warranty,
implied or otherwise.  You are free to use/modify any of the provided
code at your own risk in your applications with the expressed limitation
of liability (see below) so long as your product using the code contains
at least one uPSD products (device).

LIMITATION OF LIABILITY:   NEITHER STMicroelectronics NOR ITS VENDORS OR 
AGENTS SHALL BE LIABLE FOR ANY LOSS OF PROFITS, LOSS OF USE, LOSS OF DATA,
INTERRUPTION OF BUSINESS, NOR FOR INDIRECT, SPECIAL, INCIDENTAL OR
CONSEQUENTIAL DAMAGES OF ANY KIND WHETHER UNDER THIS AGREEMENT OR
OTHERWISE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
--------------------------------------------------------------------------

These files have been compiled using a C cross compiler from KEIL 
Software Inc. You may have to implement some syntax changes to be 
compatible with other compilers. The intent of this generated C code 
is to provide you with a core of useful broadbased functions that are 
adaptable to many vendor's compilers and microcontrollers.


Programmed by Marian Ilecko

------------------------------------------------------------------------------*/

#include "upsd3300.h"
#include "string.h"
#include "sio.h"

#define TBUF_SIZE	256		// size of RS232 transmit buffer
#define RBUF_SIZE	256		// size of RS232 receive buffer

xdata unsigned char tbuf[TBUF_SIZE]; // RS232 transmit buffer
xdata unsigned char rbuf[RBUF_SIZE]; // RS232 receive buffer
static unsigned char t_finished = 0; // flag to indicate that transmitting is done

static unsigned char tbuf_ptr = 0;   	// pointer in receive buffer
static unsigned char rbuf_ptr = 0;   	// pointer in transmit buffer
static unsigned char tbuf_datasize = 0; // length of data to transmit

//------------------------------------------------------------------------------

// serial port interrupts service routine
static void com_rx (void) interrupt 4 using 2
{
	// Received data interrupt
	if (RI != 0){
		RI = 0;
		if (rbuf_ptr < RBUF_SIZE)		// if buffer is not empty
			rbuf[rbuf_ptr++] = SBUF;    // store the received character
	}
	// Transmitted data interrupt
	if (TI != 0){
		TI = 0;
		if (tbuf_ptr != tbuf_datasize)	// if not all data has been transmitted yet
			SBUF = tbuf[tbuf_ptr++];    // transmit next character and shift pointer
		else
			t_finished = 1;				// else transmitting is done
	}
}
//------------------------------------------------------------------------------

// setup TIMER1 to generate the proper baud rate
void com_baudrate (void)
{
	EA = 0;				// disable interrupts
	TI = 0;				// clear transmit interrupt
	tbuf_ptr = 0;		// empty transmit buffer
	tbuf_datasize = 0;
	rbuf_ptr = 0;		// empty receive buffer
	t_finished = 1;		// disable transmitter

	// set timer 1 up as a baud rate generator
	TR1 = 0;			// stop timer 1
	ET1 = 0;			// disable timer 1 interrupt
	PCON |= 0x80;		// 0x80=SMOD: set serial baudrate doubler
	TMOD &= ~0xF0;		// clear timer 1 mode bits
	TMOD |= 0x20;		// put timer 1 into MODE 2

	// magic formula: (unsigned char) (256 - (OSC_FREQ / (16L * 4L * baudrate)));
	TH1 = -11; 			//for 19200 bps
	TL1 = 0x00;
	TR1 = 1;			// start timer 1
	EA = 1;             // enable interrupts
}
//------------------------------------------------------------------------------

// initialize the serial port
void com_initialize (void)
{
	com_baudrate();	// setup TIMER1 to generate the proper baud rate
	EA = 0;			// disable interrupts
	rbuf_ptr = 0; 	// clear COM buffer indexes and flag
	tbuf_ptr = 0;
	tbuf_datasize = 0;
	t_finished = 1; 

	// setup serial port registers.
	SM0 = 0; 
	SM1 = 1;		// serial port MODE 1
	SM2 = 0;
	REN = 1;		// enable serial receiver
	TI = 0;			// clear transmit interrupt
	RI = 0;			// clear receiver interrupt
	ES = 1;			// enable serial interrupts
	PS = 1;			// set serial interrupts to high priority
	IPA = 0x00;		// reset others devices' priorities
	P3SFS = 0x03;	// enable alternative function (UART RX & TX) of P3 pins
	EA = 1;         // enable interrupts
}
//------------------------------------------------------------------------------

// put the string to buffer and start to transmit it
void com_putstring (unsigned char *str, int len)
{
	while (!t_finished); 	// wait until transmitter is ready	
	EA = 0;					// disable interrupts
	memcpy(tbuf,str,len);	// add the data to the transmit buffer
	tbuf_datasize = len;
	tbuf_ptr = 0;
	t_finished = 0;
	TI = 1;					// if the transmit interrupt is disabled, enable it
	EA = 1;                 // enable Interrupts
}
//------------------------------------------------------------------------------

// read the received data from receive buffer
// the function returns amount of data received (or zero if buffer is empty)
int com_getstring (char *buf)
{
	int received_chars;
	if (rbuf_ptr == 0)			// if the buffer is empty
		return (0);				// return zero
	EA = 0;                 	// disable interrupts
	memcpy(buf,rbuf,rbuf_ptr);  // copy data from buffer
	received_chars=rbuf_ptr;    // empty the buffer
	rbuf_ptr=0;
	EA = 1;						// enable interrupts
	return (received_chars);
}
//------------------------------------------------------------------------------

