/**************** (c) 2001  STMicroelectronics ********************************
     
PROJECT  : ST7FOPTIONS-EVAL BOARD
COMPILER : (COSMIC)
MODULE   :  I2c_drv.c
VERSION  :  V 1.0
CREATION DATE :  01/11/01
AUTHOR   : STM/CMG

****************************************************************************** /

/*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

    THE SOFTWARE INCLUDED IN THIS xxx IS FOR GUIDANCE ONLY. STMicroelectronics 
    SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL 
    DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
    
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

DESCRIPTION :   ST7 I2C single master peripheral software driver.
                Polling software strategy.  
                No errors management

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

******************************************************************************/

/* EXTERNAL DECLARATIONS *****************************************************/
/* List of all the variables defined in another module and used in this one. */
                                       
#include "io72264.h"              /* Declaration of the ST72264 HW registers.*/
#include "lib_bits.h"
#include "I2c_drv.h"


/* STATIC DECLARATIONS *******************************************************/
/* List of all the variables/constant defined in another module and used in  */
/* this one.                                                                 */
/* MODEL => static type_name var_name;  or #define                           */

/* I2C_CCR possible speeds with Fcpu=8MHz  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~*/

#define I2C_SPEED     0x26                                   	/* 100.00 KHz.*/
// #define I2C_SPEED     0x37                                  /*  70.00 KHz.*/
//#define I2C_SPEED     0x4E                                   /*  50.00 KHz.*/
//#define I2C_SPEED     0x9E                                   /*  25.00 KHz.*/
//#define I2C_SPEED     0xEC                                   /*  15.5 KHz.*/
//#define I2C_SPEED     0x86                                   /* 333.00 KHz.*/
//#define I2C_SPEED     0x8C                                   /* 190.00 KHz.*/
//#define I2C_SPEED     0x8E                                   /* 167.00 KHz.*/


// I2CCR Bit definitions ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  #define PE      5                                       /* Peripheral enable.   */
  #define ENGC    4                                       /*  Enable general call.*/
  #define START   3                                /*  Start condition generation.*/
  #define ACK     2                                         /*  Acknoledge level. */
  #define STOP    1                                 /*  Stop condition generation.*/
  #define ITE     0                                         /*  Interrupt enable. */

// I2CSR1 Bit definitions ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  #define SR2F    7                              /*      Status register 2 flag  */
  #define ADD10   6                        /*     10bit master addressing mode.  */
  #define TRA     5                            /*        Transmitter / receiver. */
  #define BUSY    4            /*    Bus busy (between start and stop condition. */
  #define BTF     3                                 /*   Byte transfer finished. */
  #define ADSL    2                                   /*     Addressed as slave. */
  #define MSL     1                                   /*        Master / slave.  */
  #define SB      0                     /*   Start bit generated (master mode).  */

// I2CSR2 Bit definitions ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  #define AF      4                                    /* Acknowledge failure  */
  #define STOPF   3                         /* Stop detection flag (slave mode)*/
  #define ARLO    2                            /*              Arbitration lost*/
  #define BERR    1                               /*                Bus error  */
  #define GCAL    0                               /* General call (slave mode) */

// I2C register initial values ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
// Control register: I2CCR              ---  ---  PE  ENGC START ACK STOP  ITE 
  #define CR_INIT_VALUE   0x24        /*   0    0    1    0    0    1    0    0*/  
 
 
/* FUNCTION DESCRIPTIONS *****************************************************/
/* Description of all the functions defined in this module.                  */
/* MODEL => [static] type_name var_name;  or #define                         */

/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_Int
INPUT/OUTPUT : None.
DESCRIPTION  : I2C Interrupt routine.
COMMENTS     : Not used (polling strategy).
-----------------------------------------------------------------------------*/ 
void I2Cm_Int (void)
{
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_Init
INPUT/OUTPUT : None.
DESCRIPTION  : I2C peripheral initialisation routine.
COMMENTS     : Contains inline assembler instructions in C like mode !
-----------------------------------------------------------------------------*/ 
void I2Cm_Init (void)
{
  I2CCR   = 0x00;             /* Force reset status of the control register.*/
  I2CCCR  = I2C_SPEED;        /* Set the selected I2C-bus speed.*/
  _asm(" TNZ  _I2CDR  ");
  _asm(" TNZ  _I2CSR1 ");
  _asm(" TNZ  _I2CSR2 ");            
 /* Touch registers to remove pending interrupt.*/
 /* (Exception: written in inline assembler.*/
 
  I2CCR = CR_INIT_VALUE; 
  I2CCR = CR_INIT_VALUE;     /* Set initial control register value.*/
  I2Cm_Start();                                /* Start condition generation.*/
  I2Cm_Stop();                                 /* Stop  condition generation.*/ 
} 

/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_Start
INPUT/OUTPUT : None.
DESCRIPTION  : Generates I2C-Bus Start Condition.
COMMENTS     :
-----------------------------------------------------------------------------*/ 
void I2Cm_Start (void)
{
  SetBit(I2CCR,START);                          /* Generate start condition.*/
  while (!ValBit(I2CSR1,SB));              /* Wait the Start bit generation.*/
} 

/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_Stop
INPUT/OUTPUT : None.
DESCRIPTION  : Generates I2C-Bus Stop Condition.
COMMENTS     :
-----------------------------------------------------------------------------*/ 
void I2Cm_Stop (void)
{
  SetBit(I2CCR,STOP);                            /* Generate stop condition.*/
} 

/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_Ack
INPUT/OUTPUT : None.
DESCRIPTION  : Acknoledge generation from now.
COMMENTS     : Transfer sequence = DATA, ACK.
-----------------------------------------------------------------------------*/ 
void I2Cm_Ack (void)
{
  SetBit(I2CCR,ACK);        /* Acknoledge after the next received data byte.*/
} 

/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_nAck
INPUT/OUTPUT : None.
DESCRIPTION  : Non acknoledge generation from now.
COMMENTS     : Transfer sequence = DATA, NACK.
-----------------------------------------------------------------------------*/ 
void I2Cm_nAck (void)
{
  ClrBit(I2CCR,ACK);    /* Non acknoledge after the next received data byte.*/
} 


/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_SetAddr
INPUT/OUTPUT : external I2C device address / None.
DESCRIPTION  : Generates Start-bit and transmits the address byte.
COMMENTS     : Transfer sequence = START, EV5, ADD, ACK...
-----------------------------------------------------------------------------*/ 
void I2Cm_SetAddr (char i2c_addr)
{
  I2Cm_Start();                               /* Generates a start condition.*/
  I2CDR = i2c_addr;                      /* Write address to be transmitted.*/

} 

/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_TxData
INPUT/OUTPUT : data byte to be transfered / None.
DESCRIPTION  : Transmits a data byte.
COMMENTS     : Transfer sequence = DATA, ACK, EV8...
-----------------------------------------------------------------------------*/ 
void I2Cm_TxData (char i2c_data)
{

  do {
    if (I2CSR2)  while(1);   /* Communication error detected: infinity loop.*/
    SetBit(I2CCR,PE);           /* Touch the control register to pass "EV6".*/ 
    } while (!ValBit(I2CSR1,BTF));                        /* Wait BTF ("EV8").*/
  I2CDR = i2c_data;                    /* Write data byte to be transmitted.*/

} 

/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_RxData
INPUT/OUTPUT : Last byte to receive flag (active high) / Received data byte.
DESCRIPTION  : Receive a data byte.
COMMENTS     : Transfer sequence = DATA, ACK, EV7...
-----------------------------------------------------------------------------*/ 
char I2Cm_RxData (char last)
{
  do {
    if (I2CSR2)  while(1);   /* Communication error detected: infinity loop.*/
    SetBit(I2CCR,PE);           /* Touch the control register to pass "EV6".*/ 
  } while (!ValBit(I2CSR1,BTF));                        /* Wait BTF ("EV7").*/
  if (last) I2Cm_Stop();  /* End of communication: stop condition generation.*/
  return(I2CDR);                          /* Read/return data byte received.*/
}


/*-----------------------------------------------------------------------------
ROUTINE NAME : Wait_Ack
INPUT/OUTPUT : None.
DESCRIPTION  : Wait Ack from the slave.
COMMENTS     : 
-----------------------------------------------------------------------------*/ 
void Wait_Ack(char i2c_addr)
{
 do{
        if(I2CSR2 == 0x10)
        {
          I2CCR = CR_INIT_VALUE;
          I2CCR = CR_INIT_VALUE;
          I2Cm_SetAddr(i2c_addr);
        }          
        if(!I2CSR2) SetBit(I2CCR,PE);
        
        if ((I2CSR2) && (I2CSR2!=10)) while(1); 
        
   }while (!ValBit(I2CSR1,BTF));
                       

}        

        
/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_Tx
INPUT/OUTPUT : I2c dest @, sub @, Nb data byte to transmit, Buffer @ / None.
DESCRIPTION  : Transmit data buffer via I2C.
COMMENTS     : Low significant bytes first.
-----------------------------------------------------------------------------*/ 
void I2Cm_Tx (char * buff_add , char sub_add, char nb, char dest_add)
{

  I2Cm_SetAddr(dest_add);              /* Slave address selection on I2C bus.*/
  I2Cm_TxData(sub_add);                             /* Sub address selection.*/

  for (;nb > 0; nb--)   /* Loop to send all selected data from output buffer.*/
	  I2Cm_TxData( *(buff_add+nb-1) );  /* Next output buffer data byte sent.*/

  I2Cm_Stop();            /* End of communication: stop condition generation.*/
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : I2Cm_Rx
INPUT/OUTPUT : I2c dest @, sub @, Nb data byte to receive, Buffer @ / None.
DESCRIPTION  : Receive in data buffer via I2C.
COMMENTS     : Low significant bytes first.
-----------------------------------------------------------------------------*/ 
void I2Cm_Rx (char *buff_add , char sub_add, char nb, char dest_add)
{                            
  I2Cm_SetAddr(dest_add);              /* Slave address selection on I2C bus.*/
  Wait_Ack(dest_add);
  I2Cm_TxData(sub_add);                             /* Sub address selection.*/
  I2Cm_SetAddr(dest_add|0x01);       /* Slave address selection in read mode.*/
  do                    /* Loop to send all selected data from output buffer.*/
  {
    nb--;
    if (nb==0)                                       /* Last byte to receive.*/
    {
      I2Cm_nAck();                    /* Non acknoledge after last reception.*/
       *(buff_add+nb) = I2Cm_RxData(1);  /* Last output buffer data byte sent.*/
    }
    else
       *(buff_add+nb) = I2Cm_RxData(0);  /* Next output buffer data byte sent.*/
     
  } while (nb > 0);          /* Loop to receive the selected number of bytes.*/
  I2Cm_Ack();                                  /* Acknoledge after reception.*/
}

/************* (c) 2000 STMicroelectronics ****************** END OF FILE ***/
