/**************** (c) 2000  STMicroelectronics *******************************

NAME:		usb_ep2.c
PROJECT:	USB - ST7 FULL SPEED
VERSION:	v 1.0
CREATION:	02/01/2002
AUTHOR:		MICROCONTROLLER DIVISION / ST Rousset

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-
		Functions for sending and/or receiving data from/to EP2
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

MODIFICATIONS : 

******************************************************************************/
#include "mcu_conf.h"
#include "usb_reg.h"
#include "usb_def.h"
#include "usb_libs.h"
#include "usb_lib.h"

//#ifdef	DECLARE_EP2

//extern unsigned char _EP_RxTx_Flag;	// D2=0: Transmiter is free
									// D2=1: Transmiter is busy
									// D3=0: Receiver is free
									// D3=1: Receiver is busy
#pragma DATA_SEG USB_RAM
static unsigned char *EP2_XBuffer;	// Current pointer to transmit buffer
static unsigned char EP2_XLength;	// Remain length to be sent
static unsigned char *EP2_RBuffer;	// Current pointer to receive buffer
static unsigned char EP2_RLength;	// Remain length to be received
static unsigned char EP2_RSize;		// Bytes received in the receive buffer

#pragma CODE_SEG USB_CODE 
/*-----------------------------------------------------------------------------
ROUTINE NAME : USB_SendDataEP2
INPUT/OUTPUT : *DataAddress: points to the buffer to be sent
			   Length: gives the length of data to be sent. Range is 1-255
RETURN       : REQ_ERROR if the transmitter is busy
			   REQ_SUCCESS if the transmit starts correctly
DESCRIPTION  : Transmit "Length" data from the given buffer to host PC
-----------------------------------------------------------------------------*/
char _USB_SendDataEP2(void)
{
	if (ValBit(_EP_RxTx_Flag, EV_EP2_IN))		// Check if the transmitter is busy
		return REQ_ERROR;
	if ((EP2TXR & EP_GOOD) == 0)				// Check if the EP is in DISABLE or STALL
		return REQ_ERROR;

	if (EPs_Length > MAX_EP2_PACKET_SIZE) {
		EP2_XLength = EPs_Length - MAX_EP2_PACKET_SIZE;	// Bytes remain to xmit
		EP2_XBuffer = EPs_DataAddress + MAX_EP2_PACKET_SIZE;	// Address for the next xmit
		asm LD		X, #MAX_EP2_PACKET_SIZE;		// Bytes to copy to EP buffer
	}
	else {
		EP2_XLength = 0;						// All data can be sent in one shot
		asm LD		X, EPs_Length;			// Bytes to copy to EP buffer
	}

	asm {										// Copy data to EP buffer
		LD		CNT2TXR, X
		DEC		X
		LD		A, EPs_DataAddress
		LD		_LEX, A
		LD		A, EPs_DataAddress:1
		LD		_LEX:1, A
loop_copy:
		LD		A, ([_LEX.w], X)
		LD		(EP2_IN, X), A
		DEC		X
		JRPL	loop_copy
	}

	SetBit(_EP_RxTx_Flag, EV_EP2_IN);
	USB_SetTxEP2Status(EP_VALID);
	return REQ_SUCCESS;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : USB_EP2_isSent
RETURN       : Non-zero if the data is sent on EP2
DESCRIPTION  : This function is called to enquire if the data is sent on EP2
-----------------------------------------------------------------------------*/
//char USB_EP2_isSent(void)
//{
//	return !ValBit(_EP_RxTx_Flag, EV_EP2_IN);
//}

/*-----------------------------------------------------------------------------
ROUTINE NAME : USB_EP2_XEvent
DESCRIPTION  : This function is called when there is a USB CTR on EP2 transmitter
WARNING      : This function may be called from interrupt routine
-----------------------------------------------------------------------------*/
void _USB_EP2_XEvent(void)
{
	if (EP2_XLength == 0) {
		ClrBit(_EP_RxTx_Flag, EV_EP2_IN);		// Finish sending if there is no remain data
		return;
	}

	if (EP2_XLength > MAX_EP2_PACKET_SIZE) {	// If more than one packet to send
		EP2_XLength -= MAX_EP2_PACKET_SIZE;		// Descrease number of remain bytes
		EP2_XBuffer += MAX_EP2_PACKET_SIZE;		// Increase sending pointer for next send
		asm LD		X, #MAX_EP2_PACKET_SIZE;	// Number of bytes to copy to EP buffer
	}
	else {
		asm LD		X, EP2_XLength;				// Number of bytes to copy to EP buffer
		EP2_XLength = 0;						// All data can be sent in one shot
	}

	asm {										// Copy data from user buffer to EP buffer
		LD		A, EP2_XBuffer
		LD		_LEX, A
		LD		A, EP2_XBuffer:1
		LD		_LEX:1, A

		LD		CNT2TXR, X
		DEC		X
loop_copy:
		LD		A, ([_LEX.w], X)
		LD		(EP2_IN, X), A
		DEC		X
		JRPL	loop_copy
	}

	USB_SetTxEP2Status(EP_VALID);
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : USB_RecvDataEP2
INPUT/OUTPUT : "DataAddress" is the buffer to receive the data
			   "Length" is the size of the buffer
RETURN       : REQ_ERROR if the receiver is busy
			   REQ_SUCCESS if the receiving starts correctly
DESCRIPTION  : Enable the receiver and receive the data to the buffer
			   Use USB_TakeDataEP2() to check if the receiving is finished
-----------------------------------------------------------------------------*/
char _USB_RecvDataEP2(void)
{
	if (ValBit(_EP_RxTx_Flag, EV_EP2_OUT))	// Check if the receiver is busy
		return REQ_ERROR;
	if ((EP2RXR & EP_GOOD) == 0)			// Check if the EP is in DISABLE or STALL
		return REQ_ERROR;

	EP2_RLength = EPs_Length;				// The number of bytes to receive
	EP2_RBuffer = EPs_DataAddress;			// The user buffer pointer
	EP2_RSize = 0;							// The number of bytes received
	SetBit(_EP_RxTx_Flag, EV_EP2_OUT);		// Mark the receiver as busy

	CNT2RXR = MAX_EP2_PACKET_SIZE;
	USB_SetRxEP2Status(EP_VALID);
	return REQ_SUCCESS;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : USB_TakeDataEP2
INPUT/OUTPUT : none
RETURN       : 0xFF if the receiving does not finish
			   otherwise the number of actual received bytes
DESCRIPTION  : check if the receiving finishes
-----------------------------------------------------------------------------*/
unsigned char USB_TakeDataEP2(void)
{
	if (ValBit(_EP_RxTx_Flag, EV_EP2_OUT))
		return 0xFF;			// The receiver is empty, try later

	return EP2_RSize;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : USB_EP2_REvent
DESCRIPTION  : This function is called when there is a USB CTR on EP2 receiver
WARNING      : This function may be called from interrupt routine
-----------------------------------------------------------------------------*/
#pragma NO_OVERLAP
void _USB_EP2_REvent(void)
{
	unsigned char size = MAX_EP2_PACKET_SIZE - CNT2RXR;	// Number of bytes received

	if (size < EP2_RLength) {					// Not all bytes have been received
		EP2_RSize += size;						// Increase total byte number
		EP2_RLength -= size;					// Decrease remain byte number
		if (CNT2RXR != 0)
			ClrBit(_EP_RxTx_Flag, EV_EP2_OUT);	// Finish receiving on a short packet

		asm LD	X, size							// Going to copy data to user buffer
	}
	else {										// All bytes have been received
		ClrBit(_EP_RxTx_Flag, EV_EP2_OUT);		// Finish receiving
		EP2_RSize += EP2_RLength;				// Increase total byte number
		asm LD	X, EP2_RLength;					// Going to copy data to user buffer
		EP2_RLength = 0;						// No byte remain to receive
	}

	asm {							// Copy received bytes to the user buffer
		DEC		X
		JRMI	recv_end
		LD		A, EP2_RBuffer
		LD		_LEX, A
		LD		A, EP2_RBuffer:1
		LD		_LEX:1, A
recv_copy:
		LD		A, (EP2_OUT, X)
		LD		([_LEX.w], X), A
		DEC		X
		JRPL	recv_copy
recv_end:
	}

	if (ValBit(_EP_RxTx_Flag, EV_EP2_OUT)) {	// When there are more date to receive
		EP2_RBuffer += size;					// Increase user buffer pointer
		CNT2RXR = MAX_EP2_PACKET_SIZE;
		USB_SetRxEP2Status(EP_VALID);			// Enable the next receiving
	}
}

//#endif	// DECLARE_EP2
