/**************** (c) 2000  STMicroelectronics *******************************

NAME:		usb_lib.c
PROJECT:	USB - ST7 FULL SPEED
VERSION:	v 1.0
CREATION:	20/12/2001
AUTHOR:		MICROCONTROLLER DIVISION / ST Rousset

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-
		Functions for handling USB standard request on EP0
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

MODIFICATIONS : 

******************************************************************************/

#include "mcu_conf.h"
#include "usb_reg.h"	// Definitions for the USB registers
#include "usb_def.h"	// Definitions from the USB spec.
#include "usb_lib.h"	// Definitions of all the macros and functions exported

#include "user_usb.h"	// Definitions of functions in user space

#pragma DATA_SEG SHORT USB_RAM0		// Map USB variables into page 0 (short addressing RAM)

//***************************************************************************
//*******  INTERNAL USB Library variables, used only in the USB core  *******
//***************************************************************************
extern unsigned char _Token_OverRun;	// when it's set, stop SETUP process

// USB State Machine value, see _vUSB_StateMachine selector in Usb_def.h
static char _vUSB_StateMachine;	// States of the USB state machine
static char	_vUSB_Report[2];			// Used to report status to host.

//***************************************************************************
//******  Following variables are exported to the outside of the code *******
//***************************************************************************
#ifdef	LARGE_EP0
unsigned short	vUSB_offset;	// Offset of the data stream for data IN stage
unsigned short	vUSB_length;	// Length remain of the data stream for data IN stage
#else
unsigned char	vUSB_offset;	// Offset of the data stream for data IN stage
unsigned char	vUSB_length;	// Length remain of the data stream for data IN stage
#endif
char	*vUSB_DataToCopy;		// Data pointer for data IN stage
char	_vUSB_Data_OUT_Len;		// Used to get data length of a data OUT stage

char	vUSB_Configuration;		// Current configurationn value
char	vUSB_Current_Feature;	// D6: Power feature; 1=Self-Power, 0=Bus-Power
								// D5: Remote-wakeup: 1=Support, 0=Not support
								// D1: Remote-wakeup: 1=Enable, 0=Disable
								// Other bits: Reserved as 0

_USB_VSETUP	sUSB_vSetup;		// Internal storage for SETUP package


//************************* CODE SEGMENT ************************************
#pragma CODE_SEG USB_CODE 

#define _USB_SetEP0TxRx(Tx, Rx)	EP0R = (EP0R & 0xCC) | (Rx | (Tx << 4))
#define Non_Standard_Request	{ asm JP USER_USB_Setup; }

//***************************************************************************
//****************** Const array imported from user space *******************
//***************************************************************************
//extern ONE_DESCRIPTOR DeviceDescriptor;
//extern ONE_DESCRIPTOR ConfigDescriptor[1];
//extern ONE_DESCRIPTOR StringDescriptor[1];

//***************************************************************************
//*******  INTERNAL USB Library functions, used only in the USB core  *******
//***************************************************************************
char _USB_GetEPStatus(char Endpoint);
void _USB_Clr_Feature_EP(void);
void _USB_Set_Feature_EP(void);
void _USB_CopydataIN(void);
char _USB_CopydataOUT(void);

#pragma DATA_SEG USB_RAM
char vUSB_Num_Configuration;			// number of configuration
char vUSB_Num_Interface;				// number of interface
char vUSB_Num_Strings;				// number of string descriptor
char vUSB_Interrupt_Mask;			// USB interrupt mask

unsigned char *EPs_DataAddress;		// Used in endpoint data address
unsigned char EPs_Length;			// Used in endpoint packet length

/*-----------------------------------------------------------------------------
ROUTINE NAME : RequestError
INPUT/OUTPUT : None
DESCRIPTION  : Stall next Data or Status Stage.
				Function called when an irrelevant request has been received.
-----------------------------------------------------------------------------*/
void RequestError(void)
{
	// RequestError is called only during enumeration -> STALL EP0 both dir.
	EP0R = 0x11;	// TX Stall, RX Stall (only SETUP token will be ACKed)
	_vUSB_StateMachine = STATE_ERROR;
}

void RequestType_0x00(void);
void RequestType_0x80(void);
void RequestType_0x01(void);
void RequestType_0x81(void);
void RequestType_0x02(void);
void RequestType_0x82(void);

//----------------------------------------------------------------------------
//-------- Table_RequestType is a jump table for dispatching requests --------
//----------------------------------------------------------------------------
#pragma	NO_RETURN
void Table_RequestType(void)
{
	asm {
		JP		RequestType_0x00	// H->D Standard Device
		JP		RequestType_0x80	// D->H Standard Device
		JP		RequestType_0x01	// H->D Standard Interface
		JP		RequestType_0x81	// D->H Standard Interface
		JP		RequestType_0x02	// H->D Standard Endpoint
		JP		RequestType_0x82	// D->H Standard Endpoint
		JP		USER_USB_Setup
		JP		USER_USB_Setup
	}
}

/*****************************************************************************
ROUTINE NAME : USB_SETUP_Stage
DESCRIPTION  : This function is called when a SETUP packet is received
               The SETUP packet is copied and dispached to different routine
*****************************************************************************/
void USB_SETUP_Stage(void)
{
	vUSB_DataToCopy = NULL;
//	_vUSB_StateMachine 				= SETTING_UP;	// For testing 

	sUSB_vSetup.USBbRequest			= EP0_OUT[1];	// save current bRequest
	sUSB_vSetup.USBbmRequestType 	= EP0_OUT[0];	// save current bmRequestType

	// Copy SETUP packet to sUSB_vSetup and set the sUSB_vSetup.Flag accordingly
	asm {
		CLR		A							// A = sUSB_vSetup.Flag
		LD		X, EP0_OUT:3
		LD		sUSB_vSetup.USBwValue1, X	// Copy high byte of wValue
		JREQ	Cp_wValue0
		OR		A, #NON0_wValue1			// Set flag for high byte of wValue is not 0
Cp_wValue0:
		LD		X, EP0_OUT:2
		LD		sUSB_vSetup.USBwValue0, X	// Copy low byte of wValue
		JREQ	Cp_wIndex1
		OR		A, #NON0_wValue0			// Set flag for low byte of wValue is not 0
Cp_wIndex1:
		LD		X, EP0_OUT:5
		LD		sUSB_vSetup.USBwIndex1, X	// Copy high byte of wIndex
		JREQ	Cp_wIndex0
		OR		A, #NON0_wIndex1			// Set flag for high byte of wIndex is not 0
Cp_wIndex0:
		LD		X, EP0_OUT:4
		LD		sUSB_vSetup.USBwIndex0, X	// Copy low byte of wIndex
		JREQ	Cp_wLength1
		OR		A, #NON0_wIndex0			// Set flag for low byte of wIndex is not 0
Cp_wLength1:
		LD		X, EP0_OUT:7
		LD		sUSB_vSetup.USBwLength1, X	// Copy high byte of wLength
		JREQ	Cp_wLength0
		OR		A, #NON0_wLength1			// Set flag for high byte of wLength is not 0
Cp_wLength0:
		LD		X, EP0_OUT:6
		LD		sUSB_vSetup.USBwLength0, X	// Copy low  byte of wLength
		JREQ	Set_ZeroFlag
		OR		A, #NON0_wLength0			// Set flag for low byte of wLength is not 0
		BCP		A, #NON0_wLength1			// Check if wLength = 1 or 2
		JRNE	Set_ZeroFlag
		CP		X, #2
		JRUGT	Set_ZeroFlag
		JREQ	wLength_is_2
		OR		A, #ONE_wLength				// Set flag for wLength==1
		JRT		Set_ZeroFlag
wLength_is_2:
		OR		A, #TWO_wLength				// Set flag for wLength==2
Set_ZeroFlag:
		LD		sUSB_vSetup.Flag, A
	}

	// Stop the SETUP packet processing if SETUP overrun is detected.
	if (_Token_OverRun)
		return;

	// Preset the machine state for the next step
	if ((sUSB_vSetup.Flag & NON0_wLength) == 0)	// wLength == 0
		_vUSB_StateMachine = WAIT_STATUS_IN;
	else if ((sUSB_vSetup.USBbmRequestType & 0x80) == 0)
		_vUSB_StateMachine = OUT_DATA;	
	else
		_vUSB_StateMachine = IN_DATA;

	vUSB_offset = 0;
	// Dispatch requests to the correspond routine
	asm {
		LD		A, sUSB_vSetup.USBbmRequestType
		BCP		A, #0x7C
		JREQ	Standard_Requests
		CALL	USER_USB_Setup			// Non standard requests
		JRT		Post_Requests
Standard_Requests:
		ADD		A, #0x80
		RLC		A
		LD		X, #3
		MUL		X, A
		LD		X, A
		CALL	(Table_RequestType,X)
Post_Requests:
	}

	if ((_vUSB_StateMachine & MACHINESTATE) == IN_DATA)	// IN_DATA or ONE_MORE_IN STATE
		_USB_CopydataIN();			// Copy data to the endpoint buffer with such request
}

/*****************************************************************************
ROUTINE NAME : USB_Data_IN_Stage
DESCRIPTION  : This function is called when a data stage packet is sent
		Copy the next data packet to the endpoint buffer if it is necessary
*****************************************************************************/
void USB_Data_IN_Stage(void)
{
	if (_Token_OverRun)		// Stop the data stage procesing if SETUP overrun is detected
		return;

	switch (_vUSB_StateMachine) {
	case IN_DATA:
		if (vUSB_length == 0)
			_vUSB_StateMachine = WAIT_STATUS_OUT;
		else
			_USB_CopydataIN();
		break;
	case ONE_MORE_IN:
		CNT0TXR = 0;
		break;

	default:
		RequestError();
	}
}

/*****************************************************************************
ROUTINE NAME : USB_Data_OUT_Stage
DESCRIPTION  : This function is called when a data stage packet is received
		Copy the next data packet from the endpoint buffer if it is necessary
*****************************************************************************/
void USB_Data_OUT_Stage(void)
{
	if (_Token_OverRun)		// Stop the data stage procesing if SETUP overrun is detected
		return;

	switch (_vUSB_StateMachine & MACHINESTATE) {
	case OUT_DATA:
//	case ONE_MORE_OUT:
		if (vUSB_length < MAX_PACKET_SIZE)
			_vUSB_StateMachine = WAIT_STATUS_IN;
		if (!_USB_CopydataOUT())
			RequestError();
		break;

	default:
		RequestError();
	}
}

/*****************************************************************************
ROUTINE NAME : USB_Status_IN_Stage
DESCRIPTION  : This function is called when a status IN stage is received
*****************************************************************************/
void USB_Status_IN_Stage(void)
{
	// Successful status stage means a transaction is finished correctly
	// Do not stop the data stage procesing if SETUP overrun is detected

	switch (_vUSB_StateMachine & MACHINESTATE) {
	case ADDRESS2SET:
		DADDR = sUSB_vSetup.USBwValue0;
	/* Following lines are useless because the request is not valid in configured state
		if (DADDR == 0) {				// Device goes to address state
			vUSB_Configuration = 0;
			_EP_RxTx_Flag = 0;			// Clear flag for non-control EPs

			// Call user function to un-configure the device
			USER_USB_Set_Configuration();
		}
	*/
		_vUSB_StateMachine = WAIT_SETUP;
		break;

	case WAIT_STATUS_IN:
	case OUT_DATA:
//	case ONE_MORE_OUT:
		USER_USB_Status_In();			// Inform user function of the event
		_vUSB_StateMachine = WAIT_SETUP;
		break;

	default:
		RequestError();
	}
}

/*****************************************************************************
ROUTINE NAME : USB_Status_OUT_Stage
DESCRIPTION  : This function is called when a status OUT stage is received
*****************************************************************************/
void USB_Status_OUT_Stage(void)
{
	// Successful status stage means a transaction is finished correctly
	// Do not stop the data stage procesing if SETUP overrun is detected

	switch (_vUSB_StateMachine & MACHINESTATE) {
	case WAIT_STATUS_OUT:
	case IN_DATA:
//	case ONE_MORE_IN:
		USER_USB_Status_Out();			// Inform user function of the event
		_vUSB_StateMachine = WAIT_SETUP;
		break;

	default:
		RequestError();
	}
}

/*****************************************************************************
ROUTINE NAME : Change_EP0_State
DESCRIPTION  : The SETUP overrun event is checked after process each token
		and before change the state of EP0. The EP0 state is changed only
		when there is no SETUP overrun. (See usb_int.c & usb_poll.c)
*****************************************************************************/
void Change_EP0_State(void)
{
	if (_vUSB_StateMachine == WAIT_SETUP)
		return;

	CNT0RXR = MAX_PACKET_SIZE;
	switch (_vUSB_StateMachine & MACHINESTATE) {
	case WAIT_STATUS_IN:
	case ADDRESS2SET:
		CNT0TXR = 0;
		_USB_SetEP0TxRx(EP_VALID, EP_STALL);
		break;
	case WAIT_STATUS_OUT:
		_USB_SetEP0TxRx(EP_STALL, EP_VALID);
		break;
	case IN_DATA:
		_USB_SetEP0TxRx(EP_VALID, EP_VALID);
		break;
	case OUT_DATA:
		CNT0TXR = 0;
		_USB_SetEP0TxRx(EP_VALID, EP_VALID);
		break;
	case STATE_ERROR:
		_USB_SetEP0TxRx(EP_STALL, EP_STALL);
		break;
	}
}

/*****************************************************************************
ROUTINE NAME : RequestType_0x00
DESCRIPTION  : Standard process for request of (H->D, Standard, Device)
*****************************************************************************/
void RequestType_0x00(void)		// Host->Device, Standard, Device-Recipient
{
//	_vUSB_StateMachine = WAIT_STATUS_IN;	// Expect Status IN after the SETUP Stage

	if ((sUSB_vSetup.Flag & (NON0_wIndex | NON0_wValue1 | NON0_wLength)) != 0)
		goto Req_Error;							// (wIndex | wValue1 | wLength) !=0

	switch (sUSB_vSetup.USBbRequest) {
	case CLEAR_FEATURE:
		if (sUSB_vSetup.USBwValue0 != 1)		// Check the feature selector
			break;
		if (DADDR == 0)							// On default state, to stall the request
			break;
		if ((vUSB_Current_Feature & 0x20) == 0)	// Remote wakeup is not support
			break;

		vUSB_Current_Feature &= ~0x02;			// Disable the remote wakeup feature
		return;

	case SET_FEATURE:
		if (sUSB_vSetup.USBwValue0 != 1)		// Check the feature selector
			break;
		if (DADDR == 0)							// On default state, to stall the request
			break;
		if ((vUSB_Current_Feature & 0x20) == 0)	// Remote wakeup is not support
			break;

		vUSB_Current_Feature |= 0x02;			// Enable the remote wakeup feature
		return;

	case SET_ADDRESS:
		if (sUSB_vSetup.USBwValue0 > 127)		// Check the address range
			break;

		if (vUSB_Configuration)					// Microsoft requires this
			break;								// USB Spec. does not want it

		_vUSB_StateMachine = ADDRESS2SET;
		return;

	case SET_CONFIGURATION:
		if (DADDR == 0)							// On default state, to stall the request
			break;
		if (sUSB_vSetup.USBwValue0 > vUSB_Num_Configuration)	// Check configure value
			break;

		vUSB_Configuration = sUSB_vSetup.USBwValue0;
		if (vUSB_Configuration == 0)
			_EP_RxTx_Flag = 0;					// Clear flag for non-control EPs

		if (USER_USB_Set_Configuration() == REQ_SUCCESS)
			return;
	}
Req_Error:
	Non_Standard_Request;
}	// End of "Host->Device, Standard, Device-Recipient"

/*****************************************************************************
ROUTINE NAME : RequestType_0x80
DESCRIPTION  : Standard process for request of (D->H, Standard, Device)
*****************************************************************************/
void RequestType_0x80(void)		// Device->Host, Standard, Device-Recipient
{
//	_vUSB_StateMachine = IN_DATA;		// Expect Data IN stage after the SETUP Stage

	switch (sUSB_vSetup.USBbRequest) {
	case GET_CONFIGURATION:
		if (DADDR == 0)					// On default state, to stall the request
			goto Req_Error;
		if (sUSB_vSetup.Flag != (ONE_wLength | NON0_wLength0))
			goto Req_Error;				// (wValue | wIndex) !=0 || wLength != 1

		vUSB_DataToCopy = &vUSB_Configuration; 
		vUSB_length = 1;
		return;

	case GET_DESCRIPTOR:
		if ((sUSB_vSetup.Flag & NON0_wIndex) == 0) {
			if (sUSB_vSetup.USBwValue1 == 0x01) {		// Device Descriptor
				if (sUSB_vSetup.Flag & NON0_wValue0)
					goto Req_Error;;
				vUSB_DataToCopy = DeviceDescriptor.Descriptor;
				vUSB_length = DeviceDescriptor.Size;
			}
			else if (sUSB_vSetup.USBwValue1 == 0x02) {	// Configuration Descriptor
				// Check configuration range and return the correspond configuration descriptor
				if (sUSB_vSetup.USBwValue0 > vUSB_Num_Configuration)
					goto Req_Error;
				vUSB_DataToCopy = ConfigDescriptor[sUSB_vSetup.USBwValue0].Descriptor;
				vUSB_length = ConfigDescriptor[sUSB_vSetup.USBwValue0].Size;
			}
			else if (sUSB_vSetup.USBwValue1 == 0x03) {	// String Descriptor
				if (sUSB_vSetup.Flag & NON0_wValue0)
					goto Req_Error;;
				// Asking for the language ID
				vUSB_DataToCopy = StringDescriptor[0].Descriptor;
				vUSB_length = StringDescriptor[0].Size;
				
			}
			else {
		Req_Error:
				Non_Standard_Request;
			}
		}
		else if (sUSB_vSetup.USBwValue1 == 0x03					// String Descriptor
				&& sUSB_vSetup.USBwIndex == 0x0409				// US english only
				&& sUSB_vSetup.USBwValue0 < vUSB_Num_Strings) {	// Valid index
			vUSB_DataToCopy = StringDescriptor[sUSB_vSetup.USBwValue0].Descriptor;
			vUSB_length = StringDescriptor[sUSB_vSetup.USBwValue0].Size;
		}

		if (vUSB_DataToCopy == NULL)
			goto Req_Error;

		if (vUSB_length > sUSB_vSetup.USBwLength)
			vUSB_length = sUSB_vSetup.USBwLength;
		return;

	case GET_STATUS:
		if (DADDR == 0)				// On default state, to stall the request
			goto Req_Error;

		if (sUSB_vSetup.Flag != (TWO_wLength | NON0_wLength0))	// (wValue | wIndex) != 0 || wLength != 2
			goto Req_Error;

		// Request of Get Device Status
		_vUSB_Report[0] = vUSB_Current_Feature & 0x03;	// Remote wakeup and self power feature
		_vUSB_Report[1] = 0;
		vUSB_length = 2;
		vUSB_DataToCopy = _vUSB_Report;
		return;

	default:
		goto Req_Error;
	}
}	// Device->Host, Standard, Device-Recipient

/*****************************************************************************
ROUTINE NAME : RequestType_0x01
DESCRIPTION  : Standard process for request of (H->D, Standard, Interface)
*****************************************************************************/
void RequestType_0x01(void)		// Host->Device, Standard, Interface-Recipient
{
//	_vUSB_StateMachine = WAIT_STATUS_IN;		// Expect Status IN stage follow the Setup Stage

	if (DADDR == 0 || vUSB_Configuration == 0)	// Not on configured state
		goto Req_Error;
	if (sUSB_vSetup.Flag & NON0_wLength)
		goto Req_Error;							// wLength != 0

	if (sUSB_vSetup.USBbRequest == CLEAR_FEATURE) {
		USER_USB_Clear_Feature_Int();
		return;
	}
	else if (sUSB_vSetup.USBbRequest == SET_FEATURE) {
		USER_USB_Set_Feature_Int();
		return;
	}
	else if (sUSB_vSetup.USBbRequest == SET_INTERFACE) {
		if (sUSB_vSetup.Flag & (NON0_wIndex1 | NON0_wValue1))
			goto Req_Error;						// (wIndex1 | wValue1) != 0

		if (USER_USB_Set_Interface() == REQ_SUCCESS)
			return;
	}

Req_Error:
	Non_Standard_Request;
}	// End of "Host->Device, Standard, Interface-Recipient"

/*****************************************************************************
ROUTINE NAME : RequestType_0x81
DESCRIPTION  : Standard process for request of (D->H, Standard, Interface)
*****************************************************************************/
void RequestType_0x81(void)		// Device->Host, Standard, Interface-Recipient
{
//	_vUSB_StateMachine = IN_DATA;	// Expect Data IN stage follow the Setup Stage

	if (DADDR == 0 || vUSB_Configuration == 0)	// Not on configured state
		goto Req_Error;
	if (sUSB_vSetup.Flag & (NON0_wValue | NON0_wIndex1))
		goto Req_Error;						// (wValue | wIndex1) != 0

	if (sUSB_vSetup.USBbRequest == GET_STATUS) {
		if (!(sUSB_vSetup.Flag & TWO_wLength))
			goto Req_Error;					// wLength != 2

		// Check range of the interface number
		if (sUSB_vSetup.USBwIndex0 >= vUSB_Num_Interface)
			goto Req_Error;

		_vUSB_Report[0] = 0;				// Send a 0 Word according to spec.
		_vUSB_Report[1] = 0;				// Send a 0 Word according to spec.
		vUSB_length = 2;
		vUSB_DataToCopy = _vUSB_Report;
		return;
	}
	else if (sUSB_vSetup.USBbRequest == GET_INTERFACE) {
		if (!(sUSB_vSetup.Flag & ONE_wLength))
			goto Req_Error;					// wLength != 1

		_vUSB_Report[0] = USER_USB_Get_Interface();
		vUSB_length = 1;
		vUSB_DataToCopy = _vUSB_Report;
		return;
	}

Req_Error:
	Non_Standard_Request;
}	// End of "Device->Host, Standard, Interface-Recipient"

/*****************************************************************************
ROUTINE NAME : RequestType_0x02
DESCRIPTION  : Standard process for request of (H->D, Standard, Endpoint)
*****************************************************************************/
void RequestType_0x02(void)		// Host->Device, Standard, Endpoint-Recipient
{
//	_vUSB_StateMachine = WAIT_STATUS_IN;	// Expect Status IN follow the Setup Stage

	if (DADDR == 0 || vUSB_Configuration == 0)	// Not on configured state
		goto Req_Error;

	if ((sUSB_vSetup.Flag & (NON0_wValue | NON0_wLength | NON0_wIndex)) != NON0_wIndex0)
		goto Req_Error;						// (wValue | wIndex1 | wLength) != 0 || wIndex0 == 0

	if (sUSB_vSetup.USBbRequest == CLEAR_FEATURE) {
		_USB_Clr_Feature_EP();
		USER_USB_Clear_Feature_EP();
		return;
	}
	else if (sUSB_vSetup.USBbRequest == SET_FEATURE) {
		_USB_Set_Feature_EP();
		USER_USB_Set_Feature_EP();
		return;
	}

Req_Error:
	Non_Standard_Request;
}	// End of "Host->Device, Standard, Endpoint-Recipient"

/*****************************************************************************
ROUTINE NAME : RequestType_0x82
DESCRIPTION  : Standard process for request of (D->H, Standard, Endpoint)
*****************************************************************************/
void RequestType_0x82(void)		// Device->Host, Standard, Endpoint-Recipient
{
//	char EP_State;

//	_vUSB_StateMachine = IN_DATA;	// Expect Data IN stage follow the Setup Stage

	if (sUSB_vSetup.USBbRequest != GET_STATUS)
		goto Req_Error;

	if (DADDR == 0)
		goto Req_Error;						// Error on default state
	
	if ((sUSB_vSetup.Flag & (NON0_wValue | TWO_wLength)) != TWO_wLength)
		goto Req_Error;						// wValue != 0 || wLength != 2

	if (vUSB_Configuration == 0 && // (sUSB_vSetup.USBwIndex & ~0x80))
			(sUSB_vSetup.USBwIndex0 || (sUSB_vSetup.USBwIndex1 & ~0x80)))
		goto Req_Error;						// Error on address state if it is not EP0

	switch (_USB_GetEPStatus(sUSB_vSetup.USBwIndex0)) {
	case EP_DISABLE:
	case REQ_UNSUPPORT:
Req_Error:
		Non_Standard_Request;
	case EP_STALL:
		_vUSB_Report[0] = 1;
		break;
	default:
		_vUSB_Report[0] = 0;
		break;
	}
	_vUSB_Report[1] = 0;

	vUSB_length = 2;          
	vUSB_DataToCopy = _vUSB_Report;
}	// End of "Device->Host, Standard, Endpoint-Recipient"

/*****************************************************************************
ROUTINE NAME : _USB_CopydataIN
DESCRIPTION  : Copy data to endpoint transmit buffer
			Adjust the pointer and counter
*****************************************************************************/
void _USB_CopydataIN(void)
{
	static unsigned char TransferLength;

	if (vUSB_length >= MAX_PACKET_SIZE) {
		if (vUSB_length == MAX_PACKET_SIZE)
			_vUSB_StateMachine = ONE_MORE_IN;
		TransferLength = MAX_PACKET_SIZE;
	}
	else
		TransferLength = vUSB_length;

	CNT0TXR = TransferLength;
	vUSB_length -= TransferLength;
	if (vUSB_DataToCopy) {		// Data from the library
		asm {
			LD		X, TransferLength
			DEC		X
	_Copy_Byte:
			LD		A, ([vUSB_DataToCopy.w],X)
			LD		(EP0_IN,X), A
			DEC		X
			JRPL	_Copy_Byte
		}
		vUSB_DataToCopy += TransferLength;
	}
	else {					// Data from user space
		USER_USB_CopydataIN(TransferLength);
	}
	vUSB_offset += CNT0TXR;
}


/*****************************************************************************
ROUTINE NAME : _USB_CopydataOUT
DESCRIPTION  : Copy data from endpoint receive buffer to user space
			Adjust the pointer and counter
			Return error if the received data length is too long
*****************************************************************************/
char _USB_CopydataOUT(void)
{
	if (_vUSB_Data_OUT_Len > vUSB_length)
		return REQ_ERROR;

	USER_USB_CopydataOUT(_vUSB_Data_OUT_Len);

	vUSB_offset += _vUSB_Data_OUT_Len;
	vUSB_length -= _vUSB_Data_OUT_Len;
	if (vUSB_length == 0)
		_vUSB_StateMachine = ONE_MORE_OUT;

	return (REQ_SUCCESS);
}
 
/*****************************************************************************
ROUTINE NAME : Init_USB
DESCRIPTION  : Initialize the USB Macrocell
*****************************************************************************/
void Init_USB(void)
{
	_vUSB_StateMachine = 0;
	vUSB_Current_Feature = 0;
	vUSB_Configuration = 0;

	_EP_RxTx_Flag = 0;		// Clear flag for non-control EPs

#ifdef	USB_POLLING_MODEL
	// In polling model, set USB as highest interrupt level 3
	// All the other interrupt source are set as level 1
	ITSPR0 = 0x5D;
#else
	// In interrupt model, set USB as highest interrupt level 2
	// All the other interrupt source are set as level 1
	ITSPR0 = 0x51;
#endif
	ITSPR1 = 0x55;
	ITSPR2 = 0x55;
	ITSPR3 = 0x55;

#ifndef LOW_SPEED
	USBCTLR = 0x00;			// No-force USB reset and Power on 3.3V regulator
#else
	USBCTLR = 0x10;			// No-force USB reset and Power on 3.3V regulator
#endif
	USBISTR = 0;			// Clear pending flags
	USBIMR = IMR_RESET;		// Enable USB reset interrupt only
}

/*****************************************************************************
ROUTINE NAME : USB_Reset
DESCRIPTION  : Called when USB cell is reset
*****************************************************************************/
void USB_Reset(void)
{
	Init_USB();

	DADDR = 0x00;				// Enable this device at USB address zero
	CNT0RXR = MAX_PACKET_SIZE;	// Receive data on endpoint 0
	CNT0TXR = 0;
	EP0R = 0x01;				// TX disable, RX Stall (only SETUP token will be ACKed)

	USER_USB_Reset();
	USBIMR = vUSB_Interrupt_Mask;
}

/*****************************************************************************/
// EP_RXR_Adr has the address of all EP?RXR register.
// The value is 0x00 if the EP?RXR is not exist.
#pragma CONST_SEG	USB_CONST
const unsigned char *const near EP_RXR_Adr[6] = {
	&EP0R,
#ifdef	MCU_ST7265
	&EP1RXR,
#else
	0x00,
#endif
	&EP2RXR,
	0x00,
	0x00,
	0x00
};

// EP_TXR_Adr has the address of all EP?TXR register.
// The value is 0x00 if the EP?TXR is not exist.
const unsigned char *const near EP_TXR_Adr[6] = {
	0x00,
	&EP1TXR,
	&EP2TXR,
#ifdef	MCU_ST7265
	0x00,
	0x00,
	0x00
#else
	&EP3TXR,
	&EP4TXR,
	&EP5TXR
#endif
};

/*****************************************************************************
ROUTINE NAME : _USB_GetEPStatus
INPUT/OUTPUT :
	Endpoint: The endpoint number which the status is going to return
			  Rx status is returned if bit7 = 0
			  Tx status is returned if bit7 = 1
DESCRIPTION  : Used to get the current Status of Endpoint
*****************************************************************************/
#pragma NO_ENTRY
char _USB_GetEPStatus(char Endpoint)
{
	asm {
		TNZ		A
		JRMI	Tx_Status
//		JRNE	Non_rEP0
//		LD		A, EP0R
//		JRT		EP_Status
//Non_rEP0:
		CP		A, #5
		JRUGT	No_EPs
		LD		X, A
		LD		X, (EP_RXR_Adr,X)
		JREQ	No_EPs
gEP_Status:
		LD		A, (X)
EP_Status:
		AND		A, #0x03
		RET

Tx_Status:
		AND		A, #0x7F
		JRNE	Non_tEP0
		LD		A, EP0R
		SWAP	A
		JRT		EP_Status
Non_tEP0:
		CP		A, #5
		JRUGT	No_EPs
		LD		X, A
		LD		X, (EP_TXR_Adr,X)
		JRNE	gEP_Status
No_EPs:
	}
	return REQ_UNSUPPORT;
}

/*****************************************************************************
ROUTINE NAME : _USB_Clr_Feature_EP
DESCRIPTION  : Clear STALL condition of the specified EP if it is not DISABLED
			The specified EP is given in sUSB_vSetup.USBwIndex0
*****************************************************************************/
void _USB_Clr_Feature_EP(void)
{
	asm {
		LD		A, sUSB_vSetup.USBwIndex0
		AND		A, #0x7F
		CP		A, #6
		JRUGE	EP_Error
		LD		X, A
		LD		A, sUSB_vSetup.USBwIndex0
		JRMI	Clear_Feature_Tx
		LD		X, (EP_RXR_Adr, X)
		JRT		Set_Feature
EP_Error:
		JP		RequestError
Clear_Feature_Tx:
		LD		X, (EP_TXR_Adr, X)
Set_Feature:
		JREQ	EP_Error				// Error: if the EP is not exist
		LD		A, (X)
		AND		A, #0x03
		JREQ	EP_Error				// Error: if the EP is EP_DISABLE
		CP		A, #EP_STALL
		JRNE	End_Clear				// Return if the EP is not in STALL
		LD		A, (X)
		AND		A, #0xF8				// Clear the data toggling bit to 0
		OR		A, #EP_NAK				// Change the endpoint state to NAK
		LD		(X), A
End_Clear:
	}
}

/*****************************************************************************
ROUTINE NAME : USB_Set_Feature_EP
INPUT/OUTPUT : None
DESCRIPTION  : Set STALL condition of the specified EP if it is not DISABLED
			The specified EP is given in sUSB_vSetup.USBwIndex0
*****************************************************************************/
void _USB_Set_Feature_EP(void)
{
	asm {
		LD		A, sUSB_vSetup.USBwIndex0
		AND		A, #0x7F
		CP		A, #6
		JRUGE	EP_Error
		LD		X, A
		LD		A, sUSB_vSetup.USBwIndex0
		JRMI	Set_Feature_Tx
		LD		X, (EP_RXR_Adr, X)
		JRT		Set_Feature
EP_Error:
		JP		RequestError
Set_Feature_Tx:
		LD		X, (EP_TXR_Adr, X)
Set_Feature:
		JREQ	EP_Error				// Error: if the EP is not exist
		LD		A, (X)
		BCP		A, #0x03
		JREQ	EP_Error				// Error: if the EP is EP_DISABLE
		AND		A, #0xFC
		OR		A, #EP_STALL
		LD		(X), A
	}
}

/*****************************************************************************
ROUTINE NAME : USB_Remote_Wakeup
INPUT/OUTPUT : return SUCCESS if device is in suspend state, ERROR instead
DESCRIPTION  : Send a resume signal within 3ms for 8MHz CPU clock, 6ms for 4MHz
				12ms for 2MHz. At least the resume signal must be longer 
				than 1ms and lower than 15ms
*****************************************************************************/
char USB_Remote_Wakeup(void)
{
	if ((USBCTLR & 0x02) && (vUSB_Current_Feature & 0x02)) {	// SUSPEND MODE and RWU enabled
		USBCTLR &= ~0x02;	// Exit from SUSPEND mode
		USBCTLR |= 0x08; 	// Force RESUME signal

		// Loop to keep the Resume signal
		// CPU frequency = 6MHz => Resume signal active for about 8.2ms
		asm {
			LD		X, #32
			CLR		A
loop_delay:	INC		A
			JRNE	loop_delay
			DEC		X
			JRNE	loop_delay
		}
			
		USBCTLR &= ~0x08; // stop RESUME signal

		return 	REQ_SUCCESS;
	}
	else
		return	REQ_ERROR;
}


/**************** (c) 2000  STMicroelectronics *******************************/
