/**************** (c) 2000  STMicroelectronics *******************************

NAME:		usb_lib.h
PROJECT:	USB - ST7 FULL SPEED
VERSION:	v 1.0
CREATION:	02/01/2002
AUTHOR:		MICROCONTROLLER DIVISION / ST Rousset

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-
		All definitions and all functions from the USB library
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

MODIFICATIONS:

******************************************************************************/

#ifndef	USBLIB_DEFINE_H
#define USBLIB_DEFINE_H

/*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*/
//				Constants
/*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*/
#define NULL (void*) 0

/*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*/
//				Structures
/*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*/

typedef union {
	unsigned short w;
	struct BW 	{
		char	bb1;
		char 	bb0;
	} bw;
}WORD_BYTE;

typedef struct USB_vSetup {
	unsigned char  USBbmRequestType;
	unsigned char  USBbRequest;
	WORD_BYTE  USBwValues;
	WORD_BYTE  USBwIndexes;
	WORD_BYTE  USBwLengthes;
	unsigned char	Flag;
} _USB_VSETUP;

// Defines for Flag of USB_vSetup
#define NON0_wValue1	0x01
#define NON0_wValue0	0x02
#define NON0_wIndex1	0x04
#define NON0_wIndex0	0x08
#define NON0_wLength1	0x10
#define NON0_wLength0	0x20
#define ONE_wLength		0x40
#define TWO_wLength		0x80
#define NON0_wValue		(NON0_wValue1 | NON0_wValue0)
#define NON0_wIndex		(NON0_wIndex1 | NON0_wIndex0)
#define NON0_wLength	(NON0_wLength1 | NON0_wLength0)
// Meaning of the field 'Flag' of USB_vSetup
//	+----+----+----+----+----+----+----+----+
//	| D7 | D6 | D5 | D4 | D3 | D2 | D1 | D0 |
//	+----+----+----+----+----+----+----+----+
//     |    |    |    |    |    |    |    ^---- Set if high byte of wValue is not zero
//     |    |    |    |    |    |    +--------- Set if low byte of wValue is not zero
//     |    |    |    |    |    +-------------- Set if high byte of wIndex is not zero
//     |    |    |    |    +------------------- Set if low byte of wIndex is not zero
//     |    |    |    +------------------------ Set if high byte of wLength is not zero
//     |    |    +----------------------------- Set if low byte of wLength is not zero
//     |    +---------------------------------- Set if wLength is 1 (one)
//     +--------------------------------------- Set if wLength is 2 (two)

// Defines for USB_vSetup structure 
#define USBwValue	USBwValues.w
#define USBwValue0	USBwValues.bw.bb0
#define USBwValue1	USBwValues.bw.bb1

#define USBwIndex	USBwIndexes.w
#define USBwIndex0	USBwIndexes.bw.bb0
#define USBwIndex1	USBwIndexes.bw.bb1

#define USBwLength 	USBwLengthes.w
#define USBwLength0	USBwLengthes.bw.bb0
#define USBwLength1	USBwLengthes.bw.bb1

/*_vUSB_StateMachine selector*/
#define WAIT_SETUP			0x00		// The USB cell is waiting for a SETUP on EP0
#define SETTING_UP			0x01		// The USB Cell is computing the SETUP request
#define IN_DATA				0x02		// Wait for a IN stage to send datas
#define OUT_DATA			0x03		// Wait for a OUT stage to receive datas
#define WAIT_STATUS_IN		0x04		// Wait for a IN status stage in to close the OUT data stage
#define WAIT_STATUS_OUT		0x05		// Wait for a OUT status stage in to close the IN data stage
#define ONE_MORE_IN			0x12		// Last DATA IN Transfer before STATUS OUT.
#define ONE_MORE_OUT		0x13		// Last DATA IN Transfer before STATUS OUT.
#define ADDRESS2SET			0x06		// Device address
#define STATE_ERROR			0x07		// Error on Request
#define MACHINESTATE		0x0F		// STATE MACHINE STATUS MASK
//---------------------------------------------------------------------------

typedef struct OneDescriptor {
	char *Descriptor;
	unsigned short Size;
} ONE_DESCRIPTOR;
 
/*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*/
//				Variables in page 0
/*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*/
#pragma DATA_SEG SHORT USB_RAM0		// Map USB variables into page 0 (short addressing RAM)
#define EV_EP1_IN		0
#define EV_EP1_OUT		1
#define EV_EP2_IN		2
#define EV_EP2_OUT		3
#define EV_EP3_IN		4
#define EV_EP4_IN		5
#define EV_EP5_IN		6
extern unsigned char _EP_RxTx_Flag;

#ifdef	LARGE_EP0
extern unsigned short		vUSB_offset;
extern unsigned short		vUSB_length;
#else
extern unsigned char		vUSB_offset;
extern unsigned char		vUSB_length;
#endif
extern unsigned char		*vUSB_DataToCopy;
extern _USB_VSETUP sUSB_vSetup;

extern unsigned char		vUSB_Current_Feature;   
// D7 : Reserved
// D6 : Self-powered
// D5 : Remote-wakeup able or not
// D4-D2 : Unused
// D1 : Remote Wakeup enabled
// D0 : Unused

extern unsigned char		vUSB_Configuration;	// Current configuration value

/*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*/
//				Variables in other page
/*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*/
#pragma DATA_SEG USB_RAM
extern char vUSB_Num_Configuration;		// number of configuration
extern char vUSB_Num_Interface;			// number of interface
extern char vUSB_Num_Strings;			// number of string descriptor
extern char vUSB_Interrupt_Mask;		// USB interrupt mask

/*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*/
//				Functions
/*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*/
#define Enable_SOF()	{ USBIMR |= IMR_SOF; }
#define Disable_SOF()	{ USBIMR &= ~IMR_SOF; }

extern void USB_Reset(void);
extern void RequestError(void);   

#ifdef	MCU_ST7HUB
void Init_HUB(void);
#else
void Init_USB(void);
#endif

#ifdef	USB_POLLING_MODEL
void USB_Polling(void);
#endif

#define USB_GetRx0Status()	(EP0R & 0x03)
#ifdef	MCU_ST7265
#define USB_GetRx1Status()	(EP1RXR & 0x03)
#endif
#define USB_GetRx2Status()	(EP2RXR & 0x03)

#define	USB_GetTx0Status()	((EP0R & 0x30)>>4)
#define	USB_GetTx1Status()	(EP1TXR & 0x03)
#define USB_GetTx2Status()	(EP2TXR & 0x03)
#ifdef	MCU_ST7SCR
#define USB_GetTx3Status()	(EP3TXR & 0x03)
#define USB_GetTx4Status()	(EP4TXR & 0x03)
#define USB_GetTx5Status()	(EP5TXR & 0x03)
#endif
//------------ Macros for set status of reception endpoints -------------------
// INPUT        : EP_DISABLE, EP_STALL, EP_VALID or EP_NAK
//-----------------------------------------------------------------------------
#ifdef MCU_ST7265
#define USB_SetRxEP1Status(Status) { if (Status == EP_VALID) EP1RXR |= EP_VALID; else EP1RXR = (EP1RXR & ~0x03) | Status; }
#endif
#define USB_SetRxEP2Status(Status) { if (Status == EP_VALID) EP2RXR |= EP_VALID; else EP2RXR = (EP2RXR & ~0x03) | Status; }

//------------ Macros for set status of transmission endpoints ----------------
// INPUT        : EP_DISABLE, EP_STALL, EP_VALID or EP_NAK
//-----------------------------------------------------------------------------
#define USB_SetTxEP1Status(Status) { if (Status == EP_VALID) EP1TXR |= EP_VALID; else EP1TXR = (EP1TXR & ~0x03) | Status; }
#define USB_SetTxEP2Status(Status) { if (Status == EP_VALID) EP2TXR |= EP_VALID; else EP2TXR = (EP2TXR & ~0x03) | Status; }
#ifdef MCU_ST7SCR
#define USB_SetTxEP3Status(Status) { if (Status == EP_VALID) EP3TXR |= EP_VALID; else EP3TXR = (EP3TXR & ~0x03) | Status; }
#define USB_SetTxEP4Status(Status) { if (Status == EP_VALID) EP4TXR |= EP_VALID; else EP4TXR = (EP4TXR & ~0x03) | Status; }
#define USB_SetTxEP5Status(Status) { if (Status == EP_VALID) EP5TXR |= EP_VALID; else EP5TXR = (EP5TXR & ~0x03) | Status; }
#endif
                                        
/*-----------------------------------------------------------------------------
ROUTINE NAME : USB_isTxStatus
INPUT/OUTPUT : Endpoint number (EPn), Status to test (Status) 
DESCRIPTION  : return 1 if "EPn" is at "Status", 0 if not
-----------------------------------------------------------------------------*/
#ifndef MCU_ST7SCR
#define USB_isTxStatus(EP, Status) \
((EP) > 2) ? 0 : \
((EP) == 0) ? ((EP0R & 0x30) == (Status << 4)) : \
(((((EP) == 1) ? EP1TXR : EP2TXR) & 0x03) == Status)
#else
#define USB_isTxStatus(Endpoint, Status) \
(Endpoint == 0) ? ((EP0R & 0x30) == (Status << 4)) : \
(Endpoint == 1) ? ((EP1TXR & 0x03) == Status) : \
(Endpoint == 2) ? ((EP2TXR & 0x03) == Status) : \
(Endpoint == 3) ? ((EP3TXR & 0x03) == Status) : \
(Endpoint == 4) ? ((EP4TXR & 0x03) == Status) : \
(Endpoint == 5) ? ((EP5TXR & 0x03) == Status) : 0
#endif	

/*-----------------------------------------------------------------------------
ROUTINE NAME : USB_isRxStatus
INPUT/OUTPUT : Endpoint number (EPn), Status to test (Status) 
DESCRIPTION  : return 1 if "EPn" is at "Status", 0 if not
-----------------------------------------------------------------------------*/
#ifdef MCU_ST7265
#define USB_isRxStatus(EP, Status) \
((EP) > 2) ? 0 : \
((EP) == 0) ? ((EP0R & 0x03) == Status ) :  \
(((((EP) == 1) ? EP1RXR : EP2RXR) & 0x03) == Status )
#else
#define USB_isRxStatus(Endpoint, Status) \
(Endpoint == 0) ? ((EP0R & 0x03) == Status ) : \
(Endpoint == 2) ? ((EP2RXR & 0x03) == Status ) : 0
#endif

#endif	// USBLIB_DEFINE_H
