#include "mcu_conf.h"
#include "usb_def.h"
#include "usb_reg.h"
#include "usb_Lib.h"

#include "user_usb.h"
#include "app_usb.h"
#include "DFU.h"

#include "flashing.h"


#pragma DATA_SEG DFU_RAM

// Define some BlockNum of DNLOAD for some special meaning
//#define ST_PASSWORD		0xFFF0	// To enable the download feature
#define ST_ERASE_S1		0xFFFD		// To erase the sector 1
#define ST_ERASE_S2		0xFFFE		// To erase the sector 2
#define ST_ERASE_ALL	0xFFFF		// To erase sector 1 & 2
#define BLOCK_SIZE		8

unsigned char DFU_State;

#ifdef	ST_PASSWORD
unsigned char DFU_Capability;

typedef struct Suffix {
	unsigned int bcdDevice;
	unsigned int idProduct;
	unsigned int idVendor;
	unsigned int bcdDFU;
	unsigned char ucDfuSignature[3];
	unsigned char bLength;
	unsigned long dwCRC;
} DFU_SUFFIX;
#define DFU_Suffix	((DFU_SUFFIX*)&Buffer_0)
#else
#define DFU_Capability DFU_ConfigDescriptor[20]
#endif


#define Setup_Timer(time)	{	\
	TCR1 = 0x00;	/* disable all interrupt */	\
	TCR2 = 0x08;	/* One tick is 1us */		\
	CRL = 0x00;		/* Reset the free run counter */\
	asm LD	A, TSR;	/* clear OCF1 */	\
	OCHR1 = time >> 8;	\
	OCLR1 = (unsigned char)time; }

extern unsigned char SMC_Lookup_Table0[1024];
#define DFU_Buffer SMC_Lookup_Table0

#pragma CODE_SEG DFU_CODE
void Replug_Device()
{
	asm {		// Delay 0.7s
		LD		Y, #10
		CLR		X
		CLR		A
delay:	INC		A
		JRNE	delay
		INC		X
		JRNE	delay
		DEC		Y
		JRNE	delay
	}
	USBCTLR |= 0x04;		// Power down 3.3V
	asm {		// Delay 3s
		LD		Y, #50
		CLR		X
		CLR		A
delay1:	INC		A
		JRNE	delay1
		INC		X
		JRNE	delay1
		DEC		Y
		JRNE	delay1
	}
	USBCTLR &= ~0x04;		// Power on 3.3V again
}

unsigned int CRC_ROM(void)
{	// Calculate CRC value for 2048 bytes in the ROM
	// Pick up one byte from every 16 bytes
	// This function takes about 121ms
	asm {
		LD		A, #0x80
		LD		_LEX, A
		CLR		_LEX:1			// Get the ROM pointer of 0x8000
		LD		X, #0xFF
		LD		Y, #0xFF

crc_byte:
		LD		A, [_LEX.w]
		LD		_SEX, A			// Save CRC_data
		LD		A, #8
		LD		_R_Z, A			// Count 8 bits in a byte

shift_out_bit:
		SRL		_SEX			// shift out lower bit
		LD		A, X
		JRNC	one_bit_out
			// if the lower bit is 1, then XOR to high order bit of CRC_word
		XOR		A, #0x80
		LD		X, A

one_bit_out:
		TNZ		A
		JRPL	no_polynomial

// if the result of above XOR is 1
// then the remainder is XORed with the generator polynomial 0x8055
		LD		A, Y
		SLL		A				// shift the remainder one bit left
		XOR		A, #0x05
		LD		Y, A
		LD		A, X
		RLC		A				// lower bit set to 0 and discard higher bit
		XOR		A, #0x80
		LD		X, A
		JRA		more_bit_out

no_polynomial:
		SLL		Y			// shift the remainder one bit left
		RLC		X			// lower bit set to 0 and discard higher bit

more_bit_out:
		DEC		_R_Z
		JRNE	shift_out_bit

		LD		A, _LEX:1
		ADD		A, #0x10
		LD		_LEX:1, A
		JRNE	crc_byte
		INC		_LEX
		JRNE	crc_byte

		LD		A, Y		// return the result in X:A
		CPL		A
		CPL		X
	}
}

unsigned int CheckSum_ROM(void)
{	// Calculate Check Sum value for 2048 bytes in the ROM
	// Pick up one byte from every 16 bytes
	// This function takes about 10ms
	asm {
		LD		A, #0x80
		LD		_LEX, A
		CLR		_LEX:1			// Get the ROM pointer of 0x8000
		CLR		X
		CLR		Y

cksm_byte:
		LD		A, X
		ADD		A, [_LEX.w]
		LD		X, A
		JRNC	next_byte
		INC		Y
next_byte:
		LD		A, _LEX:1
		ADD		A, #0x10
		LD		_LEX:1, A
		JRNE	cksm_byte
		INC		_LEX
		JRNE	cksm_byte
		LD		A, X
		LD		X, Y
	}	
/*
// Calculate Check Sum value for 32KB in the ROM
// This function takes about 100ms
		LD		A, #0x80
		LD		_LEX, A
		CLR		A
		LD		_LEX:1, A
		LD		X, A
cksm_byte:
		ADD		A, [_LEX.w]
		JRNC	next_byte
		INC		X
next_byte:
		INC		_LEX:1
		JRNE	cksm_byte
		INC		_LEX
		JRNE	cksm_byte
*/
}

void Flashing_Start(void)
{
	asm	SIM;					// DisableInterrupts;
	RASS_Disable(0x56, 0xAE);	// Unlock FCSR register
	if ( PEDR & 0x10 ) {
		PEDR &= ~0x10;			// Power on 12V
		Setup_Timer(1500);
		while (!(TSR & 0x40));	// Wait 12V stable
	}
}

void Flashing_End(void)
{
	PEDR |= 0x10;				// Turn off 12V
	asm RIM;					// EnableInterrupts;
}

void DFU_Manifest(void)
{	// Calculate the CRC value for all ROM area and save the CRC word
	DFU_Timeout = CheckSum_ROM();
//	DFU_Timeout = CRC_ROM();

	Flashing_Start();

	// Program this value to X=>0xEFFE & A=>0xEFFF
	HDFlashWriteByte((unsigned char)DFU_Timeout, 0xEFFF, FCPU);			// High byte
	HDFlashWriteByte((unsigned char)(DFU_Timeout >> 8), 0xEFFE, FCPU);	// Low  byte

	Flashing_End();
}

// To validate the applicatoin ROM sector is good
// Return 0 = application ROM sector is not valid
//		non-0 = application ROM sector is valid
unsigned char DFU_App_Valid()
{
	// return valid for testing
//	return 0;

	asm {		// Get saved CRC value to DFU_Timeout
		LD		A, 0xEFFE
		LD		DFU_Timeout, A
		LD		A, 0xEFFF
		LD		DFU_Timeout:1, A
	}

	return (DFU_Timeout == CheckSum_ROM());
//	return (DFU_Timeout == CRC_ROM());
}

void DFU_Init(void)
{
	// Prepare DFU descriptor set
	DeviceDescriptor.Descriptor = DFU_DeviceDescriptor;
	DeviceDescriptor.Size = 18;

	ConfigDescriptor[0].Descriptor = DFU_ConfigDescriptor;
	ConfigDescriptor[0].Size = SIZE_DFU_CONFIG_DESC;

	StringDescriptor[0].Descriptor = DFU_StringDescriptor;
	StringDescriptor[0].Size = SIZE_DFU_STRING_DESCRIPTOR;
	StringDescriptor[1].Descriptor = DFU_StringVendor;
	StringDescriptor[1].Size = SIZE_DFU_STRING_VENDOR;
	StringDescriptor[2].Descriptor = DFU_StringProduct;
	StringDescriptor[2].Size = SIZE_DFU_STRING_PRODUCT;
	StringDescriptor[3].Descriptor = DFU_StringInterface;
	StringDescriptor[3].Size = SIZE_DFU_STRING_INTERFACE;

	// Initialize USB library variables
	vUSB_Num_Configuration = 1;
	vUSB_Num_Interface = 1;
	vUSB_Num_Strings = 4;
	vUSB_Interrupt_Mask = USBIMR = IMR_CTR | IMR_SOVR | IMR_RESET | IMR_ERR;

	DFU_Request = 0;
	DFU_Action = 0;
#ifdef	ST_PASSWORD
	DFU_Capability = DFU_ConfigDescriptor[20] & ~CAP_bitCanDnload;
#endif

	// Reset all I/O ports
	PADDR = 0;
	PAOR = 0;		// Port A
	PBDDR = 0;		// Port B
	PCDDR = 0;
	PCOR = 0;		// Port C
	PDDDR = 0;
	PDOR = 0;		// Port D
	PEDR = 0x10;	// Turn off VPP-12V
	PEDDR = 0x10;
	PEOR = 0x10;	// Port E
	PFDDR = 0;
}

void DFU_Status_In(void)
{
	switch (DFU_Request) {
	case DFU_DNLOAD | DFU_REQUEST:
		if (DFU_Action == DFU_DNLOADLAST)
			DFU_State = STATE_dfuMANIFESTSYNC;
		else {
#ifdef	ST_PASSWORD
			if (DFU_BlockNum == ST_PASSWORD) {
				// Check the DFU suffix to ensure that the firmware is for this device
				if (DFU_Suffix->bLength != sizeof(DFU_SUFFIX)) {
errTarget:			DFU_State = STATE_dfuERROR;
					DFU_Status.bStatus = DFU_STATUS_ERRTARGET;
					DFU_Capability &= ~CAP_bitCanDnload;	// Disable the download feature
					return;
				}
				if (DFU_Suffix->idVendor != *((unsigned int*)DFU_DeviceDescriptor+8))
					goto errTarget;
				if (DFU_Suffix->idProduct != *((unsigned int*)DFU_DeviceDescriptor+10))
					goto errTarget;

				// Enable the download if the descriptor says so
				if (DFU_ConfigDescriptor[20] & CAP_bitCanDnload)
					DFU_Capability |= CAP_bitCanDnload;
			}
#endif
			DFU_State = STATE_dfuDNLOADSYNC;
		}
		return;
	case DFU_DETACH | DFU_REQUEST:

	Replug_Device();// This is not DFU compliance, but MS asks to do it
	// Following lines are DFU compliance
	//	while (1) {
	//		Setup_Timer(0xBB80);
	//		while (!(TSR & 0x40));	// Wait 64ms
	//		if ( DFU_Timeout == 0)
	//			break;
	//		DFU_Timeout--;
	//	}

		while (1);

	default:
		break;
	}
}

void DFU_Status_Out(void)
{
	switch (DFU_Request) {
	case DFU_UPLOAD | DFU_REQUEST:
		if (DFU_Action == DFU_UPLOADLAST)
			DFU_State = STATE_dfuIDLE;		// go back to idle state
		return;
	case DFU_GETSTATUS | DFU_REQUEST:
		switch (DFU_Action) {
		case DFU_DNLOAD:			// We just receive the data
			Flashing_Start();

			// Erase Sector 1
			if (DFU_BlockNum == ST_ERASE_S1 || DFU_BlockNum == ST_ERASE_ALL) {
				HDFlashEraseSector(1, FCPU);
				if (Flash_Status != HDFLASH_ERASE_OK) {
					DFU_Status.bStatus = DFU_STATUS_ERRERASE;
					DFU_State = STATE_dfuERROR;
					DFU_Action = DFU_ERRORACTION;
					goto stop_flashing;
				}
			}
			// Erase Sector 2
			if (DFU_BlockNum == ST_ERASE_S2 || DFU_BlockNum == ST_ERASE_ALL) {
				HDFlashEraseSector(2, FCPU);
				if (Flash_Status != HDFLASH_ERASE_OK) {
					DFU_Status.bStatus = DFU_STATUS_ERRERASE;
					DFU_State = STATE_dfuERROR;
					DFU_Action = DFU_ERRORACTION;
					goto stop_flashing;
				}
			}

			// Program data to the flash memory
			if (DFU_BlockNum >= 0x8000 && DFU_BlockNum < 0xF000) {
#ifdef	PROG_BYTES
				unsigned char ichr;
				for (ichr = 0; ichr < DFU_BlockSiz; ichr++, DFU_BlockNum++) {
					HDFlashWriteByte(DFU_Buffer[ichr], DFU_BlockNum, FCPU);
					if (Flash_Status != HDFLASH_BYTEPROG_OK) {
						DFU_Status.bStatus = DFU_STATUS_ERRPROG;
						DFU_State = STATE_dfuERROR;
						DFU_Action = DFU_ERRORACTION;
						break;
					}
				}
#else
				HDFlashWriteBlock(DFU_Buffer, DFU_BlockNum, DFU_BlockSiz, FCPU);            
				if (Flash_Status != HDFLASH_BLOCKPROG_OK) {
					DFU_Status.bStatus = DFU_STATUS_ERRPROG;
					DFU_State = STATE_dfuERROR;
					DFU_Action = DFU_ERRORACTION;
				}
#endif
			}
			DFU_State = STATE_dfuDNLOADSYNC;
			DFU_Action = DFU_DNLOADSYNC;

stop_flashing:
			Flashing_End();
			return;

		case DFU_DNLOADSYNC:		// Received data has programmed to the flash
			DFU_Action = DFU_DNLOADIDLE;
			return;

		case DFU_DNLOADLAST:		// All the blocks have been programmed
			// calculate the CRC or the checksum of the programmed code
			// program the result to the flash to indicate the application code is good
			DFU_Manifest();

			if ((DFU_Capability & CAP_bitManifestationTolerant)) {	// test bitManifestationTolerant
				DFU_State = STATE_dfuMANIFESTSYNC;
				DFU_Action = DFU_MANIFESTSYNC;
				return;
			}

			// if bitManifestationTolerant=0, we will wait for reset definitely
			DFU_State = STATE_dfuMANIFESTWAITRESET;

			EP0R = 0;		// disable all USB traffic
		Replug_Device();// This is not DFU compliance, but MS asks to do it
			while (1);		// Stop everything and wait for USB reset
		default:
			break;
		}
		return;

	default:
		return;
	}
}


unsigned char DFU_Setup()
{
	if (vUSB_Configuration == 0)
		goto Req_Err;

	switch (sUSB_vSetup.USBbmRequestType) {
	case REQ_TYPE_CLASS | RECIPIENT_INTERFACE | 0x80:
		if (sUSB_vSetup.Flag & (NON0_wIndex | NON0_wLength1))
			break;

		switch (sUSB_vSetup.USBbRequest) {
		case DFU_UPLOAD:
			if ((DFU_Capability & CAP_bitCanUpload) == 0)	// test bitCanUpload
				break;

			if (DFU_State != STATE_dfuIDLE && DFU_State != STATE_dfuUPLOADIDLE)
				break;

			DFU_Request = DFU_UPLOAD | DFU_REQUEST;
			DFU_State = STATE_dfuUPLOADIDLE;
			vUSB_length = sUSB_vSetup.USBwLength0;		// Upload buffer size
			DFU_BlockSiz = vUSB_length;
			if (vUSB_length == 0)
				break;
			if (vUSB_length < DFU_TRANSFERSIZE)
				DFU_Action = DFU_UPLOADLAST;
			else
				DFU_Action = DFU_UPLOAD;
			DFU_BlockNum = sUSB_vSetup.USBwValue;

			vUSB_DataToCopy = (unsigned char *)(DFU_BlockNum * BLOCK_SIZE);
			return REQ_SUCCESS;

		case DFU_GETSTATUS:
			if (sUSB_vSetup.Flag & NON0_wValue)
				break;
			if (sUSB_vSetup.USBwLength0 != sizeof(DFU_Status))
				break;
			if (DFU_State == STATE_appIDLE
				|| DFU_State == STATE_dfuDNBUSY || DFU_State == STATE_dfuMANIFEST)
				break;

			DFU_Request = DFU_GETSTATUS | DFU_REQUEST;
			if (DFU_Action == DFU_DNLOAD) {
				DFU_State = STATE_dfuDNBUSY;

				if (DFU_BlockNum == ST_ERASE_S1) {
					DFU_Status.bwPollTimeout[0] = 0x10;
					DFU_Status.bwPollTimeout[1] = 0x27;
					DFU_Status.bwPollTimeout[2] = 0x00;		// 10s = 10,000ms = 0x2710
				}
				else if (DFU_BlockNum == ST_ERASE_S2 || DFU_BlockNum == ST_ERASE_ALL) {
					DFU_Status.bwPollTimeout[0] = 0x00;
					DFU_Status.bwPollTimeout[1] = 0x3B;
					DFU_Status.bwPollTimeout[2] = 0x00;		// 15s = 15,000ms = 0x3A98
				}
				else {
					// The time to program 128 bytes
					DFU_Status.bwPollTimeout[0] = 20;
					DFU_Status.bwPollTimeout[1] = 0;
					DFU_Status.bwPollTimeout[2] = 0;		// 20ms
				}
			}
			else if (DFU_Action == DFU_DNLOADSYNC) {
				DFU_State = STATE_dfuDNLOADIDLE;

				// After a download block is programmed, goto DNLOAD-IDLE state
				DFU_Status.bwPollTimeout[0] = 0x01;
				DFU_Status.bwPollTimeout[1] = 0x00;
				DFU_Status.bwPollTimeout[2] = 0;		// 1ms
			}
			else if (DFU_Action == DFU_DNLOADLAST) {
				DFU_State = STATE_dfuMANIFEST;

				// The time to calculate the CRC or checksum and program the result
				DFU_Status.bwPollTimeout[0] = 20;
				DFU_Status.bwPollTimeout[1] = 0;
				DFU_Status.bwPollTimeout[2] = 0;		// 20ms
			}
			else if (DFU_Action == DFU_MANIFESTSYNC) {
				DFU_State = STATE_dfuIDLE;

				DFU_Status.bwPollTimeout[0] = 0x01;
				DFU_Status.bwPollTimeout[1] = 0x00;
				DFU_Status.bwPollTimeout[2] = 0;		// 1ms
			}
			else {
				// The time for normal status enquires
				DFU_Status.bwPollTimeout[0] = 0x01;
				DFU_Status.bwPollTimeout[1] = 0x00;
				DFU_Status.bwPollTimeout[2] = 0;		// 1ms
			}
			DFU_Status.iString = 0;
			DFU_Status.bState = DFU_State;

			vUSB_length = sizeof(DFU_Status);
			vUSB_DataToCopy = (unsigned char *)&DFU_Status;
			return REQ_SUCCESS;

		case DFU_GETSTATE:
			if (sUSB_vSetup.Flag & NON0_wValue)
				break;
			if (sUSB_vSetup.USBwLength0 != 1)
				break;

			if (DFU_State == STATE_appIDLE 
				|| DFU_State == STATE_dfuDNBUSY || DFU_State == STATE_dfuMANIFEST)
				break;

			DFU_Request = DFU_GETSTATE | DFU_REQUEST;
			vUSB_length = 1;
			vUSB_DataToCopy = &DFU_State;
			return REQ_SUCCESS;

		default:
			break;
		}
		break;

	case REQ_TYPE_CLASS | RECIPIENT_INTERFACE:

		switch (sUSB_vSetup.USBbRequest) {
		case DFU_ABORT:
			if (sUSB_vSetup.Flag & (NON0_wIndex | NON0_wValue | NON0_wLength))
				break;
			if (DFU_State == STATE_dfuIDLE
				|| DFU_State == STATE_dfuDNLOADIDLE || DFU_State == STATE_dfuUPLOADIDLE) {
				DFU_State = STATE_dfuIDLE;
				return REQ_SUCCESS;
			}
			break;

		case DFU_DNLOAD:
			if (sUSB_vSetup.Flag & (NON0_wIndex | NON0_wLength1))
				break;

			if (DFU_State != STATE_dfuIDLE && DFU_State != STATE_dfuDNLOADIDLE)
				break;

			if ((DFU_Capability & CAP_bitCanDnload) == 0)	// test bitCanDnload
				break;

			vUSB_length = sUSB_vSetup.USBwLength0;
			if (vUSB_length)
				DFU_Action = DFU_DNLOAD;
			else {
				if (DFU_State == STATE_dfuIDLE)	// It is not allowed in dfuIDLF
					break;
				DFU_Action = DFU_DNLOADLAST;
			}
			DFU_BlockSiz = vUSB_length;
			DFU_BlockNum = sUSB_vSetup.USBwValue;
#ifdef	ST_PASSWORD
			if (DFU_BlockNum == ST_PASSWORD) {		// Receiving the DFU suffix
				if (vUSB_length != sizeof(DFU_SUFFIX))
					break;
			}
			else
#endif
			if (DFU_BlockNum != ST_ERASE_S1 && DFU_BlockNum != ST_ERASE_S2 && DFU_BlockNum != ST_ERASE_ALL ) {
				DFU_BlockNum *= BLOCK_SIZE;	// Shift left 3 bits
				// Validate the range of the block address
				if (DFU_BlockNum < 0x8000 || DFU_BlockNum >= 0xF000) {
					DFU_Status.bStatus = DFU_STATUS_ERRADDRESS;
					goto Param_Err;
				}
			}

			DFU_Request = DFU_DNLOAD | DFU_REQUEST;
			vUSB_DataToCopy = DFU_Buffer;

			return REQ_SUCCESS;

		case DFU_CLRSTATUS:
			if (sUSB_vSetup.Flag & (NON0_wIndex | NON0_wValue | NON0_wLength))
				break;
			if (DFU_State == STATE_dfuERROR) {
				DFU_Status.bStatus = DFU_STATUS_OK;
				DFU_State = STATE_dfuIDLE;
				return REQ_SUCCESS;
			}
			break;

		case DFU_DETACH:
			if (sUSB_vSetup.Flag & (NON0_wIndex1 | NON0_wLength))
				break;
			if (sUSB_vSetup.USBwIndex0 == 0)
				break;

			DFU_State = STATE_appDETACH;
			DFU_Request = DFU_DETACH | DFU_REQUEST;
			asm {		// scale the timeout value
				LD		A, sUSB_vSetup.USBwValue0
				LD		X, sUSB_vSetup.USBwValue1
				SLA		A
				RLC		X
				RLC		A
				RLC		X
				RLC		A
//				BCP		A, #0xF8
//				JRC		inc_timeout
//				JREQ	save_timeout
//inc_timeout:	INC		X
//				JRNE	save_timeout
//				INC		A
save_timeout:	AND		A, #0x07
				LD		DFU_Timeout, A
				LD		DFU_Timeout:1, X
			}
			// DFU_Timeout = (sUSB_vSetup.USBwValue + 63) / 64;
			return REQ_SUCCESS;

		default:
			break;
		}
		break;

	default:
		break;
	}

	DFU_Status.bStatus = DFU_STATUS_ERRSTALLEDPKT; // Device stalled an unexpected request
Param_Err:
	DFU_State = STATE_dfuERROR;
	DFU_Action = DFU_ERRORACTION;

Req_Err:
	RequestError();
	return REQ_ERROR;
}

void DFU_CopyDataIN(unsigned char Length)
{
}

#pragma NO_ENTRY
void DFU_CopyDataOUT(unsigned char Length)
{
	asm LD		X, A;		// Save "Length"
	if (sUSB_vSetup.USBbRequest == DFU_DNLOAD) {
		asm {
			LD		A, vUSB_DataToCopy:1
			ADD		A, vUSB_offset
			LD		_LEX:1, A
			LD		A, vUSB_DataToCopy
			ADC		A, #0
			LD		_LEX, A
			DEC		X
Dnload_Copy:
			LD		A, (EP0_OUT, X)
			LD		([_LEX.w], X), A
			DEC		X
			JRPL	Dnload_Copy
		}
	}
}

#pragma TRAP_PROC //SAVE_REGS
void INT_Flash(void)
{
}
