/*** (c) STMicroelectronics ***************************************************
#
# PROJECT : HDFlash Driver
#
# COMPILER : COSMIC / HIWARE
#
# VERSION : 1.0
#
# DESCRIPTION : Drivers to program and erase HDFlash memory. 
#								                                                              
******************************************************************************/

#include "HDFlash.h"

#ifdef HIWARE
#include MAP_FILE
#endif

#ifdef HIWARE
#pragma CODE_SEG HDFLASH_DRIVER
#endif

#ifdef USE_TIMEOUT
extern void UserWhileErase(void);      // Call-back function
extern void UserWhileWriteBlock(void); // Call-back function
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : RASS_Disable
INPUT        : First & Second keys
OUTPUT       : None
DESCRIPTION  : Send two hardware keys to unlock the FCSR
-----------------------------------------------------------------------------*/     
void RASS_Disable(unsigned char key1, unsigned char key2)
{
  FCSR = key1; // First hardware key
  FCSR = key2; // Second hardware key
}

#ifdef USE_HDFLASH_ERASESECTOR
/*-----------------------------------------------------------------------------
ROUTINE NAME : HDFlashEraseSector
INPUT        : Sector Number, Frequency
OUTPUT       : Embedded Command Status
DESCRIPTION  : Erase a Sector
-----------------------------------------------------------------------------*/     
unsigned char HDFlashEraseSector(unsigned char Sector, unsigned char Freq)
{
  EMBLG_PARAM_ECMD = EMBLG_PARAM_PTRL = 0x03;   // Command Number
  EMBLG_PARAM_SECT = EMBLG_PARAM_PTRH = Sector; // Sector Number
  EMBLG_PARAM_FREQ = Freq;                      // Frequency
  do {
    FCSR = 0x00;                                // Launch command
#ifdef USE_TIMEOUT
    UserWhileErase();                           // Call-back function
#endif
  }        
  while (EMBLG_PARAM_ECMD & HDFLASH_BUSY);      // Wait End of Operation
  return EMBLG_PARAM_ECMD;                      // Return Embedded Command Status
}
#endif

#ifdef USE_HDFLASH_WRITEBYTE
/*-----------------------------------------------------------------------------
ROUTINE NAME : HDFlashWriteByte
INPUT        : Data, Address, Frequency
OUTPUT       : Embedded Command Status
DESCRIPTION  : Write one byte
-----------------------------------------------------------------------------*/     
unsigned char HDFlashWriteByte(unsigned char Data, unsigned int Address, unsigned char Freq)
{
  EMBLG_PARAM_ECMD = 0x00;           // Command Number
  EMBLG_PARAM_PTRL = Address & 0xFF; // Address Low Byte
  EMBLG_PARAM_PTRH = Address >> 8;   // Address High Byte
  EMBLG_PARAM_DATA = Data;           // Data to write
  EMBLG_PARAM_FREQ = Freq;           // Frequency
  FCSR = 0x00;                       // Launch command (no need to wait end of operation)
  return EMBLG_PARAM_ECMD;           // Return Embedded Command Status
}
#endif

#ifdef USE_HDFLASH_WRITEBLOCK
/*-----------------------------------------------------------------------------
ROUTINE NAME : HDFlashWriteBlock
INPUT        : Buffer Address, Flash Address, Number of Bytes to write, Frequency
OUTPUT       : Embedded Command Status
DESCRIPTION  : Write N bytes
-----------------------------------------------------------------------------*/     
unsigned char HDFlashWriteBlock(unsigned char *Buffer, unsigned char *Flash, unsigned char ByteNb, unsigned char Freq)
{
  EMBLG_PARAM_ECMD = 0x01;                 // Command Number
  EMBLG_PARAM_SECT = ByteNb;               // Number of Bytes
  EMBLG_PARAM_PTRL = ((int)Flash & 0xFF);  // Flash Address Low Byte
  EMBLG_PARAM_PTRH = ((int)Flash >> 8);    // Flash Address High Byte
  EMBLG_PARAM_ENDL = ((int)Buffer & 0xFF); // Buffer Address Low Byte
  EMBLG_PARAM_ENDH = ((int)Buffer >> 8);   // Buffer Address High Byte
  EMBLG_PARAM_FREQ = Freq;                 // Frequency
  do {
    FCSR = 0x00;                           // Launch command
#ifdef USE_TIMEOUT
    UserWhileWriteBlock();                 // Call-back function
#endif
  }        
  while (EMBLG_PARAM_ECMD & HDFLASH_BUSY); // Wait End of Operation
  return EMBLG_PARAM_ECMD;                 // Return Embedded Command Status
}
#endif

#ifdef USE_HDFLASH_PROGALL
/*-----------------------------------------------------------------------------
ROUTINE NAME : HDFlashProgAll
INPUT        : Frequency
OUTPUT       : Embedded Command Status
DESCRIPTION  : Write 00h in all Flash memory
-----------------------------------------------------------------------------*/     
unsigned char HDFlashProgAll(unsigned char Freq)
{
  EMBLG_PARAM_ECMD = EMBLG_PARAM_PTRH = 0x04;
  EMBLG_PARAM_PTRL = EMBLG_PARAM_DATA = 0x00;
  EMBLG_PARAM_ENDL = EMBLG_PARAM_ENDH = 0xFF;
  EMBLG_PARAM_FREQ = Freq;
  FCSR = 0x00;             // Launch command (no need to wait end of operation)
  return EMBLG_PARAM_ECMD; // Return Embedded Command Status
}
#endif

#ifdef USE_HDFLASH_WRITEOPTBYTE
/*-----------------------------------------------------------------------------
ROUTINE NAME : HDFlashWriteOptByte
INPUT        : Option Byte 0, Option Byte 1, Frequency
OUTPUT       : Embedded Command Status
DESCRIPTION  : Write Option Bytes
-----------------------------------------------------------------------------*/     
unsigned char HDFlashWriteOptByte(unsigned char Opt0, unsigned char Opt1, unsigned char Freq)
{
  EMBLG_PARAM_ECMD = 0x02;
  EMBLG_PARAM_PTRL = 0x20;
  EMBLG_PARAM_ENDL = 0x38;
  EMBLG_PARAM_ENDH = Opt1;
  EMBLG_PARAM_DATA = Opt0;
  EMBLG_PARAM_FREQ = Freq;
  FCSR = 0x00;             // Launch command (no need to wait end of operation)
  return EMBLG_PARAM_ECMD; // Return Embedded Command Status
}
#endif

#ifdef USE_HDFLASH_READOPTBYTE
/*-----------------------------------------------------------------------------
ROUTINE NAME : HDFlashReadOptByte
INPUT        : Frequency
OUTPUT       : Embedded Command Status
               Returned values of Option Bytes are in EMBLG_PARAM_ENDH and
               EMBLG_PARAM_DATA addresses
DESCRIPTION  : Read Option Bytes
-----------------------------------------------------------------------------*/     
unsigned char HDFlashReadOptByte(unsigned char Freq)
{
  EMBLG_PARAM_ECMD = 0x05;
  EMBLG_PARAM_FREQ = Freq;
  FCSR = 0x00;             // Launch command (no need to wait end of operation)
  return EMBLG_PARAM_ECMD; // Return Embedded Command Status
}
#endif

#ifdef USE_HDFLASH_CHECKSUM
/*-----------------------------------------------------------------------------
ROUTINE NAME : HDFlashChecksum
INPUT        : Flash address, Frequency
OUTPUT       : Embedded Command Status
               Returned values of Checksum are in EMBLG_PARAM_ENDH and
               EMBLG_PARAM_ENDL addresses
DESCRIPTION  : Compute a Checksum
-----------------------------------------------------------------------------*/     
unsigned char HDFlashChecksum(unsigned char *Flash, unsigned char Freq)
{
  EMBLG_PARAM_ECMD = 0x06;
  EMBLG_PARAM_PTRL = ((int)Flash & 0xFF); // Flash Address Low Byte
  EMBLG_PARAM_PTRH = ((int)Flash >> 8);   // Flash Address High Byte
  EMBLG_PARAM_FREQ = Freq;
  FCSR = 0x00;                            // Launch command (no need to wait end of operation)
  return EMBLG_PARAM_ECMD;                // Return Embedded Command Status
}
#endif

#ifdef HIWARE
#pragma CODE_SEG DEFAULT
#endif

/*** END OF FILE ***/