/*** (c) STMicroelectronics ***************************************************
#
# PROJECT : ST7 USB LOW SPEED EVALUATION KIT + DFU
#
# COMPILER : COSMIC / HIWARE
#
# VERSION : 1.1
#
# LIB VERSION : 4.21
#
# DESCRIPTION : Contains "Handle_HID_Requests()".
#
# HISTORY :
#								                                                              
******************************************************************************/
                    
#include "HIDLayer.h"                    
#include "USB_Var.h"
#include "USB_Opts.h"
#include "Macro.h"
#include "USB_Def.h"
#include "Lib_Bits.h"
#include "USB_Lib.h"
#include "User_Def.h"
#include "User_Var.h"
#include "Applet.h"

#ifdef USE_DEVICE_FIRMWARE_UPGRADE
#include "DFULibFuncAdd.h"
#endif

#ifdef HIWARE
#include MAP_FILE
#pragma CODE_SEG USER_ROM
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : Handle_HID_Requests
INPUT/OUTPUT : None
DESCRIPTION  : Decode Human Interface Device requests
-----------------------------------------------------------------------------*/
void Handle_HID_Requests(void)
{
	switch (USBbRequest)
	{
#ifdef USE_HID_IDLE
	case SET_IDLE :	
		Idle_Value = USBwValue[1];		// REM: USBwValue[1] has been updated in the Setup decoding in the lib file usb.c
		USBLibStatus &= ~APP_REQUEST;	// Clear APP_REQUEST flag since the application request has been processed.
		return;
	case GET_IDLE :
		Write_EP_Buffer(0,&Idle_Value,USBwLength);
		Set_EP_Ready(0,EP_IN,USBwLength);
		USBLibStatus &= ~APP_REQUEST;	// Clear APP_REQUEST flag since the application request has been processed.
		return;
#endif
#ifdef USE_HID_PROTOCOL
	case SET_PROTOCOL :			
		Protocol_Value = USBwValue[0];  // REM: USBwValue[0] has been updated in the Setup decoding in the lib file usb.c
		USBLibStatus &= ~APP_REQUEST;	// Clear APP_REQUEST flag since the application request has been processed.
		return;
	case GET_PROTOCOL :
		Write_EP_Buffer(0,&Protocol_Value,USBwLength);
		Set_EP_Ready(0,EP_IN,USBwLength);			
		USBLibStatus &= ~APP_REQUEST;	// Clear APP_REQUEST flag since the application request has been processed.
		return;
#endif
#ifdef USE_HID_SET_REPORT
	case SET_REPORT:	
		switch (USBwValue[1])			// Decode Report Type
		{
		case SET_OUTPUT:	
			Set_Output();
			break;
		case SET_FEATURE:	
			Set_Feature();
			break;
		// REM : SET_INPUT is meaningless
		default:
		   	USBLibStatus |= USB_STALL;  // STALL invalid request. 
		   	// REM: When USB_STALL is set, APP_REQUEST flag is cleared by library
		}    
		return;  
#endif
	case GET_REPORT:        // MANDATORY REQUEST support for HID
		switch (USBwValue[1])			// Decode Report Type
		{
		case GET_OUTPUT:	
			Get_Output();
			break;
		case GET_INPUT:	
			Get_Input();
			break;
		case GET_FEATURE:	
			Get_Feature();
			break;  
		default:
		   	USBLibStatus |= USB_STALL;  // STALL invalid request. 
		   	// REM: When USB_STALL is set, APP_REQUEST flag is cleared by library
		}
		return;

	default :
	   	USBLibStatus |= USB_STALL;  // STALL invalid request. 
	   	// REM: When USB_STALL is set, APP_REQUEST flag is cleared by library
	} // End of CLASS requests decoding section	
}
          
/*-----------------------------------------------------------------------------
ROUTINE NAME : Get_Output
INPUT/OUTPUT : None
DESCRIPTION  : Build Output Report  (MANDATORY IF OUTPUT USAGE DEFINED)
-----------------------------------------------------------------------------*/     
void Get_Output(void)
{               
/* REM: the ST7263 demokit applet never performs Get_Output() */
#ifdef	USE_ENDPOINT2_OUT
	switch (USBwValue[0])			// Select Report according to Report_Id
	{
	case 1:		Applet_DataToSend[0] = 1;				// Report ID 
				Applet_DataToSend[1] = ((PADR & 0x80)>>7);	// LED2 status (PA7)
				break;	
	case 4:		Applet_DataToSend[0] = 4;				// Report ID 
#ifdef ST7262
                        Applet_DataToSend[1] = (PWMDCR0 * 100U) / 255;	// Led PWM on board status.
#endif
#if defined(ST7263) || defined(ST7263B)
				Applet_DataToSend[1] = 100 - TIMOC1LR;			// Led PWM on board status.
#endif
				break;
	default:
			   	USBLibStatus |= USB_STALL;  // STALL invalid request.    
				return;
	}    
	Write_EP_Buffer(0, Applet_DataToSend, USBwLength); 	// Load Data to report into DMA buffer
	Set_EP_Ready(0, EP_IN, USBwLength);		// Switch Endpoint to 'VALID' (will send data on next IN token)
	USBLibStatus &= ~APP_REQUEST;	// Clear APP_REQUEST flag since there is no more data to send
#else
   	USBLibStatus |= USB_STALL;  // STALL invalid request. 
#endif
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : Get_Input
INPUT/OUTPUT : None
DESCRIPTION  : Build Input Report  (MANDATORY IF INPUT USAGE DEFINED)
-----------------------------------------------------------------------------*/     
void Get_Input(void)
{
/* REM: the ST7263 demokit applet never performs Get_Input() */
	switch (USBwValue[0])			// Select Report according to Report_Id
	{
	case 2:		Applet_DataToSend[0] = 2;				// Report ID 
				Applet_DataToSend[1] = ((PADR & 0x20)>>2);	// Push Button SW1 (PA5) on Board status
				break;	
	case 3:		Applet_DataToSend[0] = 3;				// Report ID 
#ifdef ST7262
				Applet_DataToSend[1] = ADCDRMSB;  			// Potentiometer on board status
#endif
#if defined(ST7263) || defined(ST7263B)
				Applet_DataToSend[1] = (ADCDR & 0xFC);		// Potentiometer on board status
#endif
				break;						// REM: Skip 2 LSBs to get rid of noise
	default:
			   	USBLibStatus |= USB_STALL;  // STALL invalid request. 
				return;
	}    
	Write_EP_Buffer(0, Applet_DataToSend, USBwLength); 	// Load Data to report into DMA buffer
	Set_EP_Ready(0, EP_IN, USBwLength);		// Switch Endpoint to 'VALID' (will send data on next IN token)
	USBLibStatus &= ~APP_REQUEST;	// Clear APP_REQUEST flag since there is no more data to send
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : Get_Feature
INPUT/OUTPUT : None
DESCRIPTION  : Build feature Report  (MANDATORY IF FEATURE USAGE DEFINED)
-----------------------------------------------------------------------------*/     
void Get_Feature(void)
{ 
     
	switch (USBwValue[0])			// Select Report according to Report_Id
	{

	case 1:		Applet_DataToSend[0] = 1;				// Report ID 
			Applet_DataToSend[1] = ((PADR & 0x80)>>7);	// LED2 on board (PA7)							
			Write_EP_Buffer(0, Applet_DataToSend, USBwLength); // Load Data to report into DMA buffer
 	  		Set_EP_Ready(0, EP_IN, USBwLength);		   // Switch Endpoint to 'VALID' (will send data on next IN token)
	  		USBLibStatus &= ~APP_REQUEST;			   // Clear APP_REQUEST flag since there is no more data to send			
			break;	

	case 2:		Applet_DataToSend[0] = 2;				// Report ID 
			Applet_DataToSend[1] = ((PADR & 0x20)>>2);	// Push SW1 Button (PA5) on Board			
			Write_EP_Buffer(0, Applet_DataToSend, USBwLength); // Load Data to report into DMA buffer
 	  		Set_EP_Ready(0, EP_IN, USBwLength);		   // Switch Endpoint to 'VALID' (will send data on next IN token)
	  		USBLibStatus &= ~APP_REQUEST;			   // Clear APP_REQUEST flag since there is no more data to send
			break;	

	case 3:		Applet_DataToSend[0] = 3;				// Report ID 
#ifdef ST7262
			Applet_DataToSend[1] = ADCDRMSB;			// Potentiometer on board.
#endif
#if defined(ST7263) || defined(ST7263B)
			Applet_DataToSend[1] = (ADCDR & 0xFC);		// Potentiometer on board.
#endif			
			Write_EP_Buffer(0, Applet_DataToSend, USBwLength); // Load Data to report into DMA buffer
 	  		Set_EP_Ready(0, EP_IN, USBwLength);		   // Switch Endpoint to 'VALID' (will send data on next IN token)
	  		USBLibStatus &= ~APP_REQUEST;			   // Clear APP_REQUEST flag since there is no more data to send
			break;						// REM: Skip 2 LSBs to get rid of noise.

	case 4:		Applet_DataToSend[0] = 4;				// Report ID 
#ifdef ST7262
			Applet_DataToSend[1] = (PWMDCR0 * 100U) / 255;	// Led PWM on board.
#endif
#if defined(ST7263) || defined(ST7263B)
			Applet_DataToSend[1] = 100 - TIMOC1LR;		// Led PWM on board.
#endif			
			Write_EP_Buffer(0, Applet_DataToSend, USBwLength); // Load Data to report into DMA buffer
 	  		Set_EP_Ready(0, EP_IN, USBwLength);		   // Switch Endpoint to 'VALID' (will send data on next IN token)
	  		USBLibStatus &= ~APP_REQUEST;			   // Clear APP_REQUEST flag since there is no more data to send
			break;
					
#ifdef USE_REPORT5
	case 5:
			REQ_SendReport5ToPC();			
			break;	
#endif
	  			
	default:
			USBLibStatus |= USB_STALL;  // STALL invalid request. 
			return;
	}	        

}

/*-----------------------------------------------------------------------------
ROUTINE NAME : Set_Feature
INPUT/OUTPUT : None
DESCRIPTION  : Process feature Report 
-----------------------------------------------------------------------------*/     
void Set_Feature(void)
{    

	Read_EP_Buffer(0, Applet_DataReceived);	// Save data received on EP0 

	switch (USBwValue[0])			// Select Report ID Number
	{

	case 1:		Command_LED();			// Set LED value
			USBLibStatus &= ~APP_REQUEST;	// Clear APP_REQUEST flag since 'data stage out' is finished.	
			break;

	case 4:		Command_PWM();			// Set PWM value
			USBLibStatus &= ~APP_REQUEST;	// Clear APP_REQUEST flag since 'data stage out' is finished.	
			break;

#ifdef USE_REPORT5
	case 5:
	        	REQ_GetReport5FromPC();  										
			break;
#endif

	default:
			USBLibStatus |= USB_STALL;  // STALL invalid request.  
			return;
	}                  
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : Set_Output
INPUT/OUTPUT : None
DESCRIPTION  : Process output Report
-----------------------------------------------------------------------------*/     
void Set_Output(void)
{ 
#ifdef	USE_ENDPOINT2_OUT
	Read_EP_Buffer(0, Applet_DataReceived);	// Save data received on EP0 
	switch (USBwValue[0])			// Select Report ID Number
	{
	case 1:		Command_LED();				// Set LED value
				break;
	case 4:		Command_PWM();				// Set PWM value
				break;
	default:
			   	USBLibStatus |= USB_STALL;  // STALL invalid request. 
				return;
	}  
	USBLibStatus &= ~APP_REQUEST;	// Clear APP_REQUEST flag since 'data stage out' is finished.	
#else
   	USBLibStatus |= USB_STALL;  // STALL invalid request. 
#endif

}

#ifdef HIWARE
#pragma CODE_SEG DEFAULT
#endif

/*** END OF FILE ***/