unit uMain;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ComCtrls;

const
  WM_ONDFUDEVICECHANGED = WM_USER + 1234;

type
  TFrmMain = class(TForm)
    GrpBxDFUDevices: TGroupBox;
    LstBxDFUDevices: TListBox;
    GrpBxDeviceInfos: TGroupBox;
    LblProduct: TLabel;
    LblVersion: TLabel;
    StcTxtProd: TStaticText;
    StcTxtVers: TStaticText;
    GrpBxFlashMemory: TGroupBox;
    BtnWrite: TButton;
    BtnRead: TButton;
    ChckBxSector1: TCheckBox;
    ChckBxSector2: TCheckBox;
    PrgrsBrStatus: TProgressBar;
    SvDlgRead: TSaveDialog;
    GrpBxMessages: TGroupBox;
    LblMessage1: TLabel;
    LblMessage2: TLabel;
    OpnDlgWrite: TOpenDialog;
    LblFlash: TLabel;
    StcTxtSize: TStaticText;
    procedure BtnReadClick(Sender: TObject);
    procedure BtnWriteClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure LstBxDFUDevicesClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
  private
    procedure WMDeviceChange(var Message: TMessage); message WM_DEVICECHANGE;
    procedure WMOnDFUDeviceChanged(var Message: TMessage); message WM_ONDFUDEVICECHANGED;
    procedure RebuildListDevices;
    procedure ReadInfo;
    procedure CloseDFUDriver;
    function DFUPRT_GotoIdleState: Boolean;
    function DFUPRT_EraseSector(SectorNb: Byte): Boolean;
    function DFUPRT_WriteBlock(Buffer: PUCHAR; Address: Word): Boolean;
    procedure SaveStreamToS19(Stream: TMemoryStream; S19FileName: string; StartAdd: Word; BytesToRead: Word);
    function LoadS19ToStream(Stream: TMemoryStream; S19FileName: string): Boolean;
  public
    { Public declarations }
  end;

var
  FrmMain: TFrmMain;

implementation

uses
  uST7DFU, Math;

{$R *.DFM}

var
  TheDev: THANDLE;
  S2Start, S2Length: Word;

procedure TFrmMain.BtnReadClick(Sender: TObject);
var
  Ret: DWORD;
  SBuff: array[0..7] of Byte;
  i, Add : Word;
  DataStream: TMemoryStream;
begin

  if (ChckBxSector1.Checked) or (ChckBxSector2.Checked) then
  begin

    // Open the DFU driver for the selected device
    Ret := STDFU_Open(LstBxDFUDevices.ItemIndex, @TheDev);
    if (Ret <> STDFU_NOERROR) then
    begin
      LblMessage1.Caption := 'STDFU_Open command failed.';
      LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
      CloseDFUDriver;
      Exit;
    end;

    if (DFUPRT_GotoIdleState) then
    begin

      if (SvDlgRead.Execute) then
      begin

        DataStream := TMemoryStream.Create;
        DataStream.SetSize($10000); // 64kb
        DataStream.Position := 0;

        SBuff[0] := $FF;
        for i:=0 to $10000-1 do DataStream.Write(SBuff, 1); // clear all stream

        //---------
        // SECTOR 1
        //---------
        if (ChckBxSector1.Checked) then
        begin
          LblMessage1.Caption := 'Reading Sector 1...';
          LblMessage2.Caption := '';
          // UPLOAD : Read firmware
          PrgrsBrStatus.Max := 512; // 4096 bytes divided by 8 bytes = 512 loops
          for i := 0 to PrgrsBrStatus.Max-1 do
          begin
            Update;
            PrgrsBrStatus.Position := i+1;
            Add := $E000 + i*8; // Sector 1 starts always at E000h
            Ret := STDFU_Upload(@TheDev, @SBuff, 8, Add);
            if (Ret <> STDFU_NOERROR) then
            begin
              LblMessage1.Caption := 'STDFU_Upload command failed.';
              LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
              CloseDFUDriver;
              Exit;
            end;
            DataStream.Position := Add;
            DataStream.Write(SBuff, 8);
          end;
          LblMessage2.Caption := 'OK !';
          PrgrsBrStatus.Position := 0;
          Update;
          Sleep(100);
        end;

        //---------
        // SECTOR 2
        //---------
        if (ChckBxSector2.Checked) then
        begin
          LblMessage1.Caption := 'Reading Sector 2...';
          LblMessage2.Caption := '';
          // UPLOAD : Read firmware
          PrgrsBrStatus.Max := S2Length div 8;
          for i := 0 to PrgrsBrStatus.Max-1 do
          begin
            Update;
            PrgrsBrStatus.Position := i+1;
            Add := S2Start + i*8;
            Ret := STDFU_Upload(@TheDev, @SBuff, 8, Add);
            if (Ret <> STDFU_NOERROR) then
            begin
              LblMessage1.Caption := 'STDFU_Upload command failed.';
              LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
              CloseDFUDriver;
              Exit;
            end;
            DataStream.Position := Add;
            DataStream.Write(SBuff, 8);
          end;
          LblMessage2.Caption := 'OK !';
          PrgrsBrStatus.Position := 0;
          Update;
          Sleep(100);
        end;

        //------------
        // SAVE TO S19
        //------------
        LblMessage1.Caption := 'Saving into S19 file...';
        LblMessage2.Caption := '';
        if (ChckBxSector1.Checked) and (ChckBxSector2.Checked) then
          SaveStreamToS19(DataStream, SvDlgRead.FileName, S2Start, S2Length+4096)
        else
        if (ChckBxSector1.Checked) then
          SaveStreamToS19(DataStream, SvDlgRead.FileName, $E000, 4096)
        else
          SaveStreamToS19(DataStream, SvDlgRead.FileName, S2Start, S2Length);
        LblMessage2.Caption := 'OK !';

        DataStream.Free;

      end
      else
      begin
        LblMessage1.Caption := 'Please select a file first.';
        LblMessage2.Caption := '';
      end;
    end
    else
    begin
      LblMessage1.Caption := 'Error: Cannot go in DFU_IDLE state.';
      LblMessage2.Caption := '';
    end;
    CloseDFUDriver;
  end
  else
  begin
    LblMessage1.Caption := 'Please select at least one sector first.';
    LblMessage2.Caption := '';
  end;
end;

//==========================
procedure TFrmMain.ReadInfo;
//==========================
var
  Ret: DWORD;
  SBuff: array[0..7] of Byte;
  LBuff: array[0..255] of Byte;
  pLBuff: PChar;
  ProdAdd, VersAdd, CurrAdd: Word;
  ProdLen, VersLen, i, k: Byte;
  RemainBytes, NbBytesToRead: integer;
  FlashSize: Byte;
begin

  // Open the DFU driver for the selected device
  Ret := STDFU_Open(LstBxDFUDevices.ItemIndex, @TheDev);
  if (Ret <> STDFU_NOERROR) then
  begin
    LblMessage1.Caption := 'STDFU_Open command failed.';
    LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
    CloseDFUDriver;
    Exit;
  end;

  if (DFUPRT_GotoIdleState) then
  begin

    pLBuff := @LBuff;

    StcTxtProd.Caption := '';
    StcTxtVers.Caption := '';
    StcTxtSize.Caption := '';
    ChckBxSector1.Enabled := False;
    ChckBxSector1.Checked := False;
    ChckBxSector2.Enabled := False;
    ChckBxSector2.Checked := False;

    LblMessage1.Caption := 'Reading device '+IntToStr(LstBxDFUDevices.ItemIndex)+' informations...';
    LblMessage2.Caption := '';

    // EFF3h - EFF4h = Address of product string
    // EFF5h         = Length of product string
    // EFF6h - EFF7h = Address of version string
    // EFF8h         = Length of version string
    // EFF9h         = Flash size in Kb

    // UPLOAD : Get strings addresses + strings length + Flash Size
    Ret := STDFU_Upload(@TheDev, @SBuff, 8, $EFF3);
    if (Ret <> STDFU_NOERROR) then
    begin
      LblMessage1.Caption := 'STDFU_Upload command failed.';
      LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
      CloseDFUDriver;
      Exit;
    end;

    ProdAdd := SBuff[0]*256 + SBuff[1]; // Product string address
    ProdLen := SBuff[2];                // Product string length
    VersAdd := SBuff[3]*256 + SBuff[4]; // Version string address
    VersLen := SBuff[5];                // Version string length
    FlashSize := SBuff[6];

    FillChar(pLbuff^, 256, 0); // Fill all buffer with null character

    // UPLOAD : Get Product string
    RemainBytes := ProdLen;
    CurrAdd := ProdAdd;
    k := 0;
    while RemainBytes > 0 do
    begin
      NbBytesToRead := min(8, RemainBytes);
      Ret := STDFU_Upload(@TheDev, @SBuff, NbBytesToRead, CurrAdd);
      if (Ret <> STDFU_NOERROR) then
      begin
        LblMessage1.Caption := 'STDFU_Upload command failed.';
        LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
        CloseDFUDriver;
        Exit;
      end;
      for i := 0 to NbBytesToRead-1 do LBuff[k+i] := SBuff[i];
      CurrAdd := CurrAdd + NbBytesToRead;
      RemainBytes := RemainBytes - NbBytesToRead;
      k := k + NbBytesToRead;
    end;
    StcTxtProd.Caption := strpas(pLBuff);

    FillChar(pLbuff^, 256, 0); // Fill all buffer with null character

    // UPLOAD : Get Version string
    RemainBytes := VersLen;
    CurrAdd := VersAdd;
    k := 0;
    while RemainBytes > 0 do
    begin
      NbBytesToRead := min(8, RemainBytes);
      Ret := STDFU_Upload(@TheDev, @SBuff, NbBytesToRead, CurrAdd);
      if (Ret <> STDFU_NOERROR) then
      begin
        LblMessage1.Caption := 'STDFU_Upload command failed.';
        LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
        CloseDFUDriver;
        Exit;
      end;
      for i := 0 to NbBytesToRead-1 do LBuff[k+i] := SBuff[i];
      CurrAdd := CurrAdd + NbBytesToRead;
      RemainBytes := RemainBytes - NbBytesToRead;
      k := k + NbBytesToRead;
    end;
    StcTxtVers.Caption := strpas(pLBuff);

    LblMessage2.Caption := 'OK !';

    //-----------
    // Flash Size
    //-----------
    StcTxtSize.Caption := IntToStr(FlashSize) + 'kb';
    if (FlashSize < 8) or (FlashSize > 60) then
    begin
      LblMessage1.Caption := 'Error: Incorrect flash size.';
      LblMessage2.Caption := '(Defaulted to 16kb)';
      FlashSize := 16;
    end;
    if (FlashSize >= 8) then // Minimum 8kb to enable Sector 1
    begin
      ChckBxSector1.Enabled := True;
      ChckBxSector1.Checked := True;
    end;
    if (FlashSize >= 10) then // Minimum 10kb to enable Sector 2
    begin
      ChckBxSector2.Enabled := True;
      ChckBxSector2.Checked := True;
      // Calculate Sector 2 start address + length
      S2Length := (FlashSize*1024) - 8192;
      S2Start := $E000 - S2Length;
    end;

  end
  else
  begin
    LblMessage1.Caption := 'Error: Cannot go in DFU_IDLE state.';
    LblMessage2.Caption := '';
  end;

  CloseDFUDriver;

end;

procedure TFrmMain.BtnWriteClick(Sender: TObject);
var
  Ret: DWORD;
  DataStream: TMemoryStream;
  SBuff: array[0..7] of Byte;
  BytesRead, j: Byte;
  i: Word;
  OkToWrite: Boolean;
begin

  if (ChckBxSector1.Checked) or (ChckBxSector2.Checked) then
  begin

    // Open the DFU driver for the selected device
    Ret := STDFU_Open(LstBxDFUDevices.ItemIndex, @TheDev);
    if (Ret <> STDFU_NOERROR) then
    begin
      LblMessage1.Caption := 'STDFU_Open command failed.';
      LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
      CloseDFUDriver;
      Exit;
    end;

    if (OpnDlgWrite.Execute) then
    begin

      DataStream := TMemoryStream.Create;
      DataStream.SetSize($10000); // 64kb

      if (LoadS19ToStream(DataStream, OpnDlgWrite.FileName)) then
      begin

        //---------
        // SECTOR 1
        //---------
        if (ChckBxSector1.Checked) then
        begin
          LblMessage1.Caption := 'Erasing Sector 1...';
          LblMessage2.Caption := '';
          if (not DFUPRT_GotoIdleState) then
          begin
            LblMessage1.Caption := 'Error: Cannot go in DFU_IDLE state.';
            LblMessage2.Caption := '(erasing sector 1 aborted)';
            CloseDFUDriver;
            Exit;
          end;
          if (DFUPRT_EraseSector(1)) then
          begin
            LblMessage2.Caption := 'OK !';
            Update;
            Sleep(100);
            LblMessage1.Caption := 'Writing Sector 1...';
            LblMessage2.Caption := '';
            DataStream.Position := $E000; // Sector 1 starts always at E000h
            PrgrsBrStatus.Max := 512; // 4096 bytes divided by 8 bytes = 512 loops
            for i := 0 to PrgrsBrStatus.Max-1 do
            begin
              Update;
              PrgrsBrStatus.Position := i+1;
              BytesRead := DataStream.Read(SBuff, 8);
              if (BytesRead = 8) then
              begin
                OkToWrite := False;
                for j := 0 to 7 do if (Sbuff[j] <> $FF) then OkToWrite := True; // Blocks with only $FF are not written
                if (OkToWrite) then
                  if (not DFUPRT_WriteBlock(@SBuff, DataStream.Position-8)) then
                  begin
                    LblMessage1.Caption := 'Error: Programmation failed at @'+IntToHex(DataStream.Position,4)+'h.';
                    LblMessage2.Caption := '(writing sector 1 aborted)';
                    CloseDFUDriver;
                    Exit;
                  end;
              end
              else
              begin
                LblMessage1.Caption := 'Error: Cannot read 8 bytes from S19 file.';
                LblMessage2.Caption := '(writing sector 1 aborted)';
                CloseDFUDriver;
                Exit;
              end;
            end;
            LblMessage2.Caption := 'OK !';
            PrgrsBrStatus.Position := 0;
            Update;
            Sleep(100);
          end
          else
          begin
            LblMessage1.Caption := 'Error: Cannot erase Sector 1.';
            LblMessage2.Caption := '';
          end;
        end; // Sector 1

        //---------
        // SECTOR 2
        //---------
        if (ChckBxSector2.Checked) then
        begin
          LblMessage1.Caption := 'Erasing Sector 2...';
          LblMessage2.Caption := '';
          if (not DFUPRT_GotoIdleState) then
          begin
            LblMessage1.Caption := 'Error: Cannot go in DFU_IDLE state.';
            LblMessage2.Caption := '(erasing sector 2 aborted)';
            CloseDFUDriver;
            Exit;
          end;
          if (DFUPRT_EraseSector(2)) then
          begin
            LblMessage2.Caption := 'OK !';
            Update;
            Sleep(100);
            LblMessage1.Caption := 'Writing Sector 2...';
            LblMessage2.Caption := '';
            DataStream.Position := S2Start;
            PrgrsBrStatus.Max := S2Length div 8;
            for i := 0 to PrgrsBrStatus.Max-1 do
            begin
              Update;
              PrgrsBrStatus.Position := i+1;
              BytesRead := DataStream.Read(SBuff, 8);
              if (BytesRead = 8) then
              begin
                OkToWrite := False;
                for j := 0 to 7 do if (Sbuff[j] <> $FF) then OkToWrite := True; // Blocks with only $FF are not written
                if (OkToWrite) then
                  if (not DFUPRT_WriteBlock(@SBuff, DataStream.Position-8)) then
                  begin
                    LblMessage1.Caption := 'Error: Programmation failed at @'+IntToHex(DataStream.Position,4)+'h.';
                    LblMessage2.Caption := '(writing sector 2 aborted)';
                    CloseDFUDriver;
                    Exit;
                  end;
              end
              else
              begin
                LblMessage1.Caption := 'Error: Cannot read 8 bytes from S19 file.';
                LblMessage2.Caption := '(writing sector 2 aborted)';
                CloseDFUDriver;
                Exit;
              end;
            end;
            LblMessage2.Caption := 'OK !';
            PrgrsBrStatus.Position := 0;
            Update;
            Sleep(100);
          end
          else
          begin
            LblMessage1.Caption := 'Error: Cannot erase Sector 2.';
            LblMessage2.Caption := '';
          end;
        end; // Sector 2

        // ABORT
        Ret := STDFU_Abort(@TheDev);
        if (Ret <> STDFU_NOERROR) then
        begin
          LblMessage1.Caption := 'STDFU_Abort command failed.';
          LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
          CloseDFUDriver;
          Exit;
        end;

      end
      else
      begin
        LblMessage1.Caption := 'Error: Cannot read the S19 file.';
        LblMessage2.Caption := '';
      end;

    end
    else
    begin
      LblMessage1.Caption := 'Error: Cannot open the S19 file.';
      LblMessage2.Caption := '';
    end;

    CloseDFUDriver;

  end
  else
  begin
    LblMessage1.Caption := 'Please select at least one sector first.';
    LblMessage2.Caption := '';
  end;

end;

procedure TFrmMain.WMDeviceChange(var Message: TMessage);
begin
  STDFU_HandleDeviceChange(Message.WParam, Message.LParam);
end;

procedure TFrmMain.FormCreate(Sender: TObject);
const
  GUID_DFU : TGuid = '{3fe809ab-fb91-4cb5-a643-69670d52366e}';
  DBT_DEVTYP_DEVICEINTERFACE : DWORD = 5;
type
  DEV_BROADCAST_DEVICEINTERFACE = record
    dbcc_size: DWORD;
    dbcc_devicetype: DWORD;
    dbcc_reserved: DWORD;
    dbcc_classguid: TGuid;
    dbcc_name: char;
  end;
var
  dbch: DEV_BROADCAST_DEVICEINTERFACE;
begin
  dbch.dbcc_size := sizeof(dbch);
  dbch.dbcc_devicetype := DBT_DEVTYP_DEVICEINTERFACE;
  dbch.dbcc_classguid := GUID_DFU;
  dbch.dbcc_name := chr(0);
  RegisterDeviceNotification(Handle, @dbch, DEVICE_NOTIFY_WINDOW_HANDLE);
  TheDev := 0;
  RebuildListDevices;
end;

procedure TFrmMain.WMOnDFUDeviceChanged(var Message: TMessage);
begin
  RebuildListDevices;
end;

//====================================
procedure TFrmMain.RebuildListDevices;
//====================================
var
  i, Nb: integer;
  Desc: USB_DEVICE_DESCRIPTOR;
  szProductString: PChar;
  szProductStringArray: array[0..254] of char;
  Ret: DWORD;
begin

  LstBxDFUDevices.Items.Clear;

  StcTxtProd.Caption := '';
  StcTxtVers.Caption := '';
  StcTxtSize.Caption := '';

  ChckBxSector1.Enabled := False;
  ChckBxSector1.Checked := False;
  ChckBxSector2.Enabled := False;
  ChckBxSector2.Checked := False;

  szProductString := @szProductStringArray;

  Nb := 0;

  Ret := STDFU_EnumGetNbDevices(@Nb, Handle, WM_ONDFUDEVICECHANGED);
  if (Ret <> STDFU_NOERROR) then
  begin
    LblMessage1.Caption := 'STDFU_EnumGetNbDevices command failed.';
    LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
    Exit;
  end;

  for i := 0 to Nb-1 do
  begin
    Ret := STDFU_GetDeviceDescriptor(i, @Desc);
    if (Ret <> STDFU_NOERROR) then
    begin
      LblMessage1.Caption := 'STDFU_GetDeviceDescriptor command failed.';
      LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
      Exit;
    end;
    if (Desc.idProduct = $DF11) then
    begin
      STDFU_GetStringDescriptor(i, Desc.iProduct, szProductString, 255);
      LstBxDFUDevices.Items.Add(strpas(szProductString)+' (VID='+IntToHex(Desc.idVendor,4)+'h)');
    end;
  end;

  if (LstBxDFUDevices.Items.Count > 0) then
  begin
    LblMessage1.Caption := IntToStr(LstBxDFUDevices.Items.Count)+' device(s) connected.';
    LblMessage2.Caption := '';
    LstBxDFUDevices.ItemIndex := 0;
    ReadInfo;
  end
  else
  begin
    LblMessage1.Caption := 'No device connected.';
    LblMessage2.Caption := '';
  end;

end;

procedure TFrmMain.LstBxDFUDevicesClick(Sender: TObject);
begin
  ReadInfo;
end;

procedure TFrmMain.FormDestroy(Sender: TObject);
begin
  // Close the DFU driver if already opened
  if (TheDev <> 0) then STDFU_Close(@TheDev);
end;

//================================
procedure TFrmMain.CloseDFUDriver;
//================================
var
  Ret: DWORD;
begin
  // Close the DFU driver if already opened
  if (TheDev <> 0) then
  begin
    Ret := STDFU_Close(@TheDev);
    if (Ret <> STDFU_NOERROR) then
    begin
      LblMessage1.Caption := 'STDFU_Close command failed.';
      LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
      Exit;
    end;
    TheDev := 0;
  end;
end;

//==============================================
function TFrmMain.DFUPRT_GotoIdleState: Boolean;
//==============================================
// Return TRUE if device is in DFU_IDLE sate, otherwise FALSE.
var
  Ret: DWORD;
  DfuStatus: TDFUSTATUS;
begin

  Result := False;

  if (TheDev <> 0) then
  begin

    // GETSTATUS
    Ret := STDFU_Getstatus(@TheDev, @DfuStatus);
    if (Ret <> STDFU_NOERROR) then
    begin
      LblMessage1.Caption := 'STDFU_GetStatus command failed.';
      LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
      CloseDFUDriver;
      Exit;
    end;

    // CHECK STATE & STATUS
    if (DfuStatus.bStatus <> STATUS_OK) or (DfuStatus.bState <> STATE_DFU_IDLE) then
    begin

      if (DfuStatus.bState = STATE_DFU_ERROR) then
      begin

        // CLEARSTATUS
        Ret := STDFU_Clrstatus(@TheDev);
        if (Ret <> STDFU_NOERROR) then
        begin
          LblMessage1.Caption := 'STDFU_Clrstatus command failed.';
          LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
          CloseDFUDriver;
          Exit;
        end;

      end
      else
      begin

        // ABORT
        Ret := STDFU_Abort(@TheDev);
        if (Ret <> STDFU_NOERROR) then
        begin
          LblMessage1.Caption := 'STDFU_Abort command failed.';
          LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
          CloseDFUDriver;
          Exit;
        end;

      end;

      // GETSTATUS
      Ret := STDFU_Getstatus(@TheDev, @DfuStatus);
      if (Ret <> STDFU_NOERROR) then
      begin
        LblMessage1.Caption := 'STDFU_GetStatus command failed.';
        LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
        CloseDFUDriver;
        Exit;
      end;

      // CHECK STATE & STATUS
      if (DfuStatus.bStatus = STATUS_OK) and (DfuStatus.bState = STATE_DFU_IDLE) then
        Result := True

    end
    else Result := True;

  end;

end;

//============================================================================
procedure TFrmMain.SaveStreamToS19(Stream: TMemoryStream; S19FileName: string;
 StartAdd: Word; BytesToRead: Word);
//============================================================================

var
  BytesRead, TotalBytes: integer;
  FileStream : TFileStream;
  Tmp: array[0..200] of Byte;
  Str: string;
  pStr: array[0..50] of char;
  i: integer;
  Checksum: Byte;
  Address: integer;

begin

  FileStream := TFileStream.Create(S19FileName, fmCreate);
  FileStream.Size := 0;
  FileStream.Position := 0;

  try

    Stream.Position := StartAdd; // Set the pointer of the Stream at the start address
    TotalBytes := 0;

    while True do
    begin

      BytesRead := Stream.Read(Tmp, 32); // Read 32 bytes maximum in the Stream.
                                         // Return the number of bytes really read.
                                         // All bytes are saved in a temporary array.

      TotalBytes := TotalBytes + BytesRead;

      if (BytesRead = 0) or (TotalBytes > BytesToRead) then // No more bytes to read
      begin
        Str := 'S9030000FC';        // Last line to write in the S19 file
        StrPCopy(pStr, Str);        // Convert the null-terminated string into a Pascal string...
        FileStream.Write(pStr, 10); // Write the string in the file (10 characters)
        break;                      // Exit of the while loop
      end;

      Address := Stream.position - BytesRead; // Calculate the current address

      //=====================
      // Checksum Calculation
      //=====================
      Checksum := 0;                                            // Reset
      for i:=0 to BytesRead-1 do Checksum := Checksum + Tmp[i]; // Add all the data bytes
      Checksum := Checksum + BytesRead + 3;                     // Add the total number of bytes on the line
                                                                // total bytes = data bytes + 2 bytes address + 1 byte crc
      Checksum := Checksum + Lo(Address) + Hi(Address);         // Add the address bytes
      Checksum := not Checksum;                                 // Complement the result

      //========================
      // Write into the S19 file
      //========================

      // S1 = Address on 2 bytes
      Str := 'S1';
      StrPCopy(pStr, Str);
      FileStream.Write(pStr, 2);

      // Total number of bytes on the line (data bytes + 2 bytes address + 1 byte crc)
      Str := IntToHex(BytesRead+3, 2);
      StrPCopy(pStr, Str);
      FileStream.Write(pStr, 2);

      // MSB Address
      Str := IntToHex(Hi(Address), 2);
      StrPCopy(pStr, Str);
      FileStream.Write(pStr, 2);

      // LSB Address
      Str := IntToHex(Lo(Address), 2);
      StrPCopy(pStr, Str);
      FileStream.Write(pStr, 2);

      // Data bytes
      for i:=0 to BytesRead-1 do
      begin
        Str := IntToHex(Tmp[i], 2);
        StrPCopy(pStr, Str);
        FileStream.Write(pStr, 2);
      end;

      // Checksum
      Str := IntToHex(Checksum, 2);
      StrPCopy(pStr, Str);
      FileStream.Write(pStr, 2);

      // Carriage Return + LineFeed
      Str := #13#10;
      StrPCopy(pStr, Str);
      FileStream.Write(pStr, 2);

    end; // while

  finally
    FileStream.Free;
  end;

end;

//=====================================================================================
function TFrmMain.LoadS19ToStream(Stream: TMemoryStream; S19FileName: string): Boolean;
//=====================================================================================

const
  MAX_DATA = 64; // Maximum number of data in a line of a S19 file

var
  NbBytes: integer;
  FileStream : TFileStream;
  DataBuff : array[0..MAX_DATA-1] of byte; // Used to store data present in each S19 line
  pStr: array[0..2] of char; // Used to convert strings
  i: integer;
  Address: integer;
  BytesRead: integer;
  LineNb: integer;

begin

  //Result := False;

  FileStream := TFileStream.Create(S19FileName, fmOpenRead); // Open the S19 file in Read mode
  FileStream.Position := 0;                                  // Point to the begin of file

  //================================
  // Initializes the Stream with FFh
  //================================
  DataBuff[0] := $FF;
  for i:=0 to $10000-1 do Stream.Write(DataBuff, 1);

  LineNb := 1;

  try

    while True do // Process all lines of the S19 file
    begin

      BytesRead := FileStream.Read(pStr, 1); // Read first character

      if (pStr[0]=' ') or (pStr[0]=#13) or (pStr[0]=#10) then continue; // If first character is space or carriage return
                                                                        // or Line Feed then skip it and continue

      FileStream.Position := FileStream.Position - 1; // First character is ok, go back

      FileStream.Read(pStr, 2); // Read the next 2 characters
      pStr[2] := #0;            // Add a null character...

      if ((CompareText(StrPas(pStr), 'S9') = 0) or (BytesRead = 0)) then break; // We are at the end of the S19 file : end of while loop

      if (CompareText(StrPas(pStr), 'S0') = 0) then
      begin
        while True do
        begin
          FileStream.Read(pStr, 1); // Read next character
          if (pStr[0]=' ') or (pStr[0]=#13) or (pStr[0]=#10) then break;
        end;
        LineNb := LineNb + 1;
        continue;
      end;

      if (CompareText(StrPas(pStr), 'S1') <> 0) then // Problem: only S1 format is allowed
      begin
        MessageDlg('Only S1 format is allowed in S19 file at line '+IntToStr(LineNb)+'.',mtError,[mbOk],0);
        Result := False;
        Exit;
      end;

      // Read next 2 bytes : total number of bytes on the line
      FileStream.Read(pStr, 2);
      pStr[2] := #0;
      NbBytes := 0;
      try
        NbBytes := StrToInt('$'+StrPas(pStr)) - 3; // For us NbBytes = data bytes only
      except
        MessageDlg('Wrong number of bytes at line '+IntToStr(LineNb)+'.',mtError,[mbOk],0);
        Result := False;
        Exit;
      end;

      // Check maximum number of bytes
      if ((NbBytes > MAX_DATA) or (NbBytes < 1)) then
      begin
        MessageDlg('Wrong number of bytes (min=1, max='+IntToStr(MAX_DATA)+') at line '+IntToStr(LineNb)+'.',mtError,[mbOk],0);
        Result := False;
        Exit;
      end;

      // Read next 2 bytes : MSB address
      FileStream.Read(pStr, 2);
      pStr[2] := #0;
      Address := 0;
      try
        Address := StrToInt('$'+StrPas(pStr)) * $100;
      except
        MessageDlg('Wrong MSB address  at line '+IntToStr(LineNb)+'.',mtError,[mbOk],0);
        Result := False;
        Exit;
      end;

      // Read next 2 bytes : LSB address
      FileStream.Read(pStr, 2);
      pStr[2] := #0;
      try
        Address := StrToInt('$'+StrPas(pStr)) + Address;
      except
        MessageDlg('Wrong LSB address at line '+IntToStr(LineNb)+'.',mtError,[mbOk],0);
        Result := False;
        Exit;
      end;

      for i:=0 to MAX_DATA-1 do DataBuff[i] := $FF; // Clear buffer

      // Read next bytes : data
      for i:=0 to NbBytes-1 do
      begin
        FileStream.Read(pStr, 2);
        pStr[2] := #0;
        try
          DataBuff[i] := StrToInt('$'+StrPas(pStr)); // Data are saved in an temporary array
        except
          MessageDlg('Wrong data in S19 file at address '+IntToHex(Address+i,4)+'h.',mtError,[mbOk],0);
          Result := False;
          Exit;
        end;
      end;

      // Read next 2 bytes : Checksum (don't care)
      FileStream.Read(pStr, 2);

      //============================
      // Write bytes into the Stream
      //============================
      Stream.Position := Address; // Set the pointer at the address
      Stream.Write(DataBuff, nbBytes); // Write into the Stream the data previously stored

      LineNb := LineNb + 1;

    end; // while

    Result := True;

  finally
    FileStream.Free;
  end;

end;

//============================================================
function TFrmMain.DFUPRT_EraseSector(SectorNb: Byte): Boolean;
//============================================================
// Device must be in DFU_IDLE state prior to call this function.
// Return TRUE if sector(s) is(are) correctly erased and go in DFU_IDLE state.

const
  RASS_KEY_1 = $56;
  RASS_KEY_2 = $AE;
  ERASE_S1S2 = $FFFF;
  ERASE_S1   = $FFFE;
  ERASE_S2   = $FFFD;

var
  Ret: DWORD;
  DfuStatus: TDFUSTATUS;
  SBuff: array[0..1] of Byte;
  DfuTimeout, i: Word;

begin

  Result := False;

  if (TheDev <> 0) then
  begin

    SBuff[0] := RASS_KEY_1;
    SBuff[1] := RASS_KEY_2;

    // GETSTATUS
    Ret := STDFU_Getstatus(@TheDev, @DfuStatus);
    if (Ret <> STDFU_NOERROR) then
    begin
      LblMessage1.Caption := 'STDFU_GetStatus command failed.';
      LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
      CloseDFUDriver;
      Exit;
    end;

    // CHECK STATE & STATUS
    if (DfuStatus.bStatus <> STATUS_OK) or (DfuStatus.bState <> STATE_DFU_IDLE) then
    begin
      LblMessage1.Caption := 'Error: State or Status are not correct.';
      LblMessage2.Caption := '(must be IDLE/OK)';
      CloseDFUDriver;
      Exit;
    end;

    // DOWNLOAD : Erase command
    if (SectorNb = 1) then
      Ret := STDFU_Dnload(@TheDev, @SBuff, 2, ERASE_S1)
    else
      if (SectorNb = 2) then
        Ret := STDFU_Dnload(@TheDev, @SBuff, 2, ERASE_S2)
      else
        Ret := STDFU_Dnload(@TheDev, @SBuff, 2, ERASE_S1S2);

    if (Ret <> STDFU_NOERROR) then
    begin
      LblMessage1.Caption := 'STDFU_Dnload command failed.';
      LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
      CloseDFUDriver;
      Exit;
    end;

    // GETSTATUS
    Ret := STDFU_Getstatus(@TheDev, @DfuStatus);
    if (Ret <> STDFU_NOERROR) then
    begin
      LblMessage1.Caption := 'STDFU_GetStatus command failed.';
      LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
      CloseDFUDriver;
      Exit;
    end;

    // CHECK STATE & STATUS
    if (DfuStatus.bStatus <> STATUS_OK) or (DfuStatus.bState <> STATE_DFU_DOWNLOAD_BUSY) then
    begin
      LblMessage1.Caption := 'Error: State or Status are not correct.';
      LblMessage2.Caption := '(must be DOWNLOAD_BUSY/OK)';
      CloseDFUDriver;
      Exit;
    end;

    // WAIT TIMEOUT
    DfuTimeout := DfuStatus.bwPollTimeout[0];
    DfuTimeout := DfuTimeout or (DfuStatus.bwPollTimeout[1] shl 8);
    DfuTimeout := DfuTimeout or (DfuStatus.bwPollTimeout[2] shl 16);

    PrgrsBrStatus.Max := DfuTimeout div 100;

    for i := 1 to PrgrsBrStatus.Max do
    begin
      Update;
      Sleep(100);
      PrgrsBrStatus.Position := i;
    end;

    PrgrsBrStatus.Position := 0;

    // GETSTATUS
    Ret := STDFU_Getstatus(@TheDev, @DfuStatus);
    if (Ret <> STDFU_NOERROR) then
    begin
      LblMessage1.Caption := 'STDFU_GetStatus command failed.';
      LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
      CloseDFUDriver;
      Exit;
    end;

    // CHECK STATE & STATUS
    if (DfuStatus.bStatus <> STATUS_OK) or (DfuStatus.bState <> STATE_DFU_DOWNLOAD_IDLE) then
    begin
      LblMessage1.Caption := 'Error: State or Status are not OK.';
      LblMessage2.Caption := '(must be DOWNLOAD_IDLE/OK)';
      CloseDFUDriver;
      Exit;
    end;

    // ABORT
    Ret := STDFU_Abort(@TheDev);
    if (Ret <> STDFU_NOERROR) then
    begin
      LblMessage1.Caption := 'STDFU_Abort command failed.';
      LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
      CloseDFUDriver;
      Exit;
    end;

    // GETSTATUS
    Ret := STDFU_Getstatus(@TheDev, @DfuStatus);
    if (Ret <> STDFU_NOERROR) then
    begin
      LblMessage1.Caption := 'STDFU_GetStatus command failed.';
      LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
      CloseDFUDriver;
      Exit;
    end;

    // CHECK STATE & STATUS
    if (DfuStatus.bStatus = STATUS_OK) and (DfuStatus.bState = STATE_DFU_IDLE) then
      Result := True;

  end;

end;

//==========================================================================
function TFrmMain.DFUPRT_WriteBlock(Buffer: PUCHAR; Address: Word): Boolean;
//==========================================================================
// Device must be in DFU_IDLE or DOWNLOAD_IDLE states prior to call this function.
// Return TRUE if block is correctly programmed and exit in DOWNLOAD_IDLE state.

var
  Ret: DWORD;
  DfuStatus: TDFUSTATUS;
  DfuTimeout: Word;

begin

  Result := False;

  if (TheDev <> 0) then
  begin

    // GETSTATUS
    Ret := STDFU_Getstatus(@TheDev, @DfuStatus);
    if (Ret <> STDFU_NOERROR) then
    begin
      LblMessage1.Caption := 'STDFU_GetStatus command failed.';
      LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
      CloseDFUDriver;
      Exit;
    end;

    // CHECK STATE & STATUS
    if (DfuStatus.bStatus <> STATUS_OK) or ((DfuStatus.bState <> STATE_DFU_IDLE) and (DfuStatus.bState <> STATE_DFU_DOWNLOAD_IDLE)) then
    begin
      LblMessage1.Caption := 'Error: State or Status are not correct.';
      LblMessage2.Caption := '(must be IDLE or DOWNLOAD_IDLE/OK)';
      CloseDFUDriver;
      Exit;
    end;

    // DOWNLOAD
    Ret := STDFU_Dnload(@TheDev, Buffer, 8, Address);
    if (Ret <> STDFU_NOERROR) then
    begin
      LblMessage1.Caption := 'STDFU_Dnload command failed.';
      LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
      CloseDFUDriver;
      Exit;
    end;

    // GETSTATUS
    Ret := STDFU_Getstatus(@TheDev, @DfuStatus);
    if (Ret <> STDFU_NOERROR) then
    begin
      LblMessage1.Caption := 'STDFU_GetStatus command failed.';
      LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
      CloseDFUDriver;
      Exit;
    end;

    // CHECK STATE & STATUS
    if (DfuStatus.bStatus <> STATUS_OK) or (DfuStatus.bState <> STATE_DFU_DOWNLOAD_BUSY) then
    begin
      LblMessage1.Caption := 'Error: State or Status are not correct.';
      LblMessage2.Caption := '(must be DOWNLOAD_BUSY/OK)';
      CloseDFUDriver;
      Exit;
    end;

    // WAIT TIMEOUT
    DfuTimeout := DfuStatus.bwPollTimeout[0];
    DfuTimeout := DfuTimeout or (DfuStatus.bwPollTimeout[1] shl 8);
    DfuTimeout := DfuTimeout or (DfuStatus.bwPollTimeout[2] shl 16);

    Sleep(DfuTimeout);

    // GETSTATUS
    Ret := STDFU_Getstatus(@TheDev, @DfuStatus);
    if (Ret <> STDFU_NOERROR) then
    begin
      LblMessage1.Caption := 'STDFU_GetStatus command failed.';
      LblMessage2.Caption := 'Error #'+IntToHex(Ret,8)+'h';
      CloseDFUDriver;
      Exit;
    end;

    // CHECK STATE & STATUS
    if (DfuStatus.bStatus <> STATUS_OK) or (DfuStatus.bState <> STATE_DFU_DOWNLOAD_IDLE) then
    begin
      LblMessage1.Caption := 'Error: State or Status are not OK.';
      LblMessage2.Caption := '(must be DOWNLOAD_IDLE/OK)';
      CloseDFUDriver;
      Exit;
    end;

    Result := True;

  end;

end;

end.
