/*** (c) STMicroelectronics ***************************************************
#
# PROJECT : ST7 USB LOW SPEED EVALUATION KIT + DFU
#
# COMPILER : COSMIC / HIWARE
#
# VERSION :
#
# DESCRIPTION : Device Firmware Upgrade enable source file
#
******************************************************************************/

#include "DFUCore.h"
#include "DFUDescript.h"
#include "DFUProtocol.h"
#include "USB_Var.h"
#include "USB_Def.h"
#include "USB_Lib.h"
#include "Lib_Bits.h"
#include "Macro.h"
#include "HDFlash_macros.h"

extern void main(void);

//--------------
// DFU Variables
//--------------
Byte DFUDeviceState;
Byte DFUDeviceStatus;
Byte DFUDataReceived[8];
Byte DFUDataToSend[8];
Byte DFUPollTimeOutH;
Byte DFUPollTimeOutL;

#ifdef COSMIC
#pragma section (JUMP_MAIN)
#endif

#ifdef HIWARE
#include MAP_FILE
#pragma CODE_SEG JUMP_MAIN
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : Vec_main
INPUT/OUTPUT : None
DESCRIPTION  : Called by DFU_Main routine.
-----------------------------------------------------------------------------*/
void Vec_main(void) {
  main();
}

#ifdef COSMIC
#pragma section (DFU_CORE)
#endif
                                
#ifdef HIWARE                               
#pragma CODE_SEG DFU_CORE
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : User_Status_Out
INPUT/OUTPUT : None
DESCRIPTION  : Called by INT_USB when a Status OUT is received.
-----------------------------------------------------------------------------*/
void User_Status_Out(void) {
  if (USBbRequest == DFU_GETSTATUS) {
    if (DFUDeviceState == DFU_STATE_DFUDNBUSY) DFUAction |= DFU_ACTION_START;
    if (DFUDeviceState == DFU_STATE_DFUMANIFESTWAITRESET) DFU_ManifestWaitReset_User(); // Call-back function      
  }
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : DFU_Main
INPUT/OUTPUT : None
DESCRIPTION  : Entry point after MCU reset
-----------------------------------------------------------------------------*/
void DFU_Main(void) { 

  //-------------------
  // RAM Initialization
  //-------------------

  #asm
    // Init Page 0 (40h to FFh)
    LD   X,#0x40;
  loop0:
    CLR  (X);
    INC	 X;
    JRNE loop0;
  #endasm
  
/***
    // Init Stack (100h to 17Fh) + beginning of 16-bits RAM (180h to 1FFh)
    CLR	 A;
  loop1:
    LD	 (0x100,X),A;
    INC	 X;
    JRNE loop1;
    // Init rest of 16-bits RAM (200h to ...)
  #ifdef ST7262
  loop2:
    LD	 (0x200,X),A;
    INC	 X;
    JRNE loop2;
  loop3:
    LD	 (0x300,X),A;
    INC	 X;
    CP   X,#0x40;
    JRNE loop3;
  #endif
  #ifdef ST7263B
  loop2:
    LD	 (0x200,X),A;
    INC	 X;
    CP   X,#0x40;
    JRNE loop2;
  #endif
***/
  
  if (VbusCheck() == 0) {
    RASS_Disable(0, 0); // Protect the Flash against unwanted programmation
    Vec_main(); // Jump to the main routine if the USB cable is removed 
  }

  else { // If USB cable is plugged-in then boot in DFU Mode only
    
    // ALWAYS OK IN THIS PROJECT BECAUSE WE DON'T CHECK IF THE UPGRADE WAS CORRECT
    DFUDeviceState = DFU_STATE_DFUIDLE;
    DFUDeviceStatus = DFU_STATUS_OK;
        
    #asm
#ifdef COSMIC
      JP _DFU_Process;
#endif
#ifdef HIWARE          
      JP DFU_Process;
#endif
    #endasm

  }

}

/*-----------------------------------------------------------------------------
ROUTINE NAME : DFU_Process
INPUT/OUTPUT : None
DESCRIPTION  : This function is entered either after a DFU_DETACH when application
               is running, or when firmware is corrupted.
-----------------------------------------------------------------------------*/
void DFU_Process(void)
{

    DFU_Init();

    EnableInterrupts;

    while (1) {
    
      if (USBLibStatus & USB_SUSPEND) {
        #asm
          HALT;
        #endasm
      }

      else { // Not Suspend        		 	  		 

        Handle_USB_Events(); // Process USB Flow and STANDARD requests on EP0		

        if (USBLibStatus & APP_REQUEST) {
          if ((USBbmRequestType & TYPE) == CLASS) DFU_HandleRequests(); // Process DFU requests on EP0
          Enable_STATUS_Stage(); // Enable ACKing/STALLing STATUS Stage according to USBLibStatus
        } 		  	 		  	 	  		 		  	 	
        else HDFlashProcess(); // Call-back function

      } // Not Suspend
    
    } // while(1)

}

/*-----------------------------------------------------------------------------
ROUTINE NAME : DFU_Init
INPUT/OUTPUT : None
DESCRIPTION  : DFU variables initialization
-----------------------------------------------------------------------------*/
void DFU_Init(void)
{ 
  USBLibStatus = DFU_MODE;
  DFU_Init_User(); // Call-back function
}
      
/*-----------------------------------------------------------------------------
ROUTINE NAME : DFU_HandleRequests
INPUT/OUTPUT : None
DESCRIPTION  : Handle DFU Class specific requests
-----------------------------------------------------------------------------*/
void DFU_HandleRequests(void)
{  
  switch (USBbRequest) {
    case DFU_DNLOAD :       DFU_Download();      break;
    case DFU_UPLOAD :       DFU_Upload();        break;
    case DFU_GETSTATUS :    DFU_GetStatus();     break;
    case DFU_CLRSTATUS :    DFU_ClearStatus();   break;
    case DFU_GETSTATE :     DFU_GetState();      break;
    case DFU_ABORT :        DFU_Abort();         break;
  }
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : DFU_Download
INPUT/OUTPUT : None
DESCRIPTION  : 
-----------------------------------------------------------------------------*/
void DFU_Download(void)
{

  if (DFUDeviceState == DFU_STATE_DFUIDLE) {
    DFUDeviceState = DFU_STATE_DFUDNLOADSYNC;
  }
  else {
    if (USBwLength == 0)        
      DFUDeviceState = DFU_STATE_DFUMANIFESTSYNC;
    else
      DFUDeviceState = DFU_STATE_DFUDNLOADSYNC;
  }  

  DFUDeviceStatus = DFU_STATUS_OK;
  USBLibStatus &= ~APP_REQUEST;
  
  DFU_Download_User(); // Call-back function

}

/*-----------------------------------------------------------------------------
ROUTINE NAME : DFU_Upload
INPUT/OUTPUT : None
DESCRIPTION  : 
-----------------------------------------------------------------------------*/
void DFU_Upload(void)
{ 
  DFU_UpLoad_User(); // Call-back function
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : DFU_GetStatus
INPUT/OUTPUT : None
DESCRIPTION  : Send to the host usefull informations to facilitate synchronization.
-----------------------------------------------------------------------------*/
void DFU_GetStatus(void)
{
  
  if (DFUDeviceState == DFU_STATE_DFUDNLOADSYNC) {
    if (DFUAction != DFU_ACTION_NONE) { // Check if there is an action pending
      DFUDeviceState = DFU_STATE_DFUDNBUSY;
    }
    else DFUDeviceState = DFU_STATE_DFUDNLOADIDLE;
  }

  if (DFUDeviceState == DFU_STATE_DFUMANIFESTSYNC) {
    DFUDeviceState = DFU_STATE_DFUMANIFESTWAITRESET;
  }
    
  //---------------------
  // Prepare data to send
  //---------------------
  DFUDataToSend[0] = DFUDeviceStatus;
  DFUDataToSend[1] = DFUPollTimeOutL;
  DFUDataToSend[2] = DFUPollTimeOutH;
  DFUDataToSend[3] = 0; // PollTimeOut Upper byte (not used) 
  DFUDataToSend[4] = DFUDeviceState;
  DFUDataToSend[5] = 0; // iString : index of status description (not used)
    
  //--------------------------
  // Write data into EP buffer
  //--------------------------
  Write_EP_Buffer(0, DFUDataToSend, 6);
  Set_EP_Ready(0, EP_IN, 6);

  USBLibStatus &= ~APP_REQUEST;

}

/*-----------------------------------------------------------------------------
ROUTINE NAME : DFU_ClearStatus
INPUT/OUTPUT : None
DESCRIPTION  : 
-----------------------------------------------------------------------------*/
void DFU_ClearStatus(void)
{
  DFUDeviceStatus = DFU_STATUS_OK;
  DFUDeviceState = DFU_STATE_DFUIDLE;
  USBLibStatus &= ~APP_REQUEST;
  DFU_ClearStatus_User(); // Call-back function
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : DFU_GetState
INPUT/OUTPUT : None
DESCRIPTION  : Send current device state to the host (1 byte)
-----------------------------------------------------------------------------*/
void DFU_GetState(void)
{
  DFUDataToSend[0] = DFUDeviceState;
  Write_EP_Buffer(0, DFUDataToSend, 1);
  Set_EP_Ready(0, EP_IN, 1);
  USBLibStatus &= ~APP_REQUEST;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : DFU_Abort
INPUT/OUTPUT : None
DESCRIPTION  : 
-----------------------------------------------------------------------------*/
void DFU_Abort(void)
{
  DFUDeviceStatus = DFU_STATUS_OK;      
  DFUDeviceState = DFU_STATE_DFUIDLE;
  USBLibStatus &= ~APP_REQUEST;
  DFU_Abort_User(); // Call-back function
}

#ifdef COSMIC
#pragma section ()
#endif

#ifdef HIWARE
#pragma CODE_SEG DEFAULT
#endif

/*** END OF FILE ***/