/**************** (c) 2003 STMicroelectronics **********************

PROJECT  : DALI "gateway" Slave
COMPILER : Cosmic

MODULE  :  dali_pub.c
VERSION :  2

CREATION DATE : 11.2001 

AUTHOR : 8-bit Micro Application Team 

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

DESCRIPTION :   Dali Public

COMMENT: To understand this module,
 it is really mandatory to understand the DALI specifications and the DALI tests procedures.
         
              
 ******************************************************************************
 THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. ST MICROELECTRONICS
 SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
 WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
 ******************************************************************************

******************************************************************************/

/* Include files */  
#include "lib.h"
#include "debug.h"
                
#include "globals.h"
#include "dali_pub.h"
#include "dali_regs.h"
#include "lite_timer_8bit.h"   
#include "pwm_ar_timer_12bit.h"  
#include "eeprom.h"
#include "ports.h" /* used to debug */       

u32 DALIP_iChangeEvery;
u32 DALIP_iChangeCountdown;
u8 DALIP_bIncrease;
u8 DALIP_iMaxLevel,DALIP_iMinLevel;
u8 DALIP_DTR;
u8 DALIP_FadeGoal;

#ifdef USE_ARC_TABLE
  #define DALIP_ConvertARC(a) DALIP_ArcTable[a]
#else
  #define DALIP_ConvertARC(a) a
#endif

/**************************************************************************
 * These are the two Regs which are read only and therefore stored in ROM *
 * The first is								  *
 * - Version number   and the second					  *
 * - Physical minimum Level						  *
 * You might want to adjust these.					  *
 **************************************************************************/
#pragma INTO_ROM
const u8 ROMRegs[]={0,25}; // ALAL old /* {Version Number, Phys. Min Level} */
//const u8 ROMRegs[]={0,125}; //SESE new value for Phys. min level to test it /* {Version Number, Phys. Min Level} */
#pragma INTO_ROM
const u32 DALIP_FadeTimeTable[]={ 
      0,
    707,
   1000,
   1414,
   2000,
   2828,
   4000,
   5657,
   8000,
  11314,
  16000,
  22627,
  32000,
  45255,
  64000,
  90510
};

#pragma INTO_ROM
const u16 DALIP_FadeRateTable[]={       //ALALremark ms/step
    0,
    3,
    4,
    6,
    8,
   11,
   16,
   22,
   32,
   45,
   63,
   89,
  126,
  178,
  252,
  357
};
/**********************************************
 * Fill in the Numbers (see Spec)             *
 * then uncomment                             *
 **********************************************/
/*********************************************
 * The Numbers can be calculated like this:  *
 * - on pg. 25 of the spec, use the X values *
 * - enter here (X * 254)/100                *
 *********************************************/

#ifdef USE_ARC_TABLE
#pragma INTO_ROM
const u8 DALIP_ArcTable[]={
  0  ,1	 ,2  ,3	 ,4  ,5	 ,6  ,7	 ,8  ,9	 ,10 ,11 ,12 ,13 ,14 ,15,
  16 ,17 ,18 ,19 ,20 ,21 ,22 ,23 ,24 ,25 ,26 ,27 ,28 ,29 ,30 ,31,
  32 ,33 ,34 ,35 ,36 ,37 ,38 ,39 ,40 ,41 ,42 ,43 ,44 ,45 ,46 ,47,
  48 ,49 ,50 ,51 ,52 ,53 ,54 ,55 ,56 ,57 ,58 ,59 ,60 ,61 ,62 ,63,
  64 ,65 ,66 ,67 ,68 ,69 ,70 ,71 ,72 ,73 ,74 ,75 ,76 ,77 ,78 ,79,
  80 ,81 ,82 ,83 ,84 ,85 ,86 ,87 ,88 ,89 ,90 ,91 ,92 ,93 ,94 ,95,
  96 ,97 ,98 ,99 ,100,101,102,103,104,105,106,107,108,109,110,111,
  112,113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,
  128,129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,
  144,145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,
  160,161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,
  176,177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,
  192,193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,
  208,209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,
  224,225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,
  240,241,242,243,244,245,246,247,248,249,250,251,252,253,254,255
};
#endif





void DALIP_LaunchTimer(u8 p)
{
  RTC_LaunchUserTimer(p);
}

void DALIP_DoneTimer(void){
  RTC_DoneUserTimer();
}

/***********************************************************
 * This is a ms-Timer, meaning that the function           *
 * DALIP_TimerCallback is called every 1ms precisely by    *
 * Timer A.						   *
 * You have to activate this Timer by			   *
 *   DALIP_LaunchTimer();                                  * 
 * and you should turn it off by using			   *
 *   DALIP_DoneTimer();					   *
 *                                                         *
 * NOTE: The Controller is programmed to go to		   *
 *       SLOW-WAIT-MODE (in order to save power) if it has *
 *       nothing more to do. It can only go to this mode   *
 *       when the user-timer is deactivated, so you should *
 *       call DALIP_DoneTimer() as soon as you're finished *
 ***********************************************************/
void DALIP_TimerCallback(void)
{
    u8 zw;
  
	if (DALIP_iChangeCountdown) 
	{
	    DALIP_iChangeCountdown--;
	} 
	else 
	{
		DALIP_iChangeCountdown = DALIP_iChangeEvery;
		zw = DALIP_GetArc();
		if (DALIP_bIncrease)
		{
			if (zw < DALIP_iMaxLevel) 
			{
			    zw++;
			    DALIP_SetArc(zw);
			} 
			else 
			{ 
				DALIP_DoneTimer();
				DALIP_SetFadeReadyFlag(0); /* fade is ready */ 	
			}		  
		} 
		else 
		{	  	
			if (zw > DALIP_iMinLevel) 
			{
				zw--;
				DALIP_SetArc(zw);
			} 
			else 
			{
	            DALIP_DoneTimer();
				DALIP_SetFadeReadyFlag(0); /* fade is ready */
			}
        }
	    AR_TIMER_Set_PWM(DALIP_ConvertARC(zw));
		if (zw == DALIP_FadeGoal) 
		{  
			DALIP_DoneTimer();
			DALIP_SetFadeReadyFlag(0); /* fade is ready */
		}
    }
}

/*******************************************
 * DALI-Reg-Write-Functions                *
 *******************************************/

void DALIP_SetArc(u8 val)
{
    DALIR_WriteReg(DALIREG_ACTUAL_DIM_LEVEL,val);
}

void DALIP_SetBallastStatusFlag(u8 bit)
{
    DALIR_WriteStatusBit(0,bit);
}

void DALIP_SetLampFailureFlag(u8 bit)
{
    DALIR_WriteStatusBit(1,bit);
}

void DALIP_SetLampPowerOnFlag(u8 bit)
{
    DALIR_WriteStatusBit(2,bit);
}

void DALIP_SetFadeReadyFlag(u8 bit)
{
    DALIR_WriteStatusBit(4,bit);
}

void DALIP_SetPowerFailureFlag(u8 bit)
{
    DALIR_WriteStatusBit(7,bit);
}

/*******************************************
 * DALI-Reg-Read-Functions                 *
 *******************************************/ 
 
u8 DALIP_GetArc(void)
{
    return DALIR_ReadReg(DALIREG_ACTUAL_DIM_LEVEL);
}

u8 DALIP_GetFadeTime(void) 
{
    return DALIR_ReadReg(DALIREG_FADE_TIME);
}

u8 DALIP_GetFadeRate(void) 
{
    return DALIR_ReadReg(DALIREG_FADE_RATE);
}

u8 DALIP_GetMaxLevel(void) 
{
    return DALIR_ReadReg(DALIREG_MAX_LEVEL);
}

u8 DALIP_GetMinLevel(void) 
{
    return DALIR_ReadReg(DALIREG_MIN_LEVEL);
}

u8 DALIP_GetPowerOnLevel(void) 
{
    return DALIR_ReadReg(DALIREG_POWER_ON_LEVEL);
}

u8 DALIP_GetSysFailureLevel(void) 
{
    return DALIR_ReadReg(DALIREG_SYSTEM_FAILURE_LEVEL);
}

u8 DALIP_GetStatus(void) 
{
    return DALIR_ReadReg(DALIREG_STATUS_INFORMATION);
}

u8 DALIP_GetVersion(void) 
{
    return DALIR_ReadReg(DALIREG_VERSION_NUMBER);
}

u8 DALIP_GetPhysMinLevel(void) 
{
    return DALIR_ReadReg(DALIREG_PHYS_MIN_LEVEL);
}

/*******************************************
 * EPROM Access Functions                 *
 *******************************************/

u8 DALIP_EEPROM_Size(void)
{
	u16 zwms;
	zwms = E2_PHYSICAL_SIZE;
	zwms -= DALIREG_EEPROM_END-DALIREG_EEPROM_START+4;
	return (u8) zwms;
}

u8 DALIP_Read_E2(u8 addr)
{
	if (addr >= DALIP_EEPROM_Size()) return 0;
	return E2_ReadMem(addr+DALIREG_EEPROM_END-DALIREG_EEPROM_START);
}

void DALIP_Write_E2(u8 addr, u8 data)
{
	if (addr >= DALIP_EEPROM_Size()) return;
	E2_WriteMem(addr+DALIREG_EEPROM_END-DALIREG_EEPROM_START,data);
}

void DALIP_Write_E2_Buffer(u8 addr, u8 number, u8*buf)
{
	u8 zw;
	
	zw = DALIP_EEPROM_Size();
	if (addr >= zw) return;
	zw -= addr;
	if (number > zw) return;
	E2_WriteBurst(addr,number,buf);
}
  

/*******************************************
 * Reserved-Functions                       *
 *******************************************/

void DALIP_Reserved_Function(u8 cmd)
{

}

void DALIP_Reserved_Special_Function(u8 cmd, u8 data)
{

}

/*********************************************************************
 * Note:-The Flags
 *       "Status of Ballast",
 *	 "Lamp Failure",
 *       "Lamp arc power on",
 *       "Fade Ready" and the
 *       "Arc Power Level" Reg 
 *       have to be set by the Ballast-code!
 *       see dali_pub.h, "Write-Functions".
 *
 *      -All routines have to return AS QUICKLY AS POSSIBLE!
 *       The Slave is not able to process DALI-Commands while in a
 *       User-Function. So, for instance, if there's a fade-command,
 *       use the Timer-Interrupt rather that a waiting-loop.
 *       (Don't forget to set the "Fade Ready Flag" to false while
 *       fading).
 *********************************************************************/

void DALIP_Direct_Arc(u8 val)
{
	u8 iActVal;
	u8 iActFT;
	
	iActVal = DALIP_GetArc();
	if (iActVal == val) return;
	    iActFT = DALIP_GetFadeTime();
	
	DALIP_DoneTimer();
	if (iActFT == 0)
    {
		DALIP_SetArc(val);
		AR_TIMER_Set_PWM(DALIP_ConvertARC(val));  
	} 
    else 
    {
		if (iActVal > val)
        {
			DALIP_iMinLevel = DALIP_GetMinLevel();
			DALIP_bIncrease = 0;
			DALIP_iChangeEvery = (DALIP_FadeTimeTable[iActFT]/((long)(iActVal - val)))-1;
	    } 
        else 
        {
			DALIP_iMaxLevel = DALIP_GetMaxLevel();
			DALIP_bIncrease = 1;
			DALIP_iChangeEvery = (DALIP_FadeTimeTable[iActFT]/((long)(val - iActVal)))-1;
		}
		DALIP_iChangeCountdown = DALIP_iChangeEvery;
		DALIP_SetFadeReadyFlag(1);                       /* Fade running from now */
		DALIP_FadeGoal = val;
		DALIP_LaunchTimer(0xFF);
    }  
}

#ifdef USE_ARC_FIXPOINTS

//u8 DALIP_FixPoints[9];

//RNRN void DALIP_Init(void){
/*  DALIP_FixPoints[0]=0;
  DALIP_FixPoints[1]=1*32;
  DALIP_FixPoints[2]=2*32;
  DALIP_FixPoints[3]=3*32;
  DALIP_FixPoints[4]=4*32;
  DALIP_FixPoints[5]=5*32;
  DALIP_FixPoints[6]=6*32;
  DALIP_FixPoints[7]=7*32;
  DALIP_FixPoints[8]=254;*/
//RNRN  AR_TIMER_Init();
//}

/*
// y_x = y_1 + (xofs(y_2 - y_1)) / 32 
u8 DALIP_ConvertARC(u8 oldarc){
  u16 res;

  res = DALIP_FixPoints[(oldarc >> 5)+1] - DALIP_FixPoints[oldarc >> 5];
  res *= (u16) (oldarc & 0x1F);
  res >>= 5;
  res += DALIP_FixPoints[oldarc >> 5];
  return res;
}*/
#endif


void DALIP_Off(void)
{
	DALIP_SetArc(0);
	AR_TIMER_Set_PWM(0);
}

void DALIP_Up(void)
{
	u8 zw;
	
	DALIP_DoneTimer();
	if (DALIR_ReadStatusBit(DALIREG_STATUS_FADE_READY)==0)
    {  
	    zw = DALIP_GetFadeRate();
		if (zw == 0) 
        {
			zw = DALIP_GetArc();
			zw++;
			DALIP_SetArc(zw);
			AR_TIMER_Set_PWM(DALIP_ConvertARC(zw));
	    }
		else 
        {
			DALIP_iMaxLevel = DALIP_GetMaxLevel();
			DALIP_bIncrease = 1;
			DALIP_iChangeEvery = DALIP_FadeRateTable[zw]-1;
			DALIP_iChangeCountdown = DALIP_iChangeEvery;
			DALIP_SetFadeReadyFlag(1);      /* Fade running from now */
			DALIP_FadeGoal = 255;
			DALIP_LaunchTimer(200);
		}
	}
	else 
    {
	    DALIP_LaunchTimer(200);
	}
}       

void DALIP_Down(void)
{
	u8 zw;
	  
    DALIP_DoneTimer(); 
	if (DALIR_ReadStatusBit(DALIREG_STATUS_FADE_READY)==0)
    {
		zw = DALIP_GetFadeRate();
		if (zw == 0)
        {
			zw = DALIP_GetArc();
			zw--;
			DALIP_SetArc(zw);
			AR_TIMER_Set_PWM(DALIP_ConvertARC(zw));
		}
		else 
        {
			DALIP_bIncrease = 0;
			DALIP_iMinLevel = DALIP_GetMinLevel();
			DALIP_iChangeEvery = DALIP_FadeRateTable[zw]-1;
			DALIP_iChangeCountdown = DALIP_iChangeEvery;
			DALIP_SetFadeReadyFlag(1);      /* Fade running from now */
			DALIP_FadeGoal = 255;
			DALIP_LaunchTimer(200);
		}
	}
	else 
    {
	    DALIP_LaunchTimer(200);
	}
}  

void DALIP_Step_Up(void)
{
	u8 zw;
	
	DALIP_DoneTimer();
	zw = DALIP_GetArc();
	zw++;
	DALIP_SetArc(zw);
	AR_TIMER_Set_PWM(DALIP_ConvertARC(zw));
}

void DALIP_Step_Down(void)
{
	u8 zw;
	
	DALIP_DoneTimer();
	zw = DALIP_GetArc();
	zw--;
	DALIP_SetArc(zw);
	AR_TIMER_Set_PWM(DALIP_ConvertARC(zw));
}

void DALIP_Step_Down_And_Off(void)
{
    DALIP_Step_Down();
}

void DALIP_On_And_Step_Up(void)
{
    DALIP_Step_Up();
}

/* is THIS device of the specified device_type? (0 = NO) */
//u8 DALIP_Is_Device_Type(u8 device_type){      //ALAL function useless (DALIP_What_Device_Type can be used instead)
//  return (device_type==0); 			//ALAL 0 => fluorescent lamp
//}                                              

/* 1 for yes */
u8 DALIP_Is_Physically_Selected(void)
{
    return 0;
}

/* return the device type (see spec.) */
u8 DALIP_What_Device_Type(void)
{
    return 0; 					// 0 => fluorescent lamp
}

/********************************************************************************
 * Extended Commands -----------------------------------------------------------*
 * -----------------------------------------------------------------------------*
 *										*
 * Note:-These Commands are specific for certain device types. If you don't need*
 * extended commands, just leave those functions returning 0, they will not be  *
 * called anyway, if you don't support them. (Supposed, the Master-Software     *
 * works correctly.) See special appendixes for the distinct device types.      *
 *-the passed parameter "command" is the second byte of the received DALI-frame *
 * (the first byte contains only the short-address, according to DALI-Spec.)    *
 *******************************************************************************/

/****************************************************************************
 * should return !=0 if this extended command (param 1) Requires an answer
 * to be sent to the master.
 * If this function returns 0, the return-value of DALIP_Extended_Command
 * will be ignored.
 ****************************************************************************/
u8 DALIP_Ext_Cmd_Is_Answer_Required(u8 command)
{
    return 0;
}

/****************************************************************************
 * should return !=0 if this command's answer is only yes/no.
 * in that case, the return value of DALIP_Extended_Command will be
 * interpreted as boolean, so, if DALIP_Extended_Command returns 0, there will
 * be the timeout-answer to the master
 ****************************************************************************/
u8 DALIP_Ext_Cmd_Is_Answer_YesNo(u8 command)
{
    return 0;
}

/****************************************************************************
 * an extended command (param 1) has been called.
 * return-value will be interpreted as explained above.
 ****************************************************************************/
u8 DALIP_Extended_Command(u8 command)
{
    return 0;
}

/*** (c) 2003  STMicroelectronics ****************** END OF FILE ***/
