/******************************************************************************
COPYRIGHT 2002 STMicroelectronics
Source File Name : sci.c 
Group            : IPSW,CMG-IPDF
Author           : MCD Application Team
Date First Issued: 5/3/2002
********************************Documentation**********************************
General Purpose - Contains source code for all the functions of SCI

********************************Revision History*******************************
_______________________________________________________________________________
Date :5/3/2002			Release:1.0
Date March 2004		Release:2.0: customized for ST7MC; now uses the HW 
registers description coming with STVD7 3.0

Date Sept 2004 		Release: 2.0.1
Replace ST7FMC2N6.h by config.h for better flexibility.
Add strcatu16 prototype and replace SCI_IT_Function by interrupt 
for interrupt driven communication purposes.
Add string.h (cosmic only) to use strcat.
  
******************************************************************************/

// Library configuration and ST7FMC peripherals Hardware Registers declaration
// Also contains polling / interrupt driven communication choice
#include "config.h"
#include "Sci.h"                                   /*Declaration of functions*/
#if defined (__CSMC__)
	#include <string.h>
#endif


static volatile unsigned char SCI_Error;    /*Declaration of Global Variables*/
static volatile unsigned char SCI_MY_DATA;
static const volatile unsigned char *PtrToSCIBuffTx; 
static volatile unsigned char *PtrToSCIBuffRx;
static volatile unsigned char SCI_Tx_Buff;
static volatile unsigned char SCI_Tx_String;
static volatile unsigned int SCI_Rx_Buff;
static volatile unsigned char SCI_Rx_String;
static volatile unsigned char SCI_Pol_Stat;  


void PrintUnsignedInt( unsigned int x )
{
        char b[6], *s = b + 5;

        *s-- = '\0';
        do
        {
                *s-- = (char)(x % 10 + '0');
                x /= 10;
        } while( x );
        
        SCI_PutString( s+1 );
}

void PrintUnsignedChar( unsigned char x )
{
        char b[4], *s = b + 3;

        *s-- = '\0';
        do
        {
                *s-- = (char)(x % 10 + '0');
                x /= 10;
        } while( x );
        
        SCI_PutString( s+1 );
}

void PrintSignedChar( signed char x )
{
		char b[5], *s = b + 4;

	*s-- = '\0';

	if (x >= 0)
	{
		do
		{
			*s-- = (char)(x % 10 + '0');
			x /= 10;
		} while( x );
		*s-- = '+';
	}
	else
	{
		do
		{
			*s-- = (char)(x % 10 + '0');
			x /= 10;
		} while( x );
		*s-- = '-';
	}  
	SCI_PutString( s+1 );
}


void strcatu16(const unsigned char *PtrtoString, unsigned int x )
{
        char b[6], *s = b + 5;

        *s-- = '\0';
        do
        {
                *s-- = (char)(x % 10 + '0');
                x /= 10;
        } while( x );
        
        strcat(PtrtoString, s+1);
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SCI_Init
INPUT        : 1.SCI_DEFAULT_PARAM1, SCI_WORDLENGTH_9,SCI_ODPARITY_SELECT,
				 SCI_EVPARITY_SELECT or SCI_WAKEUP_ADDR
               2.SCI_MUTE_ENABLE,SCI_BREAK_ENABLE,SCI_DEFAULT_PARAM2
OUTPUT       : None	 		 
DESCRIPTION  : Initializes the SCI in the required mode
COMMENTS     : Must be called before starting any sci operation
-----------------------------------------------------------------------------*/
void SCI_Init ( SCI_Type_Param1  Init_Value1, SCI_Type_Param2  Init_Value2)
{   
    unsigned char temp;	
    
    SCICR1 = SCI_DEFAULT_PARAM1;                         /*Load Default value*/
    SCICR2 = SCI_DEFAULT_PARAM2;
    temp = SCISR;                    /*Clear all the flags in Status register*/
    temp = SCIDR;
    SCIDR = SCI_DEFAULT_PARAM1;
    SCICR1 |= Init_Value1;                    /*Sets SCI in the required mode*/
    SCICR2 |= Init_Value2;
    SCI_Pol_Stat = SCI_Error = 0;               /*Initializes globalvariables*/
    SCI_Tx_Buff = 0;
    SCI_Tx_String = 0;
    SCI_Rx_Buff = SCI_Rx_String = 0; 		
}
/*-----------------------------------------------------------------------------
ROUTINE NAME : SCI_Mode
INPUT        : SCI_TX_ENABLE or SCI_RX_ENABLE
OUTPUT       : None
DESCRIPTION  : Selects the transmitter or receiver mode for SCI
COMMENTS     : None
-----------------------------------------------------------------------------*/
void SCI_Mode(SCI_Mode_Type SCI_Mode_Param)
{       
    SCICR2 |= SCI_Mode_Param;          /*Selects Transmitter or Receiver mode*/
}
/*-----------------------------------------------------------------------------
ROUTINE NAME : SCI_Forced_Clear_Flag
INPUT        : None
OUTPUT       : None  	
DESCRIPTION  : Clears all status flag performs both read and write operation 
COMMENTS     : Can be used if clearing of all status flag is needed
-----------------------------------------------------------------------------*/
void SCI_Forced_Clear_Flag( void )
{
    unsigned char Temp_Variable,temp;     
	
    Temp_Variable = SCICR2; 
    SCICR2 = 0x00;      
    temp = SCISR;                       /*Clears all flags in status register*/
    SCIDR = 0x00; 
    SCICR2 = Temp_Variable;
} 
/*-----------------------------------------------------------------------------
ROUTINE NAME : SCI_IT_Enable
INPUT        : SCI_IDLE_LINE,SCI_RECEIVE_OVERRUN,SCI_TRANSMIT_REGISTER_READY,
               SCI_FRAME_TRANSMITTED,SCI_PARITY_ERROR
OUTPUT	     : None	
DESCRIPTION  : Reset the interrupt masking
COMMENTS     : None
-----------------------------------------------------------------------------*/
void SCI_IT_Enable ( SCI_IT_Type SCI_IT_Param)
{      
    SCICR1 |= (unsigned char)(SCI_IT_Param & SCI_PARITY_ERROR);   
                                             /*Enables the required interrupt*/
    SCICR2 |= (unsigned char)(SCI_IT_Param & (~SCI_PARITY_ERROR)) ; 
}                                                                                  

/*-----------------------------------------------------------------------------
ROUTINE NAME    : SCI_Select_Baudrate
INPUT           : SCI_PR_X+SCI_TR_Y+SCI_RR_Z
OUTPUT          : None
DESCRIPTION     : Selects possible baudrate for transmission and reception
COMMENTS        : Values are to be precalculated by the user
-----------------------------------------------------------------------------*/
void SCI_Select_Baudrate (SCI_Baudrate_Type Baudrate_Presc)
{
    SCIBRR =0x00;
    SCIBRR |= Baudrate_Presc;            /*Loads the required value in SCIBRR*/
}

/*-----------------------------------------------------------------------------
ROUTINE NAME    : SCI_Extend_BaudRate
INPUT           : SCI_PR_X+SCI_TR_Y+RR_Z,ETPR,ERPR 
OUTPUT          : None
DESCRIPTION     : Selects the possible baudrate with extended prescalar	
COMMENTS        : Values are to be precalculated by the user
-----------------------------------------------------------------------------*/
#if (defined SCI_72F521 | defined SCI_72F62)
void SCI_Extend_Baudrate (SCI_Baudrate_Type Baudrate_Presc,
                                         unsigned char ETPR,unsigned char ERPR)
{
    SCIBRR = 0x00;
    SCIBRR |= Baudrate_Presc;               /*Loads the required value SCIBRR*/
    SCIETPR = ETPR;                    /*loads the required value in SCIETPR */
    SCIERPR = ERPR;                    /*loads the required value in SCIERPR */
}  
#endif
/*-----------------------------------------------------------------------------
ROUTINE NAME    : SCI_PutByte
INPUT           : Tx_DATA
OUTPUT          : None		
DESCRIPTION     : Transmits a single data byte
COMMENTS        : None
-----------------------------------------------------------------------------*/
void SCI_PutByte (unsigned char Tx_DATA)
{  
    unsigned char temp;
    
#ifdef SCI_POLLING_TX
    temp = SCISR;                          /*Loads the byte to be transmitted*/
    SCIDR = Tx_DATA;
#endif

#ifdef SCI_ITDRV_WITHOUTBUF_TX	
    SCI_Tx_Buff = 1;                               /*Loads the default value */
    SCI_Tx_String = 0;
    temp = SCISR; 
    SCIDR = Tx_DATA;                       /*Loads the byte to be transmitted*/
    SCICR2 |= (unsigned char)SCICR2_TIE_OR;  //Enable INterrupt
#endif
}
/*-----------------------------------------------------------------------------
ROUTINE NAME    : SCI_IsTransmitCompleted
INPUT           : None    
OUTPUT          : TRUE or FALSE	
DESCRIPTION     : Checks if the transmission is completed or not
COMMENTS        : Must be called after Put_Byte or Put_Buffer or Put_String
-----------------------------------------------------------------------------*/
BOOL SCI_IsTransmitCompleted(void)
{
#if defined SCI_POLLING_TX
    if(SCISR & SCISR_TC_OR)                      /*Checks if transmission is completed*/
    {
        return (TRUE);
    }
    else 
    {
        return(FALSE);
    }
#else
#if defined SCI_ITDRV_WITHOUTBUF_TX 
   if (SCI_Tx_String == 1)
   {
        if((*PtrToSCIBuffTx == '\0') && (SCISR & SCISR_TDRE_OR)) 
        {
            return (TRUE);
        }
        else
        {
           return (FALSE);
        }
   }
   else if (SCI_Tx_Buff == 0)
   {
        return (TRUE);
   }
   else
   {
        return (FALSE);
   }

#else
       return(TRUE);                                 /*Dummy return statement*/
#endif    
#endif
}
/*-----------------------------------------------------------------------------
ROUTINE NAME    : SCI_IT_Disable
INPUT           : SCI_IDLE_LINE,SCI_RECEIVE_OVERRUN,SCI_TRANSMIT_REGISTER_READY
                 ,SCI_FRAME_TRANSMITTED,SCI_PARITY_ERROR
OUTPUT          : None
DESCRIPTION     : Disables the interrupt due to a particular flag
COMMENTS        : None
-----------------------------------------------------------------------------*/
void SCI_IT_Disable(SCI_IT_Type SCI_IT_Param)
{ 
   SCICR2 &= (unsigned char)(~(SCI_IT_Param & (~SCI_PARITY_ERROR)));  
                                            /*disables the selected interrupt*/
   SCICR1 &= (unsigned char)(~(SCI_IT_Param & SCI_PARITY_ERROR));     
}

/*-----------------------------------------------------------------------------
ROUTINE NAME    : SCI_PutBuffer
INPUT           : *PtrtoBuffer,NbOfBytes
OUTPUT          : None
DESCRIPTION     : Transmits  data from the user buffer in polling mode and 
                  starts transmission in interrupt driven with buffer mode
COMMENTS        : None
-----------------------------------------------------------------------------*/
void SCI_PutBuffer (const unsigned char *PtrtoBuffer, unsigned char NbOfBytes)
{  
    unsigned char temp;
#ifdef SCI_POLLING_TX
    for (; NbOfBytes > 0 ; NbOfBytes--)
    { 
   	    temp = SCISR;
        SCIDR = *PtrtoBuffer;              /*Loads the byte to be transmitted*/
        while (!(SCI_IsTransmitCompleted()));  
                                   /*Waits for the completion of transmission*/
        PtrtoBuffer++;               /*Moves the pointer to the next location*/
    } 
#endif

#ifdef SCI_ITDRV_WITHOUTBUF_TX  
    PtrToSCIBuffTx  = PtrtoBuffer; /*copies the user data to global variables*/    
    SCI_Tx_Buff = NbOfBytes;
    SCI_Tx_String = 0;
    temp = SCISR;
    SCIDR = *PtrToSCIBuffTx;        /*First byte is transmitted to generate the
                                                                    interrupt*/
   SCICR2 |= (unsigned char)SCICR2_TIE_OR;  //Enable INterrupt                                                                      
#endif
}
/*-----------------------------------------------------------------------------
ROUTINE NAME    : SCI_PutString
INPUT           : *PtrtoString
OUTPUT          : None
DESCRIPTION     : Transmits the user string in polling mode and starts 
                  transmission in interrupt driven mode
COMMENTS        : None
-----------------------------------------------------------------------------*/
void SCI_PutString(const unsigned char *PtrtoString)
{   
    unsigned char Val = *PtrtoString, temp;

#ifdef SCI_POLLING_TX 
    while (Val != '\0')                        /*Checks for the end of string*/
    {
        Val= *PtrtoString;					
        temp = SCISR;        
        SCIDR = *PtrtoString;        /*Copies the character to be transmitted*/
        while(!(SCI_IsTransmitCompleted()));   
                               /*waits for the completion of the transmission*/
        PtrtoString++;               /*Moves the pointer to the next location*/
    }	
#endif

#ifdef SCI_ITDRV_WITHOUTBUF_TX 
    PtrToSCIBuffTx = PtrtoString;   /*Copies the adress of the user string into
                                                          the global variable*/
    SCI_Tx_String = 1;               /*copies default value to recognize string 
                                     transmission in the interrupt subroutine*/
    SCI_Tx_Buff = 0;                                     
    temp = SCISR;
    SCIDR = *PtrToSCIBuffTx;/*transmits first character to generate interrupt*/
    SCICR2 |= (unsigned char)SCICR2_TIE_OR;  //Enable INterrupt
#endif
} 
/*-----------------------------------------------------------------------------
ROUTINE NAME    : SCI_9thBit_TxRx
INPUT           : TRUE or FALSE
OUTPUT          : TRUE or FALSE
DESCRIPTION     : Writes the 9th bit to be transmitted in the SCICR1 register 
                  and returns the 9th bit received
COMMENTS        : 9 bit mode must be enabled.
-----------------------------------------------------------------------------*/
BOOL SCI_9thBit_TxRx(BOOL Bit9_Val)
{
    if (Bit9_Val == TRUE)              /*If 9th bit is to be transmittes as 1*/
    {
        SCICR1 |= 0x40;
    }
    else
    {
        SCICR1 &= 0xBF;
    }
    if (SCICR1 & 0x80)                      /*If the 9th bit is received as 1*/
    {
        return (TRUE);
    }
    else
    {
        return (FALSE);
    }
}					
/*-----------------------------------------------------------------------------
ROUTINE NAME    : SCI_GetByte
INPUT           : None
OUTPUT          : Received Data
DESCRIPTION     : Retunrs a data received to the user
COMMENTS        : None
-----------------------------------------------------------------------------*/
unsigned char SCI_GetByte(void)
{  
#if defined SCI_POLLING_RX     
    return(SCIDR);                     /*Returns the current value of the Data 
                                                         register to the user*/
#else

#if defined SCI_ITDRV_WITHOUTBUF_RX 
    SCI_Error = 0x00;
    return(SCI_MY_DATA); 	  
#else	
    return (SCIDR);           
#endif
#endif
}
/*-----------------------------------------------------------------------------
ROUTINE NAME    : SCI_GetBuffer
INPUT           : *PtrtoBuffer,NbOfBytes
OUTPUT          :  Error Status
DESCRIPTION     : Receives a continuous data,stores it into the user buffer and
                  returns the error status to the user
COMMENTS        : Reception stops as soon as an error occurs
-----------------------------------------------------------------------------*/
#ifdef SCI_POLLING_RX
SCI_RxError_t SCI_GetBuffer(unsigned char *PtrtoBuffer,unsigned char NbOfBytes)
{
    unsigned char Temp = 0x00 ;
       
    for (;((NbOfBytes > 0)&& (!(Temp))); --NbOfBytes)           
                                    /*Checks if all the bytes are transmitted*/
    {   
        Temp |= (unsigned char)(SCISR & SCI_ErrValue);
        while (!(SCISR & SCISR_RDRF_OR)); 
        Temp |= (unsigned char)(SCISR & SCI_ErrValue);
                                           /*waits for the next received byte*/
        *PtrtoBuffer = SCIDR;  
                              /*copies the received byte into the user buffer*/
        PtrtoBuffer++;               /*Moves the pointer to the next location*/
    } 
    return(Temp);
}
#endif 
/*-----------------------------------------------------------------------------
ROUTINE NAME    : SCI_GetString
INPUT           : *PtrtoString
OUTPUT          : Error Status
DESCRIPTION     : Receives a string,Stores it in the user buffer and returns
                  the error status to the user
COMMENTS        : Reception stops as soon as an error occurs
-----------------------------------------------------------------------------*/
#ifdef SCI_POLLING_RX
SCI_RxError_t SCI_GetString(unsigned char *PtrtoString)
{
    unsigned char Temp = 0x00, Val = 0xff; 
 
    for (;((Val != '\0')&&(Temp == 0));PtrtoString++)
    {    
        Temp |= (unsigned char)(SCISR & SCI_ErrValue);   
        while (!(SCISR & SCISR_RDRF_OR));                /*waits for the received byte*/
        Temp |= (unsigned char)(SCISR & SCI_ErrValue);          
                   /*Temp stores any error that occurred during the reception*/
        *PtrtoString = SCIDR; 
                         /*Copies the received character into the user string*/
        Val = *PtrtoString;
    }  
    return(Temp);                      /*Returns the error status to the user*/
}
#endif 

/*-----------------------------------------------------------------------------
ROUTINE NAME	: SCI_GetBuffer
INPUT           : *PtrtoBuffer, NbOfBytes 
OUTPUT          : None	
DESCRIPTION     : Starts the reception in the interrupt subroutine
COMMENTS        : Any data received before calling this function is ignored
-----------------------------------------------------------------------------*/
#ifdef SCI_ITDRV_WITHOUTBUF_RX
void SCI_GetBuffer(unsigned char *PtrtoBuffer,unsigned char NbOfBytes) 
{
    PtrToSCIBuffRx = PtrtoBuffer;/*Copies user buffer to the global variable */
    SCI_Rx_Buff = NbOfBytes;
    SCI_Rx_Buff++;
    SCI_Error = 0x00;                                     
    SCICR2 |= (unsigned char)RDRF;    /*Enable Interrupt*/    
}                                                         

/*-----------------------------------------------------------------------------
ROUTINE NAME    : SCI_GetString
INPUT           : *PtrtoString  
OUTPUT          : None 
DESCRIPTION     : Starts the reception of a string in the interrupt subroutine
COMMENTS        : Any data received before calling this function is ignored
-----------------------------------------------------------------------------*/
void SCI_GetString(unsigned char *PtrtoString) 
{
    PtrToSCIBuffRx = (PtrtoString); /*Copies the user buffer to global buffer*/
    SCI_Rx_String = 1;                        /*For String recognition in ISR*/ 
    SCI_Error = 0x00;
    SCICR2 |= (unsigned char)RDRF;    /*Enable Interrupt*/    
}
#endif
/*-----------------------------------------------------------------------------
ROUTINE NAME    : SCI_IsReceptionCompleted
INPUT           : None
OUTPUT          : Error Status
DESCRIPTION     : Returns the error status occurred during the reception in the
                  interrupt subroutine if the reception is completed.
COMMENTS        : Must be called after SCI_GetBuffer and SCI_GetString in 
                  Interrupt Driven mode to get the reception status.
                  Must be called before SCI_GetByte in interrupt Driven mode.
-----------------------------------------------------------------------------*/
SCI_RxError_t SCI_IsReceptionCompleted (void)  
{   
#if defined SCI_ITDRV_WITHOUTBUF_RX          
    SCI_Pol_Stat = 0;          

    if ((SCI_Rx_String == 1) && (*(PtrToSCIBuffRx) != '\0') && (SCI_Error == 0))
                                        /*String reception request undergoing*/
    {
        return(SCI_STRING_ONGOING);
    }  
    else if ((SCI_Error == 0) && (SCI_Rx_Buff > 1)) 
                                        /*Buffer Reception request undergoing*/
    {   
        return(SCI_BUFFER_ONGOING) ;
    }
    else if ((SCI_Rx_Buff == 0) && (SCI_Rx_String == 0) && (SCI_Pol_Stat == 0))	
                                                   /*Single Byte not received*/
    {
        return (SCI_RX_DATA_EMPTY);	
    }
    else                                             /*Reception is completed*/
    {       
        SCI_Pol_Stat = 0; 
        SCI_Rx_Buff = 0;
        SCI_Rx_String = 0;
        return (SCI_Error);
    }		   
#else
#if defined SCI_POLLING_RX   	
    if(SCISR & SCISR_RDRF_OR) 
    {
        return ((unsigned char)(SCISR & SCI_ErrValue)); 
    }
    else
    {
        return (SCI_RX_DATA_EMPTY);    /*Returns the error status to the user*/
    }
#else 
        return (SCI_RX_DATA_EMPTY);                  /*Dummy return statement*/
#endif
#endif		
}     


#ifdef __HIWARE__		/* test for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	/* additional registers will be saved */
#else
#ifdef __CSMC__			/* test for Cosmic Compiler */
@interrupt			/* Cosmic interrupt handling */
#endif
#endif
/*-----------------------------------------------------------------------------
ROUTINE NAME    : SCI_IT_Function
INPUT           : None
OUTPUT          : None
DESCRIPTION     : Carries out communiction in Interrupt driven mode.
COMMENTS        : This function must be called inside the SCI interrupt service
                  routine for Interrupt driven communication
-----------------------------------------------------------------------------*/
void SCI_IT_Routine (void)
{                                                 
/******************************************************************************
Reception in Interrupt Driven Without Buffer mode
******************************************************************************/

#ifdef SCI_ITDRV_WITHOUTBUF_RX 
    unsigned char temp;  
    if(SCISR & ORE)          /*If Interrupt is generated due to overrun error*/
    {
        SCI_Error = (unsigned char)(SCISR & SCI_ErrValue);
        temp = SCIDR;	
        SCI_Pol_Stat = 2;
    } 
    else if (SCISR & RDRF)
    {        
        if((SCI_Rx_String == 1) && (*PtrToSCIBuffRx != '\0') && (SCI_Error == 0))     
                                                        /*If String Reception*/
        {       
            SCI_Error = (unsigned char)(SCISR & SCI_ErrValue);
            *PtrToSCIBuffRx = SCIDR;      
            if (*PtrToSCIBuffRx != '\0')
            {
                PtrToSCIBuffRx++;
            }		     
            else
            {
                SCICR2 &= (unsigned char)(~RDRF);   //Disableinterrupt
            }    
            SCI_Pol_Stat = 2;
        }
        else if ((SCI_Rx_Buff > 1) && (SCI_Error == 0))
        {              
            SCI_Error = (unsigned char)(SCISR & SCI_ErrValue) ;  
            *PtrToSCIBuffRx = SCIDR;
            SCI_Rx_Buff--;  
            SCI_Pol_Stat = 2;                       /*If a buffer is received*/
            PtrToSCIBuffRx++;  
            if(SCI_Rx_Buff == 1)
            { 
                SCICR2 &= (unsigned char)(~RDRF);   //Disableinterrupt
            }    
        }
        else if ((SCI_Rx_Buff == 0) && (SCI_Rx_String == 0))  
       /*If interrupt is generated before calling SCI_GetBuffer/SCI_GetString*/
        {  
            SCI_Error = (unsigned char)(SCISR & SCI_ErrValue) ;  
            SCI_MY_DATA = SCIDR;
            SCI_Pol_Stat = 1;
        }                    
        else
        {   
            SCI_Pol_Stat = 1;
            temp = SCIDR;
        }				
    }	
#endif 

/******************************************************************************
Transmission in Interrupt Driven without Buffer mode
******************************************************************************/

#ifdef SCI_ITDRV_WITHOUTBUF_TX 
    if((SCISR & SCISR_TC_OR)&& (SCICR2 & SCICR2_TCIE_OR))/*Checks if transmit complete flag is set*/
    {      
        if ((SCI_Tx_String == 1) && (*PtrToSCIBuffTx != '\0')) 
                                   /*Checks if complete string is transmitted*/
        {   
            PtrToSCIBuffTx++;
            SCIDR = *PtrToSCIBuffTx;
            if (*PtrToSCIBuffTx == '\0')
            {
                SCICR2 &= (unsigned char)(~SCICR2_TCIE_OR);  
                               /*Disable IT if complete string is transmitted*/
            } 
        } 
        else if (SCI_Tx_Buff > 1)   /*Checks if all the bytes are transmitted*/
        {
            SCI_Tx_Buff--; 
            PtrToSCIBuffTx++;            /*Moves the pointer to next location*/
            SCIDR = *PtrToSCIBuffTx;                /*Transmits a single byte*/
        }                  	    
        else if(SCI_Tx_Buff == 1)
        {
            SCI_Tx_Buff--;
            SCICR2 &= (unsigned char)(~SCICR2_TCIE_OR);     
                               /*Disable IT if complete string is transmitted*/
        }
    }
    else if ((SCISR & SCISR_TDRE_OR)&&(SCICR2 & SCICR2_TIE_OR))    /*Checks if TDRE Flag is set*/
    {           
        if ((SCI_Tx_String == 1)&&(*PtrToSCIBuffTx != '\0')) /*Checks if complete 
                                                        string is transmitted*/
        {   
           PtrToSCIBuffTx++;
           SCIDR = *PtrToSCIBuffTx;
           if (*PtrToSCIBuffTx == '\0')
           {    
               Nop();
               SCICR2 &= (unsigned char)(~SCICR2_TIE_OR);
                               /*Disable IT if complete string is transmitted*/
               Nop();
           }    

        }
        else if (SCI_Tx_Buff > 1)   /*Checks if all the bytes are transmitted*/
        {
            SCI_Tx_Buff--; 
            PtrToSCIBuffTx++;            /*Moves the pointer to next location*/
            SCIDR = *PtrToSCIBuffTx;                /*Transmits a single byte*/
        }
        else if(SCI_Tx_Buff == 1)
        {
            SCI_Tx_Buff--;
            SCICR2 &= (unsigned char)(~SCICR2_TIE_OR);
                               /*Disable IT if complete string is transmitted*/
        } 
    }	
#endif
}
/**** (c) 2002   STMicroelectronics *************************** END OF FILE **/