/**************** (c) 2004  STMicroelectronics ********************************
     
PROJECT  : 3-phase AC induction motor drive Software Library
COMPILER : COSMIC / METROWERKS

MODULE  :  acmotor.c
VERSION :  1.0.1

CREATION DATE : April 2004

AUTHOR :	Microcontroller Division Applications

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

DESCRIPTION :   3-phase AC motor control routines, include Start/Stop 
					management and PI regulation
              
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

MODIFICATIONS :

April 2005 (1.0.1): remove SLEW_LIMIT constant and place it in ACMparam.h.

*******************************************************************************
 THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. ST MICROELECTRONICS
 SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
 WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
******************************************************************************/

#include "lib.h"		/*General purpose typedefs and macros */
#include "mtc.h"		/*Public motor control peripheral function prototypes*/
#include "ACMparam.h"	/*AC Motor Control parameters */
#include "acmotor.h"	/*Public AC motor control function prototypes */
#include "pwmart.h"		/*Public general purpose Timebase function prototypes*/
#include "config.h"		/* AC motor library configuration keys */

#ifdef PI_PARAM_TUNING
	u8	Kp, Ki;
#else
	static u8	Kp, Ki;
#endif

static s32 VoltageIntegralTerm;                      
static s8 Error_slip;
static s32	DeltaVoltage_slip;
static s16	Voltage_slip;
static s32	DeltaVoltage_slip_s32;
static s32	Voltage_slip_s32;         
static BOOL MinPiOut;
static BOOL MaxPiOut;


#define	VOLT_INIT	((u8)0)		/* Voltage initialisation at MCU start-up */
#define	FREQ_INIT	((u16)50)	/* Freq initialisation at MCU start-up */

/*-----------------------------------------------------------------------------
ROUTINE Name : ACM_Init

Description: This function performs the initialisation of ACM module.
             Conditions for proper initialisation of ACM module:
             _ MTC functions called in below described order,
             _ Init Sinewave generation with Voltage = 0 to be sure that duty 
               cycle on each phase is 50% when starting.
-----------------------------------------------------------------------------*/
void ACM_Init(void)
{

	MTC_InitPeripheral();			// Initialize peripheral for three phase PWM generation
	MTC_InitSineGen();				// Configure SW variables for sinewave generation

   MTC_Set_ClockWise_Direction();

	ART_SetSpeedRegPeriod(SAMPLING_TIME);
	
}


/*-----------------------------------------------------------------------------
ROUTINE Name : ACM_VoltageMaxAllowed

Description: This function limits the voltage on motor stator according to
             setpoints given in ACMParam.h. It performs a linear interpolation.
             No error codes are returned if StatorFrequency is outside the 
             frequency domain given in MTCParam.h [VF_LOWFREQ_LIMIT/VF_HIGHFREQ_LIMIT].
Duration:
Important caution: when modifying setpoints, user must keep in mind that
            VF_COEFF, VF_OFFSET and local variable Buffer are declared as u16.
            They must therefore be declared as u32 there is chance for overflow to occur.
-----------------------------------------------------------------------------*/
u8 ACM_VoltageMaxAllowed(u16 StatorFrequency)
{
         u8  VoltageMax;

  if ( StatorFrequency <= VF_LOWFREQ_LIMIT )
  {
       VoltageMax = V_MIN;

  } else if ( StatorFrequency >= VF_HIGHFREQ_LIMIT )
         {
              VoltageMax = V_MAX;

         } else {
                       u16 Buffer;
                   Buffer = (u16)(StatorFrequency * VF_COEFF) - VF_OFFSET;
                   VoltageMax = (u8)(V_MIN + (u8)((u16)Buffer / (u16)256));
            /* 256 factor is included in VF COEFF and VF_OFFSET calculations */
                        /* Aim is to keep calculation accuracy */
                }

  return ( VoltageMax );
}


/*-----------------------------------------------------------------------------
ROUTINE Name : ACM_InitSoftStart

Description: This function initializes the soft start procedure: it forces 
				the voltage to be null, enable the PMW outputs and set-up the 
				timebases required to smoothly increase voltage and have a start-up
				time out.
-----------------------------------------------------------------------------*/
void ACM_InitSoftStart(u16 StatorFreq)
{
		u8 VoltageMax = ACM_VoltageMaxAllowed( StatorFreq );
		u16 TimeUnit;


	MTC_InitTachoMeasure( );
	MTC_UpdateSine(0, StatorFreq);
	MTC_EnableMCOutputs();			// PWM is present on MCO outputs

	TimeUnit = START_TIMEOUT / VoltageMax;
	if ( TimeUnit == 0 )
	{
		TimeUnit = 1;   /* Avoid applying full voltage directly */
	}
	ART_Set_TimeInMs(TimeUnit);		// Set soft start slope
		
	ART_SetSequenceDuration(START_TIMEOUT + EXTRA_TIMEOUT);

}


/*-----------------------------------------------------------------------------
ROUTINE Name : ACM_SoftStart

Description: This function provides a soft start which limits the inrush current 
				in the motor. It also monitors the speed feedback to stop 
				the voltage increase when a given minimum rotor speed is reached.
				The function ramps up the stator voltage linearly while the stator 
				frequency remains constant. The ramp ends when one of the following 
				conditions occurs:
				 the SOFTSTART_TIMEOUT is elapsed,
				 the motor starts and the rotor speed reaches MinRotorFreq.
				At the end of this ramp, if the rotor speed is too low, the tacho 
				is monitored for an additional duration (user defined OPTIONAL_TIMEOUT 
				constant). At the end of this time period, if the rotor speed is still 
				not high enough, the motor is stopped and PWM outputs are disabled.
Input:		MinRotorFreq is the minimum expected rotor frequency with [0.1Hz] unit
returns:		StartStatus_t, typedefed in acmotor.h

Note:			It is mandatory to call the ACM_InitSoftStart function before 
				calling this routine.
Caution: 	The function must be called as often as possible during the start-up
				phase to have a linear voltage profile and accurate Time out period.
-----------------------------------------------------------------------------*/
StartStatus_t ACM_SoftStart(u16 MinRotorFreq)
{
		u8 CurrentVoltage, VoltageMax;
		u16 StatorFreq;

	// Get current parameters
	StatorFreq  = MTC_GetStatorFreq();
	CurrentVoltage = MTC_GetVoltage();
	VoltageMax = ACM_VoltageMaxAllowed( StatorFreq );

	if ( MinRotorFreq >= StatorFreq )
	{
		return ( START_FAIL );	// Input parameter is not correct
	}

	if (MTC_ValidSpeedInfo(MinRotorFreq))
	{
		MTC_StartTachoFiltering();
		return (START_OK);		// Soft start completed successfully
	}
	else
	{
		if (ART_IsSequenceCompleted())
		{
			MTC_DisableMCOutputs();			// PWM is disabled on MCO outputs
			return (START_FAIL);          // Motor not correctly started after time out period
		}
		else
		{
			// Ramp-up voltage until its max value
			if ( (CurrentVoltage < VoltageMax) && (ART_Is_TimeInMsElapsed()) )
			{
				CurrentVoltage++;
				MTC_UpdateSine(CurrentVoltage, StatorFreq);
			}
			// else wait EXTRA_TIMEOUT with constant voltage
			return (START_ONGOING);
		}
	}

}

/*-----------------------------------------------------------------------------
ROUTINE Name : ACM_InitSoftStart_OL

Description: This function initializes the soft start procedure: it forces 
				the voltage to be null, enable the PMW outputs and set-up the 
				timebases required to smoothly increase voltage.
-----------------------------------------------------------------------------*/
void ACM_InitSoftStart_OL(u16 StatorFreq)
{


	MTC_UpdateSine(0, StatorFreq);
	MTC_EnableMCOutputs();			// PWM is present on MCO outputs

	if (VOLT_SLEWRATE == 0)
	{
		ART_Set_TimeInMs(1);	// Max slope: avoid applying full voltage directly
	}
	else
	{
		ART_Set_TimeInMs(VOLT_SLEWRATE);		// Set soft start slope
	}

}

/*-----------------------------------------------------------------------------
ROUTINE Name : ACM_SoftStartOL

Description: This function provides a Soft Start which limits the starting 
				torque and the inrush current in the motor. The voltage on the 
				stator windings is smoothly increased until it reaches the required
				value.
				Once the soft start period has been elapsed, the minimal interval 
				between two sinewave parameters changes (voltage, frequency) is set.
Input:		The voltage to be reached at the end of the soft start
returns:		Boolean, FALSE untill soft start completion

Note:			It is mandatory to call the ACM_InitSoftStart_OL function before 
				calling this routine.
Caution: 	The function must be called as often as possible during the start-up
				phase to have a linear voltage profile.
-----------------------------------------------------------------------------*/
BOOL ACM_SoftStartOL(u8 TargetVoltage)
{
		u8 CurrentVoltage;

	CurrentVoltage = MTC_GetVoltage();
	if (MTC_GetVoltage() < TargetVoltage)
	{
		if ( ART_Is_TimeInMsElapsed() )
		{
			CurrentVoltage++;
			MTC_UpdateSine(CurrentVoltage, MTC_GetStatorFreq());
		}
		return (FALSE);
	}
	else	// Target voltage reached -> end of Soft Start
	{
		ART_Set_TimeInMs(SLEW_LIMIT);
		return (TRUE);
	}

}

/*-----------------------------------------------------------------------------
ROUTINE Name : ACM_SustainSpeed

Description: Aim of this function is to maintain motor speed while maintaining
             the optimal slip frequency via regulation (closed loop mode).
-----------------------------------------------------------------------------*/
void ACM_SustainSpeed(u16 Time)
{
		u16 StatorFreq  = MTC_GetStatorFreq();

	ART_Set_TimeInMs(Time);

	while ( !ART_Is_TimeInMsElapsed() )		// This defines routine duration
	{
		if ( ART_IsRegPeriodElapsed() )		// Do periodically the regulation
		{
			u8  OptimumSlip, NewVoltage;

		OptimumSlip = ACM_GetOptimumSlip( StatorFreq );
		NewVoltage = ACM_SlipRegulation( OptimumSlip );

		MTC_UpdateSine(NewVoltage, StatorFreq);

		}  /* End of if: regulation performed */

	} /* End of while: completion of the routine */

}


/*-----------------------------------------------------------------------------
ROUTINE Name : ACM_InitSlipFreqReg

Description: This function must be called before starting regulation.
             It performs initialisation of the integral term of the PI regulator
             (VoltageIntegralTerm) and reset the clamping flags.
             This allow smooth transition from open loop to closed loop operation 
             (typically at the end of the starting phase).
-----------------------------------------------------------------------------*/
void ACM_InitSlipFreqReg(u8 OptimumSlip)
{
		u8 CurrentVoltage = MTC_GetVoltage();

	// PI error calculation and clamping if > u8 range
	{
			s16 Error;
		Error = MTC_GetSlip() - OptimumSlip;
		if (Error > (s16)S8_MAX)
		{
			Error_slip = S8_MAX;
		}
		else
		{
			if (Error < (s16)S8_MIN)
			{
				Error_slip = S8_MIN;
			}
			else
			{
				Error_slip = (s8)Error;
			}
		}
	}

	// Get Kp and Ki from look-up tables if not in manual tuning mode
	#if !defined PI_PARAM_TUNING
		ACM_GetPIParam( MTC_GetStatorFreq() );
	#endif
	
	// PI calculation is: Ouput = Integral term + Proportional term
	// Here we set: Integral term = Ouput - Proportional term
	Voltage_slip_s32 = 256 * (s32)Kp * (s32)Error_slip;
	Voltage_slip = (s16) ( Voltage_slip_s32 / 2560 );
	VoltageIntegralTerm = 256 * ( (s32)CurrentVoltage - (s32)Voltage_slip );

	// Reset flag for PI results clamping
	MinPiOut = FALSE;
	MaxPiOut = FALSE;

}

/*-----------------------------------------------------------------------------
ROUTINE Name : ACM_SlipRegulation

Description: This function performs closed loop slip control to maintain the 
             optimum voltage on stator windings. It can be shown that overall
             AC motor drive efficiency is roughly a function of the slip 
             frequency, having a maximum between zero and break-down values of
             slip. This function uses a PI algorithm to compute the best 
             voltage value from current and previous slip values.

Duration:
-----------------------------------------------------------------------------*/
u8 ACM_SlipRegulation(u8 OptimumSlip)
{
		s16 NewVoltage;
		u8  Voltage;

	{		// PI error calculation
			s16 Error;
		Error = MTC_GetSlip() - OptimumSlip;
		if (Error > (s16)S8_MAX)
		{
			Error_slip = S8_MAX;
		}
		else
		{
			if (Error < (s16)S8_MIN)
			{
				Error_slip = S8_MIN;
			}
			else
			{
				Error_slip = (s8)Error;
			}
		}
	}

	#if !defined PI_PARAM_TUNING
		ACM_GetPIParam( MTC_GetStatorFreq() );		// Ki and Kp coeff setup
	#endif

	/* Proportional term */
	Voltage_slip_s32 = 256 * (s32)Kp * (s32)Error_slip;
	Voltage_slip = (s16) ( Voltage_slip_s32 / 2560 );     

	NewVoltage = Voltage_slip; /* "Proportional" output */	

	/* Integral term */ 
	/* If modulation is maximum, integral term must be "frozen" */
	{  
			s32 PreviousVoltageIntegral = VoltageIntegralTerm;   

		DeltaVoltage_slip_s32 = 256 * (s32)Ki * (s32)Error_slip; 
		DeltaVoltage_slip = (s32) ( DeltaVoltage_slip_s32 / 2560 );     
		if( ((Error_slip>0) && !MaxPiOut) || ((Error_slip<0) && !MinPiOut) )
	   {
			VoltageIntegralTerm += DeltaVoltage_slip; /* "integral" output */
		}

		if ( ( DeltaVoltage_slip >= 0 ) && ( VoltageIntegralTerm < PreviousVoltageIntegral ) )
		{
			VoltageIntegralTerm = S32_MAX;	// Avoid IntTerm Overflow
		}

		if ( ( DeltaVoltage_slip < 0 ) &&  ( VoltageIntegralTerm > PreviousVoltageIntegral ) )
		{
			VoltageIntegralTerm = S32_MIN;	// Avoid IntTerm Underflow
		}    
	}

	NewVoltage += (s16)( VoltageIntegralTerm / 256 ); /* Sum "proportional" and "integral" terms */

	if ( NewVoltage < 0 )
	{
		Voltage = 0;
		MinPiOut = TRUE;
	}
	else
	{
			u8 MaxVoltage;

		MaxVoltage = ACM_VoltageMaxAllowed( MTC_GetStatorFreq() );
		if ( NewVoltage > (s16)MaxVoltage )
		{
			Voltage = MaxVoltage;
			MaxPiOut = TRUE;  /* Set ClampFlag if modulation reaches maximum value */
		}
		else
		{
			Voltage = (u8)NewVoltage;
			MinPiOut = FALSE;
			MaxPiOut = FALSE;
		}
	}

	return (Voltage);

}


/*-----------------------------------------------------------------------------
ROUTINE Name : ACM_GetOptimumSlip

Description: This function returns the Slip fequency giving the highest motor 
             efficiency. It performs a linear interpolation between the 2
             setpoints.
             No error codes are returned if StatorFrequency is outside absolute
             domain [LOWEST_FREQUENCY .. HIGHEST_FREQUENCY].

Duration:

Important caution: when modifying setpoints, user must keep in mind that
         SLIP_COEFF, SLIP_OFFSET and local variable Buffer are declared as u16.
         They must therefore be declared as u32 if overflows can occur.
-----------------------------------------------------------------------------*/
u8 ACM_GetOptimumSlip(u16 StatorFrequency)
{
         u8  OptimumSlip;

  if ( StatorFrequency <= OPT_SLIP_LOWFREQ_LIMIT )
  {
       OptimumSlip = OPT_SLIP_LOWFREQ;

  } else if ( StatorFrequency >= OPT_SLIP_HIGHFREQ_LIMIT )
         {
              OptimumSlip = OPT_SLIP_HIGHFREQ;

         } else {
                       u16 Buffer;
                   Buffer = StatorFrequency * SLIP_COEFF - SLIP_OFFSET;
                   OptimumSlip = (u8)(OPT_SLIP_LOWFREQ + (u8) (Buffer / 256));
                }

  return ( OptimumSlip );
}                                                                              


/*-----------------------------------------------------------------------------
ROUTINE Name : ACM_GetPIParam

Description: This function returns the proportional and integral parameters of
			 PI slip regulation depending on the stator frequency. It performs a 
			 linear interpolation between the two given setpoints defined in 
			 ACMparam.h file.
-----------------------------------------------------------------------------*/
void ACM_GetPIParam(u16 StatorFrequency)
{

	if ( StatorFrequency <= PI_LOWFREQ_LIMIT )
	{
	     Kp = PI_PROP_LOWSPD;
	     Ki = PI_INT_LOWSPD;	
	} 
	
	else if ( StatorFrequency >= PI_HIGHFREQ_LIMIT )
	{
	     Kp = PI_PROP_HIGHSPD;
	     Ki = PI_INT_HIGHSPD;	
	}
	
	else
	{
		u16 Buffer;
		Buffer = StatorFrequency * PI_PROP_COEFF - PI_PROP_OFFSET;
		Kp = (u8)(PI_PROP_LOWSPD - (u8) (Buffer / 256));
		Buffer = StatorFrequency * PI_INT_COEFF - PI_INT_OFFSET;
		Ki = (u8)(PI_INT_LOWSPD + (u8) (Buffer / 256));
	}

}

/*** (c) 2004  STMicroelectronics **************************** END OF FILE ***/