/**************** (c) 2004  STMicroelectronics **********************
     
PROJECT  : ST7MC demokit
COMPILER : ST7 METROWERKS C (HIWARE) / COSMIC

MODULE  :  adc.c
VERSION :  1.2.1

CREATION DATE :    08.2003
AUTHOR :      Microcontroller Application Lab  / ST Hong Kong

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

DESCRIPTION :   ADC routines              
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

01.2004 Rel 1.1: Add ADC_Init(), u8 GetHeatsinkTemp(), u8 GetBusVoltage()
03.2004 Rel 1.2: Use HW registers and bit definitions of ST7FMC2N6.h
09.2004 Rel 1.2.1: use config.h to get micro HW registers and bit definitions:
salestype definition is now done in a single place (config.h)

 ******************************************************************************
 THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. ST MICROELECTRONICS
 SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
 WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
 ******************************************************************************

******************************************************************************/

// Standard types definitions and ST7 specific macros
#include "lib.h"
// Public ADC peripheral function prototypes
#include "adc.h"
// Library configuration and ST7FMC peripherals Hardware Registers declaration
#include "config.h"

/* Private Constants --------------------------------------------------------*/

//	Heatsink Temperature monitoring
#define TEMP_SENSOR_CHANNEL		CONVERT_AIN0	/* Select here ADC channel for temperature sensor*/

// Protection set-up for ST7MC starter kit RevC; cf schematics for part numbers and values
#define NTC_THRESHOLD		110       /* Maximum allowed temperature for heatsink: 110C on IGBT package */
#define NTC_HYSTERESIS		8         /* Temperature gap before re-enabling inverter */

#define OVER_TEMPERATURE	((u8)0x01)	/* Private flag to store inverter state */

//	Inverter Bus voltage monitoring
#define BUS_VOLTAGE_CHANNEL		CONVERT_AIN1	/* Select here ADC channel for bus voltage reading */

// Protection set-up for ST7MC starter kit RevC; cf schematics for part numbers and values
/* Maximum allowed voltage for Bulk capacitors and/or power switches is around 275Vrms on AC input*/
#define HVBUS_THRESHOLD		1020      
#define HVBUS_HYSTERESIS	19        /* Voltage gap before re-enabling inverter */

#define OVER_VOLTAGE		((u8)0x02)

/* ADC peripheral bits and bitfields definitions missing in ST7FMC2N6.h -----*/
#define PRSC_MASK		((u8)0x00)		/* Define here selected prescaler ratio */
#define CS3_MSK			((u8)0x08)		/* ADC Channel selection Mask */
#define CS2_MSK			((u8)0x04)		/* ADC Channel selection Mask */
#define CS1_MSK			((u8)0x02)		/* ADC Channel selection Mask */
#define CS0_MSK			((u8)0x01)		/* ADC Channel selection Mask */

#define CONVERT_AIN0	((u8)ADCCSR_ADON_OR	+ PRSC_MASK													)
#define CONVERT_AIN1	((u8)ADCCSR_ADON_OR + PRSC_MASK										+ CS0_MSK	)
#define CONVERT_AIN2	((u8)ADCCSR_ADON_OR + PRSC_MASK							+ CS1_MSK				)
#define CONVERT_AIN3	((u8)ADCCSR_ADON_OR	+ PRSC_MASK							+ CS1_MSK	+ CS0_MSK	)
#define CONVERT_AIN4	((u8)ADCCSR_ADON_OR	+ PRSC_MASK				+ CS2_MSK							)
#define CONVERT_AIN5	((u8)ADCCSR_ADON_OR	+ PRSC_MASK				+ CS2_MSK				+ CS0_MSK	)
#define CONVERT_AIN6	((u8)ADCCSR_ADON_OR	+ PRSC_MASK				+ CS2_MSK	+ CS1_MSK				)
#define CONVERT_AIN7	((u8)ADCCSR_ADON_OR	+ PRSC_MASK				+ CS2_MSK	+ CS1_MSK	+ CS0_MSK	)
#define CONVERT_AIN8	((u8)ADCCSR_ADON_OR	+ PRSC_MASK	+ CS3_MSK										)
#define CONVERT_AIN9	((u8)ADCCSR_ADON_OR	+ PRSC_MASK	+ CS3_MSK							+ CS0_MSK	)
#define CONVERT_AIN10	((u8)ADCCSR_ADON_OR	+ PRSC_MASK	+ CS3_MSK				+ CS1_MSK				)
#define CONVERT_AIN11	((u8)ADCCSR_ADON_OR	+ PRSC_MASK	+ CS3_MSK				+ CS1_MSK	+ CS0_MSK	)
#define CONVERT_AIN12	((u8)ADCCSR_ADON_OR	+ PRSC_MASK	+ CS3_MSK	+ CS2_MSK							)
#define CONVERT_AIN13	((u8)ADCCSR_ADON_OR	+ PRSC_MASK	+ CS3_MSK	+ CS2_MSK				+ CS0_MSK	)
#define CONVERT_AIN14	((u8)ADCCSR_ADON_OR	+ PRSC_MASK	+ CS3_MSK	+ CS2_MSK	+ CS1_MSK				)
#define CONVERT_AIN15	((u8)ADCCSR_ADON_OR	+ PRSC_MASK	+ CS3_MSK	+ CS2_MSK	+ CS1_MSK	+ CS0_MSK	)

/* Private Functions prototypes -------------------------------------------- */
static u16 ADC_Get_10bits(u8);		// returns 10 bits result (full ADC resolution)
static u8 ADC_Get_8bits(u8);		// returns 8 bits result only           

/* Private Variables ------------------------------------------------------- */
static u8 InverterStatus;		// Stores current inverter state

/*-----------------------------------------------------------------------------
ROUTINE Name : ADC_Init

Description:	Start-up and initialize the ADC. Clear flags related to inverter
				monitoring
Input/Output:	None
Comments: 		None
-----------------------------------------------------------------------------*/
void ADC_Init(void)
{
	// Switch ON the ADC at max speed (4MHz with 8MHz CPU clock)
	// Heatsink temperature selected by default
	ADCCSR = TEMP_SENSOR_CHANNEL;

	InverterStatus = 0;
}


/*-----------------------------------------------------------------------------
ROUTINE Name : ADC_Get_10bits

Description:	Return the full ADC resolution value (10 bits).
Input/Output:	u8/u16 (channel number/result of conversion)
Comments: 		None
-----------------------------------------------------------------------------*/
u16 ADC_Get_10bits(u8 Channel)		
{
		u16 result = 0;
		u8 i;

	ADCCSR = Channel;		// 4 Mhz sampling, ADC on

	while (!ValBit(ADCCSR,ADCCSR_EOC));	// wait till end of conversion
	ADCDRH;                       // and ignore 1st result

	for (i=0;i<=7;i++)      // take 8 samples
	{
		while (!ValBit(ADCCSR,ADCCSR_EOC));	// wait till end of conversion
		result += ADCDRL;
		result += (u16)(ADCDRH << 2);	// clear EOC bit
	}

	result = result >> 3;	// div/8 -> smooth result
	return (result);
}
 


/*-----------------------------------------------------------------------------
ROUTINE Name : ADC_Get_8bits

Description:	Return 8 bits resolution ADC value only.
Input/Output:	u8/u8 (channel number/result of conversion)
Comments: 		None
-----------------------------------------------------------------------------*/
u8 ADC_Get_8bits(u8 Channel)	
{
		u16 result = 0;
		u8 i;

	ADCCSR = Channel;		// 4 Mhz sampling, ADC on

	while (!ValBit(ADCCSR,ADCCSR_EOC));	// wait till end of conversion
	ADCDRH;                       // and ignore 1st result

	for (i=0;i<=7;i++)      // take 8 samples
	{
		while (!ValBit(ADCCSR,ADCCSR_EOC));	// wait till end of conversion
		result += ADCDRH;	// clear EOC bit
	}

	result = result >> 3;	// div/8 -> smooth result
	return ((u8)result);
}
         

/*-----------------------------------------------------------------------------
ROUTINE Name :  ADC_GetRV1/ADC_GetRV2/ADC_GetRV3

Description:	return potentiometers value (RV1/RV2/RV3).
Input/Output:	None/u8
Comments: 		None
-----------------------------------------------------------------------------*/
u8 ADC_GetRV1(void)
{
	return(ADC_Get_8bits(CONVERT_AIN13));
}

u8 ADC_GetRV2(void)
{
	return(ADC_Get_8bits(CONVERT_AIN11));
}        

u8 ADC_GetRV3(void)
{
	return(ADC_Get_8bits(CONVERT_AIN7));
}


/*-----------------------------------------------------------------------------
ROUTINE Name :  ADC_CheckOverTemp

Description:	Return TRUE if the voltage on the thermal resistor connected 
                to channel AIN0 has reached the threshold level or if the voltage 
                has not yet reached back the threshold level minus the hysteresis 
                value after an overheat detection.
Input/Output:	none/boolean
Comments: 		None
-----------------------------------------------------------------------------*/
BOOL ADC_CheckOverTemp(void)
{
		u16 HeatsinkTemp;

	HeatsinkTemp = ADC_Get_10bits(TEMP_SENSOR_CHANNEL);

	if (HeatsinkTemp >= NTC_THRESHOLD)
	{
		InverterStatus |= OVER_TEMPERATURE;
		return(TRUE);
	}
	else
	{
		if ( (HeatsinkTemp >= (NTC_THRESHOLD - NTC_HYSTERESIS))
            && (InverterStatus & OVER_TEMPERATURE) )
		{
			return(TRUE);	// Wait heatsink cooling before re-enabling inverter
		}
		else
		{
			InverterStatus &= (u8)(~OVER_TEMPERATURE);
			return(FALSE);	// Normal operation
		}
	}

}


/*-----------------------------------------------------------------------------
ROUTINE Name :  ADC_CheckBusVoltage

Description:	Return TRUE if the voltage of the HVBUS connected to channel AIN1 
                has reached the threshold level or if the voltage has not yet
                reached back the threshold level minus the hysteresis 
                value after an overvoltage detection.

Input/Output:	none/boolean
Comments: 		None
-----------------------------------------------------------------------------*/
BOOL ADC_CheckOverVoltage(void)
{
		u16 BusVoltage;


    BusVoltage = ADC_Get_10bits(BUS_VOLTAGE_CHANNEL);

	if (BusVoltage >= HVBUS_THRESHOLD)
	{
		InverterStatus |= OVER_VOLTAGE;
		return(TRUE);
	}
	else
	{
		if ( (BusVoltage >= (HVBUS_THRESHOLD - HVBUS_HYSTERESIS))
            && (InverterStatus & OVER_VOLTAGE) )
		{
			return(TRUE);	// Wait Bus voltage decrease before re-enabling inverter
		}
		else
		{
			InverterStatus &= (u8)(~OVER_VOLTAGE);
			return(FALSE);	// Normal operation
		}
	}
}

/*-----------------------------------------------------------------------------
ROUTINE Name :  ADC_GetBusVoltage

Description:	Return the high voltage bus value voltage with arbitrary unit
				depending from external voltage divider

Input/Output:	none/u8
Comments: 		None
-----------------------------------------------------------------------------*/
u8 ADC_GetBusVoltage(void)
{
	return(ADC_Get_8bits(BUS_VOLTAGE_CHANNEL));
}


/*-----------------------------------------------------------------------------
ROUTINE Name :  ADC_GetHeatsinkTemp

Description:	Return the Heatsink temperature value with arbitrary unit
				depending from NTC value and bias resistor

Input/Output:	none/u8
Comments: 		None
-----------------------------------------------------------------------------*/
u8 ADC_GetHeatsinkTemp(void)
{
	return(ADC_Get_8bits(TEMP_SENSOR_CHANNEL));
}

/*** (c) 2004  STMicroelectronics ****************** END OF FILE ***/