/**************** (c) 2004  STMicroelectronics **********************
     
PROJECT  : 3-phase AC induction motor drive Software Library
COMPILER : COSMIC / METROWERKS

MODULE  :  main.c
VERSION :  1.0.2

CREATION DATE : April 2004

AUTHOR :	Microcontroller Division Applications

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

DESCRIPTION :   Main Routine
              
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

MODIFICATIONS :
V1.0.1 (September 04):
The SCI communication is now done at 38400 bauds and can be either by polling 
or interrupt driven (to be selected in config.h).
in DoMotorControl (OPEN LOOP mode), do nothing until the Timease counter is elapsed
Remove mention to ST7FMC2N6.h (now in config.h)
The speed range when using trimmers corresponds to the max-min speed defined in MTCparam.h

V1.0.2 (April 2005)
The rotating direction is now managed for demos, using the DIRECTION parameter
from Mainparam.h.
 ******************************************************************************
 THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. ST MICROELECTRONICS
 SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
 WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
*******************************************************************************

******************************************************************************/

// Standard types definitions and ST7 specific macros
#include "lib.h"
// General purpose peripheral function prototypes
#include "ports.h"
#include "adc.h"
#include "pwmart.h"
#include "sci.h"
// ST7 specifics related functions prototypes
#include "ST7_Misc.h"
// Motor control specific functions prototypes
#include "mtc.h"		/* Peripheral specifics */
#include "acmotor.h"	/* AC motor related */
#include "config.h"		/* AC motor library configuration keys */
#include "Mainparam.h"	/* Motor control parameters used in functions called from main.c */
#include "MTCparam.h"	/* Min and max frequencies needed for trimmer working range set-up */
#if defined (ENABLE_RS232) && defined (__CSMC__)
	#include <string.h>
#endif


///////////////////////////////////////////////////////////////////////////////
///////////////////* Private typedefs (for demo purposes only) *///////////////
typedef enum
{
IDLE, START, RUN, STOP, BRAKE, WAIT, FAULT
} SystStatus_t;

//////////////////* Private functions (for demo purposes only) *///////////////
static void DoMotorControl( void );
static SystStatus_t StartMotor( void );
static void	CheckPowerStage(void);
#ifdef ENABLE_RS232
	static void	SendSystemDataRS232(void);
#endif

//////////////////* Private constants (for demo purposes only) *///////////////
// Set LED blinking rate (x10ms), refer to ART_SetMainTimeBase prototype
#define LED_BLINKING_RATE	((u8)10)		/* 10 x 10ms = 100ms */

#define IDLE_TIME				((u16)1000)	/* 1000 ms idle time (before allowing 
													motor re-start */

//////////////////* Private variable (for demo purposes only) *////////////////
static SystStatus_t State;
#ifdef ENABLE_RS232
	static u8 UartBuffer[16];
#endif

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
/*-----------------------------------------------------------------------------
ROUTINE Name : main

Description : Demo program: refer to application note AN1904 for details.
-----------------------------------------------------------------------------*/
void main(void)
{

   ST7_IntPrioritySetUp();
	PORTS_Init();
	ADC_Init();
	ART_Init();						// PWM ART provided general purpose time bases
	ACM_Init();
	#ifdef ENABLE_RS232
		//	Init SCI for hyperterminal: 38400bps/8 data bit/no parity/1 stop bit
		SCI_Init(SCI_DEFAULT_PARAM1,SCI_DEFAULT_PARAM2); 
		SCI_Select_Baudrate(SCI_PR_13 + SCI_TR_1 + SCI_RR_1);
		SCI_Mode(SCI_TX_ENABLE);
	#endif

	ART_SetMainTimeBase(LED_BLINKING_RATE);		// Set Timebase for LED Toggle
   MTC_Set_CounterClockWise_Direction();			// This is to start CW in toggle mode
	EnableInterrupts();

	ART_Start();					// Time bases are now available
	State = IDLE;
    
	// Flash LEDs to indicate proper MCU start-up
	PORTS_GreenLedOn();
	while (!ART_IsMainTimeElapsed());
	PORTS_RedLedOn();
	while (!ART_IsMainTimeElapsed());
	PORTS_GreenLedOn();

	// Main Loop
	while(1)
	{
		switch ( State )
		{
			case IDLE:	if ( PORTS_KeyScan() == TRUE )	// Start motor
						{
								u32 temp;
							#if DIRECTION == CW
								MTC_Set_ClockWise_Direction();
							#elif DIRECTION == CCW
								MTC_Set_CounterClockWise_Direction();
							#else
									MTC_Toggle_Direction();
							#endif
							
							#ifdef CLOSED_LOOP

								ACM_InitSoftStart(START_FREQ);

							#else	/* OPEN_LOOP: Read start-up frequency on trimmer */

								// Start frequency = (ADC_GetRV1 * (HIGHEST_FREQ-LOWEST_FREQ))/256 + LOWEST_FREQ)
								temp = (u8)(ADC_GetRV1());
								temp *= (u32)(HIGHEST_FREQ-LOWEST_FREQ);
								temp >>= 8;
								temp += LOWEST_FREQ;
								ACM_InitSoftStart_OL( (u16)temp );

							#endif
							State = START;
						}
						break;

			case START:	State = StartMotor();
						break;

			case RUN:	if ( PORTS_KeyScan() == TRUE )	// Stop motor
						{
							State = STOP;
						}
						else
						{
							DoMotorControl();
						}
						break;

			case STOP:	{
								u16 BrakingTorque;

							BrakingTorque = BRAKE_DUTY_CYCLE;
							if (BrakingTorque == 0)
							{
								ART_SetTimeOutDuration(IDLE_TIME);
								MTC_DisableMCOutputs();			// PWM is disabled on MCO outputs
								State = WAIT;
							}
							else
							{
								MTC_StartBraking( BrakingTorque );
								ART_SetTimeOutDuration(BRAKE_DURATION);
								State = BRAKE;
							}
						}
						break;

			case BRAKE: if (!ART_IsTimeOutElapsed())
						{
							MTC_Brake();
						}
						else
						{
							MTC_StopBraking();
							ART_SetTimeOutDuration(IDLE_TIME);
							State = WAIT;
						}
						break;

			case WAIT:	if (ART_IsTimeOutElapsed())
						{
							State = IDLE;
						}
						break;

			case FAULT:	
			default:	MTC_DisableMCOutputs();			// PWM is disabled on MCO outputs
						State = FAULT;
						break;
		}

		// Verify that inverter is operating properly
		CheckPowerStage();

		// LED management
		if (ART_IsMainTimeElapsed())
		{
        	switch ( State )
			{
				case IDLE:	PORTS_GreenLedOn();
							break;
				case START:
				case RUN:	
				case STOP:
				case BRAKE:
				case WAIT:  PORTS_RedLedOn();
							break;
				case FAULT:	
				default:	PORTS_RedLedToggle();
							break;
			}
		}

		#ifdef ENABLE_RS232
			SendSystemDataRS232();
		#endif

	}	// End of while(1)
} // main loop


/*-----------------------------------------------------------------------------
ROUTINE Name : CheckPowerStage

Description : Verify that inverter power stage is running fine:
					heatsink temperature
					voltage bus value
					current limitation not triggerred
-----------------------------------------------------------------------------*/
void CheckPowerStage(void)
{

	if (ADC_CheckOverVoltage() || ADC_CheckOverTemp() || MTC_CheckEmergencyStop())
	{
		State = FAULT;
	}

}

/*-----------------------------------------------------------------------------
ROUTINE Name : SendSystemDataRS232

Description : upload internal data to the PC via RS232 and hyperterminal for 
debug purposes
-----------------------------------------------------------------------------*/
#ifdef ENABLE_RS232
void SendSystemDataRS232(void)
{
	#ifdef CLOSED_LOOP

		if(SCI_IsTransmitCompleted())
		{
				u32 temp;

			strcpy(UartBuffer, "S=");

			temp = (u8)(ADC_GetRV1());
			temp *= (u32)(HIGHEST_FREQ-LOWEST_FREQ);
			temp >>= 8;
			temp += LOWEST_FREQ;
			strcatu16(UartBuffer, (u16)temp );	// This is the target speed read on trimmer

			strcat(UartBuffer, " R=");		// Rotor frequency
			strcatu16(UartBuffer, MTC_GetRotorFreq());

// WARNING: If the following lines are uncommented, the UartBuffer array must be enlarged accordingly
//		  	strcat(UartBuffer, " V=");		// Voltage
//			strcatu16(UartBuffer, MTC_GetVoltage());

//			strcat(UartBuffer," Slp=");		// SlipCmd
//			strcatu16(UartBuffer, ACM_GetOptimumSlip( MTC_GetStatorFreq() ));

			#ifdef PI_PARAM_TUNING
//				strcat(UartBuffer, " Ki=");
//				strcatu16(UartBuffer, Ki);
//				strcat(UartBuffer, " Kp=");
//				strcatu16(UartBuffer, Kp);
			#endif

			strcat(UartBuffer, "\n\r");

			SCI_PutString(UartBuffer);
		}

	#else		/* OPEN_LOOP */

		if(SCI_IsTransmitCompleted())
		{
				u32 temp;
				
			strcpy(UartBuffer, "S=");
			strcatu16(UartBuffer, MTC_GetStatorFreq() );	// This is the target speed read on trimmer
			strcat(UartBuffer, " R=");		// Rotor frequency
			strcatu16(UartBuffer, MTC_GetRotorFreq());
			strcat(UartBuffer, "\n\r");
			SCI_PutString(UartBuffer);
		}

	#endif	/* OPEN_LOOP */
}
#endif	/* ENABLE_RS232 */


/* ####################### CLOSED LOOP OPERATION ######################### */
/* ####################### CLOSED LOOP OPERATION ######################### */
/* ####################### CLOSED LOOP OPERATION ######################### */
/* ####################### CLOSED LOOP OPERATION ######################### */
/* ####################### CLOSED LOOP OPERATION ######################### */
#ifdef CLOSED_LOOP
/*-----------------------------------------------------------------------------
ROUTINE Name : DoMotorControl

Description : Performs closed loop control of the AC motor: the stator frequency 
and voltage and updated according to target Speed read on RV1 trimmer.
-----------------------------------------------------------------------------*/
void DoMotorControl( void )
{
		u16 FreqCmd, StatorFreq;
		u8 SlipCmd;


	if (MTC_GetRotorFreq() == 0)
	{
		State = FAULT;
	}

	if ( ART_IsRegPeriodElapsed() )
	{
			u8 NewVoltage;
			u32 temp;

		// Get current Stator frequency
		StatorFreq  = MTC_GetStatorFreq();

		// Read Trimmers
		// FreqCmd = (ADC_GetRV1 * (HIGHEST_FREQ-LOWEST_FREQ))/256 + LOWEST_FREQ)
		temp = (u8)(ADC_GetRV1());
		temp *= (u32)(HIGHEST_FREQ-LOWEST_FREQ);
		temp >>= 8;
		temp += LOWEST_FREQ;
		FreqCmd = (u16)(temp);

		#ifdef PI_PARAM_TUNING
			Ki = ADC_GetRV2();
			Kp = ADC_GetRV3();
		#endif
		
		SlipCmd = ACM_GetOptimumSlip( StatorFreq );

		NewVoltage = ACM_SlipRegulation( SlipCmd );

		if (( (FreqCmd+SlipCmd) > StatorFreq) && (MTC_GetSlip() < ACCEL_SLIP_LIMIT))
		{
			StatorFreq++;
		}

		if (( (FreqCmd+SlipCmd) < StatorFreq) && (MTC_GetSlip() > DECEL_SLIP_LIMIT))
		{
			StatorFreq--;
		}

		MTC_UpdateSine(NewVoltage, StatorFreq);
		
	}  /* End of if: regulation performed */

}

/*-----------------------------------------------------------------------------
ROUTINE Name : StartMotor

Description : Starts the motor in closed loop: if a minimum speed has not been
reached after a given period, the start-up is considered as failed.
-----------------------------------------------------------------------------*/
SystStatus_t StartMotor( void )
{
		SystStatus_t NextState;

	switch ( ACM_SoftStart(MIN_START_FREQ) )
	{
		case START_FAIL:
							NextState = STOP;
							break;

		case START_OK:	
							ACM_InitSlipFreqReg( ACM_GetOptimumSlip( MTC_GetStatorFreq() ) );
							NextState = RUN;
							break;

		case START_ONGOING:	
							NextState = START;
							break;

		default:			NextState = FAULT;
							break;
	}

	return ( NextState );
}
#endif

/* ####################### OPEN LOOP OPERATION ######################### */
/* ####################### OPEN LOOP OPERATION ######################### */
/* ####################### OPEN LOOP OPERATION ######################### */
/* ####################### OPEN LOOP OPERATION ######################### */
/* ####################### OPEN LOOP OPERATION ######################### */
#ifdef OPEN_LOOP
/*-----------------------------------------------------------------------------
ROUTINE Name : DoMotorControl

Description : Performs open loop control of the AC motor: the stator frequency 
and voltage and updated according to trimmer reading.
-----------------------------------------------------------------------------*/
void DoMotorControl( void )
{
		u8 NewVoltage, VoltCmd, VMax;
		u16 NewFreq, SlipFreq, FreqCmd;
		u32 temp;

	if ( ART_Is_TimeInMsElapsed() )
	{

		// Read Trimmers
		// FreqCmd = (ADC_GetRV1 * (HIGHEST_FREQ-LOWEST_FREQ))/256 + LOWEST_FREQ)
		temp = (u8)(ADC_GetRV1());
		temp *= (u32)(HIGHEST_FREQ-LOWEST_FREQ);
		temp >>= 8;
		temp += LOWEST_FREQ;
		FreqCmd = (u16)(temp);
		
		VoltCmd = ADC_GetRV2();

		// Get actual values of volt and Freq
		NewVoltage = MTC_GetVoltage();
		NewFreq = MTC_GetStatorFreq();

		SlipFreq = MTC_GetSlip();

		VMax = ACM_VoltageMaxAllowed(NewFreq);
		if ( VoltCmd > VMax )
		{
			VoltCmd = VMax;
		}

		if ( VoltCmd > NewVoltage ) NewVoltage++;	// Smoothly increase voltage
		if ( VoltCmd < NewVoltage ) NewVoltage--;	// Smoothly decrease voltage

		if ( FreqCmd > NewFreq ) NewFreq++;		// Smoothly increase frequency
		// Smoothly decrease frequency unless RotorFreq>=StatorFreq (risk of reactive current)
		// if no sensor available, RotorFreq will be 0 and SlipFreq=StatorFreq
		if ( (SlipFreq != 0) && ( FreqCmd < NewFreq ) ) NewFreq--;

		MTC_UpdateSine(NewVoltage, NewFreq);

	}

}

/*-----------------------------------------------------------------------------
ROUTINE Name : StartMotor

Description : Starts the motor in open loop: with the stator frequency read 
on the trimmer, increase the stator voltage from 0 to the limit set by the 
V/f curve. When Voltage reaches this limit the routine returns TRUE.
-----------------------------------------------------------------------------*/
SystStatus_t StartMotor( void )
{
		u8 StartVolt, MaxVolt;
		u16 StartFreq;
		u32 temp;

	// Read Trimmers
	// StartFreq = (ADC_GetRV1 * (HIGHEST_FREQ-LOWEST_FREQ))/256 + LOWEST_FREQ)
	temp = (u8)(ADC_GetRV1());
	temp *= (u32)(HIGHEST_FREQ-LOWEST_FREQ);
	temp >>= 8;
	temp += LOWEST_FREQ;
	StartFreq = (u16)temp;
	
	StartVolt = ADC_GetRV2();

	// Verify that Start-up voltage is in line with V/f characteristic
	MaxVolt = ACM_VoltageMaxAllowed(StartFreq);
	if (StartVolt > MaxVolt)
	{
		StartVolt = MaxVolt;
	}
	
	if ( ACM_SoftStartOL(StartVolt) )
	{
		return(RUN);
	}
	else
	{
		return(START);
	}

}
#endif	/* OPEN_LOOP */
/*** (c) 2004  STMicroelectronics **************************** END OF FILE ***/