/**************** (c) 2004  STMicroelectronics ********************************
     
PROJECT  : 3-phase AC induction motor drive Software Library
COMPILER : COSMIC / METROWERKS

MODULE  :  mtc.c
VERSION :  1.0.2

CREATION DATE : April 2004

AUTHOR :	Microcontroller Division Applications

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

DESCRIPTION :   Low level motor control routines
              
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

MODIFICATIONS :
September 04 V1.0.1
Replace ST7FMC2N6.h by config.h for better flexibility.
Modify declaration of structure needed for speed measurement: PSpeedMeas_s 
(cosmic only) to produce more efficient code;
 add REP_RATE coming from MTCparam.h in MTC_InitPeripheral
Clear flag Sampling Out in MTC_InitPeripheral.
In MTC_StopBraking, reset also the LSBs of the compare registers.
Reset D flag in C interrupt. Reset CL flag in U interrupt.
Lock captures in C interrupt to prevent corruption of MPRSR register.
Add sine wave look-up tables at beginning of file.
Cleaner masking of U interrupt in MTC_StartBraking and C interrupt in 
GetAvrgTachoPeriod, following the section 14 "important note of the ST7MC datasheet.
In GetLastTachoPeriod, the FIFO Index is properly buffered in case a tacho 
capture occurs the frequency calculation.
                   
April 2005 V 1.0.2 Clockwise direction changed (-Phase shift) to be in-line 
with Softec motors
*******************************************************************************
 THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. ST MICROELECTRONICS
 SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
 WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
******************************************************************************/

#include "lib.h"		/* General purpose typedefs and macros */
#include "mtc.h"		/* Public motor control peripheral function prototypes */
#include "MTCparam.h"	/* Motor Control parameters */
#include "mtc_bits.h" 	/* MTC peripheral Hardware Registers declaration */
#include "pwmart.h"		/* Software Timebase needed during braking */
// Library configuration and ST7FMC peripherals Hardware Registers declaration
#include "config.h"

/* --------------------------- Private typedefs ---------------------------- */
#ifdef __HIWARE__				/* Metrowerks Compiler */
	typedef struct {
			TwoBytes Capture;
			u8 Prsc_Reg;
			} SpeedMeas_s, * near PSpeedMeas_s;
#elif defined(__CSMC__)		/* Cosmic Compiler */
	typedef struct spm {
			TwoBytes Capture;
			u8 Prsc_Reg;
			} SpeedMeas_s;
	typedef @tiny struct spm * PSpeedMeas_s;
#endif

/* -------------------------- Private variables ---------------------------- */
// Some variables accessed in assembly located in page zero to decrease CPU load
// WARNING: if SensorPeriod structure is put outside Zero page, the declaration 
// of the pointer PSpeedMeas_s has to be modified (remove near/@tiny)
#ifdef __HIWARE__				/* Metrowerks Compiler */

	#pragma DATA_SEG SHORT _ZEROPAGE
	volatile u8 	SineMag;
	volatile u16 	SineFreq;
	volatile u16 	SineFreqBuf;
	volatile s8 	PhaseShift;
	volatile u8 	MTCStatus;
	volatile u16 	Phase;
	volatile SpeedMeas_s SensorPeriod[SPEED_FIFO_SIZE];
	volatile PSpeedMeas_s SpeedFIFO_Index;
	#pragma DATA_SEG DEFAULT

#elif defined(__CSMC__)		/* Cosmic Compiler */

	@tiny volatile u8 	SineMag;
	@tiny volatile u16 	SineFreq;
	@tiny volatile u16 	SineFreqBuf;
	@tiny volatile s8 	PhaseShift;
	@tiny volatile u8 	MTCStatus;
	@tiny volatile u16 	Phase;	
	@tiny volatile SpeedMeas_s SensorPeriod[SPEED_FIFO_SIZE];
	@tiny volatile PSpeedMeas_s SpeedFIFO_Index; 	// Pointer initialized in MTC_InitPeripheral function

#endif

static 	u16	StatorFreq;
static 	u16	PrevStatorFreq;
static	u16	BrakeDutyCycle;
static	BOOL MCES_Status;

/* -------------------------- Private functions ---------------------------- */
u32 GetLastTachoPeriod(void);
u32 GetAvrgTachoPeriod(void);

/* --------------------------- Private macros ------------------------------ */
#define MTC_EnableClock()		( MCRA |= (1<<MCRA_CKE) )
#define MTC_DisableClock()		( MCRA &= ((1<<MCRA_CKE)^255) )

/* -------------------------- Private constants ---------------------------- */
// MTCStatus bitfield
#define FREQ_CHANGE			((u8)0x01)
#define BRAKE_WAIT_DEMAG 	((u8)0x02)
#define BRAKE_SET_UP   		((u8)0x04)
#define BRAKE   				((u8)0x08)
//#define NOT_USED				((u8)0x10)
#define RATIO_DEC				((u8)0x20)
#define INIT_ROLLING_AVRG 	((u8)0x40)
#define DO_ROLLING_AVRG		((u8)0x80)

#define BRAKE_MASK			((u8) BRAKE_WAIT_DEMAG + BRAKE_SET_UP + BRAKE )

#define VOLT_INIT		((u8)0)		/* Voltage initialisation at MCU start-up */
#define FREQ_INIT		((u16)50)	/* Freq initialisation at MCU start-up */

#define STATOR_DEMAG_TIME	((u16)300)	/* Time in ms before applying DC current for braking*/
#define CURRENT_SETUP_TIME ((u8)5)		/* Time in ms between two consecutive values of duty cycle during braking current increase */


#ifdef PWM_9BIT
	const u8 SINE3RDHARM[256] = {
		0,5,11,16,21,27,32,37,42,47,52,57,
		61,66,70,75,79,83,86,90,93,97,100,103,
		105,108,110,113,115,116,118,120,121,122,123,124,
		125,126,126,126,127,127,127,127,127,127,127,126,
		126,126,126,125,125,125,124,124,124,123,123,123,
		123,122,122,122,122,122,122,122,123,123,123,123,
		124,124,124,125,125,125,126,126,126,126,127,127,
		127,127,127,127,127,126,126,126,125,124,123,122,
		121,120,118,116,115,113,110,108,105,103,100,97,
		93,90,86,83,79,75,70,66,61,57,52,47,
		42,37,32,27,21,16,11,5,

		0,5,11,16,21,27,32,37,42,47,52,57,
		61,66,70,75,79,83,86,90,93,97,100,103,
		105,108,110,113,115,116,118,120,121,122,123,124,
		125,126,126,126,127,127,127,127,127,127,127,126,
		126,126,126,125,125,125,124,124,124,123,123,123,
		123,122,122,122,122,122,122,122,123,123,123,123,
		124,124,124,125,125,125,126,126,126,126,127,127,
		127,127,127,127,127,126,126,126,125,124,123,122,
		121,120,118,116,115,113,110,108,105,103,100,97,
		93,90,86,83,79,75,70,66,61,57,52,47,
		42,37,32,27,21,16,11,5	
	};

#else
#ifdef PWM_10BIT

	const u8 SINE3RDHARM[256] = {
		0,3,7,10,13,17,20,23,26,29,32,35,
		38,41,44,46,49,51,54,56,58,60,62,64,
		66,67,69,70,71,72,73,74,75,76,77,77,
		78,78,78,79,79,79,79,79,79,79,79,79,
		78,78,78,78,78,77,77,77,77,77,77,76,
		76,76,76,76,76,76,76,76,76,76,77,77,
		77,77,77,77,78,78,78,78,78,79,79,79,
		79,79,79,79,79,79,78,78,78,77,77,76,
		75,74,73,72,71,70,69,67,66,64,62,60,
		58,56,54,51,49,46,44,41,38,35,32,29,
		26,23,20,17,13,10,7,3,

		0,3,7,10,13,17,20,23,26,29,32,35,
		38,41,44,46,49,51,54,56,58,60,62,64,
		66,67,69,70,71,72,73,74,75,76,77,77,
		78,78,78,79,79,79,79,79,79,79,79,79,
		78,78,78,78,78,77,77,77,77,77,77,76,
		76,76,76,76,76,76,76,76,76,76,77,77,
		77,77,77,77,78,78,78,78,78,79,79,79,
		79,79,79,79,79,79,78,78,78,77,77,76,
		75,74,73,72,71,70,69,67,66,64,62,60,
		58,56,54,51,49,46,44,41,38,35,32,29,
		26,23,20,17,13,10,7,3	
	};

#endif	/* PWM_10BIT */
#endif	/* PWM_9BIT */

/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_ResetPeripheral

Description: This function resets the circuitry of the Motor Controller Peripheral,
				 at the sole exception of the MDTG and MPOL registers.
-----------------------------------------------------------------------------*/
void MTC_ResetPeripheral(void)
{
	SetBit(MCFR,MCFR_RST); 
	ClrBit(MCFR,MCFR_RST);
}


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_EnableMCOutputs
					MTC_DisableMCOutputs

Description: These functions allows either to output PWM signals on the 
				MCO outputs or to put these outputs in their RESET 
				configuration (user selectable state: Hi-Z, Low or High)
				MTC_DisableMCOutputs immediately disconnects the MCOx PWM outputs pins 
				from the Motor Controller peripheral. It resets the MOE bit in 
				the MCRA register, thus causing the MCOx pins to be in their 
				reset configuration, as defined in the options bytes (high 
				impedance or low impedance high/low state).
-----------------------------------------------------------------------------*/
void MTC_EnableMCOutputs(void)
{
	MCRA |= (1<<MCRA_MOE);
}

void MTC_DisableMCOutputs(void)
{
	MCRA &= ((1<<MCRA_MOE)^255);
}


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_InitPeripheral

Description: Initialization of the motor control peripheral hardware registers,
				for the sinewave general parameters (such as PWM frequency, output 
				polarity, deadtime, interrupts,...) and speed feedback processing 
				(tacho input selection, edge sensitivity,...).

Comment:		All required motor control related interrupts are enabled upon 
				completion of this routine.
-----------------------------------------------------------------------------*/
void MTC_InitPeripheral(void)
{

	MTC_ResetPeripheral();

	SET_MTC_PAGE(1);

	MPOL = DRIVERS_POLARITY;	// Set polarity according to IGBT drivers
	MDTG = DEADTIME;				// Deadtime Duration
	SetBit(MDTG,DTE);				// Deadtime enabled
	SetBit(MDTG,PCN);				// Three independant PWM configuration
	MPWME = DG_MSK;				// Use Debug outputs
	MPAR = TACHO_FALLING;		// Speed Sensor mode enabled, falling edge sensitive
	MCONF &= ((u8)~MCONF_SOI_OR);

	SET_MTC_PAGE(0);

	MPCR = PWM_PRSC;
	SetBit(MPCR,MPCR_CMS);		// Centered pattern
	MREP = REP_RATE;				// Compare preload registers loading rate
	MCP0L = (u8)PWM_MCP0;		// Set PWM frequency
	MCP0H = ((u8)(PWM_MCP0>>8));
	
	ToCMPxL(MCPUL,PWM_MCP0/2);	// 50% duty cycle on U
	ToCMPxH(MCPUH,PWM_MCP0/2);	
	ToCMPxL(MCPVL,PWM_MCP0/2);	// 50% duty cycle on V
	ToCMPxH(MCPVH,PWM_MCP0/2);
	ToCMPxL(MCPWL,PWM_MCP0/2);	// 50% duty cycle on W
	ToCMPxH(MCPWH,PWM_MCP0/2);
	
	SetBit(MCRA,MCRA_DAC);	// Direct access mode to write into MPHST
	MPHST = MCIC;				// Comparator input for Tacho signal
	ClrBit(MCRA,MCRA_DAC);	// Disable direct access

	MISR = 0;				// Clear pending interrupts
	MTCStatus = 0;
	MCES_Status = FALSE;
	// Init pointer on FIFO where speed infos are stored
	SpeedFIFO_Index = (PSpeedMeas_s)SensorPeriod;

	MTC_Set_CounterClockWise_Direction();

	MTC_EnableClock();				// Enable peripheral clock and update active 
											// registers with preloaded values
	// Unmask Update, Ratio change, Capture and emergency Stop interrupts
	MIMR = MIMR_PUM_OR + MIMR_RIM_OR + MIMR_CIM_OR + MIMR_EIM_OR;
}


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_InitSineGen

Description:	Initialization of software variables needed for sine wave 
				generation and used in the PWM update interrupt routine.
				Ensures that once the PWM update interrupts will be enabled, the 
				sinewave generated will have a zero voltage and that stator 
				frequency change will be taken into account.
-----------------------------------------------------------------------------*/
void MTC_InitSineGen(void)
{
	SineFreq = U8_MAX;				// Arbitrary value
	SineMag = 0;
	MTC_Set_ClockWise_Direction();
	Phase = 0;
	StatorFreq = FREQ_INIT-1;               // This ensure that Sinewave will be updated
   MTC_UpdateSine( VOLT_INIT, FREQ_INIT);
}


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_UpdateSine

Description: Aim of this function is to update IMCFrequency and IMCVoltage 
             variables; limitation of output frequency as defined in MTCParam.h
             is performed in this routine.
Caution:		No tests are performed in this function on the input parameters, 
				except for the frequency range. The following conditions must 
				therefore be verified by the user before calling the function:
				 voltage and frequencies must be compliant with the characteristics 
				of the motor: overvoltage can cause motor flux saturation, excessive 
				frequency is uncompatible with motor mechanicals (ball bearing for instance).
				 voltage and frequencies values should not vary too suddenly when 
				motor is running, to avoid current overshoot. This is usually handled 
				by the AC motor control software layer, by means of smoothing functions 
				and/or regulation loops.
				 Stator frequency must not be set above the rotor frequency value: 
				this would cause the motor to become generator.
-----------------------------------------------------------------------------*/
BOOL MTC_UpdateSine (u8 NewVoltage, u16 NewFrequency)
{
			u32 ExtendedFreq;
			BOOL UpdateStatus;

	// =========================================================================
	//                          FREQUENCY UPDATION PART
	// =========================================================================
         
	// Recompute frequency only if value is different from the current one...
	if (NewFrequency != StatorFreq)
	{
		if (!(MTCStatus & FREQ_CHANGE))	// ... unless an updation is on-going
		{
			// Store previous value, valid untill PWM Update event occurs
      	PrevStatorFreq = StatorFreq;	
			StatorFreq = NewFrequency;

			// Clamp value if it exceeds absolute max/min
			if ( NewFrequency >= HIGHEST_FREQ )
			{
				ExtendedFreq = (u32)HIGHEST_FREQ * (u32)256;
			}
			else
			{
				if ( NewFrequency < LOWEST_FREQ )
				{
					ExtendedFreq = (u32)LOWEST_FREQ * (u32)256;
				}
				else
				{
					// Extend to decrease quantization effects when dividing
					ExtendedFreq = (u32)NewFrequency * (u32)256;
				}
			}
		
			// Convert frequency request into time domain
			if ( ExtendedFreq > (u32)STATOR_FREQ_RESOL )
			{
				SineFreqBuf = (u16)((u32)ExtendedFreq / (u32)STATOR_FREQ_RESOL);
			}
			else
			{
				SineFreqBuf = 1; // Absolute minimum freq to avoid Freq=0 (DC current)
			}

			// New Freq value can be used in Update interrupt
			MTCStatus |= FREQ_CHANGE;
			UpdateStatus = TRUE;
		}
		else		// No frequency updation untill the previous one is completed
		{
			UpdateStatus = FALSE;
		}
	}	// End of Frequency updation part

	// =========================================================================
	//                          VOLTAGE UPDATION PART
	// =========================================================================
   SineMag = NewVoltage;
   
   return (UpdateStatus);

}


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_GetStatorFreq

Description: This function returns the value of the currently generated stator
             frequency.

Duration:
-----------------------------------------------------------------------------*/
u16  MTC_GetStatorFreq( void )
{

	if (!(MTCStatus & FREQ_CHANGE))
	{
		return(StatorFreq);
	}
	else
	{
		return(PrevStatorFreq);
	}
}


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_GetVoltage

Description: This funtion returns the current modulation index.
Duration:
-----------------------------------------------------------------------------*/
u8 MTC_GetVoltage( void )
{
    return (SineMag);
}


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_GetSlip

Description: this function returns the difference between stator and rotor
             frequencies with 0.1Hz unit.
Duration:
-----------------------------------------------------------------------------*/
u16 MTC_GetSlip( void )
{
    u16 RotorFreq  = MTC_GetRotorFreq();
    u16 StatorFreq = MTC_GetStatorFreq();

 if (StatorFreq < RotorFreq )
 {
     return (0);    /* No negative slip returned */

 } else {

        return ( StatorFreq - RotorFreq );

        }
}

 
/*-----------------------------------------------------------------------------
ROUTINE Name : IMC_GetRotorFreq

Description : This routine returns Rotor frequency with [0.1Hz] definition.
				Result is given by the following formula:
				Frotor = K x (Fosc / ([MZPRV:MZPRV]x2Power(ST[3:0])))
				where K depends on the number of motor and tacho poles pairs
				 
Comments : Result is zero in the following cases:
          _ If LastSensorPeriod is too low (glitches at start for instance)
          _ If Input frequency is too low (timer overflow)
Warning : Maximum expectable accuracy depends on Fosc: 16MHz will give the 
			best results.
-----------------------------------------------------------------------------*/
u16 MTC_GetRotorFreq ( void )
{
		u32 FreqBuffer;

	if ( MTCStatus & DO_ROLLING_AVRG )
	{
		FreqBuffer = GetAvrgTachoPeriod();
		
	} else {  // Raw tacho period
	
		FreqBuffer = GetLastTachoPeriod();
	}
	
	if ( ( FreqBuffer <= (u32)SPEED_OVERFLOW ) /*Avoid u32 DIV Overflow*/
	  || ( (MPRSR & 0x0F) >= MAX_RATIO ) ) /* Captured value not significant at start-up */
	{
			return (0);

	} else	{
			return ( (u16) ( ROTOR_SPEED_FACTOR / FreqBuffer ) );
			}
}


/*-----------------------------------------------------------------------------
ROUTINE Name : GetLastTachoPeriod

Description : returns the rotor pseudo-period based on last tacho capture
-----------------------------------------------------------------------------*/
u32 GetLastTachoPeriod(void)
{
		u8 LowNibble;
		u32 FreqBuffer;
		PSpeedMeas_s LastSpeedFIFO_Index;

	// Store pointer to prevent errors due to Tacho Capture during processing
	LastSpeedFIFO_Index = SpeedFIFO_Index;
	
	if (LastSpeedFIFO_Index != (PSpeedMeas_s)SensorPeriod)
	{
		LastSpeedFIFO_Index -= 1;
	} else 	{
			LastSpeedFIFO_Index = (PSpeedMeas_s)SensorPeriod + SPEED_FIFO_SIZE - 1;
			}

	FreqBuffer = (u32)LastSpeedFIFO_Index->Capture.w_form;
	LowNibble = LastSpeedFIFO_Index->Prsc_Reg;
	LowNibble &= 0x0F;	// Get ST[3:0] 4-bit value out of MPRSR
	while (LowNibble != 0)
	{
		FreqBuffer *= 2;
		LowNibble--;
	}

	return (FreqBuffer);

}


/*-----------------------------------------------------------------------------
ROUTINE Name : GetAvrgTachoPeriod

Description :
Side effect: the very last tacho period acquired may not be considered for the 
calculation if a capture occurs during averaging.
-----------------------------------------------------------------------------*/
u32 GetAvrgTachoPeriod(void)
{
		u8 ST3_0;
		u32 FreqBuffer, AvrgBuffer;
		PSpeedMeas_s AvrgSpeedFIFO_Index;

	AvrgBuffer = 0;
	AvrgSpeedFIFO_Index = (PSpeedMeas_s)SensorPeriod;

	while (	AvrgSpeedFIFO_Index < &SensorPeriod[SPEED_FIFO_SIZE] )
	{
		// Disable capture interrupts during acquisition
		PushCC();	// cf datasheet section 14 for explanations
		DisableInterrupts();
		ClrBit(MIMR,MIMR_CIM);
		PopCC();

		FreqBuffer = (u32)AvrgSpeedFIFO_Index->Capture.w_form;
		ST3_0 = AvrgSpeedFIFO_Index->Prsc_Reg;
		SetBit(MIMR,MIMR_CIM);	// Re-enable capture interrupts
		ST3_0 &= 0x0F;		// Get ST[3:0] 4-bit value out of MPRSR
		while (ST3_0 != 0)
		{
			FreqBuffer *= 2;
			ST3_0--;
		}
		AvrgBuffer += FreqBuffer;	// Sum the whole tacho period FIFO
		AvrgSpeedFIFO_Index++;
	}
	
	AvrgBuffer	+= (SPEED_FIFO_SIZE/2)-1; 	// Round to upper value
	AvrgBuffer /= SPEED_FIFO_SIZE;			// Average value	
	
	return (AvrgBuffer);
}


/*-----------------------------------------------------------------------------
ROUTINE Name : InitTachoMeasure

Description : Clear software FIFO used to "push" latest speed informations. This
					function must be called before starting the motor to initialize 
					the speed measurement process.
Note: First measurements will be done without filtering (no rolling average)
-----------------------------------------------------------------------------*/
void MTC_InitTachoMeasure( void )
{
	SpeedFIFO_Index = (PSpeedMeas_s)SensorPeriod;
	while (	SpeedFIFO_Index < &SensorPeriod[SPEED_FIFO_SIZE] )
	{
		SpeedFIFO_Index->Capture.w_form = U16_MAX;
		SpeedFIFO_Index->Prsc_Reg = 0x15;
		SpeedFIFO_Index++;
	}
	SpeedFIFO_Index = (PSpeedMeas_s)SensorPeriod;
	MTCStatus &= (u8)(~DO_ROLLING_AVRG);
}

/*-----------------------------------------------------------------------------
ROUTINE Name : StartTachoFiltering

Description : Set the flags to initiate properly tacho values smoothing mechanism.

Note: The initialization of the FIFO used to do the averaging will be done when 
		the next tacho capture interrupt will occur.
-----------------------------------------------------------------------------*/
void MTC_StartTachoFiltering( void )
{
	// Initialize FIFO stack where tacho periods are stored
	MTCStatus |= INIT_ROLLING_AVRG;		
	// Starting from now, the values returned by MTC_GetRotorFreq are averaged
	MTCStatus |= DO_ROLLING_AVRG;			
}


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_ValidSpeedInfo

Description : Used in Start function to know if rotor shaft turns at the right
              speed.

Warning : Since there's no way to differentiate rotation direction with a 
          tachogenerator, user must be aware that this routine may return TRUE
          in certain conditions (re-start with no/very short stop time before
          and high inertia load), even if motor is not started in the right
          condition. User should therefore manage a minimal amout of time before 
          re-starting.
          Furthermore, this function may be unefficient if start-up duration is
          far shorter than time needed to have 3 speed informations (see Soft 
          start functions in acmotor.c).
-----------------------------------------------------------------------------*/
BOOL MTC_ValidSpeedInfo( u16 MinRotorFreq )
{
	if ( MinRotorFreq == 0 ) /* Open loop or error */
	{
		return( FALSE );
	}
	else			// Right start conditions in closed loop:
	{
		return ( (BOOL)((MTC_GetRotorFreq() > MinRotorFreq)
				&& (GetAvrgTachoPeriod() >= GetLastTachoPeriod())) );//acceleration
	}
}


/*-----------------------------------------------------------------------------
ROUTINE Names : MTC_Toggle_Direction
                MTC_Set_ClockWise_Direction
                MTC_Set_CounterClockWise_Direction
                MTC_GetRotationDirection

Description : This routines are used to impose / modify / Get rotating field
              direction. The clockWise direction is set randomly (depending on 
              motor connections).

Warning: No test is performed in these routine to verify that the motor is 
			stopped. User must therefore ensures that the drive is stopped before 
			modifying the rotation direction.
-----------------------------------------------------------------------------*/
void MTC_Toggle_Direction( void )
{
    if (PhaseShift == PHASE_SHIFT)
    {
    	PhaseShift = -PHASE_SHIFT;
    } else 	{
	    	PhaseShift = PHASE_SHIFT;
    		}
}

void MTC_Set_ClockWise_Direction( void )
{
    PhaseShift = -PHASE_SHIFT;
}

void MTC_Set_CounterClockWise_Direction( void )
{
    PhaseShift = PHASE_SHIFT;
}

Direction_t MTC_GetRotationDirection( void )
{

   if ( PhaseShift == PHASE_SHIFT )
   {
      return( CLOCKWISE );
   }
   else
   {
      return( COUNTERCLOCKWISE );
   }

}


/*-----------------------------------------------------------------------------
ROUTINE Names : MTC_StartBraking

Description: This routines prepares the DC current braking:
				* it shuts down the sinewave generation and insures that when 
				U interrupts will be restarted, voltage will be null.
				* it stops PWM interrupts
				* it initializes the demagnetization waiting time base needed
				before starting DC current injection
-----------------------------------------------------------------------------*/
void MTC_StartBraking( u16 DutyCycle)
{
		u8 Counter;

	// Reset any pending reset flag
	MTCStatus &= (u8)(~(BRAKE_WAIT_DEMAG + BRAKE_SET_UP + BRAKE));

	// Prepare a clean sinewave generation re-start when braking will be completed	
	MTC_UpdateSine(0, MTC_GetStatorFreq());	// Cancel voltage on stator
	MTCStatus |= FREQ_CHANGE;	// Here this bit is used to verify that U interrupt occured
	MISR = 0xFF;					// Force generation of U interrupt
	Counter = U8_MAX;
	while( (MTCStatus & FREQ_CHANGE) && (Counter > 0) )
	{
		Counter--;		// Wait for U interrupt
	}
	// If U not generated after 256 attempts, proceed anyway (means that U have 
	// already been stopped

	// Disable PWM interrupts to be able to modify duty cycle in main program
	PushCC();	// cf datasheet section 14 for explanations
	DisableInterrupts();
	MIMR &= ((u8)~MIMR_PUM_OR);
	PopCC();

	MTC_DisableMCOutputs();		// All MCO outputs OFF (0 with positive polarity)
	
   MTCStatus |= BRAKE_WAIT_DEMAG;
	ART_SetSequenceDuration (STATOR_DEMAG_TIME);
	
	BrakeDutyCycle = DutyCycle;
}


/*-----------------------------------------------------------------------------
ROUTINE Names : MTC_Brake

Description: This routine handles the three phases of the braking procedure:
				1. It manages a waiting time for the statoric current to decrease 
				down to zero (demagnetization), all PWM being OFF.
				2. It increases smoothly the DC current up to expected value to 
				avoid inrush current in the stator
				3. It maintains this current permanently unless the MTCStatus 
				byte has not one expected value (data corruption or inadequate 
				function use.
				
Physical background:
            Braking torque is obtained by sinking DC current in one motor winding.
            Practically, constant duty cycle PWM is present on one half bridge
            while the others are grounded (low side switches continuously ON).
-----------------------------------------------------------------------------*/
void MTC_Brake( void )
{
	u8 BrakeStatus;
	u16 CurrentDutyCycle;

	BrakeStatus = (u8)(MTCStatus & BRAKE_MASK);

   switch (BrakeStatus) 
   {
      case BRAKE_WAIT_DEMAG:	if ( ART_IsSequenceCompleted() )
										{
											
											MTCStatus &= (u8)(~BRAKE_WAIT_DEMAG);
											MTCStatus |= BRAKE_SET_UP;
											ART_SetSequenceDuration (CURRENT_SETUP_TIME);
											MCPUL = 0;
											MCPUH = 0;
											MCPVL = 0;
											MCPVH = 0;
											MCPWL = 0;
											MCPWH = 0;
											MTC_EnableMCOutputs();
										}
										break;

		case BRAKE_SET_UP:		CurrentDutyCycle = ((256*MCPUH + MCPUL)>>3);
										if (CurrentDutyCycle == BrakeDutyCycle)	// Expected duty cycle reached
										{
											MTCStatus &= (u8)(~BRAKE_SET_UP);
											MTCStatus |= BRAKE;
										}
										else if (ART_IsSequenceCompleted())
										{
											CurrentDutyCycle++;
											ToCMPxL(MCPUL,CurrentDutyCycle);
											ToCMPxH(MCPUH,CurrentDutyCycle);
										}
										break;

		case BRAKE:					break;
		
		default:						MTC_DisableMCOutputs();		// All MCO outputs OFF (0 with positive polarity)
										break;
	}
}


/*-----------------------------------------------------------------------------
ROUTINE Names : MTC_StopBraking

Description: This routines disables the DC current braking:
				* it disables the PWM outputs
				* it re-starts the U interrupts.
-----------------------------------------------------------------------------*/
void MTC_StopBraking( void )
{

	MTCStatus &= (u8)(~(BRAKE_WAIT_DEMAG + BRAKE_SET_UP + BRAKE));

	MCPUL = 0;
	MCPUH = 0;
	MCPVL = 0;
	MCPVH = 0;
	MCPWL = 0;
	MCPWH = 0;
	MISR = 0xFF;					// Force PWM duty cycle updation

	MTC_DisableMCOutputs();		// All MCO outputs OFF (0 with positive polarity)
	
	MIMR |= MIMR_PUM_OR;		// Re-enable the PWM U interrupts

}


/*-----------------------------------------------------------------------------
ROUTINE Names : MTC_ClearEmergencyStop
                MTC_ClearEmergencyStop

Description : The purpose of these two functions is to provide to the higher 
				level control modules information regarding an Emergency Stop of 
				the PWM operation. This information is available once the related 
				interrupt routine has been serviced, upon function call. For user 
				requiring immediate action taken as soon as the NMCES event occurs, 
				the interrupt routine need to be used directly.
				MTC_CheckEmergencyStop: indicates if PWM outputs are enabled or not,
				and therefore if MOE bit (Main Output Enable) has been cleared by hardware.
				MTC_ClearEmergencyStop resets the boolean where the emergency Stop 
				interrupt routine execution was recorded, regardless of the MOE bit state.
Returns:		MTC_CheckEmergencyStop returns a boolean parameter, TRUE if an emergency 
				Stop interrupt has been serviced, causing the PWM outputs to be disabled.
-----------------------------------------------------------------------------*/
BOOL MTC_CheckEmergencyStop(void)
{
	return(MCES_Status);
}

void MTC_ClearEmergencyStop(void)
{
	MCES_Status = FALSE;
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : MTC_U_CL_SO_IT
DESCRIPTION  : PWM reload interrupt in which the PWM duty cycle is refreshed 
for each of the three-phase; this interrupt occurs once per PWM period.
-----------------------------------------------------------------------------*/
#ifdef __HIWARE__			/* test for HIWARE Compiler */
#pragma TRAP_PROC 		/* This is an interrupt but no additional registers will be saved */
void MTC_U_CL_SO_IT(void)
{  
#asm
	push Y	; Done inside code to avoid SAVE_REGS directive
;********************* Integrate frequency into phase **************************
	;Update Freq value is calculation has been completed in MTC_UpdateSine
	btjf MTCStatus,#0,CalcPhase
		ld A,SineFreqBuf:1
		ld SineFreq:1,A
		ld A,SineFreqBuf
		ld SineFreq,A
		bres MTCStatus,#0
	
CalcPhase:
	ld A,Phase:1
	add A,SineFreq:1
	ld Phase:1,A
	ld A,Phase
	adc A,SineFreq
	ld Phase,A				;  16-bit accumulation of FreqCmd into Phase

;********************* PHASE U duty cycle processing ***************************
	ld Y,A					; Load LUT pointer in Y
	ld A,(SINE3RDHARM,Y)	; Get entry from table
	ld X,SineMag			; Scale According to SineMag
	mul X,A					; 16-bit Result in X:A
	btjt Phase,#7,nextU	; Phase represents offset in LUT
							; jmp if entry is positive, first half of table
	cpl X					; Otherwise negate result
	neg A
	jrne nextU
	inc X					; Increase MSB if 2_s complement of A is 100
nextU:
	#ifdef PWM_9BIT
		sra X
		rrc A
	#endif
	sra X
	rrc A
	sra X
	rrc A
	sra X
	rrc A					; match 10 bit pwm scaling

	ld MCPUL,A				; Load LSB of PWM generator
	ld A,X
	add A,#OFFSET			; add in zero offset to MSB
	ld MCPUH,A				; Load MSB of PWM generator

;********************* PHASE V duty cycle processing ***************************
	ld A,Y					; Get LUT pointer
	add A,PhaseShift		; Calc Phase V angle: advance or retard +/-Phaseshift degrees
	ld Y,A					; Y is now index for V
	ld A,(SINE3RDHARM,Y)	; Get entry from table
	ld X,SineMag			; Scale According to SineMag
	mul X,A					; 16-bit Result in X:A
	
	tnz Y					; Test sign of LUT pointer
	jrmi nextV				; Jmp if entry is negative, first half of table
	cpl X					; Otherwise negate result
	neg A
	jrne nextV
	inc X					; Increase MSB if 2_s complement of A is 100
nextV:
	#ifdef PWM_9BIT
		sra X
		rrc A
	#endif
	sra X
	rrc A
	sra X
	rrc A
	sra X
	rrc A					; match 10 bit pwm scaling

	ld MCPVL,A   			; Load LSB of PWM generator
	ld A,X
	add A,#OFFSET			; add in zero offset to MSB
	ld MCPVH,A   			; Load MSB of PWM generator

;********************* PHASE W duty cycle processing ***************************
	ld A,Phase				; Get LUT pointer
	add A,PhaseShift
	add A,PhaseShift		; Calc Phase W angle: advance or retard 2x(+/-Phaseshift) degrees
	ld Y,A					; Y is now index for V
	ld A,(SINE3RDHARM,Y)	; Get entry from table
	ld X,SineMag			; Scale According to SineMag
	mul X,A					; 16-bit Result in X:A

	tnz Y					; Test sign of LUT pointer
	jrmi nextW				; Jmp if entry is negative, first half of table
	cpl X					; Otherwise negate result
	neg A
	jrne nextW
	inc X					; Increase MSB if 2_s complement of A is 100
nextW:
	#ifdef PWM_9BIT
		sra X
		rrc A
	#endif
	sra X
	rrc A
	sra X
	rrc A
	sra X
	rrc A					; match 10 bit pwm scaling

	ld MCPWL,A				; Load LSB of PWM generator
	ld A,X
	add A,#OFFSET			; add in zero offset to MSB
	ld MCPWH,A				; Load MSB of PWM generator

EndOfUInt:
	ld A,#127
	ld MISR,A				; Clr PUI flag
	pop Y
#endasm

}

#else
#ifdef __CSMC__			/* test for Cosmic Compiler */
@interrupt				/* Cosmic interrupt handling */
void MTC_U_CL_SO_IT(void)
{  
#asm
	push Y
;********************* Integrate frequency into phase **************************
	btjf _MTCStatus,#0,CalcPhase
		ld A,_SineFreqBuf+1
		ld _SineFreq+1,A
		ld A,_SineFreqBuf
		ld _SineFreq,A
		bres _MTCStatus,#0
	
CalcPhase:
		ld A,_Phase+1
		add A,_SineFreq+1
		ld _Phase+1,A
		ld A,_Phase
		adc A,_SineFreq
		ld _Phase,A				;  16-bit accumulation of FreqCmd into Phase

;********************* PHASE U duty cycle processing ***************************
	ld Y,A					; Load LUT pointer in Y
	ld A,(_SINE3RDHARM,Y)	; Get entry from table
	ld X,_SineMag			; Scale According to SineMag
	mul X,A					; 16-bit Result in X:A
	btjt _Phase,#7,nextU	; Phase represents offset in LUT
							; jmp if entry is positive, first half of table
	cpl X					; Otherwise negate result
	neg A
	jrne nextU
	inc X					; Increase MSB if 2_s complement of A is 100
nextU:    
	#ifdef PWM_9BIT
		sra X
		rrc A
	#endif
	sra X
	rrc A
	sra X
	rrc A
	sra X
	rrc A					; match 10 bit pwm scaling

	ld _MCPUL,A				; Load LSB of PWM generator
	ld A,X
	add A,#_OFFSET			; add in zero offset to MSB
	ld _MCPUH,A				; Load MSB of PWM generator

;********************* PHASE V duty cycle processing ***************************
	ld A,Y					; Get LUT pointer
	add A,_PhaseShift		; Calc Phase V angle: advance or retard +/-Phaseshift degrees
	ld Y,A					; Y is now index for V
	ld A,(_SINE3RDHARM,Y)	; Get entry from table
	ld X,_SineMag			; Scale According to SineMag
	mul X,A					; 16-bit Result in X:A
	
	tnz Y					; Test sign of LUT pointer
	jrmi nextV				; Jmp if entry is negative, first half of table
	cpl X					; Otherwise negate result
	neg A
	jrne nextV
	inc X					; Increase MSB if 2_s complement of A is 100
nextV:
	#ifdef PWM_9BIT
		sra X
		rrc A
	#endif
	sra X
	rrc A
	sra X
	rrc A
	sra X
	rrc A					; match 10 bit pwm scaling

	ld _MCPVL,A   			; Load LSB of PWM generator
	ld A,X
	add A,#_OFFSET			; add in zero offset to MSB
	ld _MCPVH,A   			; Load MSB of PWM generator

;********************* PHASE W duty cycle processing ***************************
	ld A,_Phase				; Get LUT pointer
	add A,_PhaseShift
	add A,_PhaseShift		; Calc Phase W angle: advance or retard 2x(+/-Phaseshift) degrees
	ld Y,A					; Y is now index for V
	ld A,(_SINE3RDHARM,Y)	; Get entry from table
	ld X,_SineMag			; Scale According to SineMag
	mul X,A					; 16-bit Result in X:A

	tnz Y					; Test sign of LUT pointer
	jrmi nextW				; Jmp if entry is negative, first half of table
	cpl X					; Otherwise negate result
	neg A
	jrne nextW
	inc X					; Increase MSB if 2_s complement of A is 100
nextW:
	#ifdef PWM_9BIT
		sra X
		rrc A
	#endif
	sra X
	rrc A
	sra X
	rrc A
	sra X
	rrc A					; match 10 bit pwm scaling

	ld _MCPWL,A				; Load LSB of PWM generator
	ld A,X
	add A,#_OFFSET			; add in zero offset to MSB
	ld _MCPWH,A				; Load MSB of PWM generator
	
	pop Y
EndOfUInt:
#endasm
	MISR = ((u8)~(MISR_PUI_OR + MISR_CLI_OR));
}
#endif				/* test for Cosmic Compiler */
#endif				/* test for Hiware Compiler */


#ifdef __HIWARE__		/* test for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	/* additional registers will be saved */
#pragma NO_OVERLAP
#else
#ifdef __CSMC__			/* test for Cosmic Compiler */
@interrupt			/* Cosmic interrupt handling */
#endif
#endif
/*-----------------------------------------------------------------------------
ROUTINE NAME : MTC_C_D_IT
INPUT/OUTPUT : None

DESCRIPTION  : Tacho capture interrupt; this interrupt servide routine just 
				store the values in a buffer, without any processing

COMMENTS     : 
-----------------------------------------------------------------------------*/
void MTC_C_D_IT(void)  
{  
	if ( MTCStatus & INIT_ROLLING_AVRG )
	{
			u8 BufMzprv, BufMzreg, BufMprsr;

		BufMzreg = MZREG;	// Bufferize the values in case
		BufMprsr = MPRSR;	// ... a tacho capture occurs
		BufMzprv = MZPRV;	// ... during tacho FIFO initialization
		SpeedFIFO_Index = (PSpeedMeas_s)SensorPeriod;
		while (	SpeedFIFO_Index < &SensorPeriod[SPEED_FIFO_SIZE] )
		{
			SpeedFIFO_Index->Capture.b_form.high = BufMzreg;
			SpeedFIFO_Index->Capture.b_form.low = BufMzprv;
			SpeedFIFO_Index->Prsc_Reg = BufMprsr;
			SpeedFIFO_Index++;
		}
		SpeedFIFO_Index = (PSpeedMeas_s)SensorPeriod;
		MTCStatus &= (u8)(~INIT_ROLLING_AVRG);
		
	} else {	// Store the latest speed acquisition
	         
	         // Lock the tacho capture untill MZPRV is read
				SpeedFIFO_Index->Capture.b_form.high = MZREG;
				if (MTCStatus & RATIO_DEC)
				{
					SpeedFIFO_Index->Prsc_Reg = (u8)(MPRSR + 1);
				}
				else
				{
						SpeedFIFO_Index->Prsc_Reg = MPRSR;
				}
								
				SpeedFIFO_Index->Capture.b_form.low = MZPRV;
            // Captures are now allowed (thanks to MZPRV read access)
            
				if (SpeedFIFO_Index < &SensorPeriod[SPEED_FIFO_SIZE-1])
				{
					SpeedFIFO_Index++;
			   }
			   else
            {
					SpeedFIFO_Index = (PSpeedMeas_s)SensorPeriod;
				}
	    	}

    MTCStatus &= ((u8)~(RATIO_DEC));

	MISR = ((u8)~(MISR_CI_OR + MISR_DI_OR));
}


#ifdef __HIWARE__		/* test for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	/* additional registers will be saved */
#else
#ifdef __CSMC__			/* test for Cosmic Compiler */
@interrupt			/* Cosmic interrupt handling */
#endif
#endif
/*-----------------------------------------------------------------------------
ROUTINE NAME : MTC_R_Z_IT
INPUT/OUTPUT : None

DESCRIPTION  : RATIO_DEC and RATIO_INC flags are needed to have have the 
					MPRSR value just before the R event in the capture interrupt.

COMMENTS     : 
-----------------------------------------------------------------------------*/ 
void MTC_R_Z_IT(void)
{  
	if (MISR & MISR_RMI_OR)			// R- event
	{
		MTCStatus |= RATIO_DEC;
	}

	// No Z interrupt expected

	// Clear all related flags
	MISR = ((u8)~(MISR_RMI_OR + MISR_RPI_OR + MISR_ZI_OR));
}

#ifdef __HIWARE__		/* test for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	/* additional registers will be saved */
#else
#ifdef __CSMC__			/* test for Cosmic Compiler */
@interrupt			/* Cosmic interrupt handling */
#endif
#endif
/*-----------------------------------------------------------------------------
ROUTINE NAME : MCES_SE_IT
INPUT/OUTPUT : None

DESCRIPTION  : Emergency stop interrupt

COMMENTS     : 
-----------------------------------------------------------------------------*/ 
void MCES_SE_IT(void)
{
	MCES_Status = TRUE;
	MISR = ((u8)~MISR_EI_OR);
	ClrBit(MCRC,MCRC_SEI_OI);
}
/*** (c) 2004  STMicroelectronics **************************** END OF FILE ***/