/**************** (c) 2004 STMicroelectronics **********************

PROJECT  : 3-phase AC induction motor drive Software Library
COMPILER : COSMIC / METROWERKS

MODULE  :  ports.c
VERSION :  1.2.1

CREATION DATE : 07.2003 
AUTHOR :      Microcontroller Application Lab  / ST Hong Kong

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

DESCRIPTION :  I/O line control routines

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

MODIFICATIONS

04.2004  Rel 1.1: Remove unused functions and variables
06.2004  Rel 1.2: Fix PORTS_BothLedOff, set Button and LED ports and 
									bit location as parameters. All external int. declared 
									as empty interrupt routines.
09.2004  Rel 1.2.1: replace ST7FMC2N6.h by config.h for better flexibility.

******************************************************************************
THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. ST MICROELECTRONICS
SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
******************************************************************************
              
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

******************************************************************************/
                            
#include "lib.h"
#include "ports.h"
#include "pwmart.h"
// Library configuration and ST7FMC peripherals Hardware Registers declaration
#include "config.h"

// Define here where are the LEDs connected
#define		LED				((u8)0x80)		/* LED port mask */
#define 		LED_PORT			PBDR
#define 		LED_PORT_DDR   PBDDR
#define 		LED_PORT_OR		PBOR

// Define here where is the push button connected
// It will be active with low state (connected between ground and pull-up resistor).
#define		BUTTON			((u8)0x01)		/* Push button port mask */
#define		BUTTON_PORT		PCDR

// Debouncing duration: push button must be maintained at least for this time
// without state change to be considered as pushed.
#define DEBOUNCE_TIME   ((u8)10)		/* 10ms debounce time */

#define KEY_IDLE			((u8)0)
#define KEY_IDLE_NEXT	((u8)1)
#define KEY_PUSHED		((u8)2)

/* -------------------------- Private variables ---------------------------- */

static u8 KeyState;      // Push button status



/*-----------------------------------------------------------------------------
ROUTINE NAME : PORTS_Init
INPUT/OUTPUT : None

DESCRIPTION  : Configure the ports

COMMENTS     :
-----------------------------------------------------------------------------*/
void PORTS_Init(void)           // Setting for 56 pins case of Starter kit
{

	PADDR    = 0x00;    //PA7 to PA0 floating Input
	PAOR	 	= 0x00;

  	PBDDR    = 0x80; 	//PB7 output push pull by default(LED)
	PBOR     = 0x80;  //PB6 to PB0 floating Input

	PCDDR    = 0x00; 	//PC7 to PC0 floating Input
	PCOR     = 0x00;      

	PDDDR    = 0x00; 	//PD7 to PD0 floating Input
	PDOR     = 0x00;
    
	PEDDR    = 0x00;  //PE7 to PE0 floating Input
	PEOR     = 0x00;

	PFDDR    = 0x00;	//PF7 to PF0 floating Input
	PFOR     = 0x00;

	KeyState = 0;
}


/*-----------------------------------------------------------------------------
ROUTINE NAME : PORTS_RedLedOn; PORTS_RedLedOff; PORTS_RedLedToggle
					PORTS_GreenLedOn; PORTS_GreenLedOff; PORTS_GreenLedToggle
					PORTS_RedGreenLedToggle; PORTS_BothLedOff
INPUT/OUTPUT : None

DESCRIPTION  : Set/Reset/Toggle Red or Green  or Both LED

COMMENTS     :
-----------------------------------------------------------------------------*/
void PORTS_RedLedOn(void)
{
	LED_PORT_DDR	|= LED; 				//LED I/O port: push-pull Output
	LED_PORT_OR 	|= LED;        
	LED_PORT  		|= LED;
}

void PORTS_RedLedOff(void)
{
	LED_PORT_OR  &= (u8)~LED;     // Modify I/O port's Option Register first
	LED_PORT_DDR &= (u8)~LED; 		// then Data Direction Register -> Floating Input
}

void PORTS_RedLedToggle(void)
{
	if (LED_PORT_DDR & LED)		// push-pull config?
		{
		LED_PORT_OR  &= (u8)~LED;
		LED_PORT_DDR &= (u8)~LED; //Toggle I/O port's direction (Floating Input or push-pull Output)
    	}
else
		{
		LED_PORT_DDR 	|= LED; 		//Toggle I/O port's direction (Floating Input or push-pull Output)
		LED_PORT_OR  	|= LED;
		LED_PORT  		|= LED;
		}
}

void PORTS_GreenLedOn(void)
{
	LED_PORT_DDR |= LED; 			//Set I/O port as push-pull Output
	LED_PORT_OR  |= LED;        
	LED_PORT  &= (u8)~LED;
}

void PORTS_GreenLedOff(void)
{
	LED_PORT_OR  &= (u8)~LED;     // Modify Option Register first
	LED_PORT_DDR &= (u8)~LED; 		// then LED_PORT_DDR -> PB7 Floating Input
}

void PORTS_GreenLedToggle(void)
{
	if (LED_PORT_DDR & LED)          // output config?
		{
		LED_PORT_OR  &= (u8)~LED;
		LED_PORT_DDR &= (u8)~LED; 		//Toggle PB7 direction (Floating Input or push-pull Output)
    	}
	else
		{
		LED_PORT_DDR |= LED; 			//Toggle PB7 direction (Floating Input or push-pull Output)
		LED_PORT_OR  |= LED;
		LED_PORT  &= (u8)~LED;
		}
}

void PORTS_RedGreenLedToggle(void)
{
if (!(LED_PORT_DDR & LED))
	{
	LED_PORT_DDR |= LED; 		//Toggle Port direction (Floating Input or push-pull Output)
	LED_PORT_OR  |= LED; 
	}
LED_PORT ^= LED;
}	

void PORTS_BothLedOff(void)
{
	LED_PORT_OR  &= (u8)~LED;		// Modify Option register first to avoid spurious ei2 interrupts
	LED_PORT_DDR &= (u8)~LED; 		// Set I/O port as Floating Input
}

/*-----------------------------------------------------------------------------
ROUTINE Name :  PORTS_KeyScan
Description :  return TRUE if button is read and released
-----------------------------------------------------------------------------*/ 
BOOL PORTS_KeyScan(void)
{
if (!(BUTTON_PORT & BUTTON)) // Button pushed (low state on Input)
	{
  	switch(KeyState)
		{
		case KEY_PUSHED:
               	if (DebounceButton == 0) 
		    		{
		    		KeyState = KEY_IDLE_NEXT;
		    		return (TRUE);
		    		}
				else return (FALSE);
				break;
		case KEY_IDLE:
				DebounceButton = DEBOUNCE_TIME;	// Debounce time set-up
            KeyState = KEY_PUSHED;
				return(FALSE);
				break;
        case KEY_IDLE_NEXT:
		default:
				return(FALSE);
				break;
		}
	}
else KeyState = KEY_IDLE;
return(FALSE);
}


#ifdef __HIWARE__		/* test for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	/* additional registers will be saved */
#else
#ifdef __CSMC__			/* test for Cosmic Compiler */
@interrupt			/* Cosmic interrupt handling */
#endif
#endif
/*-----------------------------------------------------------------------------
ROUTINE NAME : PORTS_ei0_Interrupt
INPUT/OUTPUT : None

DESCRIPTION  : 

COMMENTS     : 
-----------------------------------------------------------------------------*/ 
void PORTS_ei0_Interrupt(void)
{

}


#ifdef __HIWARE__		/* test for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	/* additional registers will be saved */
#else
#ifdef __CSMC__			/* test for Cosmic Compiler */
@interrupt			/* Cosmic interrupt handling */
#endif
#endif
/*-----------------------------------------------------------------------------
ROUTINE NAME : PORTS_ei1_Interrupt
INPUT/OUTPUT : None

DESCRIPTION  : 

COMMENTS     : 
-----------------------------------------------------------------------------*/ 
void PORTS_ei1_Interrupt(void)
{

}


#ifdef __HIWARE__		/* test for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	/* additional registers will be saved */
#else
#ifdef __CSMC__			/* test for Cosmic Compiler */
@interrupt			/* Cosmic interrupt handling */
#endif
#endif
/*-----------------------------------------------------------------------------
ROUTINE NAME : PORTS_ei2_Interrupt
INPUT/OUTPUT : None

DESCRIPTION  : 

COMMENTS     : 
-----------------------------------------------------------------------------*/ 
void PORTS_ei2_Interrupt(void)
{

}

/*** (c) 2004  STMicroelectronics ****************** END OF FILE ***/