/**************** (c) 2004  STMicroelectronics ********************************
     
PROJECT  : 3-phase AC induction motor drive Software Library
COMPILER : COSMIC / METROWERKS

MODULE  :  pwmart.c
VERSION :  1.0.1

CREATION DATE : April 2004

AUTHOR :	Microcontroller Division Applications

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

DESCRIPTION :   PWM Auto Reload timer low level routines
              
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

MODIFICATIONS :

September 04 V1.0.1: replace ST7FMC2N6.h by config.h for better flexibility.
Cleaner coding of lock and time elapsed flags, now in two separate variables 
to ease additional time bases insertion.

*******************************************************************************
 THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. ST MICROELECTRONICS
 SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
 WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
******************************************************************************/

#include "lib.h"		/* General purpose typedefs and macros */
#include "pwmart.h"		/* Public PWMART function prototypes */
// Library configuration and ST7FMC peripherals Hardware Registers declaration
#include "config.h"

#define TB_UNIT		((u8)0x06)	/* 1ms Time base given by 250 (256-6) clock 
												cycles @ Fin/32 (250kHz) */

volatile u8 DebounceButton;

/*--------------- Private Variables declaration section ---------------------*/

static volatile u8  TimeBaseStatus;
static volatile u8  LockStatus;

static volatile u8  RegCounter;
static volatile u8  RegTimeBase;

static volatile u8  WdgTimeBase;
static volatile u8  WdgCounter;

static volatile u16 TimeOutCounter;
static volatile u16 TimeOutPeriod;

static volatile u16 SequenceCounter;
static volatile u16 SequencePeriod;

static volatile u16 ms_Counter;
static volatile u16 ms_TimeBase;

static volatile u16  MainCounter;
static volatile u16  MainTimeBase;

/* TimeBaseStatus flags description */
#define TIMEOUT_ELAPSED			((u8)0x01)
#define REG_PERIOD_ELAPSED		((u8)0x02)
#define SEQUENCE_COMPLETED		((u8)0x04)
#define MS_TIME_ELAPSED			((u8)0x08)
#define WDG_TIMEBASE_ELAPSED	((u8)0x10)
#define MAIN_TIMEBASE_ELAPSED	((u8)0x20)
#define FREE_1						((u8)0x40)
#define FREE_2						((u8)0x80)

//LockStatus flag
#define TIMEOUT_LOCK				((u8)0x01)
#define SEQUENCE_LOCK			((u8)0x02)
#define MS_TIME_LOCK				((u8)0x04)
#define MAIN_TIME_BASE_LOCK	((u8)0x08)

/*-----------------------------------------------------------------------------
ROUTINE NAME : ART_Init
INPUT/OUTPUT : None

DESCRIPTION  : 

COMMENTS     : 
-----------------------------------------------------------------------------*/ 
void ART_Init(void)
{
	ARTICCSR = 0;	// No capture
	PWMCR = 0;		// No PWM
	ARTCAR = 0;		// Reset counter
	ARTCSR = 0;		// Clr pending OVF interrupt
	
	ARTARR = TB_UNIT;		// Time base Unit also depends from CLK prescaler below
	// Set clock Fin by 32, force reload, enable OVF int
	ARTCSR = ARTCSR_CC2_OR + ARTCSR_CC0_OR + ARTCSR_FCRL_OR + ARTCSR_OIE_OR;

	// Init software time bases variables
	TimeBaseStatus	= 0;
	LockStatus = 0;
	
	TimeOutCounter	= U16_MAX;
	TimeOutPeriod	= U16_MAX;
	RegCounter		= U8_MAX;
	RegTimeBase		= U8_MAX;
	SequenceCounter	= U16_MAX;
	SequencePeriod	= U16_MAX;
	ms_Counter		= U16_MAX; 
	ms_TimeBase		= U16_MAX;
	MainTimeBase 	= U16_MAX;
	MainCounter 	= U16_MAX;

	WdgTimeBase		= U8_MAX;
	WdgCounter		= U8_MAX;
	
	DebounceButton 	= 0;
}


/*-----------------------------------------------------------------------------
ROUTINE NAME : ART_Start
INPUT/OUTPUT : None

DESCRIPTION  : 

COMMENTS     : 
-----------------------------------------------------------------------------*/ 
void ART_Start(void)
{
	ARTCSR |= ARTCSR_TCE_OR;			// Start timer
}


/*-----------------------------------------------------------------------------
ROUTINE NAME : ART_SetTimeOutDuration
INPUT/OUTPUT : 

DESCRIPTION  : Set Time Out Duration and clear related flag.
Description : Initialize counters, Flags and autoreload values for process
              sequencing. Start time decounting once called.

COMMENTS     : A protection/lock mechanism is implemented for 16-bit Time bases
				in order to prevent any collision if the time base interrupt 
				occurs during the MSB/LSB load sequence.
				Maximum jitter is 1ms if OVF occurs just after Counter set.
-----------------------------------------------------------------------------*/
void ART_SetTimeOutDuration(u16 Duration)
{
	if ( Duration == 0)
	{
		Duration = 1;	// Min value is 1ms
	}
	LockStatus |= TIMEOUT_LOCK;		// Lock TimeOut handling in interrupt
	TimeOutPeriod = Duration;				// Value autoreloaded when counter = 0
	TimeOutCounter = Duration;  			// Reset counter to new value
	TimeBaseStatus &= (u8)(~TIMEOUT_ELAPSED); //	Clear Flag
	LockStatus &= (u8)(~TIMEOUT_LOCK);		// Unlock TimeOut handling in interrupt
}

void ART_SetSpeedRegPeriod(u8 Period)
{
	if ( Period == 0) Period = 1;	// Min value is 1ms
	RegTimeBase = Period;			// Value autoreloaded when counter = 0
	RegCounter = Period;  			// Reset counter to new value
	TimeBaseStatus &= (u8)(~REG_PERIOD_ELAPSED); //	Clear Flag
}

void ART_SetSequenceDuration(u16 Duration)
{
	if ( Duration == 0)
	{
		Duration = 1;	// Min value is 1ms
	}
	LockStatus |= SEQUENCE_LOCK;		// Lock TimeOut handling in interrupt
	SequencePeriod = Duration;				// Value autoreloaded when counter = 0
	SequenceCounter = Duration;  			// Reset counter to new value
	TimeBaseStatus &= (u8)(~SEQUENCE_COMPLETED);	//	Clear Flag
	LockStatus &= (u8)(~SEQUENCE_LOCK);	// Unlock TimeOut handling in interrupt
}

void ART_Set_TimeInMs(u16 Duration)
{
	if ( Duration == 0)
	{
		Duration = 1;	// Min value is 1ms
	}
	LockStatus |= MS_TIME_LOCK;	// Lock TimeOut handling in interrupt
	ms_TimeBase = Duration;				// Value autoreloaded when counter = 0
	ms_Counter = Duration;  			// Reset counter to new value
	TimeBaseStatus &= (u8)(~MS_TIME_ELAPSED);	//	Clear Flag
	LockStatus &= (u8)(~MS_TIME_LOCK);	// Unlock TimeOut handling in interrupt
}

void ART_SetMainTimeBase(u8 Period)
{
	if ( Period == 0) Period = 1;	// Min value is 1ms
	
	LockStatus |= MAIN_TIME_BASE_LOCK;	// Lock TimeOut handling in interrupt	
	MainTimeBase = Period*10;		// Value autoreloaded when counter = 0
	MainCounter = Period*10;  		// Reset counter to new value
	TimeBaseStatus &= (u8)(~MAIN_TIMEBASE_ELAPSED); //	Clear Flag
	LockStatus &= (u8)(~MAIN_TIME_BASE_LOCK);	// Unlock TimeOut handling in interrupt
}

void ART_SetWdgRfrshTime(u8 Period)
{
	if ( Period == 0) Period = 1;	// Min value is 1ms
	WdgTimeBase = Period;		// Value autoreloaded when counter = 0
	WdgCounter = Period;  		// Reset counter to new value
	TimeBaseStatus &= (u8)(~WDG_TIMEBASE_ELAPSED); //	Clear Flag
}


/*-----------------------------------------------------------------------------
ROUTINE NAME : ART_IsTimeOutElapsed
INPUT/OUTPUT : Boolean

DESCRIPTION  : 	Indicates that the Time out period has expired.

COMMENTS     :	This function has to be polled as often as possible to 
				minimize the error on expected Time Out period.
-----------------------------------------------------------------------------*/
BOOL ART_IsTimeOutElapsed(void)
{       
	if ( TimeBaseStatus & TIMEOUT_ELAPSED )
	{
		TimeBaseStatus &= (u8)(~TIMEOUT_ELAPSED); //	Clear Flag
		return ( TRUE );
	}
	else
 	{
		return ( FALSE );
	}
}


BOOL ART_IsRegPeriodElapsed(void)
{
	if ( TimeBaseStatus & REG_PERIOD_ELAPSED )
	{
		TimeBaseStatus &= (u8)(~REG_PERIOD_ELAPSED); //	Clear Flag
		return ( TRUE );
	}
	else
 	{
		return ( FALSE );
	}
}


BOOL ART_IsSequenceCompleted(void)
{
	if ( TimeBaseStatus & SEQUENCE_COMPLETED )
	{
		TimeBaseStatus &= (u8)(~SEQUENCE_COMPLETED); //	Clear Flag
		return ( TRUE );
	}
	else
 	{
		return ( FALSE );
	}
}


BOOL ART_Is_TimeInMsElapsed(void)
{
	if ( TimeBaseStatus & MS_TIME_ELAPSED )
	{
		TimeBaseStatus &= (u8)(~MS_TIME_ELAPSED); //	Clear Flag
		return ( TRUE );
	}
	else
 	{
		return ( FALSE );
	}
}


BOOL ART_IsMainTimeElapsed(void)
{
	if ( TimeBaseStatus & MAIN_TIMEBASE_ELAPSED )
	{
		TimeBaseStatus &= (u8)(~MAIN_TIMEBASE_ELAPSED); //	Clear Flag
		return ( TRUE );
	}
	else
 	{
		return ( FALSE );
	}

}

BOOL ART_IsWdgTimeElapsed(void)
{
	if ( TimeBaseStatus & WDG_TIMEBASE_ELAPSED )
	{
		TimeBaseStatus &= (u8)(~WDG_TIMEBASE_ELAPSED); //	Clear Flag
		return ( TRUE );
	}
	else
 	{
		return ( FALSE );
	}

}

#ifdef __HIWARE__		/* test for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	/* additional registers will be saved */
#else
#ifdef __CSMC__			/* test for Cosmic Compiler */
@interrupt			/* Cosmic interrupt handling */
#endif
#endif
/*-----------------------------------------------------------------------------
ROUTINE NAME : ART_Interrupt
INPUT/OUTPUT : None

DESCRIPTION  : 

COMMENTS     : 
-----------------------------------------------------------------------------*/ 
void ART_Interrupt(void)
{
		u8 Dummy;

	Dummy = ARTCSR;		// Clear OVF flag

	if (DebounceButton != 0) DebounceButton--;

	RegCounter--;
	if (RegCounter == 0)
	{
		TimeBaseStatus |= REG_PERIOD_ELAPSED;	// Used in IsTestTimeBaseElapsed
		RegCounter = RegTimeBase; 				// Continue with same period
	}

	if ( !(LockStatus & MAIN_TIME_BASE_LOCK))	// No Main Time base modification on-going
	{
		MainCounter--;
		if (MainCounter == 0)
		{
			TimeBaseStatus |= MAIN_TIMEBASE_ELAPSED;// Used in ART_IsMainTimeElapsed
			MainCounter = MainTimeBase;  			// Continue with same period
   	 }
   }

	WdgCounter--;
	if (WdgCounter == 0)
	{
		TimeBaseStatus |= WDG_TIMEBASE_ELAPSED;// Used in ART_SetWatchdogRefreshPeriod
		WdgCounter = WdgTimeBase; 			// Continue with same period
    }

	if ( !(LockStatus & TIMEOUT_LOCK))	// No TimeOut modification on-going
	{
		TimeOutCounter--;
		if (TimeOutCounter == 0)
		{
			TimeBaseStatus |= TIMEOUT_ELAPSED; // Used in ART_IsTimeOutElapsed
			TimeOutCounter = TimeOutPeriod;    // Reload programmed period
		}
	}

	if ( !(LockStatus & SEQUENCE_LOCK))	// No Sequence modification on-going
	{
		SequenceCounter--;
		if (SequenceCounter == 0)
		{
			TimeBaseStatus |= SEQUENCE_COMPLETED;	// Used in ART_IsTimeOutElapsed
			SequenceCounter = SequencePeriod;		// Reload programmed period
		}
	}

	if ( !(LockStatus & MS_TIME_LOCK))	// No Sequence modification on-going
	{
		ms_Counter--;
		if (ms_Counter == 0)
		{
			TimeBaseStatus |= MS_TIME_ELAPSED;	// Used in ART_IsTimeOutElapsed
			ms_Counter = ms_TimeBase;				// Reload programmed period
		}
	}

}
/*** (c) 2004  STMicroelectronics **************************** END OF FILE ***/