/**************** (c) 2005 STMicroelectronics **********************

PROJECT  : ST7MC demokit
COMPILER : ST7 METROWERKS C (HIWARE) / COSMIC

MODULE  :  mtc.c
LIBRARY VERSION  :  2.0

CREATION DATE : 07.2003 
AUTHOR :      Florent COSTE	/  Microcontroller Application Lab  / ST Hong Kong

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

DESCRIPTION :  MTC peripheral control routines

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

 ******************************************************************************
 THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. ST MICROELECTRONICS
 SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
 WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
 ******************************************************************************

******************************************************************************/

#include "version.h"
#include "lib.h"			/* General purpose typedefs and macros */
#include "ports.h"
#include "ST7MC_hr.h"
#include "mtc_hr.h"
#include "LinSCI.h"
#include "Timer.h"  
#include "adc.h"
#include "regul.h"
#include "mtc.h"			/* Public motor control function prototypes */
#include "misc.h" 
#include "opamp.h" 
#include "MTC_Settings_Sensorless.h"

                     
/* -------------------------- Private constants ---------------------------- */

/* Pattern for the refresh of high side drivers bootstrap capacitor */
const u8 BOOT_REFRESH[3] = {
	T2,
	T4,
	T6
};
 
// Step configuration for Clock Wise (CW) rotor Direction 
const u8 PHASE_CONFIG[6] =	{
                        	mem_step1,mem_step2,mem_step3,
				mem_step4,mem_step5,mem_step6
				};

const u8 ZC_DEMAG_PWM[6] =	{
                            	mem_MCRB1,mem_MCRB2,mem_MCRB3,
				mem_MCRB4,mem_MCRB5,mem_MCRB6
				}; 
        
// Step configuration for Counter Clock Wise (CCW) rotor Direction        
const u8 PHASE_CONFIG_CCW[6] =	{
                        	mem_CCW_step1,mem_CCW_step2,mem_CCW_step3,
				mem_CCW_step4,mem_CCW_step5,mem_CCW_step6
				};

const u8 ZC_DEMAG_PWM_CCW[6] =	{
                            	mem_CCW_MCRB1,mem_CCW_MCRB2,mem_CCW_MCRB3,
				mem_CCW_MCRB4,mem_CCW_MCRB5,mem_CCW_MCRB6
				};        

const u8 DEMAG_TIME[16] =	{
				Run_Soft_Dem0,Run_Soft_Dem1,Run_Soft_Dem2,Run_Soft_Dem3,
				Run_Soft_Dem4,Run_Soft_Dem5,Run_Soft_Dem6,Run_Soft_Dem7,
				Run_Soft_Dem8,Run_Soft_Dem9,Run_Soft_Dem10,Run_Soft_Dem11,
				Run_Soft_Dem12,Run_Soft_Dem13,Run_Soft_Dem14,Run_Soft_Dem15
				};

const Step_s RAMP[64] = {
			def_ramp_ratio0,def_ramp_value0,def_ramp_ratio1,def_ramp_value1,def_ramp_ratio2,def_ramp_value2,
			def_ramp_ratio3,def_ramp_value3,def_ramp_ratio4,def_ramp_value4,def_ramp_ratio5,def_ramp_value5,
			def_ramp_ratio6,def_ramp_value6,def_ramp_ratio7,def_ramp_value7,def_ramp_ratio8,def_ramp_value8,
			def_ramp_ratio9,def_ramp_value9,def_ramp_ratio10,def_ramp_value10,def_ramp_ratio11,def_ramp_value11,
			def_ramp_ratio12,def_ramp_value12,def_ramp_ratio13,def_ramp_value13,def_ramp_ratio14,def_ramp_value14,
			def_ramp_ratio15,def_ramp_value15,def_ramp_ratio16,def_ramp_value16,def_ramp_ratio17,def_ramp_value17,
			def_ramp_ratio18,def_ramp_value18,def_ramp_ratio19,def_ramp_value19,def_ramp_ratio20,def_ramp_value20,
			def_ramp_ratio21,def_ramp_value21,def_ramp_ratio22,def_ramp_value22,def_ramp_ratio23,def_ramp_value23,
			def_ramp_ratio24,def_ramp_value24,def_ramp_ratio25,def_ramp_value25,def_ramp_ratio26,def_ramp_value26,
			def_ramp_ratio27,def_ramp_value27,def_ramp_ratio28,def_ramp_value28,def_ramp_ratio29,def_ramp_value29,
			def_ramp_ratio30,def_ramp_value30,def_ramp_ratio31,def_ramp_value31,def_ramp_ratio32,def_ramp_value32,
			def_ramp_ratio33,def_ramp_value33,def_ramp_ratio34,def_ramp_value34,def_ramp_ratio35,def_ramp_value35,
			def_ramp_ratio36,def_ramp_value36,def_ramp_ratio37,def_ramp_value37,def_ramp_ratio38,def_ramp_value38,
			def_ramp_ratio39,def_ramp_value39,def_ramp_ratio40,def_ramp_value40,def_ramp_ratio41,def_ramp_value41,
			def_ramp_ratio42,def_ramp_value42,def_ramp_ratio43,def_ramp_value43,def_ramp_ratio44,def_ramp_value44,
			def_ramp_ratio45,def_ramp_value45,def_ramp_ratio46,def_ramp_value46,def_ramp_ratio47,def_ramp_value47,
			def_ramp_ratio48,def_ramp_value48,def_ramp_ratio49,def_ramp_value49,def_ramp_ratio50,def_ramp_value50,
			def_ramp_ratio51,def_ramp_value51,def_ramp_ratio52,def_ramp_value52,def_ramp_ratio53,def_ramp_value53,
			def_ramp_ratio54,def_ramp_value54,def_ramp_ratio55,def_ramp_value55,def_ramp_ratio56,def_ramp_value56,
			def_ramp_ratio57,def_ramp_value57,def_ramp_ratio58,def_ramp_value58,def_ramp_ratio59,def_ramp_value59,
			def_ramp_ratio60,def_ramp_value60,def_ramp_ratio61,def_ramp_value61,def_ramp_ratio62,def_ramp_value62,
			def_ramp_ratio63,def_ramp_value63
			};

			

/* -------------------------- Public variables ---------------------------- */

u8 Power_Motor_Status;
u16 Freq_Motor;
u8 Ki = Ki_Fmin;      // integral term
u8 Kp = Kp_Fmin;      // proportionnal term
u8 Rising_bemf = Rising_Fmin ;
u8 Falling_bemf = Falling_Fmin;
u8 Flag_MTC;
u8 Motor_Direction;  

/* -------------------------- Private variables ---------------------------- */
#ifdef __HIWARE__   // only for Metrowerks Compiler
#pragma DATA_SEG SHORT _ZEROPAGE   

static u8		RPICounter;			// ASM SW: flagR_plus
static u16	    SoftDemagTime;		// ASM SW: mem_mdreg
static u8		CeventCounter;      // ASM SW: memnbemf
static u8		BemfCounter;		// ASM SW: nbemf
static u8		RampIndex;			// ASM SW: com_t_nb
static u8		StepIndex;			// ASM SW: PhaseStep  
static u8		MotorStatus;		// ASM SW: MotorStatus
static u8		delay_counter;
static u8	    Step_Z_Counter;
                
#pragma DATA_SEG DEFAULT

#else

#ifdef __CSMC__							// only for Cosmic Compiler

@tiny static     u8		RPICounter;			// ASM SW: flagR_plus
@tiny static     u16	SoftDemagTime;		// ASM SW: mem_mdreg
@tiny static     u8		CeventCounter;      // ASM SW: memnbemf

@tiny static     u8		BemfCounter;		// ASM SW: nbemf
@tiny static     u8		RampIndex;			// ASM SW: com_t_nb
@tiny static     u8		StepIndex;			// ASM SW: PhaseStep
@tiny static     u8		MotorStatus;		// ASM SW: MotorStatus  
@tiny static     u8     delay_counter;
@tiny static     u8    	Step_Z_Counter;

#else
  #error"Unsupported Compiler!"	// Compiler Defines not found! 
#endif
#endif

static volatile u8 Step_counter;
static volatile u8 Flag_it;
#define RP_EVT		1
#define RM_EVT		2 

#define Reset_RPM_MASK	0xF9
static volatile u8 RP_counter;

#define Reset_H_SDM_MASK 	0x4f

#if (DRIVING_MODE == VOLTAGE_MODE)
static u8 Previous_Duty_L;
static u8 Previous_Duty_H;
#endif  
                           
static u8 Align_Index;

 
typedef enum
{
INIT_START, ALIGN_ON_GOING, INIT_SWITCHED_MODE, START_FINISHED
} StartStatus_t;

StartStatus_t Status_Start;


typedef enum
{
WAIT_FOR_C, ALIGN_NEXT
} AlignStatus_t;

AlignStatus_t AlignState;


typedef enum
{
BRAKE_START, BRAKE_ONGOING, BRAKE_FINISHED
} BrakeStatus_t;

BrakeStatus_t BrakeState;


/* -------------------------- Private functions ---------------------------- */
static void RefreshBootstrap(void);
static void AlignRotor(void);
static void MTC_ResetPeripheral(void);

/* -------------------------- Private macros ---------------------------- */

// number of Z event to wait before closed loop operation 
#define Z_event_for_PI 12

// Below macros allow to validate PWM outputs or put them in their RESET state
#define MTC_EnableOutputs()		( MCRA |= (1<<MOE) )
#define MTC_DisableOutputs()	( MCRA &= ((1<<MOE)^255) )

// Below macros allow to enable/disable the motor control peripheral clock
#define MTC_EnableClock()		( MCRA |= (1<<CKE) )
#define MTC_DisableClock()		( MCRA &= ((1<<CKE)^255) )

// Below macros allow to enable/disable Direct Access mode to PWM outputs
#define MTC_EnableDirectAccess()		( MCRA |= (1<<DAC) )
#define MTC_DisableDirectAccess()		( MCRA &= ((1<<DAC)^255) )

// Macro updating MTIM prescaler during acceleration ramp in switched mode:
// If the ratio required for next step is equal to the 4 LSbits of MPRSR 
// (ie presc ratio), then MISR is cleared, otherwise, the RMI bit of MISR is
// set -> the prescaler will be decreased when the next C event will occur
// Note: This function only works for the acceleration phase: no possibility 
// to increase the MTIM prescaler
#define UpdateRatio() ((RAMP[RampIndex].Ratio == (MPRSR&0x0F)) ? (MISR = 0) : (MISR = RMI_MSK) ) 

#define SW_DEMAG		((u8)4)	/* Demag Time is StepTime divided
					   by SW_DEMAG in autoswitched mode */
#define SWITCHED_SW_DEMAG	((u8)4)	/* Demag Time is StepTime divided by 
					   SWITCHED_SW_DEMAG in switched mode */ 
					

#define RAMP_SIZE		((u8)64)    	/* Acceleration Ramp size : MUST BE
						                below 254 to avoid CeventCounter overflow */
#define	START_POSITION		((u8)T1+T4+T6)	/* Defines the rotor position at start:
						                    here an intermediary position between two steps */
#define PWM_CLK_PRSC		4		 // Ftimer = Fmtc/4

#define TRANSITION_DELAY	255      // set to 255 to center Z on the step on first autoswitch    
#define AUTO_DELAY	        128      // delay use for the first electrical cycle in autoswitch
#define AUTO_DELAY_STEP     6		 //number of step using AUTO_DELAY
#define MEDIUM_DELAY        110		 // delay use for the second electrical cycle in autoswitched
#define MAX_DELAY_COUNTER   12		 //number of step for managing of delay at startup

													// b0-b2-b4:T2-T4-T6 are Low Switch -> KANDA
													// b1-b3-b5:T1-T3-T5 are High Switch -> KANDA
#define OUTPUT_PARITY		0x2A		// b7=b6=0 for BLDC mode (either sensored or sensorless)
													// b1-b3-b5:T2-T4-T6 are Low Switch
													// b0s-b2-b4:T1-T3-T5 are High Switch
#define STARTUP_DELAY		0xFF		// Default delay coefficient at start ASM SW: delay_ki1
#define ALIGN_STEP_NUM		25    		// Number of duty cycle increments during alignment phase


/* ------------------------------------------------------------------------- */
/* ------------------------------------------------------------------------- */
/* ------------------------ Functions description -------------------------- */
/* ------------------------------------------------------------------------- */
/* ------------------------------------------------------------------------- */


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_ResetPeripheral

Description: 	Reset all hardware registers related to the motor control 
				peripheral by toggling the RST bit of the MCRB register.
Input/Output:	None
-----------------------------------------------------------------------------*/
void MTC_ResetPeripheral(void)
{
	SetBit(MCFR,RST); 
	ClrBit(MCFR,RST);
}


/*-----------------------------------------------------------------------------
ROUTINE Name : RefreshBootstrap

Description:	This function ensures that all high side switch drivers are 
				correctly supplied when the motor will start. It generates a 
				1ms pulse on each of the three correspondant low side switches.
				All switches are OFF at the end of the routine.
Input/Output:	None
Comments: 		Durations are indicative and are high side driver dependant.
-----------------------------------------------------------------------------*/
void RefreshBootstrap(void)
{
		u8 i;
	for (i=0; i<3; i++)
	{
		MPHST = BOOT_REFRESH[i];	// switch ON one low side switch
		Wait1ms();
		Wait1ms();
		MPHST = 0;					// Switch OFF all switches
		Wait1ms();
		Wait1ms();
	}
	MPHST = 0;	// All switches OFF
	Wait100us();	// This is to avoid any short circuit conditions
}



/*-----------------------------------------------------------------------------
ROUTINE Name : AlignRotor

Description:	After the desired rotor position had been set, this function
				increases the stator current to align accordingly the rotor 
				and prepare the first commutation.
Input/Output:	None
Comments: 		None
-----------------------------------------------------------------------------*/
void AlignRotor(void)
{
u8 i;
u16 DutyCmd, Step_Duty;
u32 temp;

switch (AlignState)
	{
	case ALIGN_NEXT:
	default:
		#if (DRIVING_MODE == VOLTAGE_MODE)
	   		Step_Duty = ((u16)(Align_MCPUL+(u16)(Align_MCPUH<<8)))/(ALIGN_STEP_NUM-1);
	
			ToCMPxL(MCPUL,(u16)(Align_Index*Step_Duty));  // increase current (linear curve)
			ToCMPxH(MCPUH,(u16)(Align_Index*Step_Duty));
		#else
			Step_Duty = ((u16)(Align_MCPVL+(u16)(Align_MCPVH<<8)))/(ALIGN_STEP_NUM-1);
			ToCMPxL(MCPVL,(u16)(Align_Index*Step_Duty));  // increase current (linear curve)
			ToCMPxH(MCPVH,(u16)(Align_Index*Step_Duty));
		#endif
            	Align_Index++;
		if (Align_Index == ALIGN_STEP_NUM-1) Status_Start = INIT_SWITCHED_MODE;
        	AlignState = WAIT_FOR_C;
		MISR = 0;				// Reset all pending bits
	break;

	case WAIT_FOR_C:
		if (MISR & CI_MSK) AlignState = ALIGN_NEXT;
		break;
	}
}


/*-----------------------------------------------------------------------------
ROUTINE Name :	GetMotorStatus

Description:	Allows higher level SW module to get the motor drive module 
				internal state
Input:			None
Output:			MotorStatus value
Comments: 		MotorStatus flags description is public (cf mtc.h)
-----------------------------------------------------------------------------*/
u8 GetMotorStatus(void)
{
	return(MotorStatus);
}


/*-----------------------------------------------------------------------------
ROUTINE Name :	SetMotorStatus

Description:	Allows higher level SW module to set the motor drive module 
				internal state
Input:			MotorStatus value
Output:			MNone
Comments: 		MotorStatus flags description is public (cf mtc.h)
-----------------------------------------------------------------------------*/
void SetMotorStatus(u8 status)
{
	MotorStatus = (u8)(status);
}


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_InitPeripheral

Description:	This function (re)initializes the motor control peripheral from
				any	state (MCU reset, application level initialization, ...).
Input/Output:	None
Comments: 		It must be noticed that part of MPOL and MDTG registers are 
				write once, meaning they cannot be modified anymore once the 
				MTC_InitPeripheral function has been executed.
-----------------------------------------------------------------------------*/
void MTC_InitPeripheral(void)
{

	MTC_ResetPeripheral();

	// Initialize registers in page 1 first
	SET_MTC_PAGE(1);
	MCONF = mem_MCONF;
//	MPWME = (u8)(mem_MPWME | DG_MSK);      //Force output of debug signal
        MPWME = (u8)(mem_MPWME);      		//Without debug signal
        
	MPOL = DRIVERS_POLARITY;				// Switch driver to the one selected
							// ZVD bit=0; Z and D have opposite edge
							// REO bit=0;read Z event on even channels

  	MDTG = mem_MDTG; //reset PCN and write the delay value
  	MDTG = (u8)(mem_MDTG&(0x40));  // disable the complementary PWM mode
	
	MPAR = OUTPUT_PARITY;				// Define Low/High side MCOx outputs

    	MZFR = mem_MZFR;
    	MSCR = mem_MSCR;

	// Initialize registers in page 0
	SET_MTC_PAGE(0);	

	MPCR = mem_MPCR;

	MCP0L = mem_MCPOL; 
	MCP0H = mem_MCPOH;
	
	MCRA = mem_MCRA;

	MCPUL = 0;    	// 0% duty cycle on U Channel 
	MCPUH = 0;
  
	MCPVL = 0;    	// 0% duty cycle on V Channel 
	MCPVH = 0;    
	MCRC = mem_MCRC;

    	MCFR = mem_MCFR;
    	MDFR = mem_MDFR;

#ifdef EXT_AOP   // external circuitry 
	MREF = CFAV_MASK;	// No Chopper, MCCFI input
#else
	MREF &= (u8)(~CFAV_MASK);	// No Chopper, OAZ input
#endif 
   	MotorStatus = 0;	// Needed to manage switched to autoswitched transition
	    				// Indicate motor driving state (start/stop/stalled...) 
}


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_StartMotor

Description:	This function performs three actions:
		* Initializes HW registers and SW variables needed in real time	for motor drive
		* prepares the start-up by refreshing the bootstrap capacitors of the high side
		  switch drivers and aligning the rotor in a known position
		* start-up the peripheral circuitry to get expected interrupts
Input/Output:	None
-----------------------------------------------------------------------------*/
BOOL MTC_StartMotor(void)
{
switch (Status_Start)
	{
	default:
	case INIT_START:
	// SW variables init
		BemfCounter = 0;			// Counts Bemf before autoswitched mode
		CeventCounter = 0;			// Needed to be sure that Bemf regularly occurs
		RPICounter = 0;				// When we have to wait until one or several R+ 
		SoftDemagTime = 0;
		StepIndex = 0;				// Alignment step is the first of the table
		RampIndex = 0;				// To get the ramp's first step ratio
		delay_counter = 0;			//counter to manage delay at start_up
		Flag_it = 0;
		Power_Motor_Status &= Heat_Voltage_Failure;  // reset all flags except overtemp & overvoltage 
   		Step_Z_Counter = 0;
   		Step_counter = Z_event_for_PI;	     //counter use to validate PI after we pass in autoswitch
    		Freq_Motor = 0;
        
		#if (DRIVING_MODE == VOLTAGE_MODE)
		MCPVL = mem_MCPVL;    // current limitation - Voltage mode
		MCPVH = mem_MCPVH;
		#endif
		
		#ifdef CLOSED_LOOP
		Init_PI();
		#endif 

	    	PORTS_BothLedOff();     // start command -> LEDs off
		#ifndef EXT_AOP
		OPAMP_InitOffset(LOW_GAIN);     	// initialise Opamp Gain and Offset compensation
//		OPAMP_InitOffset(HIGHGAIN);     	// initialise Opamp Gain and Offset compensation
		#endif

   		// Init of MTC peripheral hardware registers used modified during run-time
		MIMR = 0;
		MPRSR = (u8)(mem_MPRSR + Align_Ratio);	// MTIM timer prescaler set-up

		MPHST = 0;											// All switches OFF
		MCRB = 0;		
 
   		MWGHT = STARTUP_DELAY;			// Delay Coefficient in switched mode
   		MCOMP = (u8)(Align_MCOMP);		// Load Alignment time 
		#if (DEMAG_TYPE == SW)
    		MDREG = DEMAG_TIME[(MPRSR & 0x0f)];  // load demagnetization time according
   		                                      // to the MTIM ratio prescaler
		#else
   		MDREG = (u8)(Align_MCOMP/4);		// Load soft Demag Time
		#endif


		MTC_EnableDirectAccess();
		MTC_EnableOutputs();
	
		RefreshBootstrap();			// To be sure high side driver correctly supplied
    
		MTC_EnableClock();
		MPHST = MCIC + START_POSITION;	// Alignment step pattern T1+T4+T6
		MTC_DisableDirectAccess();
		Align_Index = 0;
		Status_Start = ALIGN_ON_GOING;
		AlignState = WAIT_FOR_C;
		return(FALSE);
		break;
	
	case ALIGN_ON_GOING:
		AlignRotor();		// Ramp up the alignment current 
		return(FALSE);
    		break;

	case INIT_SWITCHED_MODE:
		#if (DRIVING_MODE == VOLTAGE_MODE)
	    	MCPUL = ramp_MCPUL;  // Set duty cycle for ramp up
    		MCPUH = ramp_MCPUH;
       		#else
   		MCPVL = ramp_MCPVL;   // set current for ramp up
   		MCPVH = ramp_MCPVH;
		#endif

		MTC_DisableClock();  // force 1st ramp ratio value update into MPRSR 
		MPRSR = (u8)((MPRSR & 0xf0) + RAMP[RampIndex].Ratio);	// MTIM timer prescaler set-up

		MTC_EnableClock();

    StepIndex = 1;			// Point to next step configuration
		RampIndex = 1;			// Alignment completed, prepare the second step of the ramp
		UpdateRatio();			// Prepare updation of MTIM prescaler on next C event
	
		Set_Motor_Direction();
    
    if (Motor_Direction == CW)
    {
    MPHST = PHASE_CONFIG[StepIndex];	// Preload active phase on next C event for CW Direction
    MCRB = (u8)((ZC_DEMAG_PWM[StepIndex] & Reset_H_SDM_MASK) + SDM_MSK);
                     // Expected Bemf edge, Demag type and PWM behaviour
                     // Reset both HDM and SDM flag 
                     // & Force SDM bit for next step	 for CW Direction
    }
    else
    {
    MPHST = PHASE_CONFIG_CCW[StepIndex];	// Preload active phase on next C event for CCW Direction
    MCRB = (u8)((ZC_DEMAG_PWM_CCW[StepIndex]& Reset_H_SDM_MASK) + SDM_MSK);
                      // Expected Bemf edge, Demag type and PWM behaviour
                      // Reset both HDM and SDM flag 
                      // & Force SDM bit for next step for CCW Direction
    }
    
		StepIndex++;	// This is to load the next step config
						      // when entering the coming C interrupt
					
	
		MIMR = ((u8)CIM_MSK + DIM_MSK);	// enable C & D interrupt to start commutation mechanism and motor

		//Status_Start = START_FINISHED;
		Status_Start = INIT_START;
		BrakeState = BRAKE_START;
		return(TRUE);
		break;
	}
return(FALSE);
}


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_StopMotor

Description:	This function disables all motor control related interrupts 
				and switch off all transistors. This let the windings in 
				floating state once they are completely demagnetized.
Input/Output:	None
-----------------------------------------------------------------------------*/
void MTC_StopMotor(void)
{
	MIMR = 0;                   	// Mask all MTC related interrupts
	MCRA &= (u8)(~SWA_MSK);	// Disable Autoswitched in previously enabled

	MCPUL = 0;			// Set PWM U to 0%
	MCPUH = 0;		
	MCPVL = 0;			// Set PWM V to 0%
	MCPVH = 0;		
	
	MISR =0xFF;  // force update of 12 bits Timer to 0% 	
	
	MTC_EnableDirectAccess();
	MPHST = 0;			// Switch off all drivers

	MTC_DisableClock();
	MotorStatus = 0;
	Status_Start = INIT_START;
	BrakeState = BRAKE_START;
}


#if defined ( __HIWARE__)	/* test for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	/* additional registers will be saved */
#elif defined ( __CSMC__ )	/* test for Cosmic Compiler */
@interrupt			/* Cosmic interrupt handling */
#endif
/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_U_CL_SO_IT

Description:	This is the PWM Update interrupt service routine.
Comments:	U event taken into account on No action done for BLDC motor drive, 
            but flags are reset in case the routine is entered.
-----------------------------------------------------------------------------*/
void MTC_U_CL_SO_IT(void)
{  
	if (MISR & PUI_MSK)
		{
		MISR = ((u8)~(PUI_MSK));
		}
	if (MISR & CLI_MSK) 
		{
		MIMR &= ((u8)~(CLIM_MSK));  // disable CL int
		MISR = ((u8)~(CLI_MSK));
	        SetBit(Power_Motor_Status,OverCurrent);
		timer_CL_10ms = 30;  // 300 ms temporisation 
		}
	
	SET_MTC_PAGE(1);
	if (MCONF & SOI_MSK) 
		{
		MCONF &= ((u8)~(SOI_MSK)); 
		}
	SET_MTC_PAGE(0);
}


#if defined ( __HIWARE__)	/* test for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	/* additional registers will be saved */
#elif defined ( __CSMC__ )	/* test for Cosmic Compiler */
@nosvf @interrupt		/* Cosmic interrupt handling */
#endif
/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_C_D_IT

Description:	This is the Commutation and Demagnetisation interrupt service 
		routine. 
Comments:	None
-----------------------------------------------------------------------------*/
void MTC_C_D_IT(void)  
{
	if(MISR & CI_MSK)
    	{
		MISR = ((u8)~(CI_MSK));
		RPICounter = 0;		//reset flag on ratio increment for soft demagnetisation
	
		if (Motor_Direction == CW)
    {
    MPHST = PHASE_CONFIG[StepIndex];	// Preload active phase on next C event for CW Direction
    MCRB = ZC_DEMAG_PWM[StepIndex]; 	// for CW Direction
    }
    else
    {
    MPHST = PHASE_CONFIG_CCW[StepIndex];	// Preload active phase on next C event for CCW Direction
    MCRB = ZC_DEMAG_PWM_CCW[StepIndex];   //for CCW Direction
    }
  
    #if (DRIVING_MODE == VOLTAGE_MODE)
	    if (RampIndex > 2)
	    	{
	        if (Commut_Dem_MCPUL != 0 || Commut_Dem_MCPUH != 0)   // force duty cycle during demag time?
	            {
	            Previous_Duty_L = MCPUL;
	            Previous_Duty_H = MCPUH;
	            MCPUL = Commut_Dem_MCPUL;    // update duty cycle
	            MCPUH = Commut_Dem_MCPUH;
        	    }
    	   	 } 
    #endif
     
	#if (DEMAG_TYPE == SW)
        if (MotorStatus == AUTO_SWITCH)
			{
			if ((u16)(DEMAG_TIME[(MPRSR & 0x0f)]+ MCOMP) > U8_MAX)
				{
				RPICounter ++;
				SoftDemagTime = (u16)((DEMAG_TIME[(MPRSR & 0x0f)] + MCOMP - 0xFF)/2+ 0x80);
				while ((u16)(SoftDemagTime ) > U8_MAX)
					{
		   			RPICounter ++;
		    			SoftDemagTime = (((u16)(SoftDemagTime - 0xFF)/2)+ 0x80);
			    		}
				}
			else MDREG = (u8)(DEMAG_TIME[(MPRSR & 0x0f)]+ MCOMP);    
			}			
	else MDREG = DEMAG_TIME[(MPRSR & 0x0f)];
	// load demagnetization time according to the soft demag time table 
	#endif

	SET_MTC_PAGE(1);
  if (Motor_Direction == CW)
  {
	if (ValBit(ZC_DEMAG_PWM[StepIndex],1)) SetBit(MPOL,REO);//read BEMF when PWM is on Low channel
	else ClrBit(MPOL,REO);    // read BEMF when PWM is on High channel
  }
  else
  {
  if (ValBit(ZC_DEMAG_PWM_CCW[StepIndex],1)) SetBit(MPOL,REO);//read BEMF when PWM is on Low channel
	else ClrBit(MPOL,REO);    // read BEMF when PWM is on High channel
  }
	SET_MTC_PAGE(0);

		StepIndex++;
		if (StepIndex > 5)	StepIndex = 0;
		
    

		// Autoswitched mode Init
   	if (MotorStatus & FIRST_AUTO_SWITCH)// ____________________________________
		    {
			u8 i;
			
		        MotorStatus &= ((u8)~FIRST_AUTO_SWITCH);
			MotorStatus |= AUTO_SWITCH;
			MWGHT = AUTO_DELAY;	// Set delay for number of cycle set by AUTO_DELAY_STEP
//			MCRA |= DCB;		// select MZREG to compute delay
			delay_counter = 0;	// Initialise the counter to manage delay
			
            for (i=0;i<=STEP_Z_BUFFER_SIZE-1;i++)
                {
                Step_Z[i].Ratio = RAMP[RampIndex].Ratio;    // init buffer for first time computation of the motor frequency
                Step_Z[i].StepTime = RAMP[RampIndex].StepTime; // -> avoid wrong PI output & stalled motor status in closed loop operation
                }
		
			PORTS_GreenLedOn(); // Green LED on if autoswitch mode detected 
		
		    } // end of First auto_switch commutation
	
	if (MotorStatus & AUTO_SWITCH)  // Autoswitched mode
		{	
        #if (DEMAG_TYPE == HSW)
           	if ((MCRB & SDM_MSK))   // soft demag on going
   			    {
		       	
		    	if (ValBit(Flag_it,RM_EVT)) // MTIM underflow since last Dh event 
		    		SoftDemagTime = (u16)(2*SoftDemagTime);
		    		 			
		    	else SoftDemagTime >>= (u8)(RP_counter);  //right shift base on overflow detected
       		    if ((u16)(SoftDemagTime + MCOMP) > U8_MAX)
		    	   {
			    		RPICounter ++;
			    		SoftDemagTime = (u16)((SoftDemagTime + MCOMP - 0xFF)/2+ 0x80);
			    		while ((u16)(SoftDemagTime ) > U8_MAX)
			    	   		{
				 	    	RPICounter ++;
		    		 		SoftDemagTime = (((u16)(SoftDemagTime - 0xFF)/2)+ 0x80);
		    		   		}
		    	   }    
		      else MDREG = (u8)(SoftDemagTime + MCOMP);
		        }
        #elif (DEMAG_TYPE == HW)     // do nothing! -> hard demagnetization managed by MTC peripheral
        #elif (DEMAG_TYPE == SW)     // do nothing! -> soft demagnetization managed in C IT
        #else
           #error"Wrong Demagnetization Type !"	
        #endif
        
        if ( delay_counter <= MAX_DELAY_COUNTER)
		    {
		    delay_counter ++;
		    if (delay_counter == AUTO_DELAY_STEP)
		    MWGHT = MEDIUM_DELAY;	// Set delay for number of cycle from AUTO_DELAY_STEP
		                                //  to  MAX_DELAY_COUNTER 
		    }                            
		else
		    {
			if	(MCRB & CPB_MSK)    MWGHT = Rising_bemf;
			else    MWGHT = Falling_bemf;
		    }		
					                             
		} // End of Autoswitched mode
	       
	else	// Acceleration phase processing
	   	{
		if (MotorStatus & LAST_FORCED_SWITCH)// End of switched mode without Z 
		    {
			//Calculate Demag Time since MCOMP is new
#if (DEMAG_TYPE == HSW)
			MDREG = (u8)(MCOMP / SWITCHED_SW_DEMAG); // Last Demag Time for switched mode
#endif
			MCOMP = U8_MAX;	// Wait for Z event (no possibility to have C event)
        					// Prepare autoswitched mode transition
			MotorStatus &= ((u8)~LAST_FORCED_SWITCH);
			MotorStatus |= FIRST_AUTO_SWITCH;
			MWGHT = TRANSITION_DELAY;      //set to 255/256 on first autoswitch step			
			MCRA |= SWA_MSK;	// Enable Autoswitched mode and compute delay on MZPRV
			MIMR = ((u8)(RIM_MSK + EIM_MSK + CIM_MSK + ZIM_MSK + DIM_MSK + CLIM_MSK));//...relevant int
		    }	// End of last synchronous mode commutation 

		else	// Forced switched mode without Z detection
	    	{   	
			MCOMP = RAMP[RampIndex].StepTime;
#if (DEMAG_TYPE == HSW)
			MDREG = (u8)(MCOMP / SWITCHED_SW_DEMAG);
#endif			
			RampIndex++;		// Next step on the acceleration ramp
			UpdateRatio();		// Prepare updation of MTIM prescaler on next C event
		
					
			if (RampIndex >= (RAMP_SIZE-1))	// If Ramp is finished without success
		    		{
				MTC_DisableOutputs();
				MTC_DisableClock();
				MotorStatus |= START_UP_FAILED;
			   	SetBit(Power_Motor_Status,StartUpFailed);  
                		Flash_Red_Led();
		    		}
    			if (RampIndex == bemf_blank)	// Z event masking window is finished
				{
				SetBit(MIMR,ZIM);	// Enable Z event interrupts
				}
		
			// Verify if Bemf are present and consecutive (same number of Z and C events)
			if ((BemfCounter != 0)		// If there was a Z event...
		 		&& (BemfCounter == (u8)(CeventCounter+1)))		// ... since last C
		    	{
				CeventCounter++;
			    }
			else	// Re-start Bemf counting if bemf are not consecutive
		    	{
				CeventCounter = 0;
				BemfCounter = 0;
	    		}
		    }
	}	// End of acceleration phase processing  
	
	}  // end of C IT


if(MISR & DI_MSK)	 
	{
	u8 temp_D;
	
	MISR = ((u8)~(DI_MSK));
	
  #if (DRIVING_MODE == VOLTAGE_MODE)
    if (RampIndex > 2)
	  {
	  if (Commut_Dem_MCPUL != 0 || Commut_Dem_MCPUH != 0)   // force duty cycle during demag time?
            {
            MCPUL = Previous_Duty_L;
            MCPUH = Previous_Duty_H;
            }
          }
  #endif
  
    #if (DEMAG_TYPE == HSW)
    if (!(MCRB & SDM_MSK))  // Hard/soft demagnetization
                            // hard demag done -> compute next soft demag time	
        {
        if (MotorStatus == AUTO_SWITCH)   temp_D = (u8)(MDREG - MCOMP);   // Demag.time = MDREG - MCOMP
        else temp_D = MDREG;  // synchronous mode

   	  SoftDemagTime = (temp_D >> 2); // div/4
        SoftDemagTime += temp_D;  // next MDREG value = 1.25*(hard demag.time)
        RP_counter=0;	//reset counter of RP event  coming between Dhard and next C
       
       ClrBit(Flag_it,RM_EVT); 
        }
    #endif
	}    // end of demag. interrupt
}


#if defined ( __HIWARE__)	/* test for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	/* additional registers will be saved */
#elif defined ( __CSMC__ )			/* test for Cosmic Compiler */
@interrupt			/* Cosmic interrupt handling */
#endif
/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_R_Z_IT

Description:	This is Ratio and zero Crossing interrupt service routine.
		Z event -> Autoswitched mode: save step time & ratio values for 
                speed measurement
		Z event -> Switched mode: count number of bemf and enable 
		autoswitched mode once ramp is finished
		RMI event -> no action
		RPI event -> Count number of RPI and update MDREG on last one
Comments:	The MISR flags are cleared as soon as possible to be able to
		serve a new interrupt occuring before the end of the actual one
-----------------------------------------------------------------------------*/
void MTC_R_Z_IT(void)
{  
// ZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZ Z event ZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZ
if (MISR & ZI_MSK)
    {
	MISR = ((u8)~ZI_MSK);                 // Reset Z mask

	if (MotorStatus & AUTO_SWITCH)
	    {
	    if (MZREG>=0x5)   // check MZREG value to keep good accuracy
            {
            Step_Z[Step_Z_Counter].Ratio = (u8)(MPRSR & 0x0f);   // update Z steptime buffer for PI regulation
            Step_Z[Step_Z_Counter].StepTime = MZREG;             // & motor frequency computation
           
            if (Step_Z_Counter >= STEP_Z_BUFFER_SIZE-1) Step_Z_Counter = 0;
            else Step_Z_Counter++;
      	    }	
      	    
      	if (Step_counter != 0)	Step_counter --;       
     	}
  	else    // Forced switched mode/accel phase
	    {
		BemfCounter++;
		if (BemfCounter == bemf_valid)	//Transition switched->autoswitched
		{
			if (MZREG <= (U8_MAX/2))	// No overflow when multiply
			{
				// Center last zerocrossing in the middle of two commutations
				MCOMP = (u8)(MZREG * 2);
			}
			else	MCOMP = U8_MAX-1;		// Not correct: step not centered
			MotorStatus |= LAST_FORCED_SWITCH;	// Last switched mode step
		}	// End of switched->autoswitched transition 
 	}
}	// End of Z event processing


// RRRRRRRRRRRRRRRRRRRRRRRRRRRRR RPI event RRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRR
if (MISR & RPI_MSK)
    {
    MISR = ((u8)~(RMI_MSK + RPI_MSK));
    #if (DEMAG_TYPE == HSW)
    		RP_counter ++;	
        	if (RPICounter !=0)
        		{
        		RPICounter --;        
         	    
    	   		if (RPICounter == 0)   	MDREG = (u8) SoftDemagTime;
	        	
            	}
    #elif (DEMAG_TYPE == SW)
	
	   	if (RPICounter !=0)
			{
   			RPICounter --;        
   			if (RPICounter == 0)   	MDREG = (u8) SoftDemagTime;
		   	}
    #endif

	}

// RRRRRRRRRRRRRRRRRRRRRRRRRRRRR RMI event RRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRR
if (MISR & RMI_MSK)
    {
    MISR = ((u8)~(RMI_MSK + RPI_MSK));
    SetBit(Flag_it,RM_EVT);
    }
}        



#if defined ( __HIWARE__)		/* test for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	/* additional registers will be saved */
#elif defined ( __CSMC__ )			/* test for Cosmic Compiler */
@interrupt			/* Cosmic interrupt handling */
#endif
/*-----------------------------------------------------------------------------
ROUTINE Name : MCES_SE_IT

Description:	Emergency Stop and Speed Error interrupt service routine.
        	As this routine is only entered in case of HW problem, the 
	        motor is stopped and trace of this interrupt is keeped with a 
		MotorStatus flag.

Comments:	The Speed Error Flag is only used for induction motor drive.
-----------------------------------------------------------------------------*/
void MCES_SE_IT(void)
{
	MISR = ((u8)~EI_MSK);	// Clr Emergency Stop Flag
	
	MTC_StopMotor();
	MotorStatus = EMERGENCY_STOP; 
}  


/*-----------------------------------------------------------------------------
ROUTINE Name : Set_Duty

Description:	This is the update routine of the PWM duty cycle -> feed MCPUH/L
		(voltage mode) or MCPVH/L (current mode) registers. 
		
Comments:	None
-----------------------------------------------------------------------------*/
void Set_Duty(u16 duty)
{
#if (DRIVING_MODE == VOLTAGE_MODE)
	ToCMPxL(MCPUL,(u16)duty);
	ToCMPxH(MCPUH,(u16)duty);
#elif (DRIVING_MODE == CURRENT_MODE)
	ToCMPxL(MCPVL,(u16)duty);
	ToCMPxH(MCPVH,(u16)duty);
#endif
}      
 

/*-----------------------------------------------------------------------------
ROUTINE Name : Set_Target_Electrical_Frequency

Description:	This is the update routine of the PWM duty cycle/PWM current
				limitation in closed loop mode -> feed MCPUH/L in voltage mode
				or MCPVH/L in current mode. 
				The target electrical frequency (0.1 Hz resolution, eg 1000 for 100Hz)
				is input in the the regulation routine;	result (10 bits value)
				is then used to calculate the new duty cycle.
		
Comments:	see mtc.h to for rising/falling bemf,Ki/Kp coefficient calculation.
-----------------------------------------------------------------------------*/
void Set_Target_Electrical_Frequency(u16 target_freq)
{
u32 temp;

  	if (Freq_Motor<=Freq_Min)       // Freq_Motor<=Freq_Min ??
    	{
		Rising_bemf = Rising_Fmin ;
		Falling_bemf = Falling_Fmin;
		Ki = Ki_Fmin;      // integral term
		Kp = Kp_Fmin;      // proportionnal term
    	}
    	else if (Freq_Motor<=F_1)       // Freq_Min<=Freq_Motor<=F_1 ??
		{
		Rising_bemf = (u8)(Rising_Fmin + (s32)(alpha_Rising_1*(Freq_Motor-Freq_Min)/1024)); 
		Falling_bemf = (u8)(Falling_Fmin + (s32)(alpha_Falling_1*(Freq_Motor-Freq_Min)/1024));
		Ki = (u8)(Ki_Fmin + (s32)(alpha_Ki_1*(Freq_Motor-Freq_Min)/1024));
		Kp = (u8)(Kp_Fmin + (s32)(alpha_Kp_1*(Freq_Motor-Freq_Min)/1024));
		}
	else if (Freq_Motor<=F_2)  	// F_1<Freq_Motor<=F_2 ??
		{
		Rising_bemf = (u8)(Rising_F_1 + (s16)(alpha_Rising_2*(Freq_Motor-F_1)/1024)); 
		Falling_bemf = (u8)(Falling_F_1 + (s16)(alpha_Falling_2*(Freq_Motor-F_1)/1024));
		Ki = (u8)(Ki_F_1 + (s32)(alpha_Ki_2*(Freq_Motor-F_1)/1024));
		Kp = (u8)(Kp_F_1 + (s32)(alpha_Kp_2*(Freq_Motor-F_1)/1024));
		}
	 else if (Freq_Motor<=Freq_Max)  // F_2<Freq_Motor<=Freq_Max ??
		{                       
		Rising_bemf = (u8)(Rising_F_2 + (u16)(alpha_Rising_3*(Freq_Motor-F_2)/1024)); 
		Falling_bemf = (u8)(Falling_F_2 + (u16)(alpha_Falling_3*(Freq_Motor-F_2)/1024));
		Ki = (u8)(Ki_F_2 + (u16)(alpha_Ki_3*(Freq_Motor-F_2)/1024));
		Kp = (u8)(Kp_F_2 + (u16)(alpha_Kp_3*(Freq_Motor-F_2)/1024));
		}
	 else                            // Freq_Max>Freq_Motor ??
	  	{
		Rising_bemf = Rising_Fmax ;
		Falling_bemf = Falling_Fmax;
		Ki = Ki_Fmax;      // integral term
		Kp = Kp_Fmax;      // proportionnal term
		}

	temp = PWM_FREQUENCY;              // temp =  (PWM_FREQUENCY * PIoutput)/1024
	temp *= regul_PI((u16)(target_freq));
	temp >>= 10;
	if (temp > Max_Duty) temp = Max_Duty;	

#if (DRIVING_MODE == VOLTAGE_MODE)
	ToCMPxL(MCPUL,(u16)temp);
	ToCMPxH(MCPUH,(u16)temp);
#elif (DRIVING_MODE == CURRENT_MODE)
	ToCMPxL(MCPVL,(u16)temp);
	ToCMPxH(MCPVH,(u16)temp);
#endif
}      


/*-----------------------------------------------------------------------------
ROUTINE Name : active_brake

Description:	braking routine 
Input/Output:	duty cycle, duration in ms (u16/u16)
Comments:	Return from this routine immediately if duty 
			is equal to 0x00. 
-----------------------------------------------------------------------------*/
BOOL active_brake(u16 duty, u16 time)   // duration = time * 1ms  eg: 1000 = 1sec
{
if (duty == 0) 
	{
	MCRA &= (u8)(~SWA_MSK);	// Disable Autoswitched mode and...
	MIMR = 0;
	return(TRUE);
	}
else
	{
	switch(BrakeState)
		{
    u8 temp_mphst;
		case BRAKE_START:
		default:
			MCRA &= (u8)(~SWA_MSK);	// Disable Autoswitched mode and...
			MIMR = 0;               // Disable all interruption of motor peripheral
      temp_mphst = MPHST;     // catch the actual step configuration
			MTC_EnableDirectAccess();
      MCRB &= (u8)(OS0_MSK);     // force PWM on the low side switch
			MPHST = temp_mphst;	      // Alignment step pattern

			#if (DRIVING_MODE == VOLTAGE_MODE)
			ToCMPxL(MCPUL,(u16)(duty));
			ToCMPxH(MCPUH,(u16)(duty));
			#else
			ToCMPxL(MCPVL,(u16)(duty));
			ToCMPxH(MCPVH,(u16)(duty));
			#endif
			
			Timer_1ms = time;
			BrakeState = BRAKE_ONGOING;
			return(FALSE);
			break;
		case BRAKE_ONGOING:
			if (Timer_1ms == 0)	BrakeState = BRAKE_FINISHED;
			return(FALSE);
			break;
		case BRAKE_FINISHED:
			return(TRUE);
			break;
		}
	}			  
return(FALSE);
}


/*-----------------------------------------------------------------------------
ROUTINE Name : Chk_Motor_Stalled

Description:	 
		
Comments:	None
-----------------------------------------------------------------------------*/
void Chk_Motor_Stalled(void)
{
if ((MPRSR & 0x0F) == RATIO_MAX) 
    {
    MotorStatus = MOTOR_STALLED;
    SetBit(Power_Motor_Status,MotorStalled);  
    }
}

/*-----------------------------------------------------------------------------
ROUTINE Names : Set_Motor_Direction

Description : This routines set the Motor_Direction based on Rotor_Direction state
   if Rotor_Direction = 0 we have selected CW Direction
   if Rotor_Direction = 255 we have selected CCW Direction 
   for any other value of Rotor_Direction we toggle the motor Direction
-----------------------------------------------------------------------------*/
void Set_Motor_Direction( void )
{
    if (Rotor_Direction == CW) Motor_Direction = CW;
      
    else if(Rotor_Direction == CCW) Motor_Direction = CCW;  
      
    else if(Motor_Direction == CCW) Motor_Direction = CW;   // toggle motor direction to CW
          
    else  Motor_Direction = CCW;     // toggle motor direction to CCW
}

/*** (c) 2005  STMicroelectronics ****************** END OF FILE ***/
