/**************** (c) 2005 STMicroelectronics *********************************

PROJECT  : ST7MC demokit
COMPILER : ST7 METROWERKS C (HIWARE) / COSMIC

MODULE  :  Mtc.h
LIBRARY VERSION  :  2.0

CREATION DATE :    08.2003
AUTHOR :      Florent COSTE	/  Microcontroller Application Lab  / ST Hong Kong

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

DESCRIPTION :  MTC peripheral control routine prototypes

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

 ******************************************************************************
 THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. ST MICROELECTRONICS
 SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
 WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
 ******************************************************************************
              
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

******************************************************************************/

#ifndef MTC_AC_H
#define MTC_AC_H  
/*****************************************************************************/

/* Public Constants -------------------------------------------------------- */
// MotorStatus bitfield
#define	AUTO_SWITCH			((u8)0x01)
#define	FIRST_AUTO_SWITCH		((u8)0x02)
#define LAST_FORCED_SWITCH		((u8)0x04)
#define MOTOR_STALLED			((u8)0x10)
#define HARD_FAILURE            	((u8)0x20)
#define START_UP_FAILED			((u8)0x40)
#define EMERGENCY_STOP			((u8)0x80)

#define FAULT_MSK			((u8)0xf0)    // START_UP_FAILED or MOTOR_STALLED or HARD_FAILURE or EMERGENCY_STOP

#define PWM_FREQUENCY		((u16)(mem_MCPOH<<8) + mem_MCPOL)
#define RATIO_MAX		((u8)15)		/* Max MTIM prescaler ratio */
#define STARTING_UP	

#define T1	MCO0
#define T2	MCO1
#define T3	MCO2
#define T4	MCO3
#define T5	MCO4
#define T6	MCO5


#define SAMPLING_TIME	50	// 50ms

#define CW   ((u8)0)      // Value to run Motor in Clock Wise (CW)
#define CCW   ((u8)255)     // Value to run Motor in Counter Clock Wise (CCW)
 
/******* Rising/Falling Bemf, Ki, Kp coefficient calculation **********************
/*******           	Closed loop operation		     **********************
		

              /|\               /
               |               /
  	       |	      /
               |             /
               |   _________/  
               |  /
               | /
	       |/_________________________\ 
	   Fmin   F_1      F_2  Fmax      /
				
		                                                                

We assume a linear variation of Falling/Rising, Bemf, Ki, Kp coefficients following
the motor frequency. We set 2 intermediate frequencies (see definition right after)
and 4 terms (Rising,Falling,Ki,Kp) associated with Fmin, F_1, F_2, Fmax 
(total: 4+4+4+4 terms); then we calculate the linear coefficient that will be use
to compute each term.

Example: 

Fmin = 500	50 Hz 	(reminder -> electrical frequency with 0.1 Hz resolution!)
Rising_min = 30	Falling_min = 40	Ki_min = 20		Kp_min = 40

F_1 = 2000	200 Hz 	
Rising_F_1 = 150	Falling_F_1 = 10	Ki_1 = 80	Kp_1 = 140

then:
alpha_Rising_1 = (Rising_F_1-Rising_Fmin)/(F_1-Freq_Min) = 120/1500 = 0.08
alpha_Falling_1 = (Falling_F_1-Falling_Fmin)/(F_1-Freq_Min) = -30/1500 = -0.02
alpha_Ki_1 = (Ki_1-Ki_Fmin)/(F_1-Fmin) = 60/1500 = 0.04
alpha_Kp_1 = (Kp_1-Kp_Fmin)/(F_1-Fmin) = 100/1500 = 0.066667

** Result **
From Freq_Min to F_1, Rising, Falling, Ki, Kp will then obey to:
Rising = Rising_Fmin + alpha_Rising_1*(Freq_motor-Freq_Min)
Falling = Falling_Fmin + alpha_Falling_1*(Freq_motor-Freq_Min)
Ki = Ki_Fmin + alpha_Ki_1*(Freq_motor-Freq_Min)
Kp = Kp_Fmin + alpha_Kp_1*(Freq_motor-Freq_Min)

		                                                                
*********************************************************************************/
// See 'Mtc_Settings_Sensorless.h' for Freq_Min & Freq_Max values
//Fmin
#define Rising_Fmin      20	// Frequency min coefficient settings
#define Falling_Fmin     30
#define Ki_Fmin          10	
#define Kp_Fmin          30

//F_1
#define F_1		 1000    // 100 Hz  
#define Rising_F_1       50	// Intermediate frequency 1 coefficient settings
#define Falling_F_1      40
#define Ki_F_1           20
#define Kp_F_1           10

//F_2
#define F_2		 2000    // 200 Hz
#define Rising_F_2       30    // Intermediate frequency 2 coefficient settings
#define Falling_F_2      10
#define Ki_F_2           50
#define Kp_F_2           40
  
//Fmax  
#define Rising_Fmax      10    // Frequency max coefficient settings
#define Falling_Fmax     16      
#define Ki_Fmax          13
#define Kp_Fmax          18
      
                                                                             
/********************************************************************************/      
/* linear coefficients */                                                                             
#define alpha_Rising_1  	(s32)( ((s32)((s16)Rising_F_1-(s16)Rising_Fmin)*1024) / (s32)(F_1-Freq_Min)  )   // *1024 to keep good accuracy
#define alpha_Falling_1		(s32)( ((s32)((s16)Falling_F_1-(s16)Falling_Fmin)*1024) / (s32)(F_1-Freq_Min) )
#define alpha_Ki_1		(s32)( ((s32)((s16)Ki_F_1-(s16)Ki_Fmin)*1024) / (s32)(F_1-Freq_Min) )
#define alpha_Kp_1		(s32)( ((s32)((s16)Kp_F_1-(s16)Kp_Fmin)*1024) / (s32)(F_1-Freq_Min) )

#define alpha_Rising_2  	(s32)( ((s32)((s16)Rising_F_2-(s16)Rising_F_1)*1024) / (s32)(F_2-F_1) )
#define alpha_Falling_2		(s32)( ((s32)((s16)Falling_F_2-(s16)Falling_F_1)*1024) / (s32)(F_2-F_1) )
#define alpha_Ki_2		(s32)( ((s32)((s16)Ki_F_2-(s16)Ki_F_1)*1024) / (s32)(F_2-F_1) )
#define alpha_Kp_2		(s32)( ((s32)((s16)Kp_F_2-(s16)Kp_F_1)*1024) / (s32)(F_2-F_1) )

#define alpha_Rising_3  	(s32)( ((s32)((s16)Rising_Fmax-(s16)Rising_F_2)*1024) / (s32)(Freq_Max-F_2) )
#define alpha_Falling_3		(s32)( ((s32)((s16)Falling_Fmax-(s16)Falling_F_2)*1024) / (s32)(Freq_Max-F_2) )
#define alpha_Ki_3		(s32)( ((s32)((s16)Ki_Fmax-(s16)Ki_F_2)*1024) / (s32)(Freq_Max-F_2) )
#define alpha_Kp_3		(s32)( ((s32)((s16)Kp_Fmax-(s16)Kp_F_2)*1024) / (s32)(Freq_Max-F_2) )



/* Public Variables -------------------------------------------------------- */   

typedef struct {
		u8 Ratio;
		u8 StepTime;
		} Step_s;


extern u8 Flag_MTC;	// MTC Flag
#define SAMP_EVT	        1     

extern u8 MotorCmd;
#define STOP_CMD     0x00
#define START_CMD    0xff       

extern u8 Power_Motor_Status;
#define OverHeat         0
#define OverCurrent      1
#define OverVoltage      2
#define StartUpFailed    3
#define MotorStalled     5
                                 
#define Heat_Voltage_Failure 0x05
#define Hard_Failure_MSK     0x07

extern u16 DutyCycle;
extern u16 CurrentReference;
extern u16 Target_Freq;
extern u16 Freq_Motor;
extern u8 Ki;
extern u8 Kp;
extern u8 Rising_bemf;
extern u8 Falling_bemf;

/* Public  macros ---------------------------------------------------------- */

/* Public Functions prototypes --------------------------------------------- */
void MTC_InitPeripheral(void);
BOOL MTC_StartMotor(void);
void MTC_StopMotor(void);
void Set_Duty(u16);
void Set_Target_Electrical_Frequency(u16);
BOOL active_brake(u16,u16);
u8 GetMotorStatus(void);
void SetMotorStatus(u8);
void Chk_Motor_Stalled(void);
void Set_Motor_Direction(void);
	
#endif
/*** (c) 2005  STMicroelectronics ****************** END OF FILE ***/