/**************** (c) 2005  STMicroelectronics **********************
     
PROJECT  : ST7MC demokit
COMPILER : ST7 METROWERKS C (HIWARE) / COSMIC

MODULE  :  adc.c
LIBRARY VERSION :  2.0

CREATION DATE :    08.2003
AUTHOR :      F.COSTE	/  Microcontroller Application Lab  / ST Hong Kong

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

DESCRIPTION :   ADC routines              

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

 ******************************************************************************
 THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. ST MICROELECTRONICS
 SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
 WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
 ******************************************************************************

******************************************************************************/

#include "lib.h"
#include "ST7MC_hr.h"
#include "adc.h" 
#include "mtc.h"

#define NTC_THRESHOLD    110       // temperature reference
#define NTC_HYSTERIS     8         // temperature hysteresis -> 0 <= NTC_HYSTERIS < NTC_THRESHOLD
#define HVBUS_THRESHOLD  1000      // Bus Voltage reference
#define HVBUS_HYSTERIS   19        // Bus Voltage hysteresis -> 0 <= HVBUS_HYSTERIS < HVBUS_THRESHOLD


/*-----------------------------------------------------------------------------
ROUTINE Name : ADC_Get_10bits

Description:	Return the full ADC resolution value (10 bits).
Input/Output:	u8/u16 (channel number/result of conversion)
Comments: 		None
-----------------------------------------------------------------------------*/
u16 ADC_Get_10bits(u8 Channel)		
{
u16 result;
u8 i;

result = 0;	

ADCDRH;	// clear EOC bit in case of...
ADCCSR = Channel;		// 4 Mhz sampling, ADC on

while (!ValBit(ADCCSR,EOC));	// wait till end of conversion
ADCDRH;                       // and ignore 1st result

for (i=0;i<=7;i++)      // take 8 samples
	{
	while (!ValBit(ADCCSR,EOC));	// wait till end of conversion
	result += ADCDRL;
	result += (u16)(ADCDRH << 2);	// clear EOC bit
	}

ClrBit(ADCCSR,ADON);	// shutdown ADC peripheral

result = result >> 3;	// div/8 -> smooth result
return (result);
}
 


/*-----------------------------------------------------------------------------
ROUTINE Name : ADC_Get_8bits

Description:	Return 8 bits resolution ADC value only.
Input/Output:	u8/u8 (channel number/result of conversion)
Comments: 		None
-----------------------------------------------------------------------------*/
u8 ADC_Get_8bits(u8 Channel)	
{
u16 result;
u8 i;

result = 0;	

ADCDRH;	// clear EOC bit in case of...
ADCCSR = Channel;		// 4 Mhz sampling, ADC on

while (!ValBit(ADCCSR,EOC));	// wait till end of conversion
ADCDRH;                       // and ignore 1st result

for (i=0;i<=7;i++)      // take 8 samples
	{
	while (!ValBit(ADCCSR,EOC));	// wait till end of conversion
	result += ADCDRH;	// clear EOC bit
	}

ClrBit(ADCCSR,ADON);	// shutdown ADC peripheral

result = result >> 3;	// div/8 -> smooth result
return ((u8)result);
}
         

/*-----------------------------------------------------------------------------
ROUTINE Name :  Get_RV1/Get_RV2/Get_RV3

Description:	return potentiometers value (RV1/RV2/RV3).
Input/Output:	None/u8
Comments: 		None
-----------------------------------------------------------------------------*/
u8 Get_RV1(void)
{
return(ADC_Get_8bits(CONVERT_AIN13));
}

u8 Get_RV2(void)
{
return(ADC_Get_8bits(CONVERT_AIN11));
}        

u8 Get_RV3(void)
{
return(ADC_Get_8bits(CONVERT_AIN7));
}


/*-----------------------------------------------------------------------------
ROUTINE Name :  Chk_Overtemp

Description:	Return TRUE if the voltage on the thermal resistor connected 
                to channel AIN0 has reached the threshold level or if the voltage 
                has not yet reached back the threshold level minus the hysteresis 
                value after an overheat detection.
Input/Output:	none/boolean
Comments: 		None
-----------------------------------------------------------------------------*/
BOOL Chk_Overtemp(void)
{
if (ADC_Get_8bits(CONVERT_AIN0) >= NTC_THRESHOLD)    return(TRUE);
else if ( (ADC_Get_8bits(CONVERT_AIN0) >= (NTC_THRESHOLD - NTC_HYSTERIS)) // wait for the temperature
            && ValBit(Power_Motor_Status,OverHeat) ) return(TRUE);        // to go back to normal operation
else return(FALSE);
}


/*-----------------------------------------------------------------------------
ROUTINE Name :  Chk_Overvoltage

Description:	Return TRUE if the voltage of the HVBUS connected to channel AIN1 
                has reached the threshold level.
Input/Output:	none/boolean
Comments: 		None
-----------------------------------------------------------------------------*/
BOOL Chk_Overvoltage(void)
{
if (ADC_Get_10bits(CONVERT_AIN1) >= HVBUS_THRESHOLD)    return(TRUE);
else if ( (ADC_Get_10bits(CONVERT_AIN1) >= (HVBUS_THRESHOLD - HVBUS_HYSTERIS)) // wait for the voltage
            && ValBit(Power_Motor_Status,OverVoltage) ) return(TRUE);         // to go back to normal operation
else return(FALSE);
}


/*** (c) 2005  STMicroelectronics ****************** END OF FILE ***/
