/**************** (c) 2005  STMicroelectronics **********************
     
PROJECT  : ST7MC demokit
COMPILER : ST7 METROWERKS C (HIWARE) / COSMIC

MODULE  :  mtc.c
LIBRARY VERSION :  2.0

CREATION DATE :    08.2003
AUTHOR :      Application Lab  / ST Rousset / ST Hong Kong

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

DESCRIPTION :   Motor peripheral routines
              
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

 ******************************************************************************
 THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. ST MICROELECTRONICS
 SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
 WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
 ******************************************************************************

******************************************************************************/  

#include "version.h"
#include "lib.h"			/* General purpose typedefs and macros */
#include "ports.h"
#include "ST7MC_hr.h"
#include "mtc_hr.h"
#include "LinSCI.h"
#include "timer.h" 
#include "Timer.h"  
#include "adc.h"
#include "regul.h" 
#include "Misc.h"
#include "mtc.h"			/* Public motor control function prototypes */
#include "opamp.h"   
#include "MTC_Settings_Sensor.h"
                     

/* Public Variables -------------------------------------------------------- */ 

u8 Power_Motor_Status;
u16 Freq_Motor;
u8 Ki = Ki_Fmin;      // integral term
u8 Kp = Kp_Fmin;      // proportionnal term
u8 Rising_bemf = Rising_Fmin ;
u8 Falling_bemf = Falling_Fmin;
u8 Flag_MTC;  
u8 Motor_Direction;

/* -------------------------- Private constants ---------------------------- */
/* Pattern for the refresh of high side drivers bootstrap capacitor */
const u8 BOOT_REFRESH[3] =  {T2,T4,T6};

//Step and MCRB configuration for Clock Wise (CW) Motor Direction
const u8 PHASE_CONFIG[6] = {
                           mem_step1,mem_step2,mem_step3,
			   mem_step4,mem_step5,mem_step6
			   };

const u8 ZC_EDGE[6] =	   {
                           mem_MCRB1,mem_MCRB2,mem_MCRB3,
			   mem_MCRB4,mem_MCRB5,mem_MCRB6
			   };

//Step and MCRB configuration for Counter Clock Wise (CCW) Motor Direction         
const u8 PHASE_CONFIG_CCW[6] = {
                           mem_CCW_step1,mem_CCW_step2,mem_CCW_step3,
			   mem_CCW_step4,mem_CCW_step5,mem_CCW_step6
			   };

const u8 ZC_EDGE_CCW[6] =	   {
                           mem_CCW_MCRB1,mem_CCW_MCRB2,mem_CCW_MCRB3,
			   mem_CCW_MCRB4,mem_CCW_MCRB5,mem_CCW_MCRB6
			   };         

/* -------------------------- Private variables ---------------------------- */

static u8 MotorStatus;

#ifdef __HIWARE__   // only for Metrowerks Compiler
#pragma DATA_SEG SHORT _ZEROPAGE
static u8	StepIndex;
static u8	Step_Z_Counter;
#pragma DATA_SEG DEFAULT
#endif


#ifdef __CSMC__							// only for Cosmic Compiler
@tiny static u8		StepIndex;	
@tiny static u8	        Step_Z_Counter;
#endif


typedef enum
{
BRAKE_START, BRAKE_ONGOING, BRAKE_FINISHED
} BrakeStatus_t;

BrakeStatus_t BrakeState;


/* -------------------------- Private functions ---------------------------- */
static void RefreshBootstrap(void);
static void MTC_ResetPeripheral(void);


/* -------------------------- Private macros ---------------------------- */

// Below macros allow to validate PWM outputs or put them in their RESET state
#define MTC_EnableOutputs()		( MCRA |= (1<<MOE) )
#define MTC_DisableOutputs()	( MCRA &= ((1<<MOE)^255) )

// Below macros allow to enable/disable the motor control peripheral clock
#define MTC_EnableClock()		( MCRA |= (1<<CKE) )
#define MTC_DisableClock()		( MCRA &= ((1<<CKE)^255) )

// Below macros allow to enable/disable Direct Access mode to PWM outputs
#define MTC_EnableDirectAccess()		( MCRA |= (1<<DAC) )
#define MTC_DisableDirectAccess()		( MCRA &= ((1<<DAC)^255) )


#define	BREAK_POSITION		((u8)T1+T4)	/* Defines the stator polarization during brake*/


										// b0-b2-b4:T2-T4-T6 are Low Switch -> KANDA
										// b1-b3-b5:T1-T3-T5 are High Switch -> KANDA
#define OUTPUT_PARITY		0x2A		// b7=b6=0 for BLDC mode (either sensored or sensorless)
										// b1-b3-b5:T2-T4-T6 are Low Switch
										// b0s-b2-b4:T1-T3-T5 are High Switch

/* ------------------------------------------------------------------------- */
/* ------------------------------------------------------------------------- */
/* ------------------------ Functions description -------------------------- */
/* ------------------------------------------------------------------------- */
/* ------------------------------------------------------------------------- */


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_ResetPeripheral

Description: 	Reset all hardware registers related to the motor control 
				peripheral by toggling the RST bit of the MCFR register.
Input/Output:	None
-----------------------------------------------------------------------------*/
static void MTC_ResetPeripheral(void)
{
	SetBit(MCFR,RST); 
	ClrBit(MCFR,RST);
}


/*-----------------------------------------------------------------------------
ROUTINE Name : RefreshBootstrap

Description:	This function ensures that all high side switch drivers are 
				correctly supplied when the motor will start. It generates a 
				1ms pulse on each of the three correspondant low side switches.
				All switches are OFF at the end of the routine.
Input/Output:	None
Comments: 		Durations are indicative and are high side driver dependant.
-----------------------------------------------------------------------------*/
static void RefreshBootstrap(void)
{
		u8 i;
	for (i=0; i<3; i++)
	{
		MPHST = BOOT_REFRESH[i];	// switch ON one low side switch
		Wait1ms();
		Wait1ms();
		MPHST = 0;					// Switch OFF all switches
		Wait1ms();
		Wait1ms();
	}
	MPHST = 0;	// All switches OFF
	Wait100us();	// This is to avoid any short circuit conditions
}


/*-----------------------------------------------------------------------------
ROUTINE Name : Read_Position

Description:	This function returns the sensors information
				
Input/Output:	None/unsigned byte -> xxxxx210
Comments: 	MCIA -> 2
		MCIB -> 1
		MCIC -> 0
-----------------------------------------------------------------------------*/
static u8 Read_Position(void)    
{
u8 position;

position = 0;

MPHST = MCIA;
Wait1us();
Wait1us();
Wait1us();
if (ValBit(MREF,HST))	SetBit(position,2);

MPHST = MCIB;
Wait1us();
Wait1us();
Wait1us();
if (ValBit(MREF,HST))	SetBit(position,1);

MPHST = MCIC;
Wait1us();
Wait1us();
Wait1us();
if (ValBit(MREF,HST))	SetBit(position,0);

MPHST = 0;

return(position);
}




/*-----------------------------------------------------------------------------
ROUTINE Name :	GetMotorStatus

Description:	Allows higher level SW module to get the motor drive module 
				internal state
Input:			None
Output:			MotorStatus value
Comments: 		MotorStatus flags description is public (cf mtc.h)
-----------------------------------------------------------------------------*/
u8 GetMotorStatus(void)
{
	return(MotorStatus);
}


/*-----------------------------------------------------------------------------
ROUTINE Name :	SetMotorStatus

Description:	Allows higher level SW module to set the motor drive module 
				internal state
Input:			MotorStatus value
Output:			MNone
Comments: 		MotorStatus flags description is public (cf mtc.h)
-----------------------------------------------------------------------------*/
void SetMotorStatus(u8 status)
{
	MotorStatus = (u8)(status);
}


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_InitPeripheral

Description:	This function (re)initializes the motor control peripheral from
				any	state (MCU reset, application level initialization, ...).
Input/Output:	None
Comments: 	It must be noticed that part of MPOL and MDTG registers are 
		write once, meaning they cannot be modified anymore once the 
		MTC_InitPeripheral function has been executed.
-----------------------------------------------------------------------------*/
void MTC_InitPeripheral(void)
{

	MTC_ResetPeripheral();

	// Initialize registers in page 1 first
	SET_MTC_PAGE(1);
	MCONF = mem_MCONF;
//	MPWME = ((u8)(MPWMV_MSK | MPWMU_MSK | DG_MSK));	//Enable MCPWMV & MCPWMU  ouput & debug signal 
	MPWME = ((u8)(MPWMV_MSK | MPWMU_MSK ));	//Enable MCPWMV & MCPWMU  ouput without debug signal

	MPOL = DRIVERS_POLARITY;				// Set the Output Polarity to the selected
										              // ZVD bit=0; Z and D have opposite edge
										              // REO bit=0;read Z event on even channels
	MDTG = mem_MDTG; //reset PCN and write the delay value
    	MDTG = (u8)(mem_MDTG&(0x40));  // disable the complementary PWM mode
	
	MPAR = OUTPUT_PARITY;				// Define Low/High Side MCOx outputs
   	MZFR = mem_MZFR;
   	MSCR = mem_MSCR;

	// Initialize registers in page 0
	SET_MTC_PAGE(0);	

	MPCR = mem_MPCR;

	MCP0L = mem_MCPOL; 
	MCP0H = mem_MCPOH;

	MCRA = mem_MCRA;

	MCPUL = 0;    	// 0% duty cycle on U Channel 
	MCPUH = 0;   

	MCPVL = 0;    	// 0% duty cycle on V Channel 
	MCPVH = 0;    

	MCRC = mem_MCRC;

	MCFR = mem_MCFR;

#ifdef EXT_AOP   // external circuitry used for current value
	MREF = (u8)(CFAV_MASK);					// No Chopper, MCCFI input
#else
	MREF &= (u8)(~CFAV_MASK);					// No Chopper, OAZ input
#endif 

   	MotorStatus = 0;	// Indicate motor driving state (start/stop/stalled...)
	BrakeState = BRAKE_START;
}


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_StartMotor

Description:	This function performs three actions:
		* Initializes HW registers and SW variables needed in real time	for motor drive
		* prepares the start-up by refreshing the bootstrap capacitors of the high side
		  switch drivers and aligning the rotor in a known position
		* start-up the peripheral circuitry to get expected interrupts
Input/Output:	None
-----------------------------------------------------------------------------*/
void MTC_StartMotor(void)
{
  u8	Sensor_coding_flag;
#if (FEEDBACK_TYPE == CLOSED_LOOP)
    Init_PI();
#endif
 	Flag_MTC = 0;
	Step_Z_Counter = 0;
	Power_Motor_Status &= Heat_Voltage_Failure;  // reset all flags except overtemp & overvoltage 

#ifndef EXT_AOP
		OPAMP_InitOffset(LOW_GAIN);     	// initialise Opamp Gain and Offset compensation
//		OPAMP_InitOffset(HIGHGAIN);     	// initialise Opamp Gain and Offset compensation
#endif

	// Init of MTC peripheral hardware registers used modified during run-time
    	#if (DRIVING_MODE == VOLTAGE_MODE)
    	MCPVL = mem_MCPVL;    // current limitation - Voltage mode
    	MCPVH = mem_MCPVH;
        #endif

	MIMR = 0;
  MCOMP = (u8)(MCOMP_START);		// Load Alignment time

	MTC_EnableDirectAccess();
	MPRSR = (u8)(mem_MPRSR + RATIO_START);	// MTIM timer prescaler set-up
	MTC_EnableOutputs();
	RefreshBootstrap();			// To be sure high side driver correctly supplied

    	MTC_EnableClock();

  
#if (SENSOR_TYPE == SENSOR_60)
  Sensor_coding_flag = 1;     // Flag to test erronous sensor coding read
  while ( Sensor_coding_flag == 1)
   {
    Sensor_coding_flag = 0;
    switch(Read_Position())
	  {
     case 0x04:
		    StepIndex = 1;    
		    break;
	   case 0x06:
		    StepIndex = 2;
	  	  break;
	   case 0x07:
		    StepIndex = 3;
		    break;
	   case 0x03:
		    StepIndex = 4;
		    break;
	    case 0x01:
		    StepIndex = 5;
		    break;
	    case 0x00:
		    StepIndex = 0;
		    break;
	    default:
        Sensor_coding_flag = 1;  // mark erronous sensor coding red
		    break;
	  }
   }
   
  			
#elif (SENSOR_TYPE == SENSOR_120)
  Sensor_coding_flag = 1;     // Flag to test erronous sensor coding read
  while ( Sensor_coding_flag == 1)
   {
    Sensor_coding_flag = 0;
	  switch(Read_Position())
	  {
	  case 0x04:
		  StepIndex = 2;    
		  break;
	  case 0x06:
		  StepIndex = 3;
		  break;
	  case 0x02:
		  StepIndex = 4;
		  break;
	  case 0x03:
		  StepIndex = 5;
		  break;
	  case 0x01:
		  StepIndex = 0;
		  break;
	  case 0x05:
		  StepIndex = 1;
		  break;
	  default:
      Sensor_coding_flag = 1;  // mark erronous sensor coding red
		  break;
	  }
   }			
#else 
	#error"Wrong Sensor Type Parameter!"
#endif

    Set_Motor_Direction();
    
    if (Motor_Direction == CW)
    {
    MCRB = ZC_EDGE[StepIndex];        // Load phase corresponding to the sensor word
    MPHST = PHASE_CONFIG[StepIndex];	// and to run in CW Direction
    
    MTC_DisableDirectAccess();

    StepIndex++;                  	
    if (StepIndex > 5)	StepIndex = 0;
    
    MCRB = ZC_EDGE[StepIndex];        // Load phase active on first C interruption
    MPHST = PHASE_CONFIG[StepIndex];	// and to run in CW Direction
    }
    else
    {

#if (SENSOR_TYPE == SENSOR_60)
    StepIndex++;                  	
    if (StepIndex > 5)	StepIndex = 0;
#endif   
    
    MCRB = ZC_EDGE_CCW[StepIndex];        // Load phase corresponding to the sensor word 
    MPHST = PHASE_CONFIG_CCW[StepIndex];	// and to run in CCW Direction 
    
    MTC_DisableDirectAccess();

    StepIndex++;                  	
    if (StepIndex > 5)	StepIndex = 0;
    
    MCRB = ZC_EDGE_CCW[StepIndex];        // Load phase active on first C interruption 
    MPHST = PHASE_CONFIG_CCW[StepIndex];	// and to run in CCW Direction
    }
    

    StepIndex++;                  	// Load the next step config when entering the coming C interrupt
    if (StepIndex > 5)	StepIndex = 0;

    SetBit(Flag_MTC,SAMP_EVT);  // force update of MCPxL/H registers
    
  
    MCRA |= SWA_MSK;	// Enable Autoswitched mode 
    MIMR = ((u8)(CIM_MSK + ZIM_MSK + EIM_MSK + CLIM_MSK)); // Unmask C & Z & relevant interrupts 
                                                           // to start commutation mechanism and motor
}


/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_StopMotor

Description:	This function disables all motor control related interrupts 
				and switch off all transistors. This let the windings in 
				floating state once they are completely demagnetized.
Input/Output:	None
-----------------------------------------------------------------------------*/
void MTC_StopMotor(void)
{
	MIMR = 0;           // Mask all MTC related interrupts
	MCRA &= (u8)(~SWA_MSK);	// Disable Autoswitched in previously enabled

	MCPUL = 0;			// Set PWM U to 0%
	MCPUH = 0;		

	MCPVL = 0;			// Set PWM V to 0%
	MCPVH = 0;		

	MISR =0xFF;  // force update of 12 bits Timer to 0% 	

	MTC_EnableDirectAccess();
	MPHST = 0;			// Switch off all drivers
	MTC_DisableDirectAccess();

	MotorStatus = 0;
	Freq_Motor = 0;
	BrakeState = BRAKE_START;
}


#if defined ( __HIWARE__)	/* for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	
#elif defined ( __CSMC__ )	/* for Cosmic Compiler */
@interrupt			
#endif
/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_U_CL_SO_IT

Description:	This is the PWM Update interrupt service routine.
Comments:	U event taken into account on No action done for BLDC motor drive, but flags reset in case
		the routine is entered.
-----------------------------------------------------------------------------*/
void MTC_U_CL_SO_IT(void)
{  
	if (MISR & PUI_MSK)
		{
		MISR = ((u8)~(PUI_MSK));
		}
	if (MISR & CLI_MSK) 
		{
		MIMR &= ((u8)~(CLIM_MSK));  // disable CL int
		MISR = ((u8)~(CLI_MSK));
	        SetBit(Power_Motor_Status,OverCurrent);
		timer_CL_10ms = 30;  // 300 ms temporisation 
		}
	
	SET_MTC_PAGE(1);
	if (MCONF & SOI_MSK) 
		{
		MCONF &= ((u8)~(SOI_MSK)); 
		}
	SET_MTC_PAGE(0);
}


#if defined ( __HIWARE__)	/* for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	
#elif defined ( __CSMC__ )	/* for Cosmic Compiler */
@nosvf @interrupt		
#endif
/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_C_D_IT

Description:	This is the Commutation and Demagnetisation interrupt service 
		routine. In the current software, the C event only is treated.
Comments:	None
-----------------------------------------------------------------------------*/
void MTC_C_D_IT(void)  
{  
if(MISR & CI_MSK)
	{
	MISR = ((u8)~(CI_MSK));
	
    if (Motor_Direction == CW)
    {
    MPHST = PHASE_CONFIG[StepIndex];	// Load phase corresponding to the sensor word
    MCRB = ZC_EDGE[StepIndex];        // and to run in CW Direction
    }
    else
    {
    MPHST = PHASE_CONFIG_CCW[StepIndex];	// Load phase corresponding to the sensor word
    MCRB = ZC_EDGE_CCW[StepIndex];        // and to run in CCW Direction
    }
    
 	SET_MTC_PAGE(1);
  if (Motor_Direction == CW)
    {
	  if (ValBit(ZC_EDGE[StepIndex],1)) SetBit(MPOL,REO);//read BEMF when PWM is on Low channel
	  else ClrBit(MPOL,REO);    // read BEMF when PWM is on High channel
    }
  else
    {
	  if (ValBit(ZC_EDGE_CCW[StepIndex],1)) SetBit(MPOL,REO);//read BEMF when PWM is on Low channel
	  else ClrBit(MPOL,REO);    // read BEMF when PWM is on High channel
    }
	SET_MTC_PAGE(0);

  if (Motor_Direction == CW)
    {
	  if (ValBit(StepIndex,0))    MWGHT = Rising_bemf;
	  else	MWGHT = Falling_bemf;
    }
  else
    {
    if (ValBit(StepIndex,0))    MWGHT = Falling_bemf;
	  else	MWGHT = Rising_bemf;
    }

	StepIndex++;       // Load the next step config when entering the coming C interrupt
	if (StepIndex > 5)	StepIndex = 0;
	}    
	
if(MISR & DI_MSK)
	{
	MISR = ((u8)~(DI_MSK));
    	}
}


#if defined ( __HIWARE__)	/* for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	
#elif defined ( __CSMC__ )	/* for Cosmic Compiler */
@interrupt			
#endif
/*-----------------------------------------------------------------------------
ROUTINE Name : MTC_R_Z_IT

Description:	This is Ratio and zero Crossing interrupt service routine.
				Z event -> Autoswitched mode: compute SW demag time
				Z event -> Switched mode: count number of bemf and enable 
							autoswitched mode once ramp is finished
				RMI event -> no action
				RPI event -> Count number of RPI and update MDREG on last one

Comments:		The MISR flags are cleared as soon as possible to be able to
				serve a new interrupt occuring before the end of the actual one
-----------------------------------------------------------------------------*/
void MTC_R_Z_IT(void)
{  
// ZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZ Z event ZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZZ
if (MISR & ZI_MSK)
    {
    MISR = ((u8)~ZI_MSK);   

    if (MZREG>=0x05)   // check MZREG value to keep good accuracy
        {
        Step_Z[Step_Z_Counter].Ratio = (u8)(MPRSR & 0x0f);   // update Z steptime buffer for PI regulation
        Step_Z[Step_Z_Counter].StepTime = MZREG;             // & motor frequency computation

        if (Step_Z_Counter >= STEP_Z_BUFFER_SIZE-1) Step_Z_Counter = 0;
        else Step_Z_Counter++;
      	}	
    }   // End of Z event processing

// RRRRRRRRRRRRRRRRRRRRRRRRRRRRR RPI event RRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRR
if (MISR & RPI_MSK)
	{
    MISR = ((u8)~RPI_MSK);
	}


// RRRRRRRRRRRRRRRRRRRRRRRRRRRRR RMI event RRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRRR
if (MISR & RMI_MSK)
	{
    MISR = ((u8)~RMI_MSK);
    }
}



#if defined ( __HIWARE__)		/* for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	
#elif defined ( __CSMC__ )		/* for Cosmic Compiler */
@interrupt			
#endif
/*-----------------------------------------------------------------------------
ROUTINE Name : MCES_SE_IT

Description:	Emergency Stop and Speed Error interrupt service routine.
		As this routine is only entered in case of HW problem, the 
		motor is stopped and trace of this interrupt is keeped with a 
		MotorStatus flag.

Comments:	The Speed Error Flag is only used for induction motor drive.
-----------------------------------------------------------------------------*/
void MCES_SE_IT(void)
{
	MISR = ((u8)~EI_MSK);	// Clr Emergency Stop Flag
	ClrBit(MCRC,SEI);	// Clr Speed Error Flag
	
	MTC_StopMotor();
	MotorStatus = EMERGENCY_STOP; 
}  


/*-----------------------------------------------------------------------------
ROUTINE Name : Set_Duty

Description:	This is the update routine of the PWM duty cycle -> feed MCPUH/L
		(voltage mode) or MCPVH/L (current mode) registers. 
		
Comments:	None
-----------------------------------------------------------------------------*/
void Set_Duty(u16 duty)
{
#if (DRIVING_MODE == VOLTAGE_MODE)
	ToCMPxL(MCPUL,(u16)duty);
	ToCMPxH(MCPUH,(u16)duty);
#elif (DRIVING_MODE == CURRENT_MODE)
	ToCMPxL(MCPVL,(u16)duty);
	ToCMPxH(MCPVH,(u16)duty);
#endif
}      
 

/*-----------------------------------------------------------------------------
ROUTINE Name : Set_Target_Electrical_Frequency

Description:	This is the update routine of the PWM duty cycle/PWM current
		limitation in closed loop mode -> feed MCPUH/L in voltage mode
		or MCPVH/L in current mode. 
		The target electrical frequency (0.1 Hz resolution, eg 1000 for 100Hz)
		is input in the the regulation routine;	result (10 bits value)
		is then used to calculate the new duty cycle.
		
Comments:	see mtc.h to for rising/falling bemf,Ki/Kp 
		coefficient calculation.
-----------------------------------------------------------------------------*/
void Set_Target_Electrical_Frequency(u16 target_freq)
{
u32 temp;

  	if (Freq_Motor<=Freq_Min)       // Freq_Motor<=Freq_Min ??
    		{
		Rising_bemf = Rising_Fmin ;
		Falling_bemf = Falling_Fmin;
		Ki = Ki_Fmin;      // integral term
		Kp = Kp_Fmin;      // proportionnal term
    		}
    	else if (Freq_Motor<=F_1)       // Freq_Min<=Freq_Motor<=F_1 ??
		{
		Rising_bemf = (u8)(Rising_Fmin + (s32)(alpha_Rising_1*(Freq_Motor-Freq_Min)/1024)); 
		Falling_bemf = (u8)(Falling_Fmin + (s32)(alpha_Falling_1*(Freq_Motor-Freq_Min)/1024));
		Ki = (u8)(Ki_Fmin + (s32)(alpha_Ki_1*(Freq_Motor-Freq_Min)/1024));
		Kp = (u8)(Kp_Fmin + (s32)(alpha_Kp_1*(Freq_Motor-Freq_Min)/1024));
		}
	else if (Freq_Motor<=F_2)  	// F_1<Freq_Motor<=F_2 ??
		{
		Rising_bemf = (u8)(Rising_F_1 + (s16)(alpha_Rising_2*(Freq_Motor-F_1)/1024)); 
		Falling_bemf = (u8)(Falling_F_1 + (s16)(alpha_Falling_2*(Freq_Motor-F_1)/1024));
		Ki = (u8)(Ki_F_1 + (s32)(alpha_Ki_2*(Freq_Motor-F_1)/1024));
		Kp = (u8)(Kp_F_1 + (s32)(alpha_Kp_2*(Freq_Motor-F_1)/1024));
		}
	 else if (Freq_Motor<=Freq_Max)  // F_2<Freq_Motor<=Freq_Max ??
		{                       
		Rising_bemf = (u8)(Rising_F_2 + (u16)(alpha_Rising_3*(Freq_Motor-F_2)/1024)); 
		Falling_bemf = (u8)(Falling_F_2 + (u16)(alpha_Falling_3*(Freq_Motor-F_2)/1024));
		Ki = (u8)(Ki_F_2 + (u16)(alpha_Ki_3*(Freq_Motor-F_2)/1024));
		Kp = (u8)(Kp_F_2 + (u16)(alpha_Kp_3*(Freq_Motor-F_2)/1024));
		}
	 else                            // Freq_Max>Freq_Motor ??
	  	{
		Rising_bemf = Rising_Fmax ;
		Falling_bemf = Falling_Fmax;
		Ki = Ki_Fmax;      // integral term
		Kp = Kp_Fmax;      // proportionnal term
		}

	temp = PWM_FREQUENCY;              // temp =  (PWM_FREQUENCY * PIoutput)/1024
	temp *= regul_PI((u16)(target_freq));
	temp >>= 10;
	if (temp > Max_Duty) temp = Max_Duty;	

#if (DRIVING_MODE == VOLTAGE_MODE)
	ToCMPxL(MCPUL,(u16)temp);
	ToCMPxH(MCPUH,(u16)temp);
#elif (DRIVING_MODE == CURRENT_MODE)
	ToCMPxL(MCPVL,(u16)temp);
	ToCMPxH(MCPVH,(u16)temp);
#endif
}      
 

/*-----------------------------------------------------------------------------
ROUTINE Name : active_brake

Description:	braking routine 
Input/Output:	duty cycle, duration in ms (u16/u16)
Comments:	Return from this routine immediately if duty 
		is equal to 0x00. 
-----------------------------------------------------------------------------*/
BOOL active_brake(u16 duty, u16 time)   // duration = time * 1ms  eg: 1000 = 1sec
{
if (duty == 0) 
	{
	MCRA &= (u8)(~SWA_MSK);	// Disable Autoswitched mode and...
	MIMR = 0;
	return(TRUE);
	}
else
	{
	switch(BrakeState)
		{
    u8 temp_mphst;
		case BRAKE_START:
		default:
      MCRA &= (u8)(~SWA_MSK);	// Disable Autoswitched mode and...
			MIMR = 0;               // Disable all interruption of motor peripheral
      temp_mphst = MPHST;     // catch the actual step configuration
			MTC_EnableDirectAccess();
      MCRB &= (u8)(OS0_MSK);     // force PWM on the low side switch
			MPHST = temp_mphst;	      // Alignment step pattern
//			MTC_DisableDirectAccess();

			#if (DRIVING_MODE == VOLTAGE_MODE)
			ToCMPxL(MCPUL,(u16)(duty));
			ToCMPxH(MCPUH,(u16)(duty));
			#else
			ToCMPxL(MCPVL,(u16)(duty));
			ToCMPxH(MCPVH,(u16)(duty));
			#endif
			
			Timer_1ms = time;
			BrakeState = BRAKE_ONGOING;
			return(FALSE);
			break;
		case BRAKE_ONGOING:
			if (Timer_1ms == 0)	BrakeState = BRAKE_FINISHED;
			return(FALSE);
			break;
		case BRAKE_FINISHED:
			return(TRUE);
			break;
		}
	}			  
return(FALSE);
}


/*-----------------------------------------------------------------------------
ROUTINE Name : Chk_Motor_Stalled

Description:	 
		
Comments:	None
-----------------------------------------------------------------------------*/
void Chk_Motor_Stalled(void)
{
if ((MPRSR & 0x0F) == RATIO_MAX) 
    {
    MotorStatus = MOTOR_STALLED;
    SetBit(Power_Motor_Status,MotorStalled);  
    }
}

/*-----------------------------------------------------------------------------
ROUTINE Names : Set_Motor_Direction

Description : This routines set the Motor_Direction based on Rotor_Direction state
-----------------------------------------------------------------------------*/
void Set_Motor_Direction( void )
{
    if (Rotor_Direction == CW) Motor_Direction = CW;
      
    else if(Rotor_Direction == CCW) Motor_Direction = CCW;  
      
    else if(Motor_Direction == CCW) Motor_Direction = CW;   // we toggle the motor direction
          
    else  Motor_Direction = CCW;
}

/*** (c) 2005  STMicroelectronics ****************** END OF FILE ***/


