/******************** (C) COPYRIGHT 2006 STMicroelectronics **********************
* File Name          : I2C.c
* Author             : MPA Systems Lab
* Date First Issued  : 04/13/2006 :  V1.0
* Description        : I2C low level function
                       ( POW Code : DP.APN0004.01 )
**********************************************************************************
* History:
* Date          Version     Description
* 04/13/06   :  V1.0        First version
*********************************************************************************
 THE PRESENT SOFTWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS WITH
 CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME.
 AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT
 OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE CONTENT
 OF SUCH SOFTWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING INFORMATION
 CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
**********************************************************************************/

// Public I2C peripheral function prototypes
#include "I2C.h"

#if (MOTOR_TYPE==BLDC)
		/**** Only BLDC motor*****/
		#include "st7mc_hr.h"
#else
		// Library configuration and ST7FMC peripherals Hardware Registers declaration
		#include "config.h"
#endif

                
/***************************************** Functions *********************************************************************/
static void Init_SPI_4_I2C( void );
static void Init_TIMER_4_I2C( void );
void Start_TIMER( void ); 
void Reset_TIMER( void );
static void Init_INT_SCK_4_I2C( void );
static void Init_INT_SDA_4_I2C( void );
static void Init_SPI_Ports( void );	
static void I2C_enable_SPI(void);
static void I2C_disable_SPI(void);
static void I2C_Stop_detected(void);

/***************************************** Variables *********************************************************************/
u8 I2C_Datas_W[ I2C_TABLE_SIZE ] = {0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};  // Starting Value
u8 I2C_Datas_R[ I2C_TABLE_SIZE ] ; 

static u8 I2C_Index_R = 0; // to point on the  SPI_Datas_R[10]  
static u8 I2C_Index_W = 0; // to point on the  SPI_Datas_W[10]  
                                                        
typedef enum
{
WAIT_START, 
START_VALIDATION,
COMMUNICATION
} I2CStatus_t;
                                                       
static I2CStatus_t I2C_Status = WAIT_START;  // Status of I2C BUS

static BOOL I2C_SLAVE_WRITE_flag = FALSE; 
static BOOL I2C_SLAVE_READ_flag = FALSE;
static BOOL I2C_First_Byte = TRUE;
static BOOL I2C_Master_Ack = FALSE;
static BOOL I2C_Slave_Ack = FALSE; 
static BOOL I2C_SLAVE_WAIT_MSTAK = FALSE;
BOOL New_Data_Receved = FALSE ; //flag  Needed for high level protocol  
BOOL New_Data_Sent = FALSE ;	//flag  Needed for high level protocol  

/***********************************************
 ** INIT_I2C_PROTOCOL
 *
 *  FILENAME:
 *
 *  PARAMETERS:
 *
 *  DESCRIPTION: This function initilaize the resource for I2C emulation. 
 *  SPI Peripheral, Timer A, Interrupt ei0 (SCK, SDA)
 *
 *  RETURNS:
 *
 ***********************************************/
void Init_I2C_Protocol(void)
{ 
	// Force PA6 as Push-Pull Outup and selec PA6=1 for I2C communication
	// PA6 is configured as BUS controller 
	// PA6=0 (Low) Bus is SPI (This setting can be used for external SPI memory)
	// PA6=1 (High) Bus is I2C    
	PADR	= 0x40; // Set I2C
	PADDR	= 0x40; 
	PAOR	= 0x40;

	Init_SPI_4_I2C(); 
	Init_TIMER_4_I2C();
 	Init_INT_SCK_4_I2C();
   	Init_INT_SDA_4_I2C();   
   	                                                        
   	I2C_Status = WAIT_START; // Initial status is waiting for start command
   	I2C_Slave_Ack = FALSE;  // No ACK from slave in progress
}

/***********************************************
 ** I2C_enable_SPI
 *
 *  FILENAME:
 *
 *  PARAMETERS:
 *
 *  DESCRIPTION: Enable SPI comunication 
 *		 - after a Start condition and falling of SCK
 *
 *  RETURNS:
 *
 ***********************************************/
static void I2C_enable_SPI(void)
{	    
	SPICSR; // Clear SPIF Pending Interrupt
	SPIDR;                    
	
	SetBit(SPICR,SPICR_SPE); // Enable SPI port     
}

/***********************************************
 ** I2C_disable_SPI
 *
 *  FILENAME:
 *
 *  PARAMETERS:
 *
 *  DESCRIPTION:
 *
 *  RETURNS:
 *
 ***********************************************/
static void I2C_disable_SPI(void)
{
	ClrBit(SPICR,SPICR_SPE); // Disable SPI port	
}

/***********************************************
 ** I2C_Stop_detected
 *
 *  FILENAME:
 *
 *  PARAMETERS:
 *
 *  DESCRIPTION:
 *
 *  RETURNS:
 *
 ***********************************************/
static void I2C_Stop_detected(void)
{
	Init_INT_SDA_4_I2C();
	I2C_disable_SPI();
	Init_INT_SCK_4_I2C();
	I2C_Index_R = 0;
	I2C_Index_W = 0;
   
	if (I2C_SLAVE_READ_flag)		 //for high level protocol purposes 
	{
		New_Data_Receved = TRUE ;
	}
	if (I2C_SLAVE_WRITE_flag)		 //for high level protocol purposes 
	{
		New_Data_Sent = TRUE;
	}
	
	I2C_SLAVE_WRITE_flag = FALSE;
	I2C_SLAVE_READ_flag = FALSE;
	I2C_First_Byte = FALSE;
}

/***********************************************
 ** Init_SPI_Ports
 *
 *  FILENAME:
 *
 *  PARAMETERS:
 *
 *  DESCRIPTION: Initialize port configuration for SPI
 *
 *  RETURNS:
 *
 ***********************************************/
static void Init_SPI_Ports(void)
{                        
	SetBit (PBDR,4);
	SetBit(PBDDR,4); // Configure MISO as Output Push Pull
	SetBit(PBOR,4);
	
	// Other PIN is Floating Input
	// Alternate function is enabled in SPI initialization	
}                 

/***********************************************
 ** Init_SPI_4_I2C
 *
 *  FILENAME:
 *
 *  PARAMETERS:
 *
 *  DESCRIPTION: 	SPI configuration for I2C emulation
 *
 *  RETURNS:
 *
 ***********************************************/
void Init_SPI_4_I2C(void)
{

	Init_SPI_Ports();  
	
	// Clear pending interrupt SPI
	SPICSR; // Clear SPIF + OVR
	SPIDR;  // Clear WCOL             
		
	// Enable Interrupt SPI + Disable Alternate Function (Will be enabled further)
	// CPOL = 0 CPHA = 0 SPI Clock Edge Rising
	// Slave Mode
	SPICR = SPI_CONFIG_CR;
	SPICSR = SPI_CONFIG_CSR;
} // End of Init_SPI_4_I2C  

#ifdef __HIWARE__		/* test for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	/* additional registers will be saved */
#else
#ifdef __CSMC__			/* test for Cosmic Compiler */
@interrupt			/* Cosmic interrupt handling */
#endif
#endif

/***********************************************
 ** SPI_Interrupt_Routine
 *
 *  FILENAME:
 *
 *  PARAMETERS:
 *
 *  DESCRIPTION: ISR executed each SPI end of transmission.
 *  If the received byte is the first, the slave check if it is addressed 
 *  and eventually send ack, and set flag for SLAVE_READ or SLAVE_WRITE 
 *  according the 8th bit,
 *  Else if SLAVE is reading the data is stored in I2C_Datas_R and ack,
 *  Else if SLAVE is writing set SLAVE_WAIT_MSACK
 *
 *  RETURNS:
 *
 ***********************************************/

void SPI_Interrupt_Routine(void) 
{       	
	I2C_disable_SPI();
        SPICSR;
        
        SetBit(PBDR,4); // MISO is set high when SPI is disabled before Ack  
        
        if (I2C_Status==WAIT_START)
        {
        	SPIDR;
        	return;
       	}
        
	//********************************************************************************************
	if( I2C_First_Byte )
	{
		I2C_Datas_R[0] = SPIDR;
	
		// Here we test the slave address and the read/write mode  
		switch((SLAVE_ADDRESS ^ I2C_Datas_R[0]))
		{
			//slave is adressed and write mode 
			case 0x01: 
				I2C_SLAVE_READ_flag = TRUE;
 				ClrBit(PBDR,4);  //here we ACK
 				I2C_Slave_Ack = TRUE;  
 				I2C_First_Byte = FALSE; // Next is Data to be read from slave
 				I2C_Index_R ++;
				break;

			// slave is adressed and read mode or RECEPTION  
			case 0x00:
		 		I2C_SLAVE_WRITE_flag = TRUE ;
				ClrBit(PBDR,4); // here we ACK  MISO is put low
				I2C_Slave_Ack = TRUE;
				I2C_First_Byte = FALSE; // Next is Data to be write by the slave
				I2C_Index_R ++;
				break;
		
			//slave is not adressed 
			default: 
		 		I2C_Index_R = 0;
		 		
		 		Init_INT_SCK_4_I2C();
   				Init_INT_SDA_4_I2C();
		 		I2C_Status=WAIT_START;  
				break;
	  	}
	}
	
	//******************************************************************************************
	else if ( I2C_SLAVE_READ_flag )
	{    // Reception mode  
		ClrBit(PBDR,4); // here we ACK  MISO is put low
		I2C_Slave_Ack = TRUE;  

		if(I2C_Index_R < I2C_TABLE_SIZE)  	// end of table not reached  
		{
			I2C_Datas_R[I2C_Index_R] = SPIDR;
			I2C_Index_R ++;
		}
		else
		{
			I2C_Index_R = 0;
			// we can initialise comme back to normal position  
			// for more safety communication  
			Init_INT_SCK_4_I2C();
   			Init_INT_SDA_4_I2C();
   			
   			I2C_SLAVE_READ_flag = FALSE;
			I2C_Status=WAIT_START;
		}	
	}       
	else if ( I2C_SLAVE_WRITE_flag )
	{
		SPIDR;
		
		// Trasmission Mode
		SetBit(SPICSR,2); // Disable SPI Output 
		
		I2C_SLAVE_WAIT_MSTAK = TRUE;
				
		I2C_First_Byte = FALSE;
		I2C_Index_W ++;
	}              
}

/***********************************************
 ** Init_INT_SDA_4_I2C
 *
 *  FILENAME:
 *
 *  PARAMETERS:
 *
 *  DESCRIPTION: Initialize the external interrupt SDA (ei0 PD1)
 *
 *
 *  RETURNS:
 *
 ***********************************************/
static void Init_INT_SDA_4_I2C(void)
{	
	EICR |= 0x06; // PD1 rising and falling edge
	SetBit(PDOR,1); // Set PD1 as Floating Interrupt 
}

/***********************************************
 ** INT_SDA_I2C
 *
 *  FILENAME:
 *
 *  PARAMETERS:
 *
 *  DESCRIPTION: Interrupt on each edge of SDA. It is used to detect Start or Stop condition
 *
 *  RETURNS:
 *
 ***********************************************/

#ifdef __HIWARE__		/* test for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	/* additional registers will be saved */
#else
#ifdef __CSMC__			/* test for Cosmic Compiler */
@interrupt			/* Cosmic interrupt handling */
#endif
#endif

void INT_SDA_I2C(void) 
{                        
	if (ValBit(PCDR,1))
	{
		if (ValBit(PDDR,1))
		{ 
			// Rising Edge SDA while SCK is High (STOP Condition)
        		// In any case the status is forced to WAIT_START
        		I2C_Status=WAIT_START;
        		I2C_Stop_detected();
        	}
        	else                   
	        {
	        	// Falling Edge SDA while SCK is High (START Condition)	        	
	        	if (((I2C_Status==WAIT_START)||(I2C_Status==COMMUNICATION))&&(I2C_Slave_Ack==FALSE))
	        	{
	        		I2C_Status=START_VALIDATION;
	        		I2C_First_Byte=TRUE;  
	        		Start_TIMER();
	        	}
	        } 
	}
}

/***********************************************
 ** Init_INT_SCK_4_I2C
 *
 *  FILENAME:
 *
 *  PARAMETERS:
 *
 *  DESCRIPTION:	Initialize the external interrupt SCK (ei2 PC1)
 *
 *  RETURNS:
 *
 ***********************************************/
static void Init_INT_SCK_4_I2C(void)
{
	EICR |= 0x80; // PC1 falling edge
	SetBit(PCOR,1); // Set PC1 as Floating Interrupt 
}

/***********************************************
 ** INT_SCK_I2C
 *
 *  FILENAME:
 *
 *  PARAMETERS:
 *
 *  DESCRIPTION: ISR generated each falling edge of the clock
 *  If status is STARTING_VALIDATION it will be validated and the status is set to COMMUNICATION
 *  Here we stop the Slave ACKNOWLEDGE in SLAVE_READ and 
 *  send the next byte in I2C_Datas_W and check the master ACK 
 *
 *  RETURNS:
 *
 ***********************************************/
/************************* CLOCK line  INTERRUPT ROUTINE ************************************/

#ifdef __HIWARE__		/* test for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	/* additional registers will be saved */
#else
#ifdef __CSMC__			/* test for Cosmic Compiler */
@interrupt			/* Cosmic interrupt handling */
#endif
#endif

void INT_SCK_I2C ( void ) 
{		       
       	// Falling Edge
      	if (I2C_Status==START_VALIDATION)
	{       
		// Start is validated
		// Clock falling after a Start Condition -> Start SPI comunication                      
		
		// Disable output for firs byte
		SetBit(SPICSR,2); // Disable SPI Output
		
		// Enable SPI  
		I2C_enable_SPI();
		
		I2C_Status=COMMUNICATION;
		Reset_TIMER();
	}
	else
	{   
		if (I2C_Slave_Ack)
		{
			SetBit(PBDR,4);
			I2C_Slave_Ack = FALSE;  //for next ack sinking 
			
			// If we are in Transfer mode we put the byte in SPI
			if (I2C_SLAVE_WRITE_flag)
			{
				if( I2C_Index_W < I2C_TABLE_SIZE)
				{
					SPIDR = I2C_Datas_W[I2C_Index_W];	  // data is loaded before the SPI output is enabled  
			 		ClrBit(SPICSR,2); // Enable SPI Output
			 		I2C_enable_SPI();
				}
				else
				{
					I2C_Index_W = 0;
					I2C_Status=WAIT_START;
 				}
				
			}
			if (I2C_SLAVE_READ_flag)
			{
				// Slave in read mode
				I2C_enable_SPI();
			}				
		}
		if (I2C_SLAVE_WAIT_MSTAK)
		{
			I2C_SLAVE_WAIT_MSTAK = FALSE;
			if (!ValBit(PDDR,1))
			{
				// Master ACK 
				I2C_Master_Ack=TRUE;
			}
			
			if( I2C_Index_W < I2C_TABLE_SIZE)
			{
				SPIDR = I2C_Datas_W[I2C_Index_W];	  // data is loaded before the SPI output is enabled  
		 		ClrBit(SPICSR,2); // Enable SPI Output
		 		
		 		I2C_enable_SPI();
				I2C_Master_Ack = FALSE;  //for next ack sinking 				
			}
			else
			{
				I2C_Index_W = 0;
				I2C_Status=WAIT_START;
			}							
		}			
	}		
}

/***********************************************
 ** Init_TIMER_4_I2C
 *
 *  FILENAME:
 *
 *  PARAMETERS:
 *
 *  DESCRIPTION:	Initialization of Timer A used for Time Out Purpouse
 *                      TIMEOUT defines the number of microseconds 
 *                      Timer is freezed and will be started using Start Timer
 *			After the time out the ISR will be served
 *  RETURNS:
 *
 ***********************************************/
static void Init_TIMER_4_I2C(void)
{       
	// Configure timer register
	// Fcpu/2 = 4 Mz
	// Output Compare Interrupt Enabled
	TACR1=0x40;
	TACR2=0x04;        
	
	// Freeze the Timer
	SetBit (TACSR,TACSR_TIMD);
	
	// Set Duration
	TAOC1HR=TCOUNTH; 
	TACSR;
	TAOC1LR=TCOUNTL; 
	TAOC2LR;
	TACLR;
	
	// Reset counter
	TAACLR=0;
} 

/***********************************************
 ** Start_TIMER
 *
 *  FILENAME:
 *
 *  PARAMETERS:
 *
 *  DESCRIPTION:	Unfreeze the timer
 *
 *  RETURNS:
 *
 ***********************************************/

void Start_TIMER(void)
{       
	// UnFreeze the Timer
	ClrBit (TACSR,TACSR_TIMD);
}

/***********************************************
 ** Reset_TIMER
 *
 *  FILENAME:
 *
 *  PARAMETERS:
 *
 *  DESCRIPTION: Freeze and reset the timer	
 *
 *  RETURNS:
 *
 ***********************************************/
void Reset_TIMER(void)
{       
	// Freeze the Timer
	SetBit (TACSR,TACSR_TIMD);
	
	// Reset counter
	TAACLR=0;
}

#ifdef __HIWARE__		/* test for HIWARE Compiler */
#pragma TRAP_PROC SAVE_REGS	/* additional registers will be saved */
#else
#ifdef __CSMC__			/* test for Cosmic Compiler */
@interrupt			/* Cosmic interrupt handling */
#endif
#endif

/***********************************************
 ** TIMA_Interrupt_Routine
 *
 *  FILENAME:
 *
 *  PARAMETERS:
 *
 *  DESCRIPTION: Will be executed after TIMEOUT microseconds after the Start_TIMER() command
 *               TIMEOUT procedure will be executed and the timer will be reinitialized (freezed)
 *
 *  RETURNS:
 *
 ***********************************************/
void TIMER_A_Interrupt_Routine (void)
{       
	if (ValBit(TACSR,6))
	{
		if (I2C_Status==START_VALIDATION)
		{
			I2C_Status=WAIT_START; 
		} 
			       		
		Reset_TIMER(); 
		
		// Clear Pending bit
		TACSR;
		TAOC1LR;
	}
	else
	{
	// Clear Pending bit
		TACSR; 
		TAOC2LR;
	}
}

/******************* (C) COPYRIGHT 2006 STMicroelectronics *****END OF FILE****/