/******************** (C) COPYRIGHT 2006 STMicroelectronics **********************
* File Name          : I2C.c
* Author             : MPA Systems Lab
* Date First Issued  : 04/13/2006 :  V1.0
* Description        : Main Routine common for all motor control libraries
                       ( POW Code : DP.APN0004.01 )
**********************************************************************************
* History:
* Date          Version     Description
* 04/13/06   :  V1.0        First version
*********************************************************************************
 THE PRESENT SOFTWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS WITH
 CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME.
 AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT
 OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE CONTENT
 OF SUCH SOFTWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING INFORMATION
 CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
**********************************************************************************/ 

#include "I2C.h"

#if (MOTOR_TYPE==BLDC)

#include "version.h"

#if (SENSOR_TYPE==0)

// BLDC Sensorless

#include "version.h"
#include "lib.h"
#include "mtc.h"
#include "ST7MC_hr.h"
#include "ports.h"
#include "opamp.h"
#include "timer.h"
#include "it_ST7MC.h"  
#include "LinSCI.h"
#include "misc.h"
#include "wwdg.h" 
#include "spi.h"                    
#include "regul.h"
#include "adc.h"
#include "MTC_Settings_Sensorless.h" 
#include "I2C.h"
#include "ProtoSup.h"

                                          

#define CRC	0x54	// just declare a data byte to be written in EEPROM  
  

typedef enum 
{
IDLE, START, RUN, STOP, BRAKE, WAIT, FAULT
} SystStatus_t;

static SystStatus_t State;


u16 TARGET_FREQ_CL=1500;


void main(void)
{                       
	while ( !WWD_Init() ); 

	/*** !! Check carrefully whether it is needed to change the interrupt priotities. For high electrical       !! ***/
	/*** !! frequency applications (eg 1.5Khz or more for example), it is mandatory to set the Commutation IT   !! ***/
	/*** !! event at the highest level to avoid synchronization loss (done by default in the following routine) !! ***/
	ST7_IntPrioritySetUp();
        
   	State = IDLE;
		Power_Motor_Status = 0;

	SCI_Config();
	PORTS_Init();   // initialize I/O
 	TIMB_Config();  // initialize TIMER B peripheral
	SPI_Init();
        
	MTC_InitPeripheral();	// Initialize peripheral for Sensorless BLDC drive
	Init_I2C_Protocol();
	
	EnableInterrupts();
	Flash_Both_Led();

	PORTS_RedLedOn();
	
	while(1)    // main loop
	{
	if (Chk_Timer_WDG_Elapsed() == TRUE)	WWD_Refresh();
    	Chk_Power_Motor_Status();
	
	if ((u8)(GetMotorStatus() & FAULT_MSK) != 0) State = FAULT;  // START_UP_FAILED or MOTOR_STALLED 
	// Read Set Speed Command

LINK_protocol();

if (COMMAND==CMD_SETSPEED)
{
	TARGET_FREQ_CL=(Valid_Data_In[1]<<8)+Valid_Data_In[2];
}
							    

	switch (State)
			{
			case IDLE:
       		    	if (timer_10ms == 0) PORTS_RedLedOn();  // red LED back to normal after
	    	                                                // overvoltage, overtemperature detection
					
					if ( COMMAND == CMD_START )	// Start motor
					{      
						COMMAND = 0;
						State = START;
					}

				break;
			
			case START:
					if (MTC_StartMotor() == TRUE) State = RUN;
				break;

			case RUN:
					if (SCI_Tx_Timer == 0)
					{ 
				 	#if (FEEDBACK_TYPE == CLOSED_LOOP)
					Lin_Tx_Buffer[0] = Rising_bemf;
					Lin_Tx_Buffer[1] = Falling_bemf;
					Lin_Tx_Buffer[2] = Ki;    
					Lin_Tx_Buffer[3] = Kp;
					Lin_Tx_Buffer[4] = (u8)(Freq_Motor>>8); // send MSB
					Lin_Tx_Buffer[5] = (u8)(Freq_Motor);    // send LSB
					
					Lin_Tx_Buffer[6] = 0;
					#else
					Lin_Tx_Buffer[0] = Rising_bemf;
					Lin_Tx_Buffer[1] = Falling_bemf;
					Lin_Tx_Buffer[2] = (u8)(Period_To_Frequency()>>8); // send MSB
					Lin_Tx_Buffer[3] = (u8)(Period_To_Frequency());    // send LSB
					Lin_Tx_Buffer[4] = 0;
					
					Lin_Tx_Buffer[5] = 0;
					Lin_Tx_Buffer[6] = 0;
					#endif		
					SCI_Send_Data();
					SCI_Tx_Timer = 100;    // 1 sec delay
					}

  			    	if (GetMotorStatus() & AUTO_SWITCH) 
           	   			{
						if (ValBit(Flag_MTC,SAMP_EVT))     // update PWM?
							{
							u32 temp;
					
							ClrBit(Flag_MTC,SAMP_EVT);
							#if (FEEDBACK_TYPE == CLOSED_LOOP)
						 		temp = (u8)(Get_RV1());      // target_freq = (Get_RV1 * (Freq_Max-Freq_Min))/256 + Freq_Min
								temp *= (u32)(Freq_Max-Freq_Min);
 								temp >>= 8;
 								temp += Freq_Min;
								
								Set_Target_Electrical_Frequency((u16)(TARGET_FREQ_CL));
								
 							#elif (FEEDBACK_TYPE == OPEN_LOOP)					
								Falling_bemf = (u8)(Get_RV3());   // read RV3 & set falling Bemf coefficient accordingly
								Rising_bemf = (u8)(Get_RV2());    // read RV2 & set rising Bemf coefficient accordingly

								temp = (u8)(Get_RV1());                                                                      
						
								#if (DRIVING_MODE == VOLTAGE_MODE)
 							 	temp *= (u32)(Max_Duty);          // temp =  (Max_Duty * Get_RV1)/256
								#elif (DRIVING_MODE == CURRENT_MODE)
								temp *= (u32)(PWM_FREQUENCY);	  // temp =  (PWM_FREQUENCY * Get_RV1)/256	
 							 	#else
  									#error"Wrong Driving Mode Parameter!"	
								#endif
						
								temp >>= 8;
								Set_Duty((u16)(temp));
							#else
  								#error"Wrong Feedback Type Parameter!"	
							#endif
							}
       	           	   	}
	
	   	    		Chk_Motor_Stalled();    
   	    	 		if (timer_10ms == 0) PORTS_GreenLedOn();   
				
				if ( COMMAND == CMD_BRAKE ) State=STOP;	// Start motor for I2C
				
 				break;

			case BRAKE: 
     	    			if (active_brake(Brake_Duty,Brake_Time) == TRUE) State = STOP;
				break;

			case STOP:
                		MTC_StopMotor();
	    		    	PORTS_RedLedOn();
				State = IDLE;
				break;

			case WAIT:
				break;
			case FAULT:
		        default:
			        MTC_StopMotor();
	    	   		PORTS_RedLedOn();
				if ((u8)(GetMotorStatus() & FAULT_MSK) == 0) State = IDLE;
				break;
			}

    }
} // main loop

#else

// BLDC Sensored

#include "version.h"
#include "lib.h"
#include "mtc.h"
#include "ST7MC_hr.h"
#include "ports.h"
#include "opamp.h"
#include "timer.h"
#include "it_ST7MC.h"  
#include "regul.h"
#include "adc.h"
#include "LinSCI.h"
#include "misc.h"   
#include "wwdg.h"
#include "spi.h"
#include "MTC_Settings_Sensor.h"
#include "I2C.h"
#include "ProtoSup.h"


 
#define CRC	0x54	// just declare a data byte to be written in EEPROM  

 
typedef enum 
{
IDLE, RUN, STOP, BRAKE, FAULT
} SystStatus_t;

static SystStatus_t State;

u16 TARGET_FREQ_CL=1500;

void main(void)
{   


	/*** !! Check carrefully whether it is needed to change the interrupt priorities. For high electrical       !! ***/
	/*** !! frequency applications (eg 1.5Khz or more for example), it is mandatory to set the Commutation IT   !! ***/
	/*** !! event at the highest level to avoid synchronization loss (done by default in the following routine) !! ***/
	
        
  	State = IDLE;
	Power_Motor_Status = 0;

	SCI_Config();
	PORTS_Init();
 	TIMB_Config();
 	SPI_Init();
	MTC_InitPeripheral();	// Initialize peripheral for Sensor BLDC drive
	Init_I2C_Protocol();

	EnableInterrupts();

	Flash_Both_Led();
	PORTS_RedLedOn();
	

	while(1)    // main loop
	{

    	Chk_Power_Motor_Status();
	
	if ((u8)(GetMotorStatus() & FAULT_MSK) != 0) State = FAULT;  // START_UP_FAILED or MOTOR_STALLED 
								 // or HARD_FAILURE or EMERGENCY_STOP?
	LINK_protocol(); //i2c
	// Read Set Speed Command 
	if (COMMAND==CMD_SETSPEED)
	{
		TARGET_FREQ_CL=(Valid_Data_In[1]<<8)+Valid_Data_In[2];
	}

	switch (State)
			{
			case IDLE:
        		    	if (timer_10ms == 0) PORTS_RedLedOn();  // red LED back to normal after
		    	                                                // overvoltage, overtemperature detection
				
				if ( COMMAND == CMD_START )	// Start motor	// Start motor
					{
					COMMAND = 0;	
					MTC_StartMotor();
					State = RUN;
					}
				break;
			
			case RUN:             
				if (SCI_Tx_Timer == 0)
					{ 
			 		#if (FEEDBACK_TYPE == CLOSED_LOOP)
					Lin_Tx_Buffer[0] = Rising_bemf;
					Lin_Tx_Buffer[1] = Falling_bemf;
					Lin_Tx_Buffer[2] = Ki;    
					Lin_Tx_Buffer[3] = Kp;
					Lin_Tx_Buffer[4] = (u8)(Freq_Motor>>8); // send MSB
					Lin_Tx_Buffer[5] = (u8)(Freq_Motor);    // send LSB
					
					Lin_Tx_Buffer[6] = 0;
					#else
					Lin_Tx_Buffer[0] = Rising_bemf;
					Lin_Tx_Buffer[1] = Falling_bemf;
					Lin_Tx_Buffer[2] = (u8)(Period_To_Frequency()>>8); // send MSB
					Lin_Tx_Buffer[3] = (u8)(Period_To_Frequency());    // send LSB
					Lin_Tx_Buffer[4] = 0;
					
					Lin_Tx_Buffer[5] = 0;
					Lin_Tx_Buffer[6] = 0;
					#endif		
					
					SCI_Send_Data();
					SCI_Tx_Timer = 100;    // 1 sec delay
					}

				if (ValBit(Flag_MTC,SAMP_EVT))     // update PWM?
					{
					u32 temp;
					
					ClrBit(Flag_MTC,SAMP_EVT);
					#if (FEEDBACK_TYPE == CLOSED_LOOP)
				 		temp = (u8)(Get_RV1());      // target_freq = (Get_RV1 * (Freq_Max-Freq_Min))/256 + Freq_Min
				 		temp *= (u32)(Freq_Max-Freq_Min);
 						temp >>= 8;
 						temp += Freq_Min;
            
						Set_Target_Electrical_Frequency((u16)(TARGET_FREQ_CL));
 					#elif (FEEDBACK_TYPE == OPEN_LOOP)					
						Falling_bemf = (u8)(Get_RV3());   // read RV3 & set falling Bemf coefficient accordingly
						Rising_bemf = (u8)(Get_RV2());    // read RV2 & set rising Bemf coefficient accordingly

						temp = (u8)(Get_RV1());                                                                      
						
						#if (DRIVING_MODE == VOLTAGE_MODE)
 					 	temp *= (u32)(Max_Duty);          // temp =  (Max_Duty * Get_RV1)/256
						#elif (DRIVING_MODE == CURRENT_MODE)
						temp *= (u32)(PWM_FREQUENCY);	  // temp =  (PWM_FREQUENCY * Get_RV1)/256	
 					 	#else
  							#error"Wrong Driving Mode Parameter!"	
						#endif
						
						temp >>= 8;
						Set_Duty((u16)(temp));
						Lin_Tx_Buffer[2] = (u8)(temp>>8); // send MSB
						Lin_Tx_Buffer[3] = (u8)(temp);    // send LSB

					#else
  						#error"Wrong Feedback Type Parameter!"	
					#endif
					}
					Chk_Motor_Stalled();    
   	      		if (timer_10ms == 0) PORTS_GreenLedOn();   
				
				if ( COMMAND == CMD_BRAKE ) 
				{
					State=STOP;	
				}
			break;

			case BRAKE: 
     	    			if (active_brake(Brake_Duty,Brake_Time) == TRUE) State = STOP;
				break;

			case STOP:
                    		MTC_StopMotor();
	    	    		PORTS_RedLedOn();
				State = IDLE;
				break;

			case FAULT:
		        default:
			        MTC_StopMotor();
	    	    		PORTS_RedLedOn();
				if ((u8)(GetMotorStatus() & FAULT_MSK) == 0) State = IDLE;
				break;
			}

    }
} // main loop

#endif

#endif

#if (MOTOR_TYPE==PMAC)

// PMAC 

// Standard types definitions and ST7 specific macros
#include "lib.h"
// General purpose peripheral function prototypes
#include "ports.h"
#include "adc.h"
#include "pwmart.h"
#include "sci.h"
#include "I2C.h"        // Emulated I2C
#include "ProtoSup.h"   // Emulated I2C 
// ST7 specifics related functions prototypes
#include "ST7_Misc.h"
// Motor control specific functions prototypes
#include "mtc.h"		/* Peripheral specifics */
#include "acmotor.h"	/* AC motor related */
#include "config.h"		/* AC motor library configuration keys */
#include "MainParam.h"	/* Motor control parameters used in functions called from main.c */
#include "PMACparam.h"  

// ST7FMC peripherals Hardware Registers declaration
#define __DEFINE_REGISTERS_STVD7_INCLUDE__   
#include "PMACparam.h"  
#if defined (ENABLE_RS232) && defined (__CSMC__)
	#include <string.h>
#endif 

///////////////////////////////////////////////////////////////////////////////

/* Private typedefs (for demo purposes only) */
typedef enum
{
IDLE, START, RUN, STOP, BRAKE, WAIT, FAULT
} SystStatus_t;

/* Private variable (for demo purposes only) */
static SystStatus_t State; 

#ifdef ENABLE_RS232  
	// WARNING: If RS232_SPEED and/or RS232_VOLTAGE are defined 
	// the UartBuffer array must be enlarged accordingly
	static u8 UartBuffer[32]; // Only Phase angle or Param Tuning trasmit is allowed with 16 byte of buffer
#endif

// Motor Status   
#define OVER_HEAT		((u8)0x01)
#define OVER_CURRENT		((u8)0x02)
#define OVER_VOLTAGE		((u8)0x04)
#define START_UP_FAILED	((u8)0x08)
#define MOTOR_STALLED		((u8)0x10)


/* Private functions (for demo purposes only) */
static void DoMotorControl( void );
static SystStatus_t StartMotor( void );
static void	CheckPowerStage(void);
static void	SendSystemDataRS232(void);  

////////////////////////////////////////////////


//////////////////* Private constants (for demo purposes only) *///////////////
// Set LED blinking rate (x10ms), refer to ART_SetMainTimeBase prototype
#define LED_BLINKING_RATE	((u8)10)		/* 10 x 10ms = 100ms */

#define IDLE_TIME		((u16)1000)	/* 1000 ms idle time (before allowing 
						motor re-start */   
#if (CONTROL == 0)
	static u8 Target_Curr;
#endif



volatile u8  Power_Motor_Status = ((u8)0);  //i2c
u16 TARGET_FREQ_CL;



void main(void)
{        
	              
	u16 Shift_Phase_tmp; // Temporary variable to calculate phase shift
	
	ST7_IntPrioritySetUp();
	PORTS_Init();
	ADC_Init();
	ART_Init();						// PWM ART provided general purpose time bases
	ACM_Init();
	#ifdef ENABLE_RS232
	//	Init SCI for hyperterminal: 38400bps/8 data bit/no parity/1 stop bit
	SCI_Init(SCI_DEFAULT_PARAM1,SCI_DEFAULT_PARAM2); 
	SCI_Select_Baudrate(SCI_PR_13 + SCI_TR_1 + SCI_RR_1);
	SCI_Mode(SCI_TX_ENABLE);
	#endif
	Init_I2C_Protocol();

	ART_SetMainTimeBase(LED_BLINKING_RATE);		// 100ms timebase for LED Toggle

	EnableInterrupts();

	ART_Start();					// Time bases are now available
	State = IDLE;
	Power_Motor_Status = 0;
    
	// Flash LEDs to indicate proper MCU start-up
	PORTS_GreenLedOn();
	while (!ART_IsMainTimeElapsed());
	PORTS_RedLedOn();
	while (!ART_IsMainTimeElapsed());
	PORTS_GreenLedOn(); 
	
	// Main Loop
	while(1)
	{       
	
		// Setting of Phase Angle
		#ifdef PHASE_ANGLE_RV3
			Shift_Phase_tmp = ((u16)(ADC_GetRV3()) << 8 ); // Selected By RV3     
		#else
		   	if (MTC_GetRotorFreq()<PHI_LOWFREQ)	
				Shift_Phase_tmp = (u16) PHI_MIN << 8;
                	else if (MTC_GetRotorFreq()<PHI_HIGHFREQ)	
                	{
                		Shift_Phase_tmp =  (s16)((coeff_phase*(MTC_GetRotorFreq()-PHI_LOWFREQ))/256 + PHI_MIN); 
                		Shift_Phase_tmp = Shift_Phase_tmp << 8;
			}                	
                	else Shift_Phase_tmp = (u16) PHI_MAX << 8;
		#endif
		LINK_protocol();
		
		// Read Set Speed Command "I2C Emulation"
		if (COMMAND==CMD_SETSPEED)
		{
			TARGET_FREQ_CL =(Valid_Data_In[1]<<8)+Valid_Data_In[2];
		}

		
		
		if (State == BRAKE)
			Shift_Phase = (u16)(Shift_Phase_tmp - 32768);
		else
			Shift_Phase = Shift_Phase_tmp;
	
	
	
		switch ( State )
		{
		case IDLE:	
			
			if ( COMMAND == CMD_START )	// Start motor
			{
				COMMAND = 0;
				Power_Motor_Status = 0;   
				ACM_InitSoftStart();
				State = START;  
			}
			break;

		case START:
			State = StartMotor(); 
			if (State == RUN)
			{             
				#if (CONTROL == 0)
				Target_Curr = (u8)ADC_GetRV1();
				ART_Set_TimeInMs(SLEW_LIMIT); 
				#endif
			}
			break;

		case RUN:	
			
			if ( COMMAND == CMD_BRAKE ) State=STOP;	// Start motor
			else	DoMotorControl();
			break;

		case STOP:
		{
			if (BRAKE_VOLTAGE == 0)
			{
				ART_SetTimeOutDuration(IDLE_TIME);
				MTC_DisableMCOutputs();			// PWM is disabled on MCO outputs
				State = WAIT;
			}
			else
			{   
				MTC_UpdateSine(0, MTC_GetRotorFreq());
				State = BRAKE;
			}
		}
		break;
                #if (BRAKE_VOLTAGE!=0)
		case BRAKE: 
			if (MTC_GetRotorFreq()>BRAKE_MIN_SPEED)
			{       
				MTC_Brake();
			}
			else
			{
				MTC_StopBraking();
				ART_SetTimeOutDuration(IDLE_TIME);
				State = WAIT;
			}
			break;
                #endif
		case WAIT:	
			if (ART_IsTimeOutElapsed())
			{        
				//MTC_Toggle_Direction(); /* Uncomment if you want to restart the motor in opposite direction */
				State = IDLE; 
			}
			break;

		case FAULT:	
		default:	
			MTC_DisableMCOutputs();			// PWM is disabled on MCO outputs
			State = FAULT;
			break;
		}

		// Verify that inverter is operating properly
		CheckPowerStage();
                
		// LED management
		if (ART_IsMainTimeElapsed())
		{
        	switch ( State )
			{
				case IDLE:	PORTS_GreenLedOn();
							break;
				case START:
				case RUN:	
				case STOP:
				case BRAKE:
				case WAIT:  PORTS_RedLedOn();
							break;
				case FAULT:	
				default:	PORTS_RedLedToggle();
							break;
			}
		}

		#ifdef ENABLE_RS232     
			SendSystemDataRS232();
		#endif 

	}	// End of while(1)
} // main loop


/*-----------------------------------------------------------------------------
ROUTINE Name : CheckPowerStage

Description : Verify that inverter power stage is running fine:
					heatsink temperature
					voltage bus value
					current limitation not triggerred
-----------------------------------------------------------------------------*/
void CheckPowerStage(void)
{
	if (ADC_CheckOverTemp())
	{
			State = FAULT;
			Power_Motor_Status |= OVER_HEAT;  //I2C 
	}
	
	if (ADC_CheckOverVoltage())
	{
			State = FAULT;
			Power_Motor_Status |= OVER_VOLTAGE;  //I2C 
	}
	
	if ((MTC_CheckEmergencyStop())&&(State!=FAULT))
	{
			State = FAULT;
			Power_Motor_Status |= OVER_CURRENT;  //I2C 
	}
}

/*-----------------------------------------------------------------------------
ROUTINE Name : SendSystemDataRS232

Description : upload internal data to the PC via RS232 and hyperterminal for 
debug purposes
-----------------------------------------------------------------------------*/
#ifdef ENABLE_RS232
void SendSystemDataRS232(void)
{
	if(SCI_IsTransmitCompleted())
	{		
		strcpy(UartBuffer, "");
		#ifdef PHASE_ANGLE_RV3
			strcat(UartBuffer, "PhAng=");
			strcatu16(UartBuffer, Shift_Phase >> 8);
			strcat(UartBuffer, " ");			
		#endif

		#ifdef PI_PARAM_TUNING
			strcat(UartBuffer, "Kp=");
                        strcatu16(UartBuffer, (u8)(ADC_GetRV2()));
		        
		        strcat(UartBuffer, " Ki=");
                        strcatu16(UartBuffer, (u8)(ADC_GetRV3()));
                        strcat(UartBuffer, " ");			
		#endif
                
                #ifdef RS232_SPEED  
                	strcat(UartBuffer, "ElFrq=");
                        strcatu16(UartBuffer, MTC_GetRotorFreq());
		        strcat(UartBuffer, " ");
		#endif
		
		#ifdef RS232_VOLTAGE            
			strcat(UartBuffer, "StVlt=");
                        strcatu16(UartBuffer, MTC_GetVoltage());
		        strcat(UartBuffer, " ");       
		#endif
		
		strcat(UartBuffer, "\n\r"); // Go to next line  
		
		SCI_PutString(UartBuffer);
	}
}
#endif	/* ENABLE_RS232 */

/*-----------------------------------------------------------------------------
ROUTINE Name : DoMotorControl

Description : Performs control of the AC motor: the stator frequency 
and voltage and updated according to trimmer reading.
-----------------------------------------------------------------------------*/
void DoMotorControl( void )
{
	u8 NewVoltage, TargetVolt;
	u16 NewFreq, TargetFreq;  
	
	if ( ART_IsRegPeriodElapsed() )
	{
		#if (CONTROL == 0)
        		// Open loop regulation
			TargetVolt = (u8)ADC_GetRV1();		
			if (Target_Curr > TargetVolt)
				Target_Curr--;
			else
		   	if (Target_Curr < TargetVolt)
		   		Target_Curr++;
	        	NewVoltage = Target_Curr;                     
		#endif
                                
		NewFreq = MTC_GetRotorFreq();
	        
		#if (CONTROL == 1)		
        		// Closed loop regulation
			NewVoltage = ACM_SpeedRegulation(TARGET_FREQ_CL,NewFreq);      
		#endif
		
		
		
	        // Frequency Synchronization
		if (NewFreq == 0)	
		{
        	
            COMMAND=CMD_BRAKE; 			//I2C Emulated
        	Power_Motor_Status |= MOTOR_STALLED;	//I2C Emulated
		}		
                   
        	MTC_UpdateSine(NewVoltage, NewFreq);
        }
}

#if (CONTROL == 0)
/*-----------------------------------------------------------------------------
ROUTINE Name : StartMotor

Description : Starts the motor in open loop
-----------------------------------------------------------------------------*/
SystStatus_t StartMotor( void )
{
	u8 NewVoltage;
	// NewVoltage = STATOR_VOLTAGE; // End startup when STATOR_VOLTAGE is reached
	NewVoltage = (u8)(ADC_GetRV1()); // End startup when RV1 Value is reached
        
	if ( ACM_SoftStartOL(NewVoltage) )
	{
		return(RUN);
	}
	else
	{
		return(START);
	}

}
#endif

#if (CONTROL == 1)
/*-----------------------------------------------------------------------------
ROUTINE Name : StartMotor

Description : Starts the motor in closed loop
-----------------------------------------------------------------------------*/
SystStatus_t StartMotor( void )
{
	StartStatus_t startStatus;        
        startStatus = ACM_SoftStart();
        if (startStatus == START_ONGOING)
        	return (START);
	if (startStatus == START_FAIL)
    {
		COMMAND = CMD_BRAKE;
		Power_Motor_Status |= START_UP_FAILED; //I2C
		return (FAULT);
	}

        if (startStatus == START_OK)
        {
       		
        	ACM_InitSpeedReg(TARGET_FREQ_CL);
        	return (RUN);
        }
        return (FAULT);        
}
#endif

#endif

#if (MOTOR_TYPE==AC)

// AC

// Standard types definitions and ST7 specific macros
#include "lib.h"
// General purpose peripheral function prototypes
#include "ports.h"
#include "adc.h"
#include "pwmart.h"
#include "sci.h"
//I2C Emulation
#include "I2C.h"     
#include "ProtoSup.h"

// ST7 specifics related functions prototypes
#include "ST7_Misc.h"
// Motor control specific functions prototypes
#include "mtc.h"		/* Peripheral specifics */
#include "acmotor.h"	/* AC motor related */
#include "config.h"		/* AC motor library configuration keys */
#include "Mainparam.h"	/* Motor control parameters used in functions called from main.c */
#include "MTCparam.h"	/* Min and max frequencies needed for trimmer working range set-up */
#if defined (ENABLE_RS232) && defined (__CSMC__)
	#include <string.h>
#endif


///////////////////////////////////////////////////////////////////////////////
///////////////////* Private typedefs (for demo purposes only) *///////////////
typedef enum
{
IDLE, START, RUN, STOP, BRAKE, WAIT, FAULT
} SystStatus_t;

//////////////////* Private functions (for demo purposes only) *///////////////
static void DoMotorControl( void );
static SystStatus_t StartMotor( void );
static void	CheckPowerStage(void);
#ifdef ENABLE_RS232
	static void	SendSystemDataRS232(void);
#endif

//////////////////* Private constants (for demo purposes only) *///////////////
// Set LED blinking rate (x10ms), refer to ART_SetMainTimeBase prototype
#define LED_BLINKING_RATE	((u8)10)		/* 10 x 10ms = 100ms */

#define IDLE_TIME				((u16)1000)	/* 1000 ms idle time (before allowing 
													motor re-start */

//////////////////* Private variable (for demo purposes only) *////////////////
static SystStatus_t State;
#ifdef ENABLE_RS232
	static u8 UartBuffer[16];
#endif
// Motor Status   
#define OVER_HEAT		((u8)0x01)
#define OVER_CURRENT		((u8)0x02)
#define OVER_VOLTAGE		((u8)0x04)
#define START_UP_FAILED	((u8)0x08)
#define MOTOR_STALLED		((u8)0x10)

volatile u8  Power_Motor_Status = ((u8)0);
u16 TARGET_FREQ_CL=1500;

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
/*-----------------------------------------------------------------------------
ROUTINE Name : main

Description : Demo program: refer to application note AN1904 for details.
-----------------------------------------------------------------------------*/
void main(void)
{

   ST7_IntPrioritySetUp();
	PORTS_Init();
	ADC_Init();
	ART_Init();						// PWM ART provided general purpose time bases
	ACM_Init();
	Init_I2C_Protocol();
	#ifdef ENABLE_RS232
		//	Init SCI for hyperterminal: 38400bps/8 data bit/no parity/1 stop bit
		SCI_Init(SCI_DEFAULT_PARAM1,SCI_DEFAULT_PARAM2); 
		SCI_Select_Baudrate(SCI_PR_13 + SCI_TR_1 + SCI_RR_1);
		SCI_Mode(SCI_TX_ENABLE);
	#endif
	

	ART_SetMainTimeBase(LED_BLINKING_RATE);		// Set Timebase for LED Toggle
   MTC_Set_CounterClockWise_Direction();			// This is to start CW in toggle mode
	EnableInterrupts();

	ART_Start();					// Time bases are now available
	State = IDLE;
	Power_Motor_Status = 0;
    
	// Flash LEDs to indicate proper MCU start-up
	PORTS_GreenLedOn();
	while (!ART_IsMainTimeElapsed());
	PORTS_RedLedOn();
	while (!ART_IsMainTimeElapsed());
	PORTS_GreenLedOn();

	// Main Loop
	while(1)
	{
		
	  LINK_protocol();
		
		// Read Set Speed Command
		if (COMMAND==CMD_SETSPEED)
		{
			TARGET_FREQ_CL=(Valid_Data_In[1]<<8)+Valid_Data_In[2];
		}

		switch ( State )
		{
			case IDLE:	
						
						if ( COMMAND == CMD_START )	// Start motor
						{  	
							u32 temp;
						
							COMMAND = 0;
							Power_Motor_Status = 0;
							
							#if DIRECTION == CW
								MTC_Set_ClockWise_Direction();
							#elif DIRECTION == CCW
								MTC_Set_CounterClockWise_Direction();
							#else
									MTC_Toggle_Direction();
							#endif
							
							#ifdef CLOSED_LOOP

								ACM_InitSoftStart(START_FREQ);

							#else	/* OPEN_LOOP: Read start-up frequency on trimmer */

								
								temp = (u8)(ADC_GetRV1());
								temp *= (u32)(HIGHEST_FREQ-LOWEST_FREQ);
								temp >>= 8;
								temp += LOWEST_FREQ;
								

								ACM_InitSoftStart_OL( TARGET_FREQ_CL );
							#endif
							State = START;
						}
						break;

			case START:	State = StartMotor();
						break;

			case RUN:	
						
						if ( COMMAND == CMD_BRAKE ) State=STOP;	// Start motor 
						
						else
						{
							DoMotorControl();
						}
						break;

			case STOP:	{
								u16 BrakingTorque;

							BrakingTorque = BRAKE_DUTY_CYCLE;
							if (BrakingTorque == 0)
							{
								ART_SetTimeOutDuration(IDLE_TIME);
								MTC_DisableMCOutputs();			// PWM is disabled on MCO outputs
								State = WAIT;
							}
							else
							{
								MTC_StartBraking( BrakingTorque );
								ART_SetTimeOutDuration(BRAKE_DURATION);
								State = BRAKE;
							}
						}
						break;

			case BRAKE: if (!ART_IsTimeOutElapsed())
						{
							MTC_Brake();
						}
						else
						{
							MTC_StopBraking();
							ART_SetTimeOutDuration(IDLE_TIME);
							State = WAIT;
						}
						break;

			case WAIT:	if (ART_IsTimeOutElapsed())
						{
							State = IDLE;
						}
						break;

			case FAULT:	
			default:	MTC_DisableMCOutputs();			// PWM is disabled on MCO outputs
						State = FAULT;
						break;
		}

		// Verify that inverter is operating properly
		CheckPowerStage();

		// LED management
		if (ART_IsMainTimeElapsed())
		{
        	switch ( State )
			{
				case IDLE:	PORTS_GreenLedOn();
							break;
				case START:
				case RUN:	
				case STOP:
				case BRAKE:
				case WAIT:  PORTS_RedLedOn();
							break;
				case FAULT:	
				default:	PORTS_RedLedToggle();
							break;
			}
		}

		#ifdef ENABLE_RS232
			SendSystemDataRS232();
		#endif

	}	// End of while(1)
} // main loop


/*-----------------------------------------------------------------------------
ROUTINE Name : CheckPowerStage

Description : Verify that inverter power stage is running fine:
					heatsink temperature
					voltage bus value
					current limitation not triggerred
-----------------------------------------------------------------------------*/
void CheckPowerStage(void)
{
	if (ADC_CheckOverTemp())
	{
			State = FAULT;
			Power_Motor_Status |= OVER_HEAT;  //I2C 
	}
	
	if (ADC_CheckOverVoltage())
	{
			State = FAULT;
			Power_Motor_Status |= OVER_VOLTAGE;  //I2C 
	}
	
	if ((MTC_CheckEmergencyStop())&&(State!=FAULT))
	{
			State = FAULT;
			Power_Motor_Status |= OVER_CURRENT;  //I2C 
	}
}

/*-----------------------------------------------------------------------------
ROUTINE Name : SendSystemDataRS232

Description : upload internal data to the PC via RS232 and hyperterminal for 
debug purposes
-----------------------------------------------------------------------------*/
#ifdef ENABLE_RS232
void SendSystemDataRS232(void)
{
	#ifdef CLOSED_LOOP

		if(SCI_IsTransmitCompleted())
		{
				u32 temp;

			strcpy(UartBuffer, "S=");

			temp = (u8)(ADC_GetRV1());
			temp *= (u32)(HIGHEST_FREQ-LOWEST_FREQ);
			temp >>= 8;
			temp += LOWEST_FREQ;
			strcatu16(UartBuffer, (u16)temp );	// This is the target speed read on trimmer

			strcat(UartBuffer, " R=");		// Rotor frequency
			strcatu16(UartBuffer, MTC_GetRotorFreq());

// WARNING: If the following lines are uncommented, the UartBuffer array must be enlarged accordingly
//		  	strcat(UartBuffer, " V=");		// Voltage
//			strcatu16(UartBuffer, MTC_GetVoltage());

//			strcat(UartBuffer," Slp=");		// SlipCmd
//			strcatu16(UartBuffer, ACM_GetOptimumSlip( MTC_GetStatorFreq() ));

			#ifdef PI_PARAM_TUNING
//				strcat(UartBuffer, " Ki=");
//				strcatu16(UartBuffer, Ki);
//				strcat(UartBuffer, " Kp=");
//				strcatu16(UartBuffer, Kp);
			#endif

			strcat(UartBuffer, "\n\r");

			SCI_PutString(UartBuffer);
		}

	#else		/* OPEN_LOOP */

		if(SCI_IsTransmitCompleted())
		{
				u32 temp;
				
			strcpy(UartBuffer, "S=");
			strcatu16(UartBuffer, MTC_GetStatorFreq() );	// This is the target speed read on trimmer
			strcat(UartBuffer, " R=");		// Rotor frequency
			strcatu16(UartBuffer, MTC_GetRotorFreq());
			strcat(UartBuffer, "\n\r");
			SCI_PutString(UartBuffer);
		}

	#endif	/* OPEN_LOOP */
}
#endif	/* ENABLE_RS232 */


/* ####################### CLOSED LOOP OPERATION ######################### */
/* ####################### CLOSED LOOP OPERATION ######################### */
/* ####################### CLOSED LOOP OPERATION ######################### */
/* ####################### CLOSED LOOP OPERATION ######################### */
/* ####################### CLOSED LOOP OPERATION ######################### */
#ifdef CLOSED_LOOP
/*-----------------------------------------------------------------------------
ROUTINE Name : DoMotorControl

Description : Performs closed loop control of the AC motor: the stator frequency 
and voltage and updated according to target Speed read on RV1 trimmer.
-----------------------------------------------------------------------------*/
void DoMotorControl( void )
{
		u16 FreqCmd, StatorFreq;
		u8 SlipCmd;


	if (MTC_GetRotorFreq() == 0)
	{
			COMMAND=CMD_BRAKE;	// i2c
        	Power_Motor_Status |= MOTOR_STALLED;	// i2c
		
		State = FAULT;
	}

	if ( ART_IsRegPeriodElapsed() )
	{
			u8 NewVoltage;
			u32 temp;

		// Get current Stator frequency
		StatorFreq  = MTC_GetStatorFreq();

		temp = (u8)(ADC_GetRV1());
		temp *= (u32)(HIGHEST_FREQ-LOWEST_FREQ);
		temp >>= 8;
		temp += LOWEST_FREQ;
		FreqCmd = (u16)(TARGET_FREQ_CL);

		#ifdef PI_PARAM_TUNING
			Ki = ADC_GetRV2();
			Kp = ADC_GetRV3();
		#endif
		
		SlipCmd = ACM_GetOptimumSlip( StatorFreq );

		NewVoltage = ACM_SlipRegulation( SlipCmd );

		if (( (FreqCmd+SlipCmd) > StatorFreq) && (MTC_GetSlip() < ACCEL_SLIP_LIMIT))
		{
			StatorFreq++;
		}

		if (( (FreqCmd+SlipCmd) < StatorFreq) && (MTC_GetSlip() > DECEL_SLIP_LIMIT))
		{
			StatorFreq--;
		}

		MTC_UpdateSine(NewVoltage, StatorFreq);
		
	}  /* End of if: regulation performed */

}

/*-----------------------------------------------------------------------------
ROUTINE Name : StartMotor

Description : Starts the motor in closed loop: if a minimum speed has not been
reached after a given period, the start-up is considered as failed.
-----------------------------------------------------------------------------*/
SystStatus_t StartMotor( void )
{
		SystStatus_t NextState;

	switch ( ACM_SoftStart(MIN_START_FREQ) )
	{
		case START_FAIL:
							NextState = STOP;
							COMMAND = CMD_BRAKE;	//i2c
							Power_Motor_Status |= START_UP_FAILED; //i2c
							break;

		case START_OK:	
							ACM_InitSlipFreqReg( ACM_GetOptimumSlip( MTC_GetStatorFreq() ) );
							NextState = RUN;
							break;

		case START_ONGOING:	
							NextState = START;
							break;

		default:			NextState = FAULT;
							break;
	}

	return ( NextState );
}
#endif

/* ####################### OPEN LOOP OPERATION ######################### */
/* ####################### OPEN LOOP OPERATION ######################### */
/* ####################### OPEN LOOP OPERATION ######################### */
/* ####################### OPEN LOOP OPERATION ######################### */
/* ####################### OPEN LOOP OPERATION ######################### */
#ifdef OPEN_LOOP
/*-----------------------------------------------------------------------------
ROUTINE Name : DoMotorControl

Description : Performs open loop control of the AC motor: the stator frequency 
and voltage and updated according to trimmer reading.
-----------------------------------------------------------------------------*/
void DoMotorControl( void )
{
		u8 NewVoltage, VoltCmd, VMax;
		u16 NewFreq, SlipFreq, FreqCmd;
		u32 temp;

	if ( ART_Is_TimeInMsElapsed() )
	{

		
		temp = (u8)(ADC_GetRV1());
		temp *= (u32)(HIGHEST_FREQ-LOWEST_FREQ);
		temp >>= 8;
		temp += LOWEST_FREQ;
		FreqCmd = (u16)(temp);
		
		VoltCmd = ADC_GetRV2();

		// Get actual values of volt and Freq
		NewVoltage = MTC_GetVoltage();
		NewFreq = MTC_GetStatorFreq();

		SlipFreq = MTC_GetSlip();

		VMax = ACM_VoltageMaxAllowed(NewFreq);
		if ( VoltCmd > VMax )
		{
			VoltCmd = VMax;
		}

		if ( VoltCmd > NewVoltage ) NewVoltage++;	// Smoothly increase voltage
		if ( VoltCmd < NewVoltage ) NewVoltage--;	// Smoothly decrease voltage

		if ( FreqCmd > NewFreq ) NewFreq++;		// Smoothly increase frequency
		// Smoothly decrease frequency unless RotorFreq>=StatorFreq (risk of reactive current)
		// if no sensor available, RotorFreq will be 0 and SlipFreq=StatorFreq
		if ( (SlipFreq != 0) && ( FreqCmd < NewFreq ) ) NewFreq--;

		MTC_UpdateSine(NewVoltage, NewFreq);

	}

}

/*-----------------------------------------------------------------------------
ROUTINE Name : StartMotor

Description : Starts the motor in open loop: with the stator frequency read 
on the trimmer, increase the stator voltage from 0 to the limit set by the 
V/f curve. When Voltage reaches this limit the routine returns TRUE.
-----------------------------------------------------------------------------*/
SystStatus_t StartMotor( void )
{
		u8 StartVolt, MaxVolt;
		u16 StartFreq;
		u32 temp;

	// Read Trimmers
	// StartFreq = (ADC_GetRV1 * (HIGHEST_FREQ-LOWEST_FREQ))/256 + LOWEST_FREQ)
	temp = (u8)(ADC_GetRV1());
	temp *= (u32)(HIGHEST_FREQ-LOWEST_FREQ);
	temp >>= 8;
	temp += LOWEST_FREQ;
	StartFreq = (u16)temp;
	
	StartVolt = ADC_GetRV2();

	// Verify that Start-up voltage is in line with V/f characteristic
	MaxVolt = ACM_VoltageMaxAllowed(StartFreq);
	if (StartVolt > MaxVolt)
	{
		StartVolt = MaxVolt;
	}
	
	if ( ACM_SoftStartOL(StartVolt) )
	{
		return(RUN);
	}
	else
	{
		return(START);
	}

}
#endif	/* OPEN_LOOP */

#endif

/******************* (C) COPYRIGHT 2006 STMicroelectronics *****END OF FILE****/
