/******************** (C) COPYRIGHT 2005 STMicroelectronics ********************
* File Name          : E3PROM.c
* Author             : MCD Application Team
* Date First Issued  : 01/06/2005
* Description        : This file provides all the EEPROM software functions
********************************************************************************
* History:
*  01/06/2005 : V1.0
*******************************************************************************
 THE PRESENT SOFTWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS WITH
 CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME.
 AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT
 OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE CONTENT
 OF SUCH SOFTWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING INFORMATION
 CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
*******************************************************************************/

#include "E3PROM.h"


/*******************************************************************************
* Function Name  : EEPROM_Init
* Description    : Initialize the EEPROM 
* Input          : None
* Return         : None
*******************************************************************************/
void EEPROM_Init(void)
{
  /* Initialize the Flash */
  FLASH_Init();
  /* Erase the sectors B0F0 & B0F1 */
  FLASH_SectorErase(FLASH_B1F0|FLASH_B1F1);
  /* Mark the Sector B0F0 as valid */
  FLASH_WordWrite(ValidSectorAddress_1,0x600D);
}

/*******************************************************************************
* Function Name  : EERPOM_GetValidSector
* Description    : Get the valid sector 
* Input          : None
* Return         : The valid sector 
*******************************************************************************/
u32 EEPROM_GetValidSector(void)
{
  if (ValidSector1 == 0x600D)
  {
    return EEPROM_SECTOR_1_VALID;        /* Return 0 if the valid sector is the B0F0 */
  }
  else if (ValidSector2 == 0x600D)       /* Return 1 if the valid sector is the B0F1 */
  {
    return EEPROM_SECTOR_2_VALID;
  }
  else
  {
    return EEPROM_FULL_ERASE_REQUIRED;   /* Return 2 if no valid sector (a full erase is required) */
  }
}

/*******************************************************************************
* Function Name  : EEPROM_ValidDataSet
* Description    : Get the valid data set 
* Input 1        : The valid sector
* Input 2        : The valid data set
* Return         : The status of the data set
*******************************************************************************/
u32 EEPROM_ValidDataSet(vu32 Valid_Sector,vu32 *Valid_Data_Set)
{
  vu32 Valid_Data_Count;
  vu32 Word_Count=0;
  vu32 Address;
  
  switch (Valid_Sector)
  {
    case EEPROM_SECTOR_1_VALID :

         /* Test if the first sector is empty */ 
         if (*(vu32*)Valid_Region_1 == 0xFFFFFFFF)
           return (EEPROM_NO_DATA_SET);   /* Return an eeprom no data set status */
         else
         {
           /* Initialize the Address of the data set */
           Address = Valid_Region_1;

           /* Find the Valid Data */
           for (Valid_Data_Count = 1;Valid_Data_Count <= Max_Count_1;Valid_Data_Count++)
           {
             /* Increment the Word Counter */
             Word_Count++;

             /* Select the correspondent Word */
             if (!(Valid_Data_Count % 32))
             {
               Address -= 4;       /* Decrement the Address */
               Word_Count = 0;     /* Reset the Word Counter */
             }

             /* Find the First "01" */
             if ((*(vu32*)Address)&(0x1<<Word_Count))
             {
               /* Valid data set */
               *Valid_Data_Set = Valid_Data_Count;
               /* Return a successful data set status */
               return (EEPROM_SUCCESSFUL_DATA_SET);
             }
           }
         }
    case EEPROM_SECTOR_2_VALID :

         /* Test if the first sector is empty */ 
         if (*(vu32*)Valid_Region_2 == 0xFFFFFFFF)
           return (EEPROM_NO_DATA_SET);    /* Return an eeprom no data set status */
         else
         {
           /* Initialize the Address of the data set */
           Address = Valid_Region_2; 

           /* Find the Valid Data */
           for (Valid_Data_Count = 1;Valid_Data_Count <= Max_Count_2;Valid_Data_Count++)
           {
             /* Increment the Word Counter */
             Word_Count++;

             /* Select the correspondent Word */
             if (!(Valid_Data_Count % 32))
             {
               Address -= 4;       /* Decrement the Address */
               Word_Count = 0;     /* Reset the Word Counter */
             }

             /* Find the First "01" */
             if ((*(vu32*)Address)&(0x1<<Word_Count))
             {
               /* Valid data set */
               *Valid_Data_Set = Valid_Data_Count;
               /* Return a successful data set status */
               return (EEPROM_SUCCESSFUL_DATA_SET);
             }
           }
         }  
  }
  /* No valid sector ==> Return a full erase status */
  return (EEPROM_FULL_ERASE_REQUIRED);      
}

/*******************************************************************************
* Function Name  : EEPROM_DataRead
* Description    : Read the valid data 
* Input          : The Data
* Return         : The status of the read operation
*******************************************************************************/
u32 EEPROM_DataRead(vu32 *Data)
{
  vu32 Valid_Data_Set;
  vu32 Valid_Sector;
  vu32 Address = 0x400C0000;
  vu32 Status;

  /* Get the valid Sector */
  Valid_Sector = EEPROM_GetValidSector();
  
  /* Check for the Valid Sector status */
  if (Valid_Sector == EEPROM_FULL_ERASE_REQUIRED)
    return(EEPROM_FULL_ERASE_REQUIRED);

  /* Get the valid data-set */
  Status = EEPROM_ValidDataSet(Valid_Sector,&Valid_Data_Set);

  /* Check for the data set */
  if (Status == EEPROM_NO_DATA_SET)
    return(EEPROM_NO_VALID_DATA);
  else
  {
    /* Calculate the valid address */
    Address += ((Valid_Data_Set) * 4) + (Valid_Sector * 0x2000) - 4;

    /* Update the data */
    *Data = *(vu32*)Address;
  
    /* Return a valid status */
    return EEPROM_SUCCESSFUL_DATA_READ;
  }
}

/*******************************************************************************
* Function Name  : EEPROM_DataWrite
* Description    : Write the new data
* Input          : The data to be written
* Return         : The status of the write operation
*******************************************************************************/
u32 EEPROM_DataWrite(vu32 Data)
{
  vu32 Valid_Data_Set;
  vu32 Valid_Sector;
  vu32 Address;
  vu32 Status;

  /* Get the valid Sector */
  Valid_Sector = EEPROM_GetValidSector();

  /* Check for the Valid Sector status */
  if (Valid_Sector == EEPROM_FULL_ERASE_REQUIRED)
    return EEPROM_FULL_ERASE_REQUIRED;

  /* Get the valid data-set */
  Status = EEPROM_ValidDataSet(Valid_Sector,&Valid_Data_Set);

  /* Check for the data set */
  if (Status == EEPROM_NO_DATA_SET)
  {
    Valid_Data_Set = 0;
    Address = Valid_Sector * 0x2000;
  }
  else
    /* Calculate the valid address */
    Address = (Valid_Data_Set * 4) + (Valid_Sector * 0x2000);

  /* Check for data transfer */
  switch(Valid_Sector)
  {
    case EEPROM_SECTOR_1_VALID:
       if(Address >= Max_Data_Set_1)
       {
         /* Write the last data to sector B0F1 */
         FLASH_WordWrite(Offset_Sector_2,*(vu32*)(0x400C0000 + Address - 4));
         /* Write the new data to sector B0F1 */
         FLASH_WordWrite(Offset_Sector_2 + 4,Data);
         /* Update the data set of the sector B0F1 */
         FLASH_WordWrite(Valid_Region_2,0xFFFFFFFC);
         /* Mark the sector B0F1 as valid */
         FLASH_WordWrite(ValidSectorAddress_2,0x600D);
         /* Erase the sector B0F0 */
         FLASH_SectorErase(FLASH_B1F0);
         return (EEPROM_SUCCESSFUL_DATA_WRITE);
       }

    case EEPROM_SECTOR_2_VALID:
       if(Address >= Max_Data_Set_2)
       {
         /* Write the last data to sector B0F0 */
         FLASH_WordWrite(Offset_Sector_1,*(vu32*)(0x400C0000 + Address - 4));
         /* Write the new data to sector B0F0 */
         FLASH_WordWrite(Offset_Sector_1 + 4,Data);
         /* Update the data set of the sector B0F0 */
         FLASH_WordWrite(Valid_Region_1,0xFFFFFFFC);
         /* Mark the sector B0F0 as valid */
         FLASH_WordWrite(ValidSectorAddress_1,0x600D);
         /* Erase the sector B0F1 */
         FLASH_SectorErase(FLASH_B1F1);
         return (EEPROM_SUCCESSFUL_DATA_WRITE);
       }
  }
  /* Write the new Data */
  FLASH_WordWrite(0x400C0000 + Address,Data);
  /* Update the data set */
  EEPROM_UpdateDataSet(Valid_Data_Set,Valid_Sector);
  /* Return Successful data write state */
  return (EEPROM_SUCCESSFUL_DATA_WRITE);
}

/*******************************************************************************
* Function Name  : EEPROM_UpdateDataSet
* Description    : Update the data set
* Input  1       : The current valid data set  
* Input  2       : The valid sector
* Return         : None
*******************************************************************************/
void  EEPROM_UpdateDataSet(vu32 Valid_Data_Set, vu32 Valid_Sector)
{
  vu32 Word_Number;
  vu32 Word_Offset;
  vu32 Address;
  vu32 Data;
  /* Calculate the Word number and the offset */
  Word_Number = Valid_Data_Set / 32;
  Word_Offset = Valid_Data_Set % 32;
  if (Valid_Sector)
  {
    /* Calclulate the address */
    Address = Valid_Region_2 - (Word_Number * 4);
    Data=(~(0x1<<Word_Offset))&(0xFFFFFFFF<<Word_Offset);
    /* Update the correspondent data set */
    FLASH_WordWrite(0x40000000 + Address,Data); 
  }
  else
  {
    /* Calclulate the address */
    Address = Valid_Region_1 - (Word_Number * 4);
    Data=(~(0x1<<Word_Offset))&(0xFFFFFFFF<<Word_Offset);
    /* Update the correspondent data set */
    FLASH_WordWrite(0x40000000 + Address,Data); 
  }
}

/******************* (C) COPYRIGHT 2005 STMicroelectronics *****END OF FILE****/
