
/***************** (c) 2006  STMicroelectronics, Inc. **********************/

/**************************************************************************
     PROGRAM  : SENSORLESS CONTROL OF 3 PHASE BLDC MOTOR WITH INNER
                CURRENT LOOP
     COMPILER : COSMIC
     FILENAME : MOTCON.C
     RELEASE  : 1.0
     DATE     : 3 Jan 2006
     AUTHOR   : Power Systems Application Lab
                STMicroelectronics, Inc.
                Schaumburg, IL - 60173

***************************************************************************

 DISCLAIMER :
 ============
 THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. ST MICROELECTRONICS
 SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
 WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
***************************************************************************


      *******************************************************
      *   This file contains                                *
      *      - all motor control function definitions       *
      *        and its associated ISRs                      *
      *******************************************************

MODIFICATIONS :

/******************************************************************************
... Test status: .............................................................*
******************************************************************************/

#define _MOTCON_C

#include "include.inc"

// ramp table column names definition
#define  RATIO          0
#define  COUNT          1

/*****************************************************************************/
/***********************  GLOBAL VARIABLES DEFINITIONS  **********************/
/*****************************************************************************/
u8
//    ramp[32][2],  /* ramp table - stores step [ratio] & [time] for start up */
    mem_ratio,  /* mem of ratio		 */
    mem_mprsr,  /* mem of prescaler and sampling register */
    nbemf,      /* number of bemf=0 observed. */
    memnbemf,   /* mem for test of successive bemf */
    sync_step,  /* number of active sync comutation step during start up */
    phasestep,  /* phase step from 0 to 5. */

    /* FLAGS
    **********/
    Status_Step; /* Status register for step behaviour during start up */

/************************************************************************
*************************************************************************/

/* General Flag Variables
**************************/
u8    key1count = DEBOUNCE_TIME_COUNT,
      flag,               // on / off command
      crntFlag,           // flag to hold reading curRef until startup complete
      starting_failed;    // motor running condition flag

/* Speed feedback calc Variables
*********************************/
u8    ratio0,
      ratio1,
      count0,
      count1,
      spdSample;

/* Speed Loop Variables
*************************/
s16        speedRef,
           speedFb,
           speedFb_temp,
           speedError,
           speedErrorPrev;

sQUAD_8    PIconSpeed,
           productSpeed;

const u16  Ka_spdLoop = (Kp_spdLoop + Ki_spdLoop*Ts_spdLoop)*0xffff,  // Q16
           Kb_spdLoop =  Kp_spdLoop * 0xffff ;   // Q16
           maxCurRef = MAX_CURRENT_REF,
           minCurRef = MIN_CURRENT_REF;

/* Current Loop Variables
***************************/
s16        currentRef,
           currentFb,
           curError,
           curErrorPrev;

sQUAD_8    PIconCur,
           productCur;

const u16  Ka_curLoop = (Kp_curLoop + Ki_curLoop*Ts_curLoop)*0xffff,   // Q16
           Kb_curLoop =  Kp_curLoop * 0xffff ;   // Q16
           max_duty = MAX_DUTY_CYCLE,
           min_duty = MIN_DUTY_CYCLE;
/*****************************************************************************/
/*************************  CONSTANTS  - ST72MC  *****************************/
/*****************************************************************************/
/* MPHST register values from step 1 to step 6 for bldc sensorLESS motor */
const u8 phreg_SL[] = {
               /* MPHST  ===>>>  IS1 IS0 T5 T4 T3 T2 T1 T0   */
	0x89,  /* 10001001	;T1-T4 - Valid Comparator C  */
	0x61,  /* 01100001	;T1-T6 - Valid Comparator B  */
	0x24,  /* 00100100	;T3-T6 - Valid Comparator A  */
	0x86,  /* 10000110	;T3-T2 - Valid Comparator C  */
	0x52,  /* 01010010	;T5-T2 - Valid Comparator B  */
	0x18   /* 00011000	;T5-T4 - Valid Comparator A  */
	};

/* MCRB reg values from steps 1 to 6 for bldc sensorLESS motor */
/* CPB, HDM, SDM and OS2 bits*/
/* MCRB ===>>> -0-  CPB  HDM  SDM  OCV  OS2  OS1  OS0
               - CPB=1 0->1 !! CPB=0 1->0 is BEMF on selected phase
	       - OS2=0 not alternate, OS2=1 Alternate during demagnetisation
	       - EVEN group= high side switches, ODD group=low side switches
               - SD only without forced demag
               - Simulated demagn with PWM on EVEN switch */

const u8 volt_crbreg_SL[] = {
    0x10,  /* 00010000 ;CPB=0 Simulated demagn  */
    0x50,  /* 01010000 ;CPB=1 Simulated demagn  */
    0x10,  /* 00010000 ;CPB=0 Simulated demagn  */
    0x50,  /* 01010000 ;CPB=1 Simulated demagn  */
    0x10,  /* 00010000 ;CPB=0 Simulated demagn  */
    0x50   /* 01010000 ;CPB=1 Simulated demagn  */
    };

//Ramp Table from Motor Controller Setup File

const u8 ramp[][2] =
{
/* {Number   Value}      alignment      */
      { 10,200 },  /*   ;0  	 */
      {  9,142 },  /*	;1	 */
      {  9,114 },  /*	;2	 */
      {  9, 95 },  /*	;3	 */
      {  9, 82 },  /*	;4	 */
      {  9, 72 },  /*	;5	 */
      {  9, 64 },  /*	;6	 */
      {  9, 57 },  /*	;7	 */
      {  8,104 },  /*	;8	 */
      {  8, 96 },  /*	;9	 */
      {  8, 88 },  /*	;10	 */
      {  8, 82 },  /*	;11	 */
      {  8, 76 },  /*	;12	 */
      {  8, 72 },  /*	;13	 */
      {  8, 68 },  /*	;14	 */
      {  8, 64 },  /*	;15	 */
      {  8, 60 },  /*	;16	 */
      {  8, 56 },  /*	;17	 */
      {  8, 54 },  /*	;18	 */
      {  7,104 },  /*	;19	 */
      {  7,100 },  /*	;20	 */
      {  7, 96 },  /*	;21	 */
      {  7, 92 },  /*	;22	 */
      {  7, 88 },  /*	;23	 */
      {  7, 84 },  /*	;24	 */
      {  7, 80 },  /*	;25	 */
      {  7, 76 },  /*	;26	 */
      {  7, 76 },  /*	;27	 */
      {  7, 72 },  /*	;28	 */
      {  7, 72 },  /*	;29	 */
      {  7, 68 },  /*	;30	 */
      {  7, 68 },  /*	;31	 */
      {  7, 64 },  /*	;32	 */
      {  7, 64 },  /*	;33	 */
      {  7, 60 },  /*	;34	 */
      {  7, 60 },  /*	;35	 */
      {  7, 56 },  /*	;36	 */
      {  7, 56 },  /*	;37	 */
      {  7, 56 },  /*	;38	 */
      {  7, 52 },  /*	;39	 */
      {  7, 52 },  /*	;40	 */
      {  7, 52 },  /*	;41	 */
      {  7, 48 },  /*	;42	 */
      {  7, 48 },  /*	;43	 */
      {  7, 48 },  /*	;44	 */
      {  7, 48 },  /*	;45	 */
      {  7, 44 },  /*	;46	 */
      {  7, 44 },  /*	;47	 */
      {  7, 44 },  /*	;48	 */
      {  7, 44 },  /*	;49	 */
      {  7, 44 },  /*	;50	 */
      {  7, 40 },  /*	;51	 */
      {  7, 40 },  /*	;52	 */
      {  7, 40 },  /*	;53	 */
      {  7, 40 },  /*	;54	 */
      {  7, 40 },  /*	;55	 */
      {  7, 39 },  /*	;56	 */
      {  7, 39 },  /*	;57	 */
      {  7, 39 },  /*	;58	 */
      {  7, 39 },  /*	;59	 */
      {  7, 39 },  /*	;60	 */
      {  7, 39 },  /*	;61	 */
      {  7, 39 },  /*	;62	 */
      {  7, 38 },  /*	;63	 */
      {  7, 38 }   /*   ;64      */
};

/****************************************************************************/
/*****************************  SUBROUTINES  ********************************/
/****************************************************************************/

/*************************
 calculate ramp-up table
**************************
void ramp_cal (void)	{
u8           step,
             accelration;
u16          denominator;
uDOUBLE_8    TEMP;

  ramp[0][RATIO] = 10;    //alignment prescaler
  ramp[0][COUNT] = 200;   //alignment timer

  step = 1;
  accelration = 78;	//ADCDR;  // 90
  denominator = accelration + 250; //base frequency is 5hz , (number must >130)
  while (step<32)  {
    ramp[step][RATIO] = 9;
    TEMP.word     = 32552/denominator;
    if (TEMP.byte.b0 < 4)
      TEMP.byte.b0 = 4;
    ramp[step++][COUNT] = TEMP.byte.b0;
    denominator     = denominator + accelration;
  }
  return;
}

/***********************************************************
; stop the motor
;***********************************************************/
void stop (void)	{
  sim();
  //MCPUH = 1;        // set PWM to 0%
  MCRA  = MOE | CKE | DAC;  // Output and clok enabled  ;direct access
                            // voltage mode selected  ;commutated mode
  MPHST  = 0;       // all driver OFF
  wait_1ms(1200);   // wait 1.2s before refresh
  MCsoft_reset();
  MCU_reset();
  while (1);
}


/*********************************************************************
Initialise 72MC for BLDC sensorless control
**********************************************************************/
void MCinit_bldc(void)  {

  // Initialize registers in page 1 first
  // *************************************
  bitSet(MCFR, RPGS);

  // MDTG  ===>>>   PCN  DTE  DTG[5:4:3:2:1:0]
  // RESET STATE -   0    0      [0 0 0 0 0 0]
  MDTG = MDTG_init;

  // MPOL  ===>>>   ZVD  REO  OP[5:4:3:2:1:0]
  // RESET STATE -   0    0     [1 1 1 1 1 1]
  MPOL = 0x3f;               // Switch driver input polarity (L6386D)
                             // ZVD=0; Z and D have opposite edge
                             // REO=0; read Z on PWM off of even channel

  // MPWME  ===>>>   DG  PWMW  PWMV  PWMU  OT[3:2:1:0]
  // RESET STATE  -  0    0     0     0      [0 0 0 0]
  MPWME = MPWME_init;

  // MCONF  ===>>>   DS[3:2:1:0]  SOI  SOM  XT16  XT8
  // RESET STATE  -    [0 0 0 0]   0    0    1     0
  MCONF = MCONF_init;

  // MPAR  ===>>>  TES1  TES0  OE5  OE4  OE3  OE2  OE1  OE0
  // RESET STATE  -  0     0     0    0    0    0    0    0
  MPAR = OE5 + OE3 + OE1;       // T5 T3 T1 == ODD
                                // T4 T2 T0 == EVEN

  // MZFR  ===>>>  ZEF[3:2:1:0]  ZWF[3:2:1:0]
  // RESET STATE  -    0 0 0 0       1 1 1 1
  MZFR = MZFR_init;

  // MSCR  ===>>>  ZSV  -  -  -  SCF1  SCF0  ECM  DISS
  // RESET STATE  - 0   0  0  0    0     0    0     0
  MSCR = MSCR_init;
                               // D and Z counter sampling clk = 250KHz
                               // Unused MCIx pins are HiZ

  // Initialize registers in page 0
  // ********************************
  bitClr(MCFR, RPGS);

  // MCRC  ===>>>  SEI  HZ  SZ  SC  SPLG  VR2  VR1  VR0
  // RESET STATE  - 0   0   0    0    0     0   0    0
  MCRC = MCRC_low_duty_cycle;

  // MDFR  ===>>>  DEF[3:2:1:0]  DWF[3:2:1:0]
  // RESET STATE  -    0 0 0 0       1 1 1 1
  MDFR = MDFR_init;

  // MCFR  ===>>>  RPGS  RST  CFF2  CFF1  CFF0  CFW2  CFW1  CFW0
  // RESET STATE  -  0    0     0     0     0     0     0     0
  MCFR = MCFR_init;

  // MREF  ===>>>  HST  CL  CFAV  HFE1  HFE0  HFRQ2  HFRQ1  HFRQ0
  // RESET STATE  - 0   0    0     0     0      0      0      0
  MREF = MREF_init;

  /* PWM carrier initialisation
  ******************************/
  // MPCR  ===>>>  PMS  OFLU  OFLV  OFLW  CMS  PCP2  PCP1  PCP0
  // RESET STATE  - 0    0     0     0     0    0     0     0
  MPCR  = MPCR_init;
  MCP0L = Carrier_mag % 256;	   // this value decides both ....
  MCP0H = Carrier_mag / 256;     //    ... carrier mag and freq
  MREP  = 16 - 1;                // Tupdate = 16 * Carrier_mag * Tclk

  return;
}

/***********************************************************************
OPEN LOOP - start the motor from zero speed
***********************************************************************/
void start_from_zero (void)	{
u8          bemfcpt;
uDOUBLE_8   uTEMP;

  MCsoft_reset();          // reset of all the peripheral registers
  MCinit_bldc();
  //ramp_cal();              // calculate the ramp table

  /* variables initialisation
  *****************************/
  nbemf = memnbemf = Status_Step = 0;        // flag initialization
  sync_step = phasestep = 0;

  /* bootstrap charging
  ***********************/
  MCRA  = DAC;       // direct access
  MPHST = 0;         // no output "ON"
  MCRA  = MOE | DAC; // output enabled
  wait_100us();
  MPHST = 0x2a;      // on-off of low side
  wait_100us();
  MPHST = 0;	       // put "OFF" all power switch
  wait_1ms(1);       // all drivers off during 1mS to avoid short ckt

  /* register initialisation
  ***************************/
  mem_ratio = ramp[0][RATIO];    // get ratio from ramp table for alignment
  mem_mprsr = mem_ratio;         // memo starting Step Ratio
  MPRSR     = mem_ratio;         // load MPRSR with first starting value
  MCOMP     = ramp[0][COUNT];    // select step time from ramp
  MDREG     = MCOMP >> 2;        // demag time = step time / 4

  // MCRA  ===>>>  MOE  CKE  SR  DAC  V0C1  SWA  PZ  DCB
  // RESET STATE  - 0    0   0    0    0     0   0    0

  // MCRB  ===>>>   -   CPB  HDM  SDM  OCV  OS2  OS1  OS0
  // RESET STATE  - 0    0    0    0    0     0   0    0

  /* alignment initialisation
  *****************************/
  MPHST = phreg_SL[0];           //output alignment configuration T1-T4
  MCRB  = volt_crbreg_SL[0];     //demag type and z edge settings

  uTEMP.word = (start_duty << 3);  //starting dutycycle
  MCPUL = uTEMP.byte.b0;           //..start with a lower current
  MCPUH = uTEMP.byte.b1;           //..for vibration free alignment
  MCRA  = MOE | CKE | Z_select;
                           //enable output; start clk;  sensorless; volt mode
                           //DAC=0;--> MPHST xfer to MCOx on C
                           //Z previous or Z actual for computation

  /* aligment current ramp up
  *****************************/
  for (bemfcpt = n_align_step;   bemfcpt-- ; ) {
    MISR = 0;                        //reset all event pending bit
    while ( (MISR & CI) == 0 )
      ;                            //wait for C event
    uTEMP.word += (duty_ramp << 3);
    MCPUL = uTEMP.byte.b0;
    MCPUH = uTEMP.byte.b1;
  }

  MIMR = CIM;       //IT only on:"C"
  wait_1ms(500);
  return;
}

/**********************************************************************
CURRENT LOOP - start the motor from zero speed
***********************************************************************/
void current_loop_init(void)  {
u8  bemfcpt;

  MCsoft_reset();          // reset of all the peripheral registers
  MCinit_bldc();

  /* variables initialisation
  *****************************/
  nbemf = memnbemf = Status_Step = 0;        // flag initialization
  sync_step = phasestep = 0;

  /* bootstrap charging
  ***********************/
  MCRA  = DAC;       // direct access
  MPHST = 0;         // no output "ON"
  MCRA  = MOE | DAC; // output enabled
  wait_100us();
  MPHST = 0x2a;      // on-off of low side
  wait_100us();
  MPHST = 0;	       // put "OFF" all power switch
  wait_1ms(1);       // all drivers off during 1mS to avoid short ckt

  /* register initialisation
  ***************************/
  mem_ratio = ramp[0][RATIO];    // get ratio from ramp table for alignment
  mem_mprsr = mem_ratio;         // memo starting Step Ratio
  MPRSR     = mem_ratio;         // load MPRSR with first starting value
  MCOMP     = ramp[0][COUNT];    // select step time from ramp
  MDREG     = MCOMP >> 2;        // demag time = step time / 4

  // MCRA  ===>>>  MOE  CKE  SR  DAC  V0C1  SWA  PZ  DCB
  // RESET STATE  - 0    0   0    0    0     0   0    0

  // MCRB  ===>>>   -   CPB  HDM  SDM  OCV  OS2  OS1  OS0
  // RESET STATE  - 0    0    0    0    0     0   0    0

  /* alignment initialisation
  *****************************/
  MPHST = phreg_SL[0];           //output alignment configuration T1-T4
  MCRB  = volt_crbreg_SL[0];     //demag type and z edge settings

  MCPUL = 40;           //..start with a lower current
  MCPUH = 0;            //..for vibration free alignment
  MCRA  = MOE | CKE | Z_select;
                           //enable output; start clk;  sensorless; volt mode
                           //DAC=0;--> MPHST xfer to MCOx on C
                           //Z previous or Z actual for computation

  /* Alignment
  **************/
  currentRef = crntFlag = 0;
  MIMR = PUM;
  for (bemfcpt = 20;   bemfcpt-- ; ) {
    MISR = 0xff - CI;                        //reset C event pending bit
    while ( (MISR & CI) == 0 )
      ;                            //wait for C event
    currentRef += (0x3ff/20);
  }

  MIMR = PUM | CIM;
  wait_1ms(500);
  crntFlag = 1;

  return;
}

/**********************************************************************
Calculate motor speed from 2 successive Z times
***********************************************************************/
void speedFb_Cal(void) {
u16   totalCount;

  if (spdSample == 2) {
    if (ratio1 < ratio0) {
      do
        count0 = count0 >> 1;
      while (--ratio0 != ratio1);
    } else if (ratio0 < ratio1) {
      do
        count1 = count1 >> 1;
      while (--ratio1 != ratio0);
    }
    totalCount = count0 + count1;

    speedFb_temp = ( ( 2*1000000*SPEED_SCALE/3 ) >> ratio0 )/totalCount;
    if ( speedFb_temp > SPEED_FB_MAX )
      speedFb_temp = SPEED_FB_MAX;
    //( * (static u16 *) aMCPVH )  = speedFb_temp << 3;

#if SPEED_LOOP
    spdSample = 3;     // clear capture data available flag
#else
    spdSample = 0;
#endif
  }

  return;
}
/*****************************************************************************/

   /************************************************
   ************  INTERRUPT FUNCTIONS   *************
   *************************************************
   * - LOCAL VARIABLES INSIDE INTERRUPT ROUTINE    *
   *      HAS TO BE 'STATIC'                       *
   *************************************************/


/***********************************************************************
         Motor control Emergency Stop /
         Speed Error /
         FLASH start programming Interrupt
************************************************************************/
@interrupt   void tli_ISR(void)  {      /* TLI interrupt */

  if (bitTest_TRUE(MISR, EI) )  {  //if E is active EMERGENCY STOP
    //MCPUH = 1;                   // set PWM to 0%
    MCRA = 0;   // disable MOE ;; MCOx pins set to reset state
    MISR = 0xff - EI;      //reset IT flag
    while (1)
      ;   //infinite loop till reset
  }
  return;
}

/****************************************************************************
           Commutation & Demagnetisation Interrupt Routine
                    Event C or Event D interrupt
                    *** Exec time for C
                       @ start up -- ~30uS
                       @ run time -- ~14uS
****************************************************************************/
@interrupt  void  mtcCD_ISR(void)  {
static u8  prev_ratio;

  while ( MISR & (DI + CI) )	{
    // C EVENT
    if (bitTest_TRUE(MISR, CI) )   {
      //WWDGCR = 0xff;          // activate and restart watchdog
      starting_failed = 40;

      // preload for next commutation Cn+1
      // get the preload values for next step C event (Cn+2)
      // MCRB ==>> -0-  CPB  HDM  SDM  OCV  OS2  OS1  OS0
      MCRB  = volt_crbreg_SL[phasestep];
      MPHST = phreg_SL[phasestep];
      if ( ++phasestep > 5 )
        phasestep = 0;

      /**********************************************************************
      * Status_Step bits                                                    *
      * =================                                                   *
      *   b7=1 :AUTOCOM bit set : full-running Autoswitched mode            *
      *      0 :AUTOCOM bit reset : not in full_running Autoswitched mode   *
      *   b6=1 :FAS bit set : First Autoswitched Step                       *
      *      0 :FAS bit reset : not in First Autoswitched Step              *
      *   b5=1 :LCS bit set : Last Switched Step                            *
      *      0 :LCS bit reset : not in Last Switched Step                   *
      *   b4-b0:FREE FLAGS                                                  *
      **********************************************************************/

      // FULL AUTO COMMUTATION MODE
      if (Status_Step == 1<<7)  {
        MDREG = ((u16)(MZPRV * (70*0xff/100))) >> 8;   //MDREG = MCOMP+5; //TEST
        //bitToggle (PEDR, BIT(3));
      }

      // FIRST AUTO COMMUTATION STEP
      else if (Status_Step == 1<<6)  {
        //bitSet(PCDR, BIT(5));
        Status_Step = 1<<7;            //clear FAS bit //set AUTOCOM bit
        MWGHT = delay_ki;              //set delay to the value of the Ki
        MDREG = MCOMP+(MCOMP>>1);
        //bitClr(PCDR, BIT(5));
      }

      // LAST SWITCHED STEP
      else if (Status_Step == 1<<5)  {
        //bitSet(PCDR, BIT(5));
        MDREG = MCOMP;  //mem_mdreg;   //last DREG value for Switched mode
        MCOMP = 0xff;                  //switch before Z event
        MWGHT = start_ki;              //select Ki when pass in Auto_com
        MCRA |= (SWA);                 //autoswitch mode // DCB=0/1--MZPRV/MZREG
        MIMR  = PUM | RIM | EIM | ZIM | DIM | CIM;
        Status_Step = 1<<6;            //clear LCS bit  //set FAS bit
        //bitClr(PCDR, BIT(5));
      }

      // FORCED SWITCHED MODE
      else if (Status_Step == 0)  {
        // test if end of ramp
        if (sync_step > 62)
          stop();

        //enable Z IT after stepping through "Val_Z_read" steps
        if ( sync_step > Val_Z_read )
          MIMR = PUM | ZIM | CIM;    // IT on "C" and "Z"

        // find new values for mcomp, mdreg and timer prescale ratio value
        MCOMP = ramp[sync_step][COUNT];        //load current step time
        MDREG = MCOMP >> 2;                    //Tsoft_ demag = step time/4
        mem_ratio = ramp[++sync_step][RATIO];  //get ratio for next step

        // compare curernt step ratio with next step ratio to adjust timer
        // prescaler for next step by setting appropriate flag in MISR
        prev_ratio = mem_mprsr & 0x0f;
        if (prev_ratio > mem_ratio)
          MISR = RMI;
        else if (prev_ratio < mem_ratio)
          MISR = RPI;
        else
          MISR = 0;
        mem_mprsr = (mem_mprsr & 0xf0) | mem_ratio;   //memo clock and ratio

        //test if bemf are successive
        if ( nbemf && (++memnbemf != nbemf) ) {
          memnbemf = 0;  //c
          nbemf = 0;   //z
        }
      }

      MISR = 0xff - CI;           //reset flag of C IT
    }

    // D EVENT
    if (bitTest_TRUE(MISR, DI) )   {
      MISR = 0xff - DI;           //reset flag of D IT
    }

  } //while ( MISR & (DI + CI) )

  return;
}

/***********************************************************************
            Ratio & Zero crossing  Interrupt Routine
                  Event U or Event Z interrupt
                  *** Exec Time for Z --  ~15uS  ***
***********************************************************************/
@interrupt  void  mtcRZ_ISR(void)  {

  while (MISR & ( RPI | RMI | ZI )  )	{
    /***************
      BEMF Z EVENT
    ****************/
    if (bitTest_TRUE (MISR, ZI) )	{
      /***************************************************************
      ; if the number of BEMF Z observed is < nbemf (min number of bemf)
      ; then stay in SWITCHED MODE, else, change to AUTOSWITCHED MODE
      ;**************************************************************/
      if (Status_Step == 0)   {  // if synchronous mode
        if (++nbemf >= Minbemf_SL)    {  // Min_nbemf = 4 for SensorLess
          // bitSet(PCDR, BIT(5));
          Status_Step = 1<<5;         //set Last Commutated Step bit
          // bitClr(PCDR, BIT(5));
        }
      }
      //bitToggle(PCDR, BIT(5));
      /***********************************************
      ;  BEMF input scaling based on PWM dutycycle
      ***********************************************/
      // MCRC  ===>>>  SEI  HZ  SZ  SC  SPLG  VR2  VR1  VR0
      // MCONF  ===>>>   DS[3:2:1:0]  SOI  SOM  XT16  XT8
      if (MCPUH > 17) {
        BEMF_DDR |= (BEMF_INPUTS);         // scaledown bemf inputs
        MCRC = MCRC_high_duty_cycle;
        bitSet(MCFR, RPGS);
          MCONF |= Ton_sample_delay;       // sample at ON after delay
        bitClr(MCFR, RPGS);
      }  /**/
      else if (MCPUH < 16)  {
        BEMF_DDR &= (0xff - BEMF_INPUTS);   // unscale bemf inputs
        MCRC = MCRC_low_duty_cycle;         // sample at end of PWM OFF
        bitSet(MCFR, RPGS);
          MCONF &= (0xff - Ton_sample_delay);
        bitClr(MCFR, RPGS);
      }/**/

      /*****************************************************
      ; Commtn delay coefficient tuning based on dutycycle
      *****************************************************/
      if (MCPUH > 24)
        MWGHT = 110;
      else if (MCPUH > 16)
        MWGHT = 110;
      else
        MWGHT = 110;

      /***************************************************
      ; BEMF Time Period sampling for speed calculation
      ***************************************************/
      if (spdSample == 0)   {
        count0 = MZREG;
        ratio0 = MPRSR & 0x0f;
        spdSample = 1;
      } else if (spdSample == 1)   {
        count1 = MZREG;
        ratio1 = MPRSR & 0x0f;
        spdSample = 2;
      }  /**/

      MISR = 0xff - ZI;     //clear ZI IT
    }// end Z

    MISR = 0xff - RPI - RMI;  //ratio ITs not used, clear RPI, RMI IT

    /*
    if (MIMR & RIM) {  //- no task right now - can be commented out
      /***************
         RPI EVENT
      ****************
      if (bitTest_TRUE (MISR, RPI) )  {
        // Ratio update routines for soft DREG update
        MISR = 0xff-RPI;
      }

      /***************
         RMI EVENT
      ****************
      if (bitTest_TRUE (MISR, RMI) )	{
        // Ratio update routines for soft DREG update // ZREG <= 55h
        MISR = 0xff-RMI;
      }
    } */
  }// end while(MISR & ( RPI | RMI | ZI )  )

  return;
}
/***********************************************************************
          Motor control - Event U or crnt loop or samp out interrupt
              ***  Exec time for U with assembly -- ~90-110uS  ***
************************************************************************/
@interrupt @nosvf  void mtcU_CL_SO_ISR(void)  {

  /**************
    U EVENT
  ***************/
  if (bitTest_TRUE(MISR, PUI) )	{
#if CUR_LOOP
    /*********************************
    // CURRENT LOOP - PI CONTROLLER //
    *********************************/
    //bitToggle (PCDR, BIT(5));
    currentFb = ((ADCDRMSB << 2) + ADCDRLSB) << CURRENT_FB_SCALE;
    piconCur();

    MCPUL = PIconCur.byte.b2;
    MCPUH = PIconCur.byte.b3;

    /****  Current Loop PI controller ends here *****/

    if (crntFlag) {
#if SPEED_LOOP
      getADC_10bit (speedRef , SPEED_REF_CHNL);
      if (speedRef > SPEED_REF_MAX)
        speedRef = SPEED_REF_MAX;
#else if CUR_LOOP
        getADC_10bit (currentRef , CURRENT_REF_CHNL);
        if (currentRef > MAX_CURRENT_REF)
          currentRef = MAX_CURRENT_REF;
#endif
    }

    // Current Feedback measurement setup for next cycle
    ADCCSR = ADON + CURRENT_FDBK_CHNL;
    ADCDRMSB;  // to clear EOC of prev conv
#endif
    MISR = 0xff - PUI;            //reset IT flag
  }

  /**************
    CL EVENT
  ***************/
  if (bitTest_TRUE(MISR, CLI) )	{
    MISR = 0xff - CLI;            //reset IT flag
  }

  /**************
    SO EVENT
  ***************/
  /*  This interrupt is generated only when sampling at Ton enabled and
    using DS[3:0] to set sampling delay counting from turn ON instant
    REGISTERS USED ARE
    PAGE1
    - MCONF ==>> DS[3:0]  SOI  SOM  XT16  XT8
    - Clearing SOI is not given in detail  */

  return;
}

/************************************************************************/
/* Main Clock Control/ Real time clock interrupt *
            ***   Exec time is --  ~80uS   ***
/************************************************************************/
@interrupt @nosvf  void mccRtc_ISR(void)	{

  if (starting_failed)
    --starting_failed;

  /* ON/OFF switch - 100ms debounce time
  ***************************************/
  if (PB3_PORT & PB3)   {
    if (key1count != DEBOUNCE_TIME_COUNT)
      if (++key1count == DEBOUNCE_TIME_COUNT)
        flag ^= 0xff;
  } else 
    key1count = 0;
  /**/

#if SPEED_LOOP

  if (spdSample == 3) {
    speedFb = speedFb_temp;
    spdSample = 0;
  }

  /*************************************************/
  /********   SPEED LOOP - PI CONTROLLER   *********/
  /*************************************************/
  if (crntFlag) {
    speedErrorPrev = speedError;
    speedError = (speedRef - speedFb) << SPEED_PI_SCALE;
    piconSpd();
    currentRef = PIconSpeed.word.w1;
  } else
    PIconSpeed.word.w1 = currentRef;   // init PIconSpeed with latest currentRef
  //( * (static u16 *) aMCPVH )  = currentRef << 3;

#endif
  /********  Speed loop PI controller ends here  ********/

  MCCSR;	/* read MCCSR to clear interrupt  */
  return;
}

/************************************************************************/
@interrupt void ext1_ISR(void)		{;}		/* External1 interrupt */
@interrupt void pwmArt_ISR(void)  {;}   /* PWM ART o/f interrupt */

/************************************************************************
  Code ends here
************************************************************************/
