/******************** (C) COPYRIGHT 2006 STMicroelectronics ********************
* File Name          : My_functions.c
* Author             : MPA Systems Lab
* Date First Issued  : 01/02/2005
* Description        : Motor control routines and medium layer functions
********************************************************************************
* THE PRESENT SOFTWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
* WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME.
* AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT,
* INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE
* CONTENT OF SUCH SOFTWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING
* INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
*******************************************************************************/

/* Includes ------------------------------------------------------------------*/
#include "My_functions.h"  
#include "lib.h" 
#include "PMAC_Param.h"
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
u16 Start_up_volt=(u16)(START_VOLT<<8); 
u16 period;

extern u8 PhaseShift,counter,SinAmp,updated_semiperiod;
extern u16 Freq_buf,timebase_1ms,measure[4];
extern _Bool start_up_on_going, count_carry, Is_first_measurement;

#if DRIVING_MODE==CLOSED_LOOP
extern s16 old_error, error, delta_error,prop_term,integral_term;  
#endif 
/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/

/*-----------------------------------------------------------------------------
ROUTINE NAME : Set_Spinning_Direction
INPUT				 : Spinning direction
OUTPUT 			 : None
DESCRIPTION  : It sets the motor spinning direction
COMMENTS     : It must be called before main while loop i.e. when motor is still
-----------------------------------------------------------------------------*/ 

void Set_Spinning_Direction(direction dir)
{
 if (dir==CW)
  PhaseShift=85;
 else 
   if (dir==CCW)
    PhaseShift=170;        
	start_up_on_going = 1;
	count_carry = 0;
	Is_first_measurement = 1;
}               

/*-----------------------------------------------------------------------------
ROUTINE NAME : Soft_start
INPUT/OUTPUT : None
DESCRIPTION  : It slowly increments stator voltage
COMMENTS     : Start-up ends when START_UP_DURATION elapsed 
							 or, in closed loop, if rotor frequency exceeds
							 CL_FREQ_VALIDATION.
-----------------------------------------------------------------------------*/ 
 void Soft_start(void) 
 {
  u8 static old_timebase_1ms;

#if DRIVING_MODE==OPEN_LOOP     
  if(timebase_1ms<START_UP_DURATION)  
#else if DRIVING_MODE==CLOSED LOOP
  if (timebase_1ms<START_UP_DURATION && Freq_buf<CL_FREQ_VALIDATION)
#endif
   { 
     if(old_timebase_1ms !=(u8)(timebase_1ms))
      {
       Start_up_volt+=SLEW_RATE; 
       SinAmp=(u8)(Start_up_volt>>8);
       old_timebase_1ms++;
      }
   }
  else
   {
    start_up_on_going=0;
#if DRIVING_MODE==CLOSED_LOOP  
    timebase_1ms= SAMPLING_TIME+1; //this ensures istantaneous action of PI regulator
#else if DRIVING_MODE==OPEN_LOOP   //at the end of the soft start
    timebase_1ms=0;
#endif
   }    
 }

/*-----------------------------------------------------------------------------
ROUTINE NAME : Freq_computing
INPUT				 : divisor
OUTPUT			 : quotient
DESCRIPTION  : It convert the period into frequency 
							 implementing 3FFFF/period division
-----------------------------------------------------------------------------*/ 
 
u16 Freq_computing(u16 divisor)
{ 
 unsigned int quot1;
 unsigned int quot2;
 
 //if (divisor>0x03 && divisor<0x3FFF) //in normal conditions the condition is always true
 {
 quot1=0xFFFF/divisor; 
 quot2=quot1*divisor; 
 quot2=((unsigned int)(0xFFFF-quot2)<<4)+0x0F;
 quot2=(unsigned int)(quot2)/divisor;
 quot2+=(quot1<<4);
 return (quot2);
 }
// else 
// return(Freq_buf); //Freq_buf is not update
}


/*-----------------------------------------------------------------------------
ROUTINE NAME : Get_Phase_angle
INPUT				 : None / rotor frequency
OUTPUT			 : Phase angle
DESCRIPTION  : Phase angle is computed starting from 
							 potentiometer value or Ph/f curve
-----------------------------------------------------------------------------*/ 

#ifdef PHASE_READING_FROM_POT2

u16 Get_Phase_angle(void)
{ 
  u32 volatile aux_32;
  u8 Ch6_Voltage;
  u16 aux_16;
  
  Ch6_Voltage=Get_Pot2();
  
  aux_16=Ch6_Voltage*(u8)(DELTA_PHASE);
  aux_32=Ch6_Voltage*(u8)(DELTA_PHASE>>8);   
  aux_32<<=8;
  aux_32+=aux_16;
  aux_32>>=8;
  aux_32+=MIN_PHASE;         
  return ((u16)(aux_32));  
}

#else

u16 Get_Phase_angle(u16 frequency)
{ 
  s32 Phase_angle_tmp;  
  u16 aux_16;   
  
  if (frequency<=PH_LOWFREQ)	
   Phase_angle_tmp = PH_AT_LOWSPEED;
    else if (frequency<=PH_HIGHFREQ)	
      	 {
          aux_16=(u16)(frequency-PH_LOWFREQ);
          Phase_angle_tmp = (s32)(COEFF_PHASE)*aux_16;
					Phase_angle_tmp = Phase_angle_tmp/32;
          Phase_angle_tmp=(u16)((s16)(Phase_angle_tmp)+PH_AT_LOWSPEED);
         }                	
        else Phase_angle_tmp = PH_AT_HIGHSPEED;
 
  return (Phase_angle_tmp);           
}
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : DoRotorAlignment
INPUT				 : None 
OUTPUT			 : None
DESCRIPTION  : It implementents rotor alignment befor voltage
							 start-up 
-----------------------------------------------------------------------------*/ 

#ifdef ROTOR_ALIGNMENT
void DoRotorAlignment (void)
{ 
  u16 duty=0;
  while (timebase_1ms < ALIGNMENT_DURATION)  
  {
   duty+=DELTA_ALIGN;
   if (duty<=32768)
   DCR0L=(u8)(duty>>7); 
   else
   {
   DCR0H=0x0F;
   DCR0L=(u8)(duty>>7);
   }
   Enable_Transfer(); 
    
   _asm("WFI");     //Waits for interrupts
  }                                      

  counter=0;
  timebase_1ms=0;
  ATCSR=0x12;    //It enables  AT Overflow Interrupt
  LTCSR1= 0x90; // and Lite Timer Input Capture Interrupt  
  LTICR;
  return;
}
#endif    

/*-----------------------------------------------------------------------------
ROUTINE NAME : EnableTimersInterrupts
INPUT				 : None 
OUTPUT			 : None
DESCRIPTION  : It implementents pheripheral and variable 
							 initialization if rotor alignment is not 
							 enabled
-----------------------------------------------------------------------------*/ 

#ifndef ROTOR_ALIGNMENT
void EnableTimersInterrupts(void)  
{
 counter=0;
 ATCSR=0x12;    //It enables  AT Overflow Interrupt
 LTCSR1= 0x90; // and Lite Timer Input Capture Interrupt  
 LTICR; 
} 
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : Get_Target_Voltage
INPUT				 : None 
OUTPUT			 : Stator voltage
DESCRIPTION  : In open loop, it computes stator voltage
							 starting from potentiometer converted value. 
							 If enabled, V/f limitation is actuated.
-----------------------------------------------------------------------------*/ 

#if DRIVING_MODE==OPEN_LOOP   

u8 Get_Target_Voltage(void)
{ 
  u8 target;
  u16 aux;
  
  target=Get_Pot1();
  
  aux=((u16)(target)+1)*DELTA_VOLT;
  target=(u8)(aux>>8);    
  target+=MIN_VOLTAGE;

#ifdef V_by_f_limitation 
  aux=Get_Max_volt(Freq_buf);    
 
  if (target> aux)          
   target=(u8)(aux);
#endif

  return (target);
  
  }
#endif 
  

/*-----------------------------------------------------------------------------
ROUTINE NAME : Get_Target_Speed
INPUT				 : None 
OUTPUT			 : Target speed
DESCRIPTION  : In closed loop, it computes target speed 
							 starting from potentiometer converted value
-----------------------------------------------------------------------------*/ 

#if DRIVING_MODE==CLOSED_LOOP
u16 Get_Target_Speed()
{ 
  u32 target;
  u8 Ch5_Voltage;
  u16 aux,aux_16;
  
  Ch5_Voltage=Get_Pot1();
  
  aux = Ch5_Voltage * (u8)(DELTA_FREQ);
  target = Ch5_Voltage * (u8)(DELTA_FREQ>>8);   
  target = (u32)(aux) + (target<<8);
  target >>= 8;
  target += MIN_FREQ;       
   
  return ((u16)target);
  
  }

/*-----------------------------------------------------------------------------
ROUTINE NAME : DoMotorControl
INPUT				 : Rotor target electrical frequency 
OUTPUT			 : Regulated stator voltage
DESCRIPTION  : In closed loop, it implements a PI controller
							 for regulating stator voltage 
-----------------------------------------------------------------------------*/ /

void DoMotorControl(u16 target_freq)
{ 
  #ifdef V_by_f_limitation 
  u16 aux_16;
  #endif
	
  old_error=error;
  error = target_freq-Freq_buf; //update actual errors
  delta_error=error-old_error;
        
  if (error>127)error=127;
    else if (error<-127) error=-127;     //errors saturation
  if (delta_error>127)delta_error=127;
    else if (delta_error<-127) delta_error=-127;

  prop_term = (s16)(kp* delta_error);
  integral_term = (s16) (ki*error); 
  prop_term=(signed int)(prop_term+integral_term);
  prop_term= (signed int) (prop_term/64);
  integral_term=(signed char)prop_term+SinAmp;
        
  if (integral_term<0) SinAmp=0;
   else if (integral_term>255) 
		SinAmp=255;

#ifdef V_by_f_limitation 
  aux_16=Get_Max_volt(Freq_buf);    
 
  if (integral_term> aux_16)          
   SinAmp=(u8)aux_16;
#endif
  else
   SinAmp=(u8)integral_term;  
               
}  
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : Get_Max_volt
INPUT				 : Rotor actual electrical frequency 
OUTPUT			 : Maximum allowed stator voltage
DESCRIPTION  : It computes the maximum allowed stator 
							 voltage for a given rotor speed
-----------------------------------------------------------------------------*/ 

#ifdef V_by_f_limitation

u8 Get_Max_volt(u16 frequency)    
 { 
  u16 SinAmp_tmp;
  u16 aux_16;   
  
  if (frequency<=V_LOWFREQ)	
   SinAmp_tmp = V_AT_LOWSPEED;
    else if (frequency<=V_HIGHFREQ)	
      	 {
          aux_16=(u16)(frequency-V_LOWFREQ);
          SinAmp_tmp = COEFF_V*aux_16;  
          SinAmp_tmp>>=8;
          SinAmp_tmp=(u16)(SinAmp_tmp+V_AT_LOWSPEED);
         
         }                	
        else SinAmp_tmp = V_AT_HIGHSPEED;
 
  return ((u8)(SinAmp_tmp)); 
}
#endif   

/*-----------------------------------------------------------------------------
ROUTINE NAME : Get_Rotor_Freq
INPUT				 : None 
OUTPUT			 : Rotor actual frequency
DESCRIPTION  : It returns rotor electrical frequency based 
						   on latest 2 Hall sensors signal period 
							 measurements. Latest measurement is returned 
							 only during start-up 
-----------------------------------------------------------------------------*/ 

u16 Get_Rotor_Freq(void)
{ 	
	if (updated_semiperiod == 4)
		Freq_buf = START_FREQ;
	else
	{	
		u8 i;
		u32 period_32=0;
		
		for (i=0 ; i<4; i++)
		{
			if (measure[i]==0)
				{
					period = (measure[updated_semiperiod]);
					Freq_buf = Freq_computing(period);
					return (Freq_buf);
				}
			else
				period_32 += measure[i];
		}
		period = period_32>>2;
		Freq_buf = Freq_computing(period);
	}
		return (Freq_buf);
}

