/******************** (C) COPYRIGHT 2003 STMicroelectronics ********************
* File Name          : flash.h
* Author             : MCD Application Team
* Date First Issued  : 28/07/2003
* Description        : This file contains all the functions prototypes for the
*                      Embedded Flash software library.
********************************************************************************
* History:
*  01/01/2004 : V1.2
*  14/07/2004 : V1.3
********************************************************************************
 THE PRESENT SOFTWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS WITH
 CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME.
 AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT
 OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE CONTENT
 OF SUCH SOFTWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING INFORMATION
 CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
*******************************************************************************/
#ifndef __FLASH_H
#define __FLASH_H

#include "71x_lib.h"

#define FLASH_B0F0 0x00000001
#define FLASH_B0F1 0x00000002
#define FLASH_B0F2 0x00000004
#define FLASH_B0F3 0x00000008
#define FLASH_B0F4 0x00000010
#define FLASH_B0F5 0x00000020
#define FLASH_B0F6 0x00000040
#define FLASH_B0F7 0x00000080

#define FLASH_B1F0 0x00010000
#define FLASH_B1F1 0x00020000

#define FLASH_B0 (FLASH_B0F0 | FLASH_B0F1 | FLASH_B0F2 | FLASH_B0F3 | FLASH_B0F4 | FLASH_B0F5 | FLASH_B0F6 | FLASH_B0F7)
#define FLASH_B1 (FLASH_B1F0 | FLASH_B1F1)

typedef enum
{
  FLASH_FINISHED,
  FLASH_ONGOING
} flashwriteoperation;

typedef enum
{
  FLASH_BANK0 = 0x1000000,
  FLASH_BANK1 = 0x2000000
} flashbanks;

typedef enum
{
  FLASH_LPS   = 0x2F, /* 001-01111 = 0010 1111 = 0x2F */
  FLASH_LOCK  = 0x24, /* 001-00100 = 0010 0100 = 0x24 */
  FLASH_BSY1  = 0x22, /* 001-00010 = 0010 0010 = 0x22 */
  FLASH_BSY0  = 0x21, /* 001-00001 = 0010 0001 = 0x21 */
  FLASH_WPF   = 0x48, /* 010-01000 = 0100 1000 = 0x48 */
  FLASH_RESER = 0x47, /* 010-00111 = 0100 0111 = 0x47 */
  FLASH_SEQER = 0x46, /* 010-00110 = 0100 0110 = 0x46 */
  FLASH_10ER  = 0x43, /* 010-00011 = 0100 0011 = 0x43 */
  FLASH_PGER  = 0x42, /* 010-00010 = 0100 0010 = 0x42 */
  FLASH_ERER  = 0x41, /* 010-00001 = 0100 0001 = 0x41 */
  FLASH_ERR   = 0x40  /* 010-00000 = 0100 0000 = 0x40 */
} flashflags;

#define FLASH_WMS_Mask   0x80000000
#define FLASH_SUSP_Mask  0x40000000
#define FLASH_WPG_Mask   0x20000000
#define FLASH_DWPG_Mask  0x10000000
#define FLASH_SER_Mask   0x08000000
#define FLASH_SPR_Mask   0x01000000
#define FLASH_BER_Mask   0x04000000
#define FLASH_MER_Mask   0x02000000
#define FLASH_BSYA1_Mask 0x00000002
#define FLASH_BSYA2_Mask 0x00000004

#define FLASH_DBGP_Mask  0x00000002
#define FLASH_ACCP_Mask  0x00000001

#define FLASH_Reg_Mask   0xE0
#define FLASH_Flag_Mask  0x1F

#define FLASH_INTM_Mask  0x00200000
#define FLASH_INTP_Mask  0x00100000

/*******************************************************************************
* Function Name  : FLASH_Init
* Description    : Initialise the Flash
* Input          : None
* Return         : None
*******************************************************************************/
void FLASH_Init(void);

/*******************************************************************************
* Function Name  : FLASH_FlagStatus
* Description    : Returns the NewState of Flash flags
* Input 1        : Flash Flag
* Return         : flagstate
*******************************************************************************/
FlagStatus FLASH_FlagStatus(flashflags Xflag);

/*******************************************************************************
* Function Name  : FLASH_WriteOpStatus
* Description    : Checks the write operation status
* Input 1        : flash banck
* Return         : write operation status
*******************************************************************************/
inline FlagStatus FLASH_WriteOpStatus(flashbanks Xbank)
{
  return Xbank == FLASH_BANK0 ? FLASH_FlagStatus(FLASH_BSY0) : FLASH_FlagStatus(FLASH_BSY1);
}

/*******************************************************************************
* Function Name  : FLASH_WordWrite
* Description    : Writes a Word to the Flash
* Input 1        : Address of the Destination
* Input 2        : Word to program
* Return         : None
*******************************************************************************/
void FLASH_WordWrite(u32 TargetAdd, u32 Data);

/*******************************************************************************
* Function Name  : FLASH_DWordWrite
* Description    : Writes Double Word to the Flash
* Input 1        : Address of the Destination
* Input 2        : Word 1 To program
* Input 3        : Word 2 To program
* Return         : None
*******************************************************************************/
void FLASH_DWordWrite(u32 TargetAdd, u32 Data0, u32 Data1);

/*******************************************************************************
* Function Name  : FLASH_BlockWrite
* Description    : Writes Data To the Flash
* Input 1        : Address of the Data source
* Input 2        : Address of the Destination
* Input 3        : Nbr of words to be stored
* Return         : None
*******************************************************************************/
void FLASH_BlockWrite(u32 SourceAdd, u32 TargetAdd, u32 DataLength);

/*******************************************************************************
* Function Name  : FLASH_EraseSector
* Description    : Erases a Flash sector
* Input 1        : Sectors to be Erased
* Return         : None
*******************************************************************************/
void FLASH_SectorErase(u32 Xsectors);

/*******************************************************************************
* Function Name  : FLASH_EraseBank
* Description    : Erases a Bank of the Flash
* Input 1        : Bank to be Erased
* Return         : None
*******************************************************************************/
inline void FLASH_BankErase(flashbanks Xbank)
{
  FLASH_SectorErase(Xbank == FLASH_BANK0 ? FLASH_B0 : FLASH_B1);
}

/*******************************************************************************
* Function Name  : FLASH_EraseModule
* Description    : Erases a flash module
* Input          : None
* Return         : None
*******************************************************************************/
void FLASH_ModuleErase(void);

/*******************************************************************************
* Function Name  : FLASH_Delay
* Description    : Add the delay required for the Flash Write & Erase operation
* Input 1        : None
* Return         : None
*******************************************************************************/
void FLASH_Delay(void);

/*******************************************************************************
* Function Name  : FLASH_Suspend
* Description    : Suspends the current program or erase operation
* Input 1        : None
* Return         : None
*******************************************************************************/
void FLASH_Suspend(void);

/*******************************************************************************
* Function Name  : FLASH_Resume
* Description    : Resume a Suspended program or erase operation
* Input 1        : None
* Return         : None
*******************************************************************************/
void FLASH_Resume(void);

/*******************************************************************************
* Function Name  : FLASH_ReadWord
* Description    : Read a single word of the flash
* Input 1        : Source Address
* Return         : Word
*******************************************************************************/
u32 FLASH_WordRead(u32 SourceAdd);

/*******************************************************************************
* Function Name  : FLASH_ReadBlock -> Block Read
* Description    : Block Read from the flash
* Input 1        : Destination Address where the Data will be Stored
* Input 2        : Data Source Address
* Input 3        : Nbr of word to be Read
* Return         : Word
*******************************************************************************/
void FLASH_BlockRead(u32 DestAdd, u32 SourceAdd, u32 NbrData);

/*******************************************************************************
* Function Name  : FLASH_WritePrConfig
* Description    : Configures The Write Protection Bits
* Input 1        : Flash Bank
* Input 2        : Enable or disable Protection
* Return         : None
*******************************************************************************/
void FLASH_WritePrConfig(u32 Xsectors, FunctionalState NewState);

/*******************************************************************************
* Function Name  : FLASH_DebugPrConfig
* Description    : Configures The Debug Protection Bits
* Input 1        : ENABLE or DISABLE
* Return         : None
*******************************************************************************/
void FLASH_DebugPrConfig(FunctionalState NewState);

/*******************************************************************************
* Function Name  : FLASH_FlagClear
* Description    : Clears a flash flag
* Input 1        : Flash Flag
* Return         : None
*******************************************************************************/
void FLASH_FlagClear(flashflags Xflag);

/*******************************************************************************
* Function Name  : FLASH_ITConfig
* Description    : Enables Or Disables the write-end interrupt
* Input 1        : FunctionalState(Enable, Disable)
* Return         : None
*******************************************************************************/
inline void FLASH_ITConfig(FunctionalState NewState)
{
  if (NewState == ENABLE) FLASHR->CR0 |= FLASH_INTM_Mask;
  else FLASHR->CR0 &= ~FLASH_INTM_Mask;
}

/*******************************************************************************
* Function Name  : FLASH_ITStatus
* Description    : Checks if the write-end interrupt is enabled or not
* Input 1        : None
* Return         : Enable, Disable
*******************************************************************************/
inline FunctionalState FLASH_ITStatus(void)
{
  return (FLASHR->CR0 & FLASH_INTM_Mask)==0 ? DISABLE : ENABLE;
}

/*******************************************************************************
* Function Name  : FLASH_ITClear
* Description    : Clears an interrupt pending flag
* Input 1        : None
* Return         : None
*******************************************************************************/
inline void FLASH_ITClear(void)
{
  FLASHR->CR0 &= ~FLASH_INTM_Mask;
}

/*******************************************************************************
* Function Name  : ResetBit
* Description    : Resets a single bit in a 32 bit register value
* Input 1        : The Value to be modified.
* Input 2        : The Index of the bit to be Reset.
* Return         : The value passed in parameter with the bit (Bitindex) reset
*******************************************************************************/
inline u32 ResetBit(u32 MyValue, u8 BitIndex)
{
  return MyValue & ~(0x1 << BitIndex);
}

/*******************************************************************************
* Function Name  : ProtectionLevel
* Description    : Gives the level of protection in the PDS PEN registers
* Input 1        : The Protection Registers
* Return         : The last bit not yet reset
*******************************************************************************/
u16 ProtectionLevel(u16 ProtectionRegs);

/*******************************************************************************
* Function Name  : Wait For Last Task
* Description    : Waits for the end of last task on a Flash Bank
* Input 1        : Bank number.
* Return         : The value passed in parameter with the bit (Bitindex) reset
*******************************************************************************/
void WaitForLastTask(flashbanks Xbank);

#endif  // __FLASH_H

/*******************(C) COPYRIGHT 2003 STMicroelectronics *****END OF FILE****/
