/************************************************************************/
/*                                                                      */
/*   MODULE:     lan8900.c                                              */
/*   DATE:       5/7/96                                                 */
/*   PURPOSE:    Crystal Semiconductor CS8900 driver 		        */
/*   PROGRAMMER: Quentin Stephenson                                     */
/*                                                                      */
/*----------------------------------------------------------------------*/
/*                                                                      */
/*               Copyright 1996, Crystal Semiconductor Corp.            */
/*                      ALL RIGHTS RESERVED                             */
/*                                                                      */
/*                                                                      */
/************************************************************************/
/*                                                                      */
/*    Change Log:                                                       */
/*                9/15/99 James Ayres added support for the RDY4TXNow   */
/*                interrupt.  All changes marked with @jla              */
/*                                                                      */
/*               Week 20~23-2005 Porting for STR71x and uIP integration */
/************************************************************************/


#include "bsp.h"
#include "lan8900.h"
#include "71x_type.h"
#include "71x_lib.h"

#define		ETHER_BASE	0x64800000


#if (BSP_LAN1 == YES) && (BSP_LANCARD_MODEL == CS8900)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* Note on naming conventions used in this driver.                     */
/*                                                                     */
/* All routines and global variables in this driver start with a lower */
/* case "cs", which stands for Crystal Semiconductor.  This identifies */
/* these globally existing objects as belonging to this driver.        */
/*                                                                     */
/* All variables which contain an address instead of a value, start    */
/* with a lower case "p", which stands for pointer.  If a global       */
/* variable contains an address then it starts with "cs_p".            */
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* External variables */

extern long PsosIsUp;


/* Prototypes of external routines */

extern unsigned short inw( unsigned long Port );
extern unsigned char inb( unsigned long Port );
extern void outw( unsigned long Port, unsigned short Value );
extern void outb( unsigned long Port, unsigned char  Value );
extern void delay_ms(void); //changed May-17-2005


/* Prototypes of internal routines */

//unsigned long ( void (*ap_addr)(mblk_t *, PIA, PIA) );
unsigned long csVerifyChip(void);
unsigned long csResetChip(void);
unsigned long csInitFromEEPROM(void);
int csReadEEPROM( unsigned short Offset, unsigned short *pValue );
unsigned long csInitFromBSPDefs(void);
char *csHexWord( char *pChar, unsigned short *pWord );
char *csHexByte( char *pChar, unsigned char *pByte );
unsigned long csInitInterrupt(void);
unsigned long csSend( PIA pDestAddr, mblk_t *pSendParms );
unsigned short csRequestTransmit(void);
void csCopyTxFrame(void);
//unsigned long csPoll();
//unsigned long csIoctl( struct niioctl *pIoctlParms );
void csProcessISQ(void);
void csReceiveEvent( unsigned short RxEvent );
void csBufferEvent( unsigned short BufEvent );
void csTransmitEvent(void);
unsigned short csReadPacketPage( unsigned short Offset );
void csWritePacketPage( unsigned short Offset, unsigned short Value );
unsigned long csEnqueueTxReq( PIA pDestAddr, mblk_t *pSendParms );
void csDequeueTxReq(void);


/* Global variables */

IA csBroadcastAddr;
IA csHardwareAddr;
unsigned short csIntNumber;
unsigned short *cs_pPacketPage;
int csInMemoryMode;
int csPsosIsInit;
long csNINumber;
mblk_t RxMsg;
mblk_t TxMsg;

short config;
short dbg1, dbg2;


void (*cs_pAnnounce_Packet)( mblk_t *pMsg );
/*
mblk_t *(*cs_p_allocb)( int Size, int Priority );
void (*cs_p_freemsg)( mblk_t *pMsg );
*/

PTXREQ cs_pTxReqQueue;       /* Linked list of outstanding transmit requests */
PTXREQ cs_pTxFreeQueue;      /* Linked list of unused tx request structures */
TXREQ  csTxReq[MAXTXREQ];    /* Array of transmit request structures */

/* The transmit request at the head of the cs_pTxReqQueue is the request */
/* that is currently being transmitted.  If the cs_pTxReqQueue is empty  */
/* then there is not a transmission in progress. */





/*****************************************************************************/
/*  csInitialize()                                                           */
/*                                                                           */
/*  This routine initializes the network interface driver.                   */
/*  If the driver is successfully initialized, then the address of the       */
/*  driver's hardware address is returned, else FAILURE is returned.         */
/*                                                                           */
/*****************************************************************************/

unsigned long csInitialize( void (*ap_addr)(mblk_t *) )
{
   int x;
   
   csInMemoryMode = FALSE;  /* Must start out using IO mode */
   csPsosIsInit = FALSE;    /* Assume pSOS kernel is not initialized */

    /* Verify that it is the correct chip */
    if ( csVerifyChip() == FAILURE )
       return FAILURE;

   /* Reset the chip */
   if ( csResetChip() == FAILURE )
      return FAILURE;

   #if (BSP_CS8900_EEPROM == YES)

     /* Initialize using data in the EEPROM */
     if ( csInitFromEEPROM() == FAILURE )
        return FAILURE;

   #else  /* BSP_CS8900_EEPROM == NO */

      /* Initialize using defines in the BSP.H file */
      if ( csInitFromBSPDefs() == FAILURE )
         return FAILURE;

   #endif  /* BSP_CS8900_EEPROM */

   /* Initialize the config and control registers */
   csWritePacketPage( PKTPG_RX_CFG, RX_CFG_RX_OK_IE); //|RX_CFG_CRC_ERR_IE|RX_CFG_RUNT_IE|RX_CFG_X_DATA_IE );
   csWritePacketPage( PKTPG_RX_CTL, RX_CTL_RX_OK_A|RX_CTL_IND_A|RX_CTL_BCAST_A);
   csWritePacketPage( PKTPG_TX_CFG, TX_CFG_ALL_IE );

	// @jla added to support rdy4tx interrupt
   csWritePacketPage( PKTPG_BUF_CFG, BUF_CFG_RDY4TX_IE ); 

   /* Put hardware address into the Individual Address register */
   csWritePacketPage( PKTPG_IND_ADDR,   csHardwareAddr.word[0] );
   csWritePacketPage( PKTPG_IND_ADDR+2, csHardwareAddr.word[1] );
   csWritePacketPage( PKTPG_IND_ADDR+4, csHardwareAddr.word[2] );

   /* Set broadcast address to all F's */
   for ( x=0; x<BSP_LAN1_HWALEN/2; x++ )
      csBroadcastAddr.word[x] = 0xFFFF;

   /* Initialize the transmit request queue to empty */
   cs_pTxReqQueue = NULL;

   /* Put all the transmit requests on the free queue */
   cs_pTxFreeQueue = csTxReq; 
   for ( x=0; x<MAXTXREQ-1; x++ )
      csTxReq[x].pNext = &csTxReq[x+1];
   csTxReq[MAXTXREQ-1].pNext = NULL;

   cs_pAnnounce_Packet = ap_addr;  //Notify function


   /* Initialize the interrupt */

   if ( csInitInterrupt() == FAILURE )
      return FAILURE;

   return (unsigned long)&csHardwareAddr;  /* Successfull initialization! */
}


/*****************************************************************************/
/*  csVerifyChip()                                                           */
/*                                                                           */
/*  This routine verifies that the Crystal Semiconductor chip is present     */
/*  and that it is a CS8900.                                                 */
/*                                                                           */
/*  IMPORTANT!  This routine will fail if the IO base address programmed     */
/*              in the chip's EEPROM does not match the IO base address      */
/*              specified with BSP_CS8900_IO_BASE in the BSP.H file.         */
/*                                                                           */
/*****************************************************************************/

unsigned long csVerifyChip(void)
{
   /* See if the PacketPage Pointer port contains the correct signature */
   if ( (inw(PORT_PKTPG_PTR)&SIGNATURE_PKTPG_PTR) != SIGNATURE_PKTPG_PTR )
      return FAILURE;

   /* Verify that the chip is a Crystal Semiconductor chip */
   if ( csReadPacketPage(PKTPG_EISA_NUM) != EISA_NUM_CRYSTAL )
      return FAILURE;

   /* Verify that the chip is a CS8900 */
   if ( (csReadPacketPage(PKTPG_PRODUCT_ID)&PROD_ID_MASK) != PROD_ID_CS8900 )
      return FAILURE;

   return SUCCESS;
}


/*****************************************************************************/
/*  csResetChip()                                                            */
/*                                                                           */
/*  This routine resets the chip and initializes it for 16-bit operation.    */
/*                                                                           */
/*****************************************************************************/

unsigned long csResetChip(void)
{
   int x;

   /* Issue a reset command to the chip */
   csWritePacketPage( PKTPG_SELF_CTL, SELF_CTL_RESET );

   /* Delay for 125 micro-seconds */
   delay_ms(); //changed May-17-2005

   /* Transition SBHE to switch chip from 8-bit to 16-bit */
   inb( PORT_PKTPG_PTR   );
   inb( PORT_PKTPG_PTR+1 );
   inb( PORT_PKTPG_PTR   );
   inb( PORT_PKTPG_PTR+1 ); 

#if (BSP_CS8900_EEPROM == YES)
   /* Wait until the EEPROM is not busy */
   for ( x=0; x<MAXLOOP; x++ )
      if ( !(csReadPacketPage(PKTPG_SELF_ST)&SELF_ST_SI_BUSY) )
         break;
   if ( x == MAXLOOP ) return FAILURE;

   /* Wait until initialization is done */
   for ( x=0; x<MAXLOOP; x++ )
      if ( csReadPacketPage(PKTPG_SELF_ST)&SELF_ST_INIT_DONE )
         break;
   if ( x == MAXLOOP ) return FAILURE;
#endif  /* BSP_CS8900_EEPROM */

   return SUCCESS;
}


#if (BSP_CS8900_EEPROM == YES)

/*****************************************************************************/
/*  csInitFromEEPROM()                                                       */
/*                                                                           */
/*  This routine initializes the chip using configuration information        */
/*  obtained from the EEPROM attached to the chip.  This routine also reads  */
/*  the interrupt level and hardware address from the EEPROM and saves them  */
/*  in the csHardwareAddr and csIntNumber global variables.                  */
/*                                                                           */
/*****************************************************************************/

unsigned long csInitFromEEPROM(void)
{
   unsigned short SelfStatus;
   unsigned short ISAConfig;
   unsigned short MemBase;
   unsigned short BusCtl;
   unsigned short AdapterConfig;
   unsigned short SelfCtl;
   unsigned short XmitCtl;

   /* Verify that the EEPROM is present and OK */
   SelfStatus = csReadPacketPage( PKTPG_SELF_ST );
   if ( !((SelfStatus & SELF_ST_EEP_PRES) && (SelfStatus & SELF_ST_EEP_OK)) )
      return FAILURE;

   /* Get ISA configuration from the EEPROM */
   if ( csReadEEPROM(EEPROM_ISA_CFG,&ISAConfig) == FAILURE )
      return FAILURE;

   /* If memory mode is enabled */
   if ( ISAConfig & ISA_CFG_MEM_MODE )
   {
      /* Get memory base address from EEPROM */
      if ( csReadEEPROM(EEPROM_MEM_BASE,&MemBase) == FAILURE )
         return FAILURE;

      MemBase &= MEM_BASE_MASK;  /* Clear unused bits */

      /* If external logic is present for address decoding */
      if ( csReadPacketPage(PKTPG_SELF_ST) & SELF_ST_EL_PRES )
      {
         /* Program the external logic to decode address bits SA20-SA23 */
         csWritePacketPage( PKTPG_EEPROM_CMD, (MemBase>>12) | EEPROM_CMD_ELSEL);
      }

      /* Setup chip for memory mode */
      csWritePacketPage( PKTPG_MEM_BASE,  (MemBase<<8) & 0xFFFF );
      csWritePacketPage( PKTPG_MEM_BASE+2, MemBase>>8 );
      BusCtl = BUS_CTL_MEM_MODE;
      if ( ISAConfig & ISA_CFG_USE_SA )
         BusCtl |= BUS_CTL_USE_SA;
      csWritePacketPage( PKTPG_BUS_CTL, BusCtl );

      /* Setup global variables for memory mode */
      cs_pPacketPage = (unsigned short *)((((unsigned long)MemBase)<<8)) + ETHER_BASE;
      csInMemoryMode = TRUE;  /* We are in memory mode now! */
   }

   /* If IOCHRDY is enabled then clear the bit in the BusCtl register */
   BusCtl = csReadPacketPage( PKTPG_BUS_CTL );
   if ( ISAConfig & ISA_CFG_IOCHRDY )
      csWritePacketPage( PKTPG_BUS_CTL, BusCtl & ~BUS_CTL_IOCHRDY );
   else
      csWritePacketPage( PKTPG_BUS_CTL, BusCtl | BUS_CTL_IOCHRDY );

   /* Save the interrupt number to be initialized later */
   csIntNumber = ISAConfig & ISA_CFG_IRQ_MASK;
   if ( csIntNumber == 3 )
      csIntNumber = 5;
   else
      csIntNumber += 10;

   /* Get adapter configuration from the EEPROM */
   if ( csReadEEPROM(EEPROM_ADPTR_CFG,&AdapterConfig) == FAILURE )
      return FAILURE;

   /* Set the Line Control register to match the media type */
   if ( (AdapterConfig & ADPTR_CFG_MEDIA) == ADPTR_CFG_10BASET )
      csWritePacketPage( PKTPG_LINE_CTL, LINE_CTL_10BASET );
   else
      csWritePacketPage( PKTPG_LINE_CTL, LINE_CTL_AUI_ONLY );

   /* Set the BSTATUS/HC1 pin to be used as HC1 */
   /* HC1 is used to enable the DC/DC converter */
   SelfCtl = SELF_CTL_HC1E;

   /* If the media type is 10Base2 */
   if ( (AdapterConfig & ADPTR_CFG_MEDIA) == ADPTR_CFG_10BASE2 )
   {
      /* Enable the DC/DC converter */
      /* If the DC/DC converter has a low enable */
      if ( (AdapterConfig & ADPTR_CFG_DCDC_POL) == 0 )
         /* Set the HCB1 bit, which causes the HC1 pin to go low */
         SelfCtl |= SELF_CTL_HCB1;
   }
   else  /* Media type is 10BaseT or AUI */
   {
      /* Disable the DC/DC converter */
      /* If the DC/DC converter has a high enable */
      if ( (AdapterConfig & ADPTR_CFG_DCDC_POL) != 0 )
         /* Set the HCB1 bit, which causes the HC1 pin to go low */
         SelfCtl |= SELF_CTL_HCB1;
   }
   csWritePacketPage( PKTPG_SELF_CTL, SelfCtl );

   /* If media type is 10BaseT */
   if ( (AdapterConfig & ADPTR_CFG_MEDIA) == ADPTR_CFG_10BASET )
   {
      /* Get transmission control from the EEPROM */
      if ( csReadEEPROM(EEPROM_XMIT_CTL,&XmitCtl) == FAILURE )
         return FAILURE;

      /* If full duplex mode then set the FDX bit in TestCtl register */
      if ( XmitCtl & XMIT_CTL_FDX )
         csWritePacketPage( PKTPG_TEST_CTL, TEST_CTL_FDX );
   }

   /* Get Individual Address from the EEPROM */
   if ( csReadEEPROM(EEPROM_IND_ADDR_H,&csHardwareAddr.word[0]) == FAILURE )
      return FAILURE;
   if ( csReadEEPROM(EEPROM_IND_ADDR_M,&csHardwareAddr.word[1]) == FAILURE )
      return FAILURE;
   if ( csReadEEPROM(EEPROM_IND_ADDR_L,&csHardwareAddr.word[2]) == FAILURE )
      return FAILURE;

   return SUCCESS;
}


/*****************************************************************************/
/*  csReadEEPROM()                                                           */
/*                                                                           */
/*  Read the specified offset within the EEPROM.                             */
/*                                                                           */
/*****************************************************************************/

int csReadEEPROM( unsigned short Offset, unsigned short *pValue )
{
   int x;

   /* Ensure that the EEPROM is not busy */
   for ( x=0; x<MAXLOOP; x++ )
      if ( !(csReadPacketPage(PKTPG_SELF_ST)&SELF_ST_SI_BUSY) )
         break;
   if ( x == MAXLOOP ) return FAILURE;

   /* Issue the command to read the offset within the EEPROM */
   csWritePacketPage( PKTPG_EEPROM_CMD, Offset | EEPROM_CMD_READ );

   /* Wait until the command is completed */
   for ( x=0; x<MAXLOOP; x++ )
      if ( !(csReadPacketPage(PKTPG_SELF_ST)&SELF_ST_SI_BUSY) )
         break;
   if ( x == MAXLOOP ) return FAILURE;

   /* Get the EEPROM data from the EEPROM Data register */
   *pValue = csReadPacketPage( PKTPG_EEPROM_DATA );

   return SUCCESS;
}


#else  /* BSP_CS8900_EEPROM == NO */


/*****************************************************************************/
/*  csInitFromBSPDefs()                                                      */
/*                                                                           */
/*  This routine initializes the chip using definitions in the BSP.H file.   */
/*  This routine also converts the hardware address string specified by the  */
/*  BSP_CS8900_IND_ADDR definition and stores it in the csHardwareAddr       */
/*  global variable.  The defined interrupt level is stored in the           */
/*  csIntNumber global variable.                                             */
/*                                                                           */
/*****************************************************************************/

unsigned long csInitFromBSPDefs(void)
{
   unsigned short BusCtl;
   unsigned short SelfCtl;
   char *pChar;

   #if (BSP_CS8900_MEM_MODE == YES)

      /* If external logic is present for address decoding */
      if ( csReadPacketPage(PKTPG_SELF_ST) & SELF_ST_EL_PRES )
      {
         /* Program the external logic to decode address bits SA20-SA23 */
         csWritePacketPage( PKTPG_EEPROM_CMD,
               (BSP_CS8900_MEM_BASE>>20) | EEPROM_CMD_ELSEL );
      }

      /* Setup chip for memory mode */
      csWritePacketPage( PKTPG_MEM_BASE,   BSP_CS8900_MEM_BASE & 0xFFFF );
      csWritePacketPage( PKTPG_MEM_BASE+2, BSP_CS8900_MEM_BASE>>16 );
      BusCtl = BUS_CTL_MEM_MODE;
      #if (BSP_CS8900_USE_SA == YES)
         BusCtl |= BUS_CTL_USE_SA;
      #endif
      csWritePacketPage( PKTPG_BUS_CTL, BusCtl );

      /* Setup global variables for memory mode */
      cs_pPacketPage = (unsigned short *)(BSP_CS8900_MEM_BASE + ETHER_BASE); //added base May-23-2005
      csInMemoryMode = TRUE;  /* We are in memory mode now! */

   #endif  /* BSP_CS8900_MEM_MODE */

   /* If IOCHRDY is enabled then clear the bit in the BusCtl register */
   BusCtl = csReadPacketPage( PKTPG_BUS_CTL );
   #if (BSP_CS8900_IOCHRDY == YES)
      csWritePacketPage( PKTPG_BUS_CTL, BusCtl & ~BUS_CTL_IOCHRDY );
   #else
      csWritePacketPage( PKTPG_BUS_CTL, BusCtl | BUS_CTL_IOCHRDY );
   #endif

   /* Save the interrupt number to be initialized later */
   csIntNumber = BSP_CS8900_IRQ;

   /* Set the Line Control register to match the media type */
   #if (BSP_CS8900_MEDIA_TYPE == TEN_BASE_T )
      csWritePacketPage( PKTPG_LINE_CTL, LINE_CTL_10BASET );
   #else
      csWritePacketPage( PKTPG_LINE_CTL, LINE_CTL_AUI_ONLY );
   #endif

   /* Set the BSTATUS/HC1 pin to be used as HC1 */
   /* HC1 is used to enable the DC/DC converter */
   SelfCtl = SELF_CTL_HC1E;

   /* If the media type is 10Base2 */
   #if (BSP_CS8900_MEDIA_TYPE == TEN_BASE_2 )
      /* Enable the DC/DC converter */
      /* If the DC/DC converter has a low enable */
      #if (BSP_CS8900_DCDC_POL == LOW)
         /* Set the HCB1 bit, which causes the HC1 pin to go low */
         SelfCtl |= SELF_CTL_HCB1;
      #endif
   #else  /* Media type is 10BaseT or AUI */
      /* Disable the DC/DC converter */
      /* If the DC/DC converter has a high enable */
      #if (BSP_CS8900_DCDC_POL == HIGH)
         /* Set the HCB1 bit, which causes the HC1 pin to go low */
         SelfCtl |= SELF_CTL_HCB1;
      #endif
   #endif  /* BSP_CS8900_MEDIA_TYPE */
   csWritePacketPage( PKTPG_SELF_CTL, SelfCtl );

   /* If media type is 10BaseT */
   #if (BSP_CS8900_MEDIA_TYPE == TEN_BASE_T )
      /* If full duplex mode then set the FDX bit in TestCtl register */
      #if (BSP_CS8900_FDX == YES )
         csWritePacketPage( PKTPG_TEST_CTL, TEST_CTL_FDX );
      #endif
   #endif
   
   //config = csReadPacketPage(PKTPG_LINE_ST);
   /* Convert and save the Individual Address string */
   pChar = BSP_CS8900_IND_ADDR;
   pChar = csHexWord( pChar, &csHardwareAddr.word[0] );
   if ( pChar == NULL ) return FAILURE;
   pChar = csHexWord( pChar, &csHardwareAddr.word[1] );
   if ( pChar == NULL ) return FAILURE;
   pChar = csHexWord( pChar, &csHardwareAddr.word[2] );
   if ( pChar == NULL ) return FAILURE;

   return SUCCESS;
}


/*****************************************************************************/
/*  csHexWord()                                                              */
/*                                                                           */
/*  This routine converts a sequence of hex characters to the 16-bit value   */
/*  that they represent.  The address of the first hex character is passed   */
/*  in the pChar parameter and the address just beyond the last hex          */
/*  character is returned.  The 16-bit variable pointed to by the pWord      */
/*  parameter is updated with the converted 16-bit value.  If an error       */
/*  occurred then NULL is returned.                                          */
/*                                                                           */
/*****************************************************************************/

char *csHexWord( char *pChar, unsigned short *pWord )
{
   union
   {
      unsigned short word;
      unsigned char  byte[2];
   } Value;

   /* Get the value of the first hex byte */
   pChar = csHexByte( pChar, &Value.byte[0] );
   if ( pChar==NULL || *pChar==0 ) return NULL;

   /* Get the value of the second hex byte */
   pChar = csHexByte( pChar, &Value.byte[1] );
   if ( pChar==NULL ) return NULL;

   /* Save value of the hex word */
   *pWord = Value.word;

   return pChar;
}


/*****************************************************************************/
/*  csHexByte()                                                              */
/*                                                                           */
/*  This routine converts a sequence of hex characters to the 8-bit value    */
/*  that they represent.  There may be zero, one or two hex characters and   */
/*  they may be optionally terminated with a colon or a zero byte.           */
/*  The address of the first hex character is passed in the pChar parameter  */
/*  and the address just beyond the last hex character is returned.          */
/*  The 8-bit variable pointed to by the pByte parameter is updated with     */
/*  the converted 8-bit value.  If an error occurred then NULL is returned.  */
/*                                                                           */
/*****************************************************************************/

char *csHexByte( char *pChar, unsigned char *pByte )
{
   int x;

   /* Inititalize the byte value to zero */
   *pByte = 0;

   /* Process two hex characters */
   for ( x=0; x<2; x++,pChar++ )
   {
      /* Stop early if find a colon or end of string */
      if ( *pChar==':' || *pChar==0 ) break;

      /* Convert the hex character to a value */
      if ( *pChar >= '0' && *pChar <= '9' )
         *pByte = (*pByte * 16) + *pChar - '0';
      else if ( *pChar >= 'a' && *pChar <= 'f' )
         *pByte = (*pByte * 16) + *pChar - 'a' + 10;
      else if ( *pChar >= 'A' && *pChar <= 'F' )
         *pByte = (*pByte * 16) + *pChar - 'A' + 10;
      else return NULL;  /* Illegal character */
   }

   /* Skip past terminating colon */
   if ( *pChar == ':' ) pChar++;

   return pChar;
}

#endif  /* BSP_CS8900_EEPROM */


/*****************************************************************************/
/*  csInitInterrupt()                                                        */
/*                                                                           */
/*  This routine initializes the chip, interrupt descriptor table and the    */
/*  programmable interrupt controller for the interrupt level contained in   */
/*  in the csIntNumber global variable.                                      */
/*                                                                           */
/*  If the interrupt level is not valid then FAILURE is returned.            */
/*                                                                           */
/*****************************************************************************/

unsigned long csInitInterrupt(void)
{
   /* Verify that the interrupt number is vaild */
   if ( !(csIntNumber==5||csIntNumber==10||csIntNumber==11||csIntNumber==12) )
      return FAILURE;

   /* Set the interrupt number in the chip */
   if ( csIntNumber == 5 )
      csWritePacketPage( PKTPG_INT_NUM, 3 );
   else
      csWritePacketPage( PKTPG_INT_NUM, csIntNumber-10 );

   /* Setup the interrupt descriptor table */
   

   /* Enable the interrupt at the PIC */
   

   /* Enable interrupt at the chip */
   csWritePacketPage( PKTPG_BUS_CTL,
         csReadPacketPage(PKTPG_BUS_CTL) | BUS_CTL_INT_ENBL );

   /* Enable reception and transmission of frames */
   csWritePacketPage( PKTPG_LINE_CTL,
         csReadPacketPage(PKTPG_LINE_CTL) | LINE_CTL_RX_ON | LINE_CTL_TX_ON );

   return SUCCESS;
}


/*****************************************************************************/
/*  csSend()                                                                 */
/*                                                                           */
/*  This routine is called when pNA requests that a new message (packet) be  */
/*  transmitted.  The new transmit request is placed on the transmit request */
/*  queue.  If a transmit request is not currently in progress, then the new */
/*  transmit request is started, else it waits its turn in the queue.        */
/*                                                                           */
/*****************************************************************************/

unsigned long csSend( PIA pDestAddr, mblk_t *pSendParms )
{
   int TxInProgress;
   unsigned short BusStatus;
   unsigned long IntState;

   /* Disable interrupts at the CPU */
   

   /* If the Transmit Request queue is empty then */
   /* there is no transmit in progress */
   if ( cs_pTxReqQueue == NULL )
      TxInProgress = FALSE;
   else
      TxInProgress = TRUE;

   /* Put this transmit request on the Transmit Request queue */
   if ( csEnqueueTxReq(pDestAddr,pSendParms) == FAILURE )
   {
      //splx( IntState );
      return FAILURE; //ENOBUFS;  /* Transmit request queue is full */
   }

   /* If a transmit request is not currently being transmitted */
   if ( !TxInProgress )
   {
      /* Request that a transmit be started */
      BusStatus = csRequestTransmit();

      /* If there was an error with the transmit bid */
      if ( BusStatus & BUS_ST_TX_BID_ERR )
      {
         /* Free the transmit request message */
         // NON NECESSARIO (*cs_p_freemsg)( cs_pTxReqQueue->pMsg ); 
         /* Remove the transmit request from the transmit request queue */
         csDequeueTxReq();
         //splx( IntState );
         return FAILURE; //EMSGSIZE;  /* Bad transmit length */
      }
      else if ( BusStatus & BUS_ST_RDY4TXNOW )
      {
         /* The chip is ready for transmission now */
         /* Copy the message to the chip to start the transmit */
         csCopyTxFrame();
         /* Free the transmit request message */
         // NON NECESSARIO (*cs_p_freemsg)( cs_pTxReqQueue->pMsg );
      }
   }

   /* Restore the interrupt state at the CPU */
   //splx( IntState );

   return SUCCESS;
}


/*****************************************************************************/
/*  csRequestTransmit()                                                      */
/*                                                                           */
/*  This routine requests that a transmit be started by writing a transmit   */
/*  command and a frame length to the chip.  The contents of the BusStatus   */
/*  register is returned which indicates the success of the request.         */
/*                                                                           */
/*****************************************************************************/

unsigned short csRequestTransmit(void)
{
   /* Request that the transmit be started after all data has been copied */
   if ( csInMemoryMode )
   {
      csWritePacketPage( PKTPG_TX_CMD, TX_CMD_START_ALL );
      csWritePacketPage( PKTPG_TX_LENGTH, cs_pTxReqQueue->Length+HEADER_LENGTH);
   }
   else  /* In IO mode */
   {
      outw( PORT_TX_CMD, TX_CMD_START_ALL );
      outw( PORT_TX_LENGTH, cs_pTxReqQueue->Length+HEADER_LENGTH );
   }

   /* Return the BusStatus register which indicates success of the request */
   return csReadPacketPage( PKTPG_BUS_ST );
}


/*****************************************************************************/
/*  csCopyTxFrame()                                                          */
/*                                                                           */
/*  This routine builds an ethernet header in the chip's transmit frame      */
/*  buffer and then copies the frame data from the list of message blocks    */
/*  to the chip's transmit frame buffer.  When all the data has been copied  */
/*  then the chip automatically starts to transmit the frame.                */
/*                                                                           */
/*  The reason why this "simple" copy routine is so long and complicated is  */
/*  because all reads and writes to the chip must be done as 16-bit words.   */
/*  If a message block has an odd number of bytes, then the last byte must   */
/*  be saved and combined with the first byte of the next message block.     */
/*                                                                           */
/*****************************************************************************/

void csCopyTxFrame(void)
{
   unsigned short *pFrame;
   unsigned short *pBuff;
   unsigned short *pBuffLimit;
   unsigned char  *pStart;
   mblk_t *pMsg;
   int HaveExtraByte;
   union
   {
      unsigned char  byte[2];
      unsigned short word;
   } Straddle;


   /* Put the header into the frame buffer */
   if ( csInMemoryMode )
   {
      pFrame = cs_pPacketPage + (PKTPG_TX_FRAME/2);
      *pFrame++ = cs_pTxReqQueue->pDestAddr->word[0];
      *pFrame++ = cs_pTxReqQueue->pDestAddr->word[1];
      *pFrame++ = cs_pTxReqQueue->pDestAddr->word[2];
      *pFrame++ = csHardwareAddr.word[0];
      *pFrame++ = csHardwareAddr.word[1];
      *pFrame++ = csHardwareAddr.word[2];
      //*pFrame++ = htons( cs_pTxReqQueue->Type );
	  *pFrame++ = cs_pTxReqQueue->Type;
   }
   else  /* In IO mode */
   {
      outw( PORT_RXTX_DATA, cs_pTxReqQueue->pDestAddr->word[0] );
      outw( PORT_RXTX_DATA, cs_pTxReqQueue->pDestAddr->word[1] );
      outw( PORT_RXTX_DATA, cs_pTxReqQueue->pDestAddr->word[2] );
      outw( PORT_RXTX_DATA, csHardwareAddr.word[0] );
      outw( PORT_RXTX_DATA, csHardwareAddr.word[1] );
      outw( PORT_RXTX_DATA, csHardwareAddr.word[2] );
      //outw( PORT_RXTX_DATA, htons(cs_pTxReqQueue->Type) );
	  outw( PORT_RXTX_DATA, cs_pTxReqQueue->Type );
   }

   //HaveExtraByte = FALSE;  /* Start out with no extra byte */

	pMsg=cs_pTxReqQueue->pMsg; // reference to message...	

   /* Process the message */
   
	pStart = pMsg->buff;  /* Start copying from the first byte */

      /* Point pBuff to the correct starting point */
      pBuff = (unsigned short *)pStart;

      /* If there are odd bytes remaining in the buffer */
      if ( pMsg->dim & 1 )
      {
         /* Point pBuffLimit to the extra byte */
         pBuffLimit = (unsigned short *)(pMsg->buff + pMsg->dim - 2);
		 HaveExtraByte = TRUE;
      }
      else  /* There is an even number of bytes remaining */
      {
         /* Point pBuffLimit to just beyond the last word */
         pBuffLimit = (unsigned short *)(pMsg->buff + pMsg->dim - 1);
		 HaveExtraByte = FALSE;
      }

      /* Copy the words in the buffer to the frame */
      if ( csInMemoryMode )
         while ( pBuff < pBuffLimit ) *pFrame++ = *pBuff++;
      else
         while ( pBuff < pBuffLimit ) outw( PORT_RXTX_DATA, *pBuff++ );

      /* If there is an extra byte left over in this buffer */
      if ( HaveExtraByte ){
         /* Save the extra byte for later */
         Straddle.byte[0] = *(unsigned char *)pBuff;
     	 Straddle.byte[1] = 0;
     

    /* Save the last word */
        if ( csInMemoryMode )
          *pFrame = Straddle.word;
        else
          outw( PORT_RXTX_DATA, Straddle.word );
    }
}



/*****************************************************************************/
/*  csProcessISQ()                                                           */
/*                                                                           */
/*  This routine processes the events on the Interrupt Status Queue.  The    */
/*  events are read one at a time from the ISQ and the appropriate event     */
/*  handlers are called.  The ISQ is read until it is empty.  If the chip's  */
/*  interrupt request line is active, then reading a zero from the ISQ will  */
/*  deactivate the interrupt request line.                                   */
/*                                                                           */
/*****************************************************************************/

void csProcessISQ(void)
{
   unsigned short Event;
   unsigned short dummy;

   /* Read an event from the Interrupt Status Queue */

   if ( csInMemoryMode )
      Event = csReadPacketPage( PKTPG_ISQ );
   else
      Event = inw( PORT_ISQ );

   /* Process all the events in the Interrupt Status Queue */
   while ( Event != 0 )
   {
      /* Dispatch to an event handler based on the register number */
      //printf("Event1: %X\n", Event);
      switch ( Event & REG_NUM_MASK )
      {
         case REG_NUM_RX_EVENT:
            csReceiveEvent( Event );
            break;
         case REG_NUM_TX_EVENT:
	    GPIO_BitWrite(GPIO1, 12, SET);
            csTransmitEvent();
	    GPIO_BitWrite(GPIO1, 12, RESET);
            break;
         case REG_NUM_BUF_EVENT:
		 // @jla add for rdy4txnow interrupt
		    csBufferEvent( Event );
         case REG_NUM_RX_MISS:
         case REG_NUM_TX_COL:
            /* Unused events */
            break;
      }

      /* Read another event from the Interrupt Status Queue */
      if ( csInMemoryMode )
         Event = csReadPacketPage( PKTPG_ISQ );
      else
         Event = inw( PORT_ISQ );
   }
}


/*****************************************************************************/
/*  csReceiveEvent()                                                         */
/*                                                                           */
/*  This routine is called whenever a frame has been received at the chip.   */
/*  This routine gets a data buffer from pNA, copies the received frame      */
/*  into the data buffer and passes the frame up to pNA by calling pNA's     */
/*  Announce_Packet() entry point.                                           */
/*                                                                           */
/*****************************************************************************/

void csReceiveEvent( unsigned short RxEvent )
{
   IA SourceAddr, DestAddr;
   unsigned short *pFrame;
   unsigned short *pBuff;
   unsigned short *pBuffLimit;
   unsigned short MsgLength;
   unsigned short Type;
   mblk_t *pMsg;
   int i;
   unsigned short *tmp;
   int add;
   union
   {
      unsigned char  byte[2];
      unsigned short word;
   } Last;

   /* Verify that it is an RxOK event */
   if ( !(RxEvent & RX_EVENT_RX_OK) ) return;

   pMsg = &RxMsg;   
   if ( pMsg == NULL )  /* If pNA doesn't have any more buffers */
   {
      /* Discard the received frame */
      csWritePacketPage( PKTPG_RX_CFG,
            csReadPacketPage(PKTPG_RX_CFG) | RX_CFG_SKIP );
      return;
   }

   /* Setup pointers to the buffer for copying */
   pBuff = (unsigned short *)pMsg->buff;
   tmp = pBuff;
   

   /* Read the header from the frame buffer */
   if ( csInMemoryMode )
   {
      pFrame = cs_pPacketPage + (PKTPG_RX_LENGTH/2);
      MsgLength = *pFrame++ - HEADER_LENGTH;
      DestAddr.word[0]   = *pFrame++;
      DestAddr.word[1]   = *pFrame++;
      DestAddr.word[2]   = *pFrame++;
      SourceAddr.word[0] = *pFrame++;
      SourceAddr.word[1] = *pFrame++;
      SourceAddr.word[2] = *pFrame++;
      Type = *pFrame++;
   }
   else  /* In IO mode */
   {
      inw( PORT_RXTX_DATA );  /* Discard RxStatus */
      MsgLength = inw( PORT_RXTX_DATA );

      DestAddr.word[0]   = inw( PORT_RXTX_DATA );
      DestAddr.word[1]   = inw( PORT_RXTX_DATA );
      DestAddr.word[2]   = inw( PORT_RXTX_DATA );
      SourceAddr.word[0] = inw( PORT_RXTX_DATA );
      SourceAddr.word[1] = inw( PORT_RXTX_DATA );
      SourceAddr.word[2] = inw( PORT_RXTX_DATA );

      Type = inw( PORT_RXTX_DATA );
      
   }

   /* Swap the the bytes around for Intel CPU (little endian)*/

  *pBuff++ = DestAddr.word[0]; 
  *pBuff++ = DestAddr.word[1];
  *pBuff++ = DestAddr.word[2];
  *pBuff++ = SourceAddr.word[0];
  *pBuff++ = SourceAddr.word[1];
  *pBuff++ = SourceAddr.word[2];
  *pBuff++ = Type;
  Type = htons( Type );
   /* Verify that the frame type is acceptable */
   if ( !(Type==FRAME_TYPE_IP || Type==FRAME_TYPE_ARP) )
   {
      /* Discard the received frame */
      csWritePacketPage( PKTPG_RX_CFG,
      				csReadPacketPage(PKTPG_RX_CFG) | RX_CFG_SKIP );
      return;
   }

   pBuffLimit = tmp + (MsgLength/2);   

   /* Copy the frame from the chip to the pNA buffer */
   if ( csInMemoryMode )
      while ( pBuff < pBuffLimit ) *pBuff++ = *pFrame++;
   else
      while ( pBuff < pBuffLimit ) *pBuff++ = inw( PORT_RXTX_DATA );
	
   
   
   /* If there is an extra byte at the end of the frame */
   if ( MsgLength & 1 )
   {
      /* Read the byte as a word */
      if ( csInMemoryMode )
         Last.word = *pFrame;
      else
         Last.word = inw( PORT_RXTX_DATA );

      /* Write the byte as a byte */
      *pBuff++ = Last.byte[0];
   }
   pMsg->type = Type;
   pMsg->dim = MsgLength;// - HEADER_LENGTH;

   (*cs_pAnnounce_Packet)( pMsg );
   
}


// @jla added this routine to send frame on rdy4tx interrupt
/*****************************************************************************/
/*  csBufferEvent()                                                         */
/*                                                                           */
/*  This routine is called whenever a buffer event has occurred.             */
/*  It only handles Rdy4TXNow events.  All other buffer events are ignored.  */
/*****************************************************************************/

void csBufferEvent( unsigned short BufEvent )
{

	if ( BufEvent & BUF_EVENT_RDY4TX )
	{
	    /* The chip is ready for transmission now */
	    /* Copy the message to the chip to start the transmit */
	    csCopyTxFrame();
	}

}


/*****************************************************************************/
/*  csTransmitEvent()                                                        */
/*                                                                           */
/*  This routine is called whenever the transmission of a frame has          */
/*  completed (either successfully or unsuccessfully).  This routine         */
/*  removes the completed transmit request from the transmit request queue.  */
/*  If there are more transmit requests waiting, then start the transmission */
/*  of the next transmit request.                                            */
/*                                                                           */
/*****************************************************************************/

void csTransmitEvent(void)
{
   unsigned short BusStatus;

   /* Remove the completed transmit request from the transmit request queue */
   csDequeueTxReq();

   /* While there is still another transmit request to do */
   while ( cs_pTxReqQueue != NULL )
   {
      /* Request to start a new transmit */
      BusStatus = csRequestTransmit();

      /* If there was an error in the transmit bid */
      if ( BusStatus & BUS_ST_TX_BID_ERR )
      {
         /* Discard the bad transmit request */
         csDequeueTxReq();
         /* Loop up to do the next transmit request */
      }
      else if ( BusStatus & BUS_ST_RDY4TXNOW )
      {
         /* The chip is ready for transmit now */
         /* Copy the frame to the chip to start transmission */
         csCopyTxFrame();
         break;  /* Exit this routine */
      }
   }
}


/*****************************************************************************/
/*  csReadPacketPage()                                                       */
/*                                                                           */
/*  Reads the PacketPage register at the specified offset.                   */
/*                                                                           */
/*****************************************************************************/

unsigned short csReadPacketPage( unsigned short Offset )
{
//short tmp;
   if ( csInMemoryMode )
   {
      return *(cs_pPacketPage+(Offset/2));
   }
   else  /* In IO mode */
   {
      outw( PORT_PKTPG_PTR, Offset );
      //tmp = inw( PORT_PKTPG_DATA );
      return inw( PORT_PKTPG_DATA );
   }
}


/*****************************************************************************/
/*  csWritePacketPage()                                                      */
/*                                                                           */
/*  Writes to the PacketPage register at the specified offset.               */
/*                                                                           */
/*****************************************************************************/

void csWritePacketPage( unsigned short Offset, unsigned short Value )
{
   if ( csInMemoryMode )
   {
      *(cs_pPacketPage+(Offset/2)) = Value;
   }
   else  /* In IO mode */
   {
      outw( PORT_PKTPG_PTR, Offset );
      outw( PORT_PKTPG_DATA, Value );
   }
}


/*****************************************************************************/
/*  csEnqueueTxReq()                                                         */
/*                                                                           */
/*  This routine places a transmit request at the end of the Transmit        */
/*  Request queue.                                                           */
/*                                                                           */
/*****************************************************************************/

unsigned long csEnqueueTxReq( PIA pDestAddr, mblk_t *pSendParms )
{
   PTXREQ pTxReq;

   /* If there are no more transmit request structures available */
   if ( cs_pTxFreeQueue == NULL )
      return FAILURE;

   /* If the transmit request queue is empty */
   if ( cs_pTxReqQueue == NULL )
   {
      /* Move a transmit request structure from the head of the */
      /* free queue to the head of the request queue */
      pTxReq = cs_pTxReqQueue = cs_pTxFreeQueue;
      cs_pTxFreeQueue = cs_pTxFreeQueue->pNext;
   }
   else  /* Transmit request queue is not empty */
   {
      /* Move a transmit request structure from the head of the */
      /* free queue to the tail of the request queue */
      for ( pTxReq=cs_pTxReqQueue; pTxReq->pNext!=NULL; pTxReq=pTxReq->pNext );
      pTxReq->pNext = cs_pTxFreeQueue;
      cs_pTxFreeQueue = cs_pTxFreeQueue->pNext;
      pTxReq = pTxReq->pNext;
   }

   /* Fill in the new transmit request structure */
   pTxReq->pNext     = NULL;
   pTxReq->pDestAddr = pDestAddr;
   pTxReq->pMsg      = pSendParms;
   pTxReq->Length    = pSendParms->dim;
   pTxReq->Type      = pSendParms->type;

   return SUCCESS;
}


/*****************************************************************************/
/*  csDequeueTxReq()                                                         */
/*                                                                           */
/*  This routine removes a transmit request from the head of the Transmit    */
/*  Request queue.                                                           */
/*                                                                           */
/*****************************************************************************/

void csDequeueTxReq(void)
{
   PTXREQ pTxReq;

   /* If the transmit request queue is not empty */
   if ( cs_pTxReqQueue != NULL )
   {
      /* Move the transmit request structure at the head of the */
      /* transmit queue to the head of the free queue */
      pTxReq = cs_pTxReqQueue;
      cs_pTxReqQueue = pTxReq->pNext;
      pTxReq->pNext = cs_pTxFreeQueue;
      cs_pTxFreeQueue = pTxReq;
   }
}


#endif /* (BSP_LAN1 == YES) && (BSP_LANCARD_MODEL == CS8900) */

