/******************** (C) COPYRIGHT 2003 STMicroelectronics ********************
* File Name          : lcd.c
* Author             : MCD Application Team
* Date First Issued  : 20/05/2003
* Description        : This file provides all the functions to drive the LCD
*                      16 characters x 2 lines
********************************************************************************
* History:
*  20/05/03 : First Version.
*******************************************************************************/
#include <stdio.h>
#include <string.h>
#include "lcd.h"

#ifdef LCD_I2C
// ********************************
// LCD is driven through an I2C bus
// ********************************

#include "i2c.h"
#include "gpio.h"

#define LCD_I2C_BUS  I2C0

#define I2C_SCL 0x2000
#define I2C_SDA 0x4000

static void LCD_Delay_1ms(void)
{
  int j;
  for(j=0; j<5592; j++);
}

void LCD_Delay(u16 time)
{
  int i=0;
  while (i!=time)
  {
    LCD_Delay_1ms();
    i++;
  }
}

void LCD_Init(u32 speed)
{
  GPIO_Config (GPIO1, I2C_SCL, GPIO_AF_OD);
  GPIO_Config (GPIO1, I2C_SDA, GPIO_AF_OD);

  I2C_OnOffConfig (LCD_I2C_BUS, DISABLE);
  I2C_Init (LCD_I2C_BUS);
  I2C_FCLKConfig (LCD_I2C_BUS);
  I2C_OnOffConfig (LCD_I2C_BUS, ENABLE);
  // Set the Highest Speed
  I2C_SpeedConfig (LCD_I2C_BUS, speed);
  I2C_AcknowledgeConfig (LCD_I2C_BUS, ENABLE);
  // Enable Start generation
  I2C_STARTGenerate (LCD_I2C_BUS, ENABLE);
  while (I2C_FlagStatus (LCD_I2C_BUS, DIRECT, I2C_SB)==RESET);
  // Send the eeprom address
  I2C_AddressSend (LCD_I2C_BUS,0x50,I2C_Mode7,I2C_TX);
  // Test the end of Address transmission
  while (I2C_FlagStatus (LCD_I2C_BUS, DIRECT, I2C_ENDAD )==RESET);
  I2C_FlagClear (LCD_I2C_BUS, I2C_ENDAD);
}

void LCD_AutoScrollOn(void)
{
  I2C_ByteSend (LCD_I2C_BUS, 0xFE);
  I2C_ByteSend (LCD_I2C_BUS, 0x51);
}

void LCD_AutoScrollOff(void)
{
  I2C_ByteSend (LCD_I2C_BUS, 0xFE);
  I2C_ByteSend (LCD_I2C_BUS, 0x52);
}

void LCD_SetCursorPosition(u8 col, u8 row)
{
  I2C_ByteSend (LCD_I2C_BUS, 0xFE);
  I2C_ByteSend (LCD_I2C_BUS, 0x47);
  I2C_ByteSend (LCD_I2C_BUS, col);
  I2C_ByteSend (LCD_I2C_BUS, row);
}

void LCD_SetCursorHome(void)
{
  I2C_ByteSend (LCD_I2C_BUS, 0xFE);
  I2C_ByteSend (LCD_I2C_BUS, 0x48);
}

void LCD_UnderlineCursorOn(void)
{
  I2C_ByteSend (LCD_I2C_BUS, 0xFE);
  I2C_ByteSend (LCD_I2C_BUS, 0x4A);
} 

void LCD_UnderlineCursorOff(void)
{
  I2C_ByteSend (LCD_I2C_BUS, 0xFE);
  I2C_ByteSend (LCD_I2C_BUS, 0x4B);
}

void LCD_BlockCursorOn(void)
{
  I2C_ByteSend (LCD_I2C_BUS, 0xFE);
  I2C_ByteSend (LCD_I2C_BUS, 0x53);
}

void LCD_BlockCursorOff(void)
{
  I2C_ByteSend (LCD_I2C_BUS, 0xFE);
  I2C_ByteSend (LCD_I2C_BUS, 0x54);
}

void LCD_CursorLeft(void)
{
  I2C_ByteSend (LCD_I2C_BUS, 0xFE);
  I2C_ByteSend (LCD_I2C_BUS, 0x4C);
}

void LCD_CursorRight(void)
{
  I2C_ByteSend (LCD_I2C_BUS, 0xFE);
  I2C_ByteSend (LCD_I2C_BUS, 0x4D);
}

void LCD_DisplayCharacter (u8 c, u8 col, u8 row)
{
  LCD_SetCursorPosition(col, row);
  I2C_ByteSend (LCD_I2C_BUS, c);
}

void LCD_CreateCharacter(u8 c, u8 *pattern)
{
  // not implemented
}

void LCD_String(char *String, u8 col, u8 row)
{
  LCD_SetCursorPosition(col, row);
  I2C_StringSend(LCD_I2C_BUS, String);
}

#endif // LCD_I2C


#ifdef LCD_EMI
// *********************************
// LCD is driven through an EMI bank
// *********************************

#include "emi.h"
#include "gpio.h"
#include "wdg.h"
#include "rccu.h"

#define LCD_EMI_BANK_NUMBER   2
#define LCD_EMI_BANK_ADDRESS  EMI_BANK2

// ***************************************************************************************************
// IMPORTANT NOTE:
// The LCD EMI does not work at 48 MHz (timings out of spec), so the main clock is slowed down
// at each high-level function entry, and switched back again to 48 MHz before returning.
// The LCD EMI works with the following configurations:
// PLL1 div  clock(MHz)  wait states
//        3         32            15
//        6         16            10
// ***************************************************************************************************

int slow_clock; // counter of nested calls to LCD_SlowClock() function
RCCU_RCLK_Clocks oldRCLKsource;
int oldPLL1Mul, oldPLL1Div;

static void LCD_SlowClock(void)
{
  if (slow_clock++ == 0)
  {
    oldRCLKsource = RCCU_RCLKClockSource();
    oldPLL1Mul = (RCCU->PLL1CR & 0x30) >> 4;  //PLLCONF
    oldPLL1Div =  RCCU->PLL1CR & 0x07;

    // set main clock to 32 MHz
    RCCU_RCLKSourceConfig ( RCCU_CLOCK2 );
    RCCU_PLL1Config ( RCCU_PLL1_Mul_12, RCCU_Div_3 );
    RCCU_RCLKSourceConfig ( RCCU_PLL1_Output );
  }
}

static void LCD_FastClock(void)
{
  if (--slow_clock == 0)
  {
    if (oldRCLKsource == RCCU_PLL1_Output)
    {
      // restore old clock
      RCCU_RCLKSourceConfig ( RCCU_CLOCK2 );
      RCCU_PLL1Config ( oldPLL1Mul, oldPLL1Div );
    }
    RCCU_RCLKSourceConfig(oldRCLKsource);
  }
}

/*
void LCD_Delay ( u32 time) //time in us
{
  WDG_ECFlagClear();
  WDG_PeriodValueConfig (time); 
  WDG_CntOnOffConfig ( ENABLE );
  while ( !(WDG_ECStatus() & 0x0001) );
  WDG_CntOnOffConfig ( DISABLE );
  WDG_ECFlagClear();
}
*/

static void LCD_Delay_1ms(void)
{
  int j;
  for(j=0; j<850; j++); // value adjusted for MCLK=32 MHz
}

void LCD_Delay(u16 time)
{
  int i=0;
  while (i!=time)
  {
    LCD_Delay_1ms();
    i++;
  }
}

static void LCD_Command ( u8 command )
{
  *(u8 *)(LCD_EMI_BANK_ADDRESS) = command;
  //wait
  LCD_Delay (1);
}

static void LCD_Data ( u8 data )
{
  *(u8 *)(LCD_EMI_BANK_ADDRESS + 4) = data;
  //wait
  LCD_Delay (1);
}

static void LCD_SetCharacterAddress ( u8 address )
{
  LCD_SlowClock();
  LCD_Command (0x40 + 8*address);
  LCD_FastClock();
}

/*
** LCD_AutoScrollOn: Activate the automatic scroll
*/
void LCD_AutoScrollOn(void)
{
  LCD_SlowClock();
  LCD_Command (0x07);
  LCD_FastClock();
}

/*
** LCD_AutoScrollOff: Deactivate the automatic scroll
*/
void LCD_AutoScrollOff(void)
{
  LCD_SlowClock();
  LCD_Command (0x04);
  LCD_FastClock();
}

/*
** LCD_SetCursorPosition: Position the LCD cursor at a desired row and column.
*/
void LCD_SetCursorPosition(u8 col, u8 row)
{
  LCD_SlowClock();
  switch (row) {
    case 1: LCD_Command (0x80 + col - 1); LCD_Delay (10); break;
    case 2: LCD_Command (0xc0 + col - 1); LCD_Delay (10); break;
    default: break;
  }
  LCD_FastClock();
}

/*
** LCD_SetCursorHome: Return the LCD cursor to the original position.
*/
void LCD_SetCursorHome(void)
{
  LCD_SlowClock();
  LCD_Command (0x02);
  //wait for 1.6 ms
  LCD_Delay (16);  
  LCD_FastClock();
}

/*
** LCD_UnderlineCursorOn: the cursor is displayed
*/ 
void LCD_UnderlineCursorOn(void)
{
  LCD_SlowClock();
  LCD_Command (0x0E);
  //wait for 1.6 ms
  LCD_Delay (16);  
  LCD_FastClock();
}

/*
** LCD_UnderlineCursorOff: the cursor disappears
*/ 
void LCD_UnderlineCursorOff(void)
{
  LCD_SlowClock();
  LCD_Command (0x0C);
  //wait for 1.6 ms
  LCD_Delay (16);  
  LCD_FastClock();
}

/*
** LCD_BlockCursorOn: the blinking is set
*/
void LCD_BlockCursorOn(void)
{
  LCD_SlowClock();
  LCD_Command (0x0D);
  //wait for 1.6 ms
  LCD_Delay (16);  
  LCD_FastClock();
}

/*
** LCD_BlockCursorOff: the blinking is over
*/
void LCD_BlockCursorOff(void)
{
  LCD_SlowClock();
  LCD_Command (0x0C);
  //wait for 1.6 ms
  LCD_Delay (16);  
  LCD_FastClock();
}

/*
** LCD_CursorLeft: move the cursor one character to the left
*/
void LCD_CursorLeft(void)
{
  LCD_SlowClock();
  LCD_Command (0x10);
  //wait for 1.6 ms
  LCD_Delay (16);  
  LCD_FastClock();
}

/*
** LCD_CursorRight: move the cursor one character to the right
*/
void LCD_CursorRight(void)
{
  LCD_SlowClock();
  LCD_Command (0x14);
  //wait for 1.6 ms
  LCD_Delay (16);  
  LCD_FastClock();
}

/*
** LCD_Init: initialize the LCD screen
*/
void LCD_Init(void)
{
  GPIO_Config (GPIO2, 0x0004, GPIO_AF_PP);
  EMI_Config ( LCD_EMI_BANK_NUMBER, EMI_ENABLE | EMI_WAITSTATE (0xF) | EMI_SIZE_8 );

  slow_clock = 0;
  LCD_SlowClock();

  //wait for 15 ms
  LCD_Delay (150);
  //function set
  LCD_Command (0x38);

  //wait for 4.1 ms
  LCD_Delay (41);  
  //function set 
  LCD_Command (0x38);

  //wait for 100 us
  LCD_Delay (2);
  //function set 
  LCD_Command (0x38);
 
  //function set, 8 bits long, 2 display lines, 5x8 dots display font type
  LCD_Command (0x38);
  //wait 1.6ms
  LCD_Delay (16);
 
  //display off 
  LCD_Command (0x08);
  //wait 1.6ms
  LCD_Delay (16);
 
  //display clear 
  LCD_Command (0x01);
  //wait 1.6ms  
  LCD_Delay (16);

  //entry mode set, assign cursor moving direction and disable the shift of entire display 
  LCD_Command (0x06);
  //wait 1.6ms
  LCD_Delay (1600);

  LCD_FastClock();
}

/*
** LCD_DisplayCharacter: Display a character at a given position.
*/
void LCD_DisplayCharacter (u8 c, u8 col, u8 row)
{
  LCD_SlowClock();
  LCD_SetCursorPosition(col, row);
  LCD_Data (c);
  //wait 1.6ms
  LCD_Delay (1);
  LCD_FastClock();
}

/*
** LCD_CreateCharacter: Create a character in the CGRAM (character generator RAM)
*/
void LCD_CreateCharacter(u8 c, u8 *pattern)
{
  int i;
  LCD_SlowClock();
  LCD_SetCharacterAddress (c);
  for (i=0; i<7; i++)
    LCD_Data (pattern[i]);
  LCD_Data (0x00);
  LCD_SlowClock();
}

/*
** LCD_String: Display a string at a given position.
*/
void LCD_String(char *String, u8 col, u8 row)
{
  int i;
  LCD_SlowClock();
  LCD_SetCursorPosition(col, row);
  for (i=0; String[i] != '\0'; i++)
  {
    LCD_Data (String[i]);
    //wait 1.6ms
    LCD_Delay (1);
  }
//    LCD_DisplayCharacter(String[i], col+i, row);
  LCD_FastClock();
}

#endif // LCD_EMI

// ***************************************************************
// Part of the LCD driver that is not dependant from its interface
// ***************************************************************

/*
** LCD_ClearLine: Clear a given line
*/
void LCD_ClearLine (u8 line)
{
  LCD_String("                ", 1, line);
}

/*
** LCD_ScrollMessage: Scroll a message string from right to left
*/
void LCD_ScrollMessage(char *String, u8 row, u16 time_1ms)
{
  #define LCD_SCROLL_MAX_LENGTH  80
  char szPadding[16+1]="                ";
  char szScroll[16+LCD_SCROLL_MAX_LENGTH+16+1];
  char save_char, *pc = szScroll+16;
  int nLoops = 16+strlen(String)+1;
  int i;

  sprintf(szScroll, "%s%s%s", szPadding, String, szPadding);  

  #ifdef LCD_EMI
  LCD_SlowClock();
  #endif

  LCD_ClearLine (3 - row);

  for (i=0; i<nLoops; i++)
  {
    save_char = *pc;
    *pc = '\0';
    LCD_String(szScroll+i, 1, row);
    *pc++ = save_char;
    LCD_Delay(time_1ms);
  }

  #ifdef LCD_EMI
  LCD_FastClock();
  #endif
}

/******************* (C) COPYRIGHT 2003 STMicroelectronics *****END OF FILE****/
