
/**************** (c) 2005  STMicroelectronics, Inc. **********************/

/**************************************************************************
     PROGRAM  : SINUSOIDAL CONTROL OF 3 PHASE BLDC MOTOR WITH HALL SENSOR
     COMPILER : COSMIC
     FILENAME : MAIN.C 
     RELEASE  : 1.0
     DATE     : May 2005
     AUTHOR   : Power Systems Application Lab
                STMicroelectronics, Inc.
                Schaumburg, IL - 60173
***************************************************************************

MODIFICATIONS :


******************************************************************************
 THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. ST MICROELECTRONICS
 SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES
 WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.
******************************************************************************/


#include "include.inc"

/*****************************************************************************/
/****************  DEMOBOARD USER INTERFACE HARDWARE MAPPING  ****************/
/*****************************************************************************/
/* TQFP 32 DEMOBOARD
**********************/
#define   CH_RV1     13
#define   CH_RV2     11
#define   CH_RV3     7

#define   SW1_PORT    PCDR
#define   SW1         BIT(0)

/* Functional Assignments for user interface hardware
******************************************************/

#define   ON_OFF_PORT    SW1_PORT
#define   ON_OFF_BIT     SW1

#define   CH_LOAD_ANGLE  CH_RV2
#define   CH_SINE_MAG    CH_RV1 

/*****************************************************************************/
/***********************  SYSTEM PARAMETER DEFINITIONS  **********************/
/*****************************************************************************/
/* ADC bit definitions
**********************/
#define   ADON       BIT(4)
#define   EOC        BIT(7)

/* Toggle switch debounce time settings
***************************************/
#define   DEBOUNCE_TIME_COUNT  50     /* at 2mS RTC intrpt, 
                                         50 --> 100mS debounce time */

/* Motor rotation direction looking at shaft
*********************************************/
#define  CW    0
#define  CCW   1
#define  DIRECTION  CCW

/* Motor speed estimation method
*********************************/
#define  PLL   1
#define  DIV   0
#define  MODE  DIV

/* Sine table settings
**********************/
#define   DEG_360           256          /* table size is 256 entries to 
                                            represent from 0 to 360 deg */
#define   DEG_120           (DEG_360/3)

#define   MCP0_COUNT        625          /* mag of triangular carrier,
                                            also sets sets carrier freq
                                            Fcarrier = 16MHz/(2*MCP0_COUNT) */
#define   CARRIER_MAX       MCP0_COUNT
#define   CARRIER_MIDDLE    ((CARRIER_MAX<<3)/2)   /* to match MCPU,V,W reg */
#define   SINE_MAG          (long)CARRIER_MIDDLE/65535   /* to scale sine table */

/*****************************************************************************/
/**************************  CONSTANTS DEFINITIONS  **************************/
/*****************************************************************************/

/* Sine Table (3rd harmonic injected)
**************************************/
const u16  uiSineTable[128] = {
    0 * SINE_MAG,   2555 * SINE_MAG,   5105 * SINE_MAG,   7645 * SINE_MAG,
10170 * SINE_MAG,  12676 * SINE_MAG,  15157 * SINE_MAG,  17609 * SINE_MAG,
20027 * SINE_MAG,  22407 * SINE_MAG,  24745 * SINE_MAG,  27036 * SINE_MAG,
29277 * SINE_MAG,  31464 * SINE_MAG,  33594 * SINE_MAG,  35663 * SINE_MAG,
37669 * SINE_MAG,  39609 * SINE_MAG,  41481 * SINE_MAG,  43282 * SINE_MAG,
45010 * SINE_MAG,  46665 * SINE_MAG,  48245 * SINE_MAG,  49749 * SINE_MAG,
51176 * SINE_MAG,  52527 * SINE_MAG,  53801 * SINE_MAG,  54998 * SINE_MAG,
56119 * SINE_MAG,  57165 * SINE_MAG,  58137 * SINE_MAG,  59036 * SINE_MAG,
59864 * SINE_MAG,  60623 * SINE_MAG,  61315 * SINE_MAG,  61942 * SINE_MAG,
62506 * SINE_MAG,  63011 * SINE_MAG,  63459 * SINE_MAG,  63852 * SINE_MAG,
64195 * SINE_MAG,  64490 * SINE_MAG,  64740 * SINE_MAG,  64949 * SINE_MAG,
65120 * SINE_MAG,  65256 * SINE_MAG,  65362 * SINE_MAG,  65439 * SINE_MAG,
65491 * SINE_MAG,  65522 * SINE_MAG,  65535 * SINE_MAG,  65532 * SINE_MAG,
65516 * SINE_MAG,  65491 * SINE_MAG,  65459 * SINE_MAG,  65423 * SINE_MAG,
65383 * SINE_MAG,  65344 * SINE_MAG,  65306 * SINE_MAG,  65271 * SINE_MAG,
65240 * SINE_MAG,  65215 * SINE_MAG,  65197 * SINE_MAG,  65185 * SINE_MAG,
65182 * SINE_MAG,  65185 * SINE_MAG,  65197 * SINE_MAG,  65215 * SINE_MAG,
65240 * SINE_MAG,  65271 * SINE_MAG,  65306 * SINE_MAG,  65344 * SINE_MAG,
65383 * SINE_MAG,  65423 * SINE_MAG,  65459 * SINE_MAG,  65491 * SINE_MAG,
65516 * SINE_MAG,  65532 * SINE_MAG,  65535 * SINE_MAG,  65522 * SINE_MAG,
65491 * SINE_MAG,  65439 * SINE_MAG,  65362 * SINE_MAG,  65256 * SINE_MAG,
65120 * SINE_MAG,  64949 * SINE_MAG,  64740 * SINE_MAG,  64490 * SINE_MAG,
64195 * SINE_MAG,  63852 * SINE_MAG,  63459 * SINE_MAG,  63011 * SINE_MAG,
62506 * SINE_MAG,  61942 * SINE_MAG,  61315 * SINE_MAG,  60623 * SINE_MAG,
59864 * SINE_MAG,  59036 * SINE_MAG,  58137 * SINE_MAG,  57165 * SINE_MAG,
56119 * SINE_MAG,  54998 * SINE_MAG,  53801 * SINE_MAG,  52527 * SINE_MAG,
51176 * SINE_MAG,  49749 * SINE_MAG,  48245 * SINE_MAG,  46665 * SINE_MAG,
45010 * SINE_MAG,  43282 * SINE_MAG,  41481 * SINE_MAG,  39609 * SINE_MAG,
37669 * SINE_MAG,  35663 * SINE_MAG,  33594 * SINE_MAG,  31464 * SINE_MAG,
29277 * SINE_MAG,  27036 * SINE_MAG,  24745 * SINE_MAG,  22407 * SINE_MAG,
20027 * SINE_MAG,  17609 * SINE_MAG,  15157 * SINE_MAG,  12676 * SINE_MAG,
10170 * SINE_MAG,   7645 * SINE_MAG,   5105 * SINE_MAG,   2555 * SINE_MAG
};

/*

const uBYTE step[2][8] =  {   // C B A
   {0, 1, 'x', 2, 5, 'x', 4, 3},
   {4, 3, 'x', 2, 5, 'x', 0, 1}
};

/**/
const u8 
    /* Hall output A, B, C select table for Hall Edge detection for
       each direction of rotation */
    phase_input_sel[2][3] = {
                               {  /* CW rotation */
                                  0x40,     /* phase B */
                                  0x80,     /* phase C */
                                  0         /* phase A */
                               },    
                               {  /* CCW rotation */
                                  0x80,     /* phase C */
                                  0,        /* phase A */
                                  0x40      /* phase B */
                               }
                            },
    
    /* Table contains sector information (0-5) where the table index is 
       framed by the code the three HALL SENSOR outputs CBA make 
       for each direction of rotation (CW / CCW) */
    sectorTable[2][8] =  {    /* sector #,    table index */
                             {                /* C B A --- CW direction */
                                 0,         /* 0 0 0  => 0 */
                                 1,           /* 0 0 1  => 1 */
                                 'x',           /* 0 1 0  => 2 */
                                 2,           /* 0 1 1  => 3 */
                                 5,           /* 1 0 0  => 4 */
                                 'x',           /* 1 0 1  => 5 */
                                 4,           /* 1 1 0  => 6 */
                                 3          /* 1 1 1  => 7 */
                             },
                             {                /* C B A --- CCW direction */
                                 4,         /* 0 0 0  => 0 */
                                 3,           /* 0 0 1  => 1 */
                                 'x',           /* 0 1 0  => 2 */
                                 2,           /* 0 1 1  => 3 */
                                 5,           /* 1 0 0  => 4 */
                                 'x',           /* 1 0 1  => 5 */
                                 0,           /* 1 1 0  => 6 */
                                 1          /* 1 1 1  => 7 */
                             }
                         },
                         
    /* angle table contains angular information at the beginning of each sector, 
       ie., each edge transition of Hall Sensor outputs on any phase. These are 
       obtained by matching the transition to phase A bemf */
    angleTable[6] = {
                        21,     /*  30 degrees  */
                        64,     /*  90 degrees  */
                        106,    /* 150 degrees  */
                        149,    /* 210 degrees  */
                        192,    /* 270 degrees  */
                        234     /* 330 degrees  */
                    };
    

/*****************************************************************************/
/***********************  GLOBAL VARIABLES DEFINITIONS  **********************/
/*****************************************************************************/
static u8   load_angle,    /* stores load angle set by pot */
            ucSineMagRef,  /* sine mag ref set by pot, going into soft start */
            ucSineMag,     /* output from soft start, used for PWM cal */
            flag,          /* motor run/stop flag */
            running,       /* non zero value represents motor is running */
            key1count = DEBOUNCE_TIME_COUNT;

static uDOUBLE_8  rotor_position;    /* 2 byte storage for rotor position,
                                        MSByte used to read from sine LUT */

                  
#if (MODE == DIV)

   static uQUAD_8    capture_period;  /* store captured value of free running 
                                         timer bytes at hall edge capture event
                                         (interrupt) */
   static u8   step_ratio,     /* store free running timer scaling ratio 
                                  at hall edge capture event (interrupt) */
               cTick;          /* Hall edge capture event flag */

   static u16  cur_speed,      /* Est speed based on current data */
               ave_speed,      /* Est speed based on previous 6 data */
               speed_table[6]; /* to store 6 latest current speed */


#else

   static uQUAD_8    pll_speed;  /* Est speed based on software PLL */

#endif

/*****************************************************************************/



/*****************************************************************************/
/**************************  INTERRUPT FUNCTIONS   ***************************/
/*****************************************************************************/

/********************************************************************
.  Motor control - Event C and D ...................................*
.................. Event C - HALL EDGE CAPTURE interrupt............*
.................. Event D - Demag interrupt (non existent here) ...*
. Functions: .......................................................*
. - Identify the new sector and corresponding rotor position .......*
. - Identify the next HALL Sensor phase that will toggle and  ......*
.   set the peripheral to look out for that phase ..................*
. - In PLL mode, perform PLL to find motor speed ...................*
. - In DIV mode, buffer the timer registers and ratio register for .*
.   using in main() to estimate motor speed ........................*
.  NOTE: ...........................................................*
.  ***** ...........................................................*
...  Exec time is 17uS in DIV MODE and .............................*
................. 27uS in PLL MODE .................................*
*********************************************************************/
@interrupt  void  mtcCD_ISR(void)  {
static u8           sector, ref_angle;
static sDOUBLE_8    dif_angle;

   while ( MISR & (DI + CI) )	{
      /*************************************
      ***  Hall Edge Capture Interrupt   ***
      **************************************/
      sector    = sectorTable[DIRECTION][(PBDR >> 1) & 0x7];
      ref_angle = angleTable[sector];
      MPHST     = phase_input_sel[DIRECTION][MPHST >> 6];

#if (MODE == DIV)
      /* NOTE:-
         Captured values of Timer and ratio register are buffered here.
         Speed estimation requires a 32b division.  As it takes more time,
         it is implemented in main().  This frees up the CPU to handle other
         interrupts asap.
      */
      if (cTick == 0)   {
         capture_period.byte.b1 = MZREG;
         step_ratio             = MPRSR & 0x0f;
         capture_period.byte.b0 = MZPRV;
         cTick =1;
      }
#else
      /* PLL 32-b implementation
      ///======================= *
      dif_angle.word = (ref_angle<<8) - rotor_position.word;
      if (dif_angle.word > 0x200)
        pll_speed.dw += 0x20000;
      else if (dif_angle.word < -0x200)
        pll_speed.dw -= 0x20000;
      else
        pll_speed.dw += dif_angle.word*0x400;   /**/

      /* PLL 16-b (error 16-b) implementation
      ///==================================== *
      dif_angle.word = (ref_angle<<8) - rotor_position.word;
      if (dif_angle.word > 0x200)
        pll_speed.word.w1 += 0x2;
      else if (dif_angle.word < -0x200)
        pll_speed.word.w1 -= 0x2;
      else
        pll_speed.word.w1 += dif_angle.word >> 8;  // *0x400;   /**/

      /* PLL 16-b (error 8-b) implementation
      ///=================================== */
      dif_angle.byte.b1 = ref_angle - rotor_position.byte.b1;
      if (dif_angle.byte.b1 > 2)
        pll_speed.word.w1 += 0x2;
      else if (dif_angle.byte.b1 < -0x2)
        pll_speed.word.w1 -= 0x2;
      else
        pll_speed.word.w1 += dif_angle.byte.b1;  // >> 8;     /**/

      // ***  PLL limit clamp  ***
      if (pll_speed.word.w1 > 0x2000)
        pll_speed.word.w1 = 0x2000;
      else if (pll_speed.word.w1 <= 0x10) 
        pll_speed.word.w1 = 0x10;           /**/
#endif

      rotor_position.byte.b1 = ref_angle;  // each hall edge represent unique..
                                           // .. actual rotor position, 
      running  = 100; // Presence of hall edge interrupt means motor is running
                      // set (motor) running flag
                      
      MISR = 0xff - DI - CI;     // reset hall edge capture interrupt flag
   }

   return;
}

/*****************************************************************************/

/*****************************************************************
. Motor control - Event U and CURRENT LOOP interrupt ............* 
. ...............................................................*
. EVENT U FUNCTIONS : ...........................................*
. - Update PWM U, V and W phase compare registers with new set ..*
.   of values corresponding to a new phase angle ................*
.  NOTE: ........................................................*
.  ***** ........................................................*
...  Exec time is 47uS ..........................................*
...  ***   Exec time is 43uS when sine value is 8 bit ...........*
. ...............................................................*
. EVENT CURRENT LOOP FUNCTIONS : ................................*
. - Reduce sine magnitude to perform voltage foldback ...........*
*****************************************************************/
@interrupt void mtcU_CL_SO_ISR(void)  {
  static u8   	sucPhasePtr, ptr;
  static s8          stator_voltage_position;
  static uDOUBLE_8   out16;

#define MCPU   ( * (static u16 *) aMCPUH )
#define MCPV   ( * (static u16 *) aMCPVH )
#define MCPW   ( * (static u16 *) aMCPWH )

/* Macro for PWM pulse count calculation
*****************************************/
#define PWM_CALC( REG_PHASE )                                               \
  {                                                                         \
    ptr = stator_voltage_position + stator_voltage_position;                \
    out16.word = *((u8 *)uiSineTable + ptr + 1) * ucSineMag;                \
    out16.word = (*((u8 *)uiSineTable + ptr) * ucSineMag) + out16.byte.b1;  \
    if ( stator_voltage_position < 0 )                                      \
       out16.word = - out16.word;                                           \
    (REG_PHASE) = CARRIER_MIDDLE + out16.word;                              \
  }
                     
  while (MISR & (PUI + CLI)) {
    /************************************
    ****** PWM Update Interrupt *********
    *************************************/
    if (MISR & PUI) {
      #if (MODE == DIV)
         rotor_position.word += ave_speed;
      #else
         rotor_position.word += pll_speed.word.w1; 
      #endif
      
      // phase U duty update
      //=====================
      stator_voltage_position = rotor_position.byte.b1 + load_angle;
      PWM_CALC(MCPU);

      // phase V duty update
      ///====================
      #if (DIRECTION == CCW)
         stator_voltage_position += DEG_120;    
      #else
         stator_voltage_position -= DEG_120;    
      #endif
      PWM_CALC(MCPV);
  
      // phase W duty update
      //=====================
      #if (DIRECTION == CCW)
         stator_voltage_position += DEG_120;    
      #else
         stator_voltage_position -= DEG_120;    
      #endif
      PWM_CALC(MCPW);

      MISR = (0xff - PUI);		/* RESET PUI to clear interrupt*/
    }

    /***************************************
    ******* Current Limit Interrupt ********
    ****************************************/
    if (MISR & CLI) {         // reduce sine mag ref on current limit intrpt
      if (ucSineMag > 25)     // to give a voltage foldback effect
        ucSineMag -= 1;       
      MISR = 0xff - CLI;
    }
  } // end while (MISR & (PUI+CLI))
  return;
}

/*****************************************************************************
............. Main Clock Control/ Real time clock interrupt .................*
....................  ***   Exec time is < 12.5uS in DIV MODE  ***  .........*
........................................ < 14.0us in PLL MODE  ***  .........*
. Functions: ................................................................*
.. - Part of stall detection  ...............................................*
.. - Key debounce for ON/OFF switch .........................................*
.. - Soft start for sine magnitude reference from pot .......................*
*****************************************************************************/
@interrupt void mccRtc_ISR(void)	{
   static u8  counter;
   static s8  sucTemp;

   if (running)     // stall detection -- if running == 0, restart attempt ...
      --running;    // ....  is done in main() .............

   /* ON/OFF switch - 100ms debounce time
   ***************************************/
   if ( !(ON_OFF_PORT & ON_OFF_BIT) )  {
     if (key1count != DEBOUNCE_TIME_COUNT)
       if (++key1count == DEBOUNCE_TIME_COUNT) {
         flag ^= 0xff;
       }
   } else 
     key1count = 0;/**/

   /**********************************************************************
   *..... Soft start on Sine Magnitude reference from potentiometer .....*
   **********************************************************************/
#define MAG_ERR_LIMIT  1
   if (++ counter == 10 ) {
      counter = 0;

      sucTemp = (signed)(ucSineMagRef - ucSineMag)>>1;
      if (sucTemp > MAG_ERR_LIMIT)
        ucSineMag += MAG_ERR_LIMIT;
      else if (sucTemp < -MAG_ERR_LIMIT)
        ucSineMag -= MAG_ERR_LIMIT;
      else
        ucSineMag = ucSineMagRef;
      
#if (MODE != DIV)
      if (ucSineMag == 0)
        pll_speed.dw = 0;
#endif
   }

   sucTemp = MCCSR;	/* read SR to clear interrupt  */
   return;
}

/*****************************************************************************/
@interrupt void mtcRZ_ISR(void)		{;}
@interrupt void tli_ISR(void)  		{;}
/*****************************************************************************/
/**************************  FUNCTION DEFINITIONS  ***************************/
/*****************************************************************************/

/********************/
/*  delay function  */
/********************/
void delay (u8 loop) {
  while (loop>0)
    loop--;
  return;
  }

/*************************************************/
/*  Initialise 72MC for BLDC sinusoidal control  */
/*************************************************/
void MC_init_pmsm(void)  {

    // MCFR  ===>>>  RPGS  RST  CFF2  CFF1  CFF0  CFW2  CFW1  CFW0
    // RESET STATE  -  0    0     0     0     0     0     0     0
    MCFR = RST;                // Reset motor control macro cell
    MCFR = 0;
    MCFR = CFW0 + CFF0;        // current sense blank window = 0.5us
                               // current filter count = 2 samples

    // Initialize registers in page 1 first
    bitSet(MCFR, RPGS);

    // MDTG  ===>>>   PCN  DTE  DTG[5:4:3:2:1:0]
    // RESET STATE -   0    0      [0 0 0 0 0 0]
    MDTG = PCN + DTE + DTG2;   // PCN=1; 3 phase sine PWM 
                               // DTE=1; dead time enabled
                               // Tdeadtime = 0.5uS

    // MPOL  ===>>>   ZVD  REO  OP[5:4:3:2:1:0]
    // RESET STATE -   0    0     [1 1 1 1 1 1]
    MPOL = 0x3f;               // IGBT driver input's active polarity (L6386D)

    // MPAR  ===>>>  TES1  TES0  OE5  OE4  OE3  OE2  OE1  OE0
    // RESET STATE  -  0     0     0    0    0    0    0    0
    MPAR = TES1 + TES0 ;       // Hall feedback rising and falling edge select

    // Initialize registers in page 0
    bitClr(MCFR, RPGS);

    // MREF  ===>>>  HST  CL  CFAV  HFE1  HFE0  HFRQ2  HFRQ1  HFRQ0
    // RESET STATE  - 0   0    0     0     0      0      0      0
    MREF &= (0xff - CFAV);     // OAZ is current comparator entry
                               // pulse chopper disabled

    // MPCR  ===>>>  PMS  OFLU  OFLV  OFLW  CMS  PCP2  PCP1  PCP0
    // RESET STATE  - 0    0     0     0     0    0     0     0
    MPCR = CMS;       // PMS=0/1; 16bit/8bit mode
                      // CMS=0/1; edge/centre aligned mode
                      // counter freq = Fmtc = 16MHz (resonator clock)

    MREP = 3;	      // repetition counter - U event & PWM compare reg update
                      // updates every 2nd carrier cycle, ie., N = (MREP+1)/2,
                      // update ferq = 16MHz/( 2*MCP0*N ) = 6.4KHz currently

    MCP0L = CARRIER_MAX % 256;
    MCP0H = CARRIER_MAX / 256;
    MCPUL = CARRIER_MIDDLE % 256;
    MCPVL = CARRIER_MIDDLE % 256;
    MCPWL = CARRIER_MIDDLE % 256;
    MCPUH = CARRIER_MIDDLE / 256;
    MCPVH = CARRIER_MIDDLE / 256;
    MCPWH = CARRIER_MIDDLE / 256;

    // MCRA  ===>>>  MOE  CKE  SR  DAC  V0C1  SWA  PZ  DCB
    // RESET STATE  - 0    0   0    0    0     0   0    0
    MCRA = DAC;   // For instant update of MPHST below
   // MPHST = IS1;  // Tacho input is MCIC

    MCRA = MOE + CKE;

    // MIMR  ===>>>  PUM  SEM  RIM  CLM  EIM  ZIM  DIM  CIM
    // RESET STATE  - 0    0    0    0    0    0    0    0
    MIMR = PUM + CIM + CLIM;

    return;
}

/*****************************************************************/
/************************  MAIN FUNCTION   ***********************/
/*****************************************************************/
void main(void) {

u8   ucTemp, ptr;

   sim();

   // OPAM settings
   delay(100);
   OACSR  = OAON;
   delay(200);
   OACSR = OAON + OFFCMP + AVGCMP;
   delay(200);
   //OACSR = HIGHGAIN + OAON + OFFCMP + AVGCMP;
   while ( !( OACSR& CMPOVR));

   /* Init motor control macro cell */
   MC_init_pmsm();

   /* Enable timer interrupt */
   MCCSR = 0x02;

   /* init SCI to tx at 2400 baud */
   //init_SCI_tx_2400();        // used primarily for debug purpose
   
   // variables initialisation
   flag = 0;
   running   = 0;
   ucSineMag = 0;
   rotor_position.word = 0;

   rim();  
   
   /**************************************************************************
   ****************   Program entering into INFINITE LOOP   ******************
   **************************************************************************/

   while (1) {
     
     if (flag)  {  // 'flag' is handled in mccRtc_ISR() based on on/off switch
                   // if flag != 0, run the motor
                   
       if (running == 0) {    // 'running' is handled in mccRtc_ISR() and 
                              // mtcCD_ISR() to determine stall condition 
                              // running == 0 means motor not in motion
                              // set initial conditions to get the motor 
                              // up and running
          ucSineMagRef = 25;
          load_angle = 10;
          rotor_position.byte.b1 =     /* get initial position of rotor */
              angleTable[ sectorTable[DIRECTION][(PBDR >> 1) & 0x7] ];

       } else {
          
          // running != 0 means motor is in motion
          
          /* load angle reading
          **********************/
          ADCCSR = ADON + CH_LOAD_ANGLE;       // scan load angle ref pot
          ucTemp = ADCDRMSB;                   // clear EOC for prev conv
          while (bitTest_FALSE(ADCCSR, EOC)) ; // wait for new EOC
          load_angle = (ADCDRMSB-128) >> 1;    /* chose to vary angle 
                                                  from -45 to +45 deg */

          /* speed ref reading
          *********************/
          ADCCSR = ADON + CH_SINE_MAG;         // scan sine mag ref pot
          ucTemp = ADCDRMSB;                   // clear EOC for prev conv
          while (bitTest_FALSE(ADCCSR, EOC)) ; // wait for new EOC
          if (ADCDRMSB < 15)                   // clamp sine mag ref ....
             ucSineMagRef = 15;                // ... 15 <= sine mag ref <= 250
          else if (ADCDRMSB > 250)
             ucSineMagRef = 250;
          else
             ucSineMagRef = ADCDRMSB;          /**/
          
       } // end of else (running == 0)
     
     }  // end of if (flag)
     else {
       // if == 0, stop the motor
       ucSineMagRef = 0;
     } // end of else (flag)
     
#if (MODE == DIV)
     if (cTick) {
     
       /* 'cTick' is handled in mtcCD_ISR() to flag hall edge capture, and to 
          indicate new samples available for current speed estimation

          SPEED AVERAGING APPROACH
          =========================
          - speed table contains latest 6 current speed estimations
          - speed table maintained by purging oldest sample out and inserting
            new sample in that place
          - average speed is identified by summing up all 6 entries
          - post sum div by 6 not needed as each entry is already div by 6  */
          
       cur_speed = (u16)( ((0x1a0aa00/6) >> step_ratio)/(capture_period.dw) );
       if (cur_speed > 0x6aa)
         cur_speed = 0x6aa;
       if ((cur_speed > 0x32a) && (ave_speed < 0x2000))
         cur_speed = speed_table[ptr];
       ave_speed = ave_speed - speed_table[ptr] + cur_speed;
       speed_table[ptr] = cur_speed;
       if (++ptr > 5)
          ptr = 0;
          
       /*/
       
       // No averaging approach
       // =====================
       // - no need for speed table 
       // - current speed is taken as average speed
       
       cur_speed = (u16)( (0x1a0aa00 >> step_ratio)/(capture_period.dw) );  
       if (cur_speed > 0x13ff)
         cur_speed = 0x13ff;
//     if ( (cur_speed > 0x1300) && (ave_speed < 0x1000) )
//       cur_speed = ave_speed;
       ave_speed = cur_speed;
       /**/
              
       cTick = 0;     // clear capture data available flag
     }
#endif
     
   } // end of while (1) or INFINITE LOOP
   
   return;
}
/*****************************************************************************/
/**************************   PROGRAM ENDS HERE   ****************************/
/*****************************************************************************/
