/******************** (C) COPYRIGHT 2007 STMicroelectronics ********************
* File Name          : spi_flash.c
* Author             : MCD Application Team
* Version            : V1.0
* Date               : 12/06/2007
* Description        : This file provides all firmware functions needed to read 
*                      data from M25P64 SPI based Flash memory.
********************************************************************************
* THE PRESENT SOFTWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
* WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME.
* AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT,
* INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE
* CONTENT OF SUCH SOFTWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING
* INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
*******************************************************************************/

/* Includes ------------------------------------------------------------------*/
#include "spi_flash.h"

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* BSPI0 pins */
#define S0_MISO    0x01
#define S0_MOSI    0x02
#define S0_SCLK    0x04
#define S0_SS      0x08

/* SPI Flash ChipSelect */
#define SPI_FLASH_CS    0x02

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/

/*******************************************************************************
* Function Name  : SPI_FLASH_Init
* Description    : Configures BSPI0 and its associated pin.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void SPI_FLASH_Init(void)
{
/* GPIO configuration --------------------------------------------------------*/
  /* Configure BSPI0 pins as alternate function push-pull */
  GPIO_Config(GPIO0, S0_MISO | S0_MOSI | S0_SCLK | S0_SS, GPIO_AF_PP); 
  /* P1.01 as output push-pull, used to drive the Flash ChipSelect line */
  GPIO_Config(GPIO1, SPI_FLASH_CS, GPIO_OUT_PP);

/* BSPI0 configuration --------------------------------------------------------*/  
  /* Enable the BSPI0 */
  BSPI_BSPI0Conf(ENABLE);

  /* Initialize BSPI0 */
  BSPI_Init(BSPI0);

  /* BSPI0 Baud Rate : ---> PCLK1 / 6 */
  BSPI_ClockDividerConfig(BSPI0, 6);

  /* Configure BSPI0 as a Master */
  BSPI_MasterEnable(BSPI0, ENABLE);

  /* Configure the clock to be active low */
  BSPI_ClkActiveHigh(BSPI0, DISABLE);

  /* Enable capturing the first Data sample on the first edge of SCK */
  BSPI_ClkFEdge(BSPI0, ENABLE);

  /* Set the word length to 8 bit */
  BSPI_8bLEn(BSPI0, ENABLE); 
  
  /* Enable BSPI0 */
  BSPI_Enable(BSPI0, ENABLE);
}

/*******************************************************************************
* Function Name  : SPI_FLASH_ChipSelect
* Description    : Selects or deselects the FLASH.
* Input          : State : level to be applied on the FLASH's ChipSelect line.
* Output         : None
* Return         : None
*******************************************************************************/
void SPI_FLASH_ChipSelect(u8 State)
{
  if(State == Low)
  {
    GPIO1->PD &= ~SPI_FLASH_CS;   
  }
  else if(State == High)
  {
    GPIO1->PD |= SPI_FLASH_CS;
  }
}

/*******************************************************************************
* Function Name  : SPI_FLASH_SendByte
* Description    : Sends a byte through the SPI interface and return the byte 
*                  received from the SPI bus.
* Input          : byte : byte to send.
* Output         : None
* Return         : The value of the received byte.
*******************************************************************************/
u8 SPI_FLASH_SendByte(u8 byte)
{
  /* send byte through the BSPI0 peripheral */	
  BSPI_WordSend(BSPI0, byte); 	
  
  /* loop while Transmit FIFO is full */
  while(BSPI_FlagStatus(BSPI0, BSPI_TFF) == SET);   

  /* loop while Receive FIFO is empty */
  while(BSPI_FlagStatus(BSPI0, BSPI_RFNE) == RESET);	
  
  /* return the byte read from the SPI bus */
  return(BSPI_WordReceive(BSPI0)); 	   
}

/******************* (C) COPYRIGHT 2007 STMicroelectronics *****END OF FILE****/
