/******************** (C) COPYRIGHT 2007 STMicroelectronics ********************
* File Name          : waveplayer.c
* Author             : MCD Application Team
* Version            : V1.0
* Date               : 12/06/2007
* Description        : This file provides all the firmware function for "STR7 
*                      audio generation with PWM" application note.
********************************************************************************
* THE PRESENT SOFTWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
* WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME.
* AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT,
* INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE
* CONTENT OF SUCH SOFTWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING
* INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
*******************************************************************************/

/* Includes ------------------------------------------------------------------*/
#include "waveplayer.h"

/* Private define ------------------------------------------------------------*/
#define  ChunkID             0x52494646  /* correspond to letters 'RIFF' */
#define  FileFormat          0x57415645  /* correspond to letters 'WAVE' */
#define  FormatID            0x666D7420  /* correspond to letters 'fmt ' */
#define  DataID              0x64617461  /* correspond to letters 'data' */
#define  FactID              0x66616374  /* correspond to letters 'fact' */

#define  WAVE_FORMAT_PCM     0x01
#define  FormatChunkSize     0x10
#define  Channel_Mono        0x01

#define  SampleRate_8000     8000
#define  SampleRate_11025    11025
#define  SampleRate_22050    22050
#define  SampleRate_44100    44100
#define  Bits_Per_Sample_8   8

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
WAVE_FormatTypeDef WAVE_Format;
vu32 WAVEDataLength = 0;
u16 TIM0_OCAValue = 0;

/* Private function prototypes -----------------------------------------------*/
static u32 ReadUnit(u8 NbrOfBytes, Endianness BytesFormat);
static void Timers_Config(void);

/* Private functions ---------------------------------------------------------*/

/*******************************************************************************
* Function Name  : Timers_Config
* Description    : Configures TIM0 and TIM3.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
static void Timers_Config(void)
{
  /* TIM0 used as system timer -----------------------------------------------*/
  /* Initialize the Timer */
  TIM_Init(TIM0);

  /* Timer clock: 32MHz/(0+1) = 32MHz */
  TIM_PrescalerConfig(TIM0, 0x00);
  TIM_ClockSourceConfig(TIM0, TIM_INTERNAL);
  
  /* Enable TIM0 clock on APB2 */
   APB_ClockConfig (APB2, ENABLE, TIM0_Periph);

  /* TIM0 frequency depends on the .WAV file sample rate value; the TIM0_OCAValue
  variable wich defines the OCAR value will be filled by WAVE_Parsing function */
  TIM_OCMPModeConfig (TIM0, TIM_CHANNEL_A, TIM0_OCAValue-5, TIM_WAVE, TIM_HIGH);

  /* Enable TIM0 Output Compare A interrupt */
  TIM_ITConfig(TIM0, TIM_OCA_IT, ENABLE);

  /* Enable TIM0 FIQ channel */
  EIC_FIQChannelConfig(T0TIMI_FIQChannel, ENABLE);

  /* TIM3 used in PWM mode ---------------------------------------------------*/
  /* Configure P1.02 as alternate function (T3_OCMPA) */
  GPIO_Config(GPIO1, T3_OCMPA, GPIO_AF_PP);

  /* Initialize the Timer */
  TIM_Init(TIM3);

  /* Timer clock: 32MHz/(0+1) = 32MHz */
  TIM_PrescalerConfig(TIM3, 0x0);
  
  /* Enable TIM3 clock on APB2 */
   APB_ClockConfig (APB2, ENABLE, TIM3_Periph);

  /* Generate a PWM Signal: freq = 32MHz/255 = ~125.5KHz */
  TIM_PWMOModeConfig(TIM3, 128, TIM_HIGH, 255, TIM_LOW);

  /* Start TIM3 Counter */
  TIM_CounterConfig(TIM3, TIM_START);

  /* Start TIM0 Counter */
  TIM_CounterConfig(TIM0, TIM_START);

  /* Enable FIQ output request to CPU */
  EIC_FIQConfig(ENABLE);
}

/*******************************************************************************
* Function Name  : WAVE_PlayBack
* Description    : Play back the .WAV file stored in the SPI Flash. This function
*                  assumes that the .WAV file is stored at address 0x00 of the Flash.
* Input          : None
* Output         : None
* Return         : Zero value if the function succeed, otherwise it return
*                  a nonzero value which specifies the error code.
*******************************************************************************/
ErrorCode WAVE_PlayBack(void)
{
  ErrorCode WAVEFileStatus;
	
  /* check the .WAV file format */		
  WAVEFileStatus = WAVE_Parsing();
	
  if(WAVEFileStatus == Valid_WAVE_File) /* the .WAV file is valid */
  {
    /* Get the length of the .WAV file data */
    WAVEDataLength = WAVE_Format.DataSize;

    /* start audio playback */
    Timers_Config();
  }
  else  /* the .WAV file is corrupted */
  {    				
    /* Deselect the SPI Flash */
    SPI_FLASH_ChipSelect(High);

    /* Disable BSPI0 */
    BSPI_Enable(BSPI0, DISABLE);
  }

  return WAVEFileStatus;
}

/*******************************************************************************
* Function Name  : WAVE_Parsing
* Description    : Checks the format of the .WAV file and gets information about
*                  the audio format. This is done by reading the value of a
*                  number of parameters stored in the file header and comparing
*                  these to the values expected authenticates the format of a
*                  standard .WAV  file (44 bytes will be read). If it is a valid
*                  .WAV file format, it continues reading the header to determine
*                  the audio format such as the sample rate and the sampled data
*                  size. If the audio format is supported by this application,
*                  it retrieves the audio format in the WAVE_Format structure and
*                  returns a zero value. Otherwise the function fails and the
*                  return value is nonzero.In this case, the return value specifies
*                  the cause of  the function fails. The error codes that can be
*                  returned by this function are declared within this file.
* Input          : None
* Output         : None
* Return         : Zero value if the function succeed, otherwise it return
*                  a nonzero value which specifies the error code.
*******************************************************************************/
ErrorCode WAVE_Parsing(void)
{
  u32 index = 0x00;
  u32 Temp = 0x00;
  bool ExtraFormatBytes = FALSE;

  /* Start a read data byte sequence from the Flash starting from @0x00 */
  Start_Read_Sequence(0x00);

/* Read chunkID, must be 'RIFF'	----------------------------------------------*/
  Temp = ReadUnit(4, BigEndian);
  if(Temp != ChunkID)
  {
    return(Unvalid_RIFF_ID);
  }
  	
  /* Read the file length */
  WAVE_Format.RIFFchunksize = ReadUnit(4, LittleEndian);
	
  /* Read the file format, must be 'WAVE' */
  Temp = ReadUnit(4, BigEndian);
  if(Temp != FileFormat)
  {
    return(Unvalid_WAVE_Format);
  }
	
/* Read the format chunk, must be'fmt ' --------------------------------------*/
  Temp = ReadUnit(4, BigEndian);
  if(Temp != FormatID)
  {
    return(Unvalid_FormatChunk_ID);
  }

  /* Read the length of the 'fmt' data, must be 0x10 */
  Temp = ReadUnit(4, LittleEndian);
  if(Temp != 0x10)
  {
    ExtraFormatBytes = TRUE;
  }

  /* Read the audio format, must be 0x01 (PCM) */
  WAVE_Format.FormatTag = ReadUnit(2, LittleEndian);
  if(WAVE_Format.FormatTag != WAVE_FORMAT_PCM)
  {
    return(Unsupporetd_FormatTag);	
  }
	
  /* Read the number of channels, must be 0x01 (Mono) */
  WAVE_Format.NumChannels = ReadUnit(2, LittleEndian);
  if(WAVE_Format.NumChannels != Channel_Mono)
  {
    return(Unsupporetd_Number_Of_Channel);	
  }
	
  /* Read the Sample Rate */
  WAVE_Format.SampleRate = ReadUnit(4, LittleEndian);

  /* Update the TIM0 OCA value according to the .WAV file Sample Rate */
  switch(WAVE_Format.SampleRate)
  {
    case SampleRate_8000 : TIM0_OCAValue = 4000; break;  /* 8KHz = 32MHz /4000 */
    case SampleRate_11025: TIM0_OCAValue = 2902; break;  /* 11.025KHz = 32MHz /2902 */
    case SampleRate_22050: TIM0_OCAValue = 1451; break; /* 22.05KHz = 32MHz /1451 */
    case SampleRate_44100: TIM0_OCAValue = 725; break;  /* 44.1KHz = 32MHz /725 */		
    default: return(Unsupporetd_Sample_Rate);
  }
	
  /* Read the Byte Rate */
  WAVE_Format.ByteRate = ReadUnit(4, LittleEndian);
	
  /* Read the block alignment */
  WAVE_Format.BlockAlign = ReadUnit(2, LittleEndian);
	
  /* Read the number of bits per sample */
  WAVE_Format.BitsPerSample = ReadUnit(2, LittleEndian);
  if(WAVE_Format.BitsPerSample != Bits_Per_Sample_8)
  {
    return(Unsupporetd_Bits_Per_Sample);	
  }

/* If there is Extra format bytes, these bytes will be defined in "Fact Chunk" */
  if(ExtraFormatBytes == TRUE)
  {
    /* Read th Extra format bytes, must be 0x00 */
    Temp = ReadUnit(2, LittleEndian);
    if(Temp != 0x00)
    {
      return(Unsupporetd_ExtraFormatBytes);
    }

    /* Read the Fact chunk, must be 'fact' */
    Temp = ReadUnit(4, BigEndian);
    if(Temp != FactID)
    {
      return(Unvalid_FactChunk_ID);
    }

    /* Read Fact chunk data Size */
    Temp = ReadUnit(4, LittleEndian);
    /* Read */
    for(index = 0; index < Temp; index++)
    {
      SPI_FLASH_SendByte(Dummy_Byte);
    }
  }

/* Read the Data chunk, must be 'data' ---------------------------------------*/
  Temp = ReadUnit(4, BigEndian);
  if(Temp != DataID)
  {
    return(Unvalid_DataChunk_ID);
  }
	
  /* Read the number of sample data */
  WAVE_Format.DataSize = ReadUnit(4, LittleEndian);

  return(Valid_WAVE_File);	
}

/*******************************************************************************
* Function Name  : Start_Read_Sequence
* Description    : Initiates a read data byte (READ) sequence from the Flash.
*                  This is done by driving the /CS line low to select the device,
*                  then the READ instruction is transmitted followed by 3 bytes
*                  address. This function exit and keep the /CS line low, so the
*                  Flash still being selected. With this technique the whole
*                  content of the Flash is read with a single READ instruction.
* Input          : - ReadAddr : FLASH's internal address to read from.
* Output         : None
* Return         : None
*******************************************************************************/
void Start_Read_Sequence(u32 ReadAddr)
{
  /* Select the FLASH: Chip Select low */
  SPI_FLASH_ChipSelect(Low);	   
  
  /* Send "Read from Memory " instruction */
  SPI_FLASH_SendByte(READ);	 

/* Send the 24-bit address of the address to read from -----------------------*/  
  /* Send ReadAddr high nibble address byte */
  SPI_FLASH_SendByte((ReadAddr & 0xFF0000) >> 16);
  /* Send ReadAddr medium nibble address byte */
  SPI_FLASH_SendByte((ReadAddr& 0xFF00) >> 8);  
  /* Send ReadAddr low nibble address byte */
  SPI_FLASH_SendByte(ReadAddr & 0xFF);  
}

/*******************************************************************************
* Function Name  : ReadUnit
* Description    : Reads a number of bytes from the SPI Flash.
*                  This function must be used only if the Start_Read_Sequence
*                  function has been previously called.
* Input          : - NbrOfBytes : number of bytes to read.
*                    This parameter must be a number between 1 and 4.
*                  - Endians : specifies the bytes endianness.
*                    This parameter can be one of the following values:
*                          - LittleEndian
*                          - BigEndian
* Output         : None
* Return         : Data read from the SPI Flash.
*******************************************************************************/
static u32 ReadUnit(u8 NbrOfBytes, Endianness BytesFormat)
{
  u32 index = 0;
  u32 Temp = 0;

  if(BytesFormat == LittleEndian)
  {
    for(index = 0; index < NbrOfBytes; index++)
    {
      Temp |= ReadByte() << (index * 8);
    }
  }
  else
  {
    for(index = NbrOfBytes; index != 0; index--)
    {
      Temp |= ReadByte() << ((index-1) * 8);
    }
  }

  return Temp;
}

/*******************************************************************************
* Function Name  : ReadByte
* Description    : Reads a byte from the SPI Flash.
*                  This function must be used only if the Start_Read_Sequence
*                  function has been previously called.
* Input          : None
* Output         : None
* Return         : Byte Read from the SPI Flash.
*******************************************************************************/
u8 ReadByte(void)
{
  return (SPI_FLASH_SendByte(Dummy_Byte));
}

/******************* (C) COPYRIGHT 2007 STMicroelectronics *****END OF FILE****/
