/******************** (C) COPYRIGHT 2006 STMicroelectronics ********************
* File Name          : device_RFID.c
* Author             : MPA Systems Lab
* Date First Issued  : 31/01/2006
* Description        : RFID Management for SR176 tag
********************************************************************************
* History:
*  31/01/2006 : V1.0
*******************************************************************************
 THE PRESENT SOFTWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS WITH
 CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME.
 AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT
 OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE CONTENT
 OF SUCH SOFTWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING INFORMATION
 CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
*******************************************************************************/

/* Includes ------------------------------------------------------------------*/#include "device_RFID.h"
#include "i2c.h"

/* Defines ------------------------------------------------------------------*/
#define TEST_RFID 0
#define I2C0_SCL 0x2000
#define I2C0_SDA 0x4000
#define I2C1_SCL 0x0004
#define I2C1_SDA 0x0008
#define RFID64UIDmask 0x1
#define RFIDReadmask  0x2
#define RFIDWritemask 0x4
#define TIMEOUT 100000


/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
UCHAR ADDR=0xA0;													// RFID ADDR						
u8 bTableau[10] = {1,2,3,4,15};
char *pBuffer= "RFIDexample";
u8 result;

/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/



/*******************************************************************************
* Function Name  : RFID_Handler
* Description    : Manage SR176 RFID tag
* Input          : (RFID_Action, TX_frame, TX_lenght, RX_frame, RX_lenght, RX_offset, RFID_UID, RFID_timeout)
*	  				- RFID_Action: 3 bit													   
*		  						   bit 0: 1=ENABLE 64UID request, 0=DISABLE 64UID request  
*		  						   bit 1: 1=ENABLE READ  request, 0=DISABLE READ  request  
*		  						   bit 2: 1=ENABLE WRITE request, 0=DISABLE WRITE request  	  						   	  						   
*																						   
*					- TX_frame:	   TX buffer pointer (write operations)					   
*					- TX_lenght:   TX buffer lenght (write operations)	  				   
*					- TX_offset:   TX offset for write operations 	  					   
*																						   
*					- RX_frame:	   RX buffer pointer (read operations)  				   
*					- RX_lenght:   RX buffer lenght (read operations)	  				   
*					- RX_offset:   RX offset for read operations 	  					   
*	  						   														  	   
*					- RFID_UID:	   64 bits RFID UID										   
*					- RFID_timeout:timeout for RFID polling (number of times STR7 is trying to search tag)
* Output         : (RX_frame, RFID_UID)
*					- RX_frame:	   RX buffer pointer (read operations)   				   
*					- RFID_UID:	   64 bits RFID UID										   
* Return         : Status.
*					- Status:		   														   
*							bit0: 1-No tag selected		0-tag selected				  	   
*							bit1: 1-Error on 64UID		0-no error					  	   
*							bit2: 1-Error on READ		0-no error					  	   
*							bit3: 1-Error on WRITE		0-no error					  	   
*							bit4: 1-Error on Completion 0-no error					  	   
*							bit5: 1-Error on RF_OFF  	0-no error					  	   
*******************************************************************************/
u8 RFID_Handler(u8 RFID_Action, u8 *TX_frame, u8 TX_lenght, u8 TX_offset, u8 *RX_frame, u8 RX_lenght, u8 RX_offset, u32 *RFID_UID, u8 RFID_timeout){

	/*------------------------------------------------------------------
			LOCAL VARIABLES DEFINITION
	--------------------------------------------------------------------*/
  UCHAR RX_framebuf[3];
  UCHAR RFID_CHIPID;
  u8 	sts=0;
  SR176_State SR176State=SR176_POWEROFF;

  

  
	/*------------------------------------------------------------------
	  		init and start I2C0
	--------------------------------------------------------------------*/
	  init_CRX14_I2C();

	/*------------------------------------------------------------------
	  		search device, activate and select it
	--------------------------------------------------------------------*/
	  while((SR176State!=SR176_SELECTED)&&(!sts)&&((RFID_timeout--)>1))
		  if(init_RFID( &RFID_CHIPID, &SR176State)) sts=1;

	/*------------------------------------------------------------------
	  		if device is selected start to read/write...
	--------------------------------------------------------------------*/
	  if((!sts)&&(RFID_timeout)) {
	  
		    if (RFID_Action & RFID64UIDmask) 		  sts|=GetSR176_64UID(&RFID_UID[0])<<1;
			if (RFID_Action & RFIDReadmask) 		  sts|=ReadSR176_EEPROM(&RX_frame[0],RX_offset,RX_lenght)<<2;
			if (RFID_Action & RFIDWritemask)
			{
					  I2C_TIMEOUT_SEND=100000;
			 		  sts|=WriteSR176_EEPROM(&TX_frame[0],TX_offset,TX_lenght)<<3;
					  sts|=ReadSR176_EEPROM(&RX_frame[0],TX_offset,TX_lenght)<<2;			 		  
					  I2C_TIMEOUT_SEND=5000;			 		  
			}
			/*------------------------------------------------------------------
			  		DEACTIVATE TAG
			--------------------------------------------------------------------*/
		    sts|=Completion ( &RX_framebuf[0])<<4;  

			/*------------------------------------------------------------------
			  		CUT OFF 13.56MHz RF CARRIER
			--------------------------------------------------------------------*/
			sts|=RF_OFF()<<5;
	  }
		/*------------------------------------------------------------------
	  		...else CUT OFF 13.56MHz RF CARRIER
		--------------------------------------------------------------------*/
	  else sts|=RF_OFF()<<5;
	  


	/*-------------------------------------------------------------------------------------
	Disable I2C0
	--------------------------------------------------------------------------------------*/
	I2C_OnOffConfig (I2C0, DISABLE);	
return sts;
}




/*******************************************************************************
* Function Name  : delay
* Description    : delay program execution of "value" count
* Input          : value
* Output         : None.
* Return         : None.
*******************************************************************************/
void delay(u32 value){
    u32 icount=0;
	while(icount<=value)	icount++;							//delay
}
	
/*******************************************************************************
* Function Name  : ReadSR176_EEPROM
* Description    : Read SR176 RFID EEPROM
* Input          : 
*					RFID_Data	- buffer pointer to RX frame where to read in SR176 EEPROM		   
*					EEPROMoffset- EEPROM offset where to write									   
*					TX_frame_lenght	- buffer lenght												   
* Output         : None.
* Return         : Status.
*					1			- error  														   
*					0			- no error														   
*******************************************************************************/
u8 ReadSR176_EEPROM(u8 *RFID_Data, u8 EEPROMoffset,u8 RX_frame_lenght){
	u8 sts=0;
	UCHAR RX_frame[3];
	u8 i=0;

			/*------------------------------------------------------------------
			  		READ RFID
			--------------------------------------------------------------------*/
		  if((EEPROMoffset+RX_frame_lenght)<=20){
		  	  sts=Time_Read();		  
			  for(i=0;i<(RX_frame_lenght>>1);i++){
				  sts=ReadBlock ((4+i+EEPROMoffset), &RX_frame[0]);						// FUNCTION TO READ BLOCK TAG
				  RFID_Data[i<<1]=RX_frame[1];
				  RFID_Data[(i<<1)+1]=RX_frame[2];
			  }
		  }
		  else sts=1;
		  return sts;
}

/*******************************************************************************
* Function Name  : WriteSR176_EEPROM
* Description    : Write SR176 RFID EEPROM
* Input          : 
*					TX_frame	- buffer pointer to TX frame to write in SR176 EEPROM			   
*					EEPROMoffset- EEPROM offset where to write									   
*					TX_frame_lenght	- buffer lenght												   
* Output         : None.
* Return         : Status.
*					1			- error  														   
*					0			- no error														   
*******************************************************************************/
u8 WriteSR176_EEPROM(u8 *TX_frame, u8 EEPROMoffset,u8 TX_frame_lenght){
	u8 sts=0;
	UCHAR RX_frame[3];
	u8 i=0;
			/*------------------------------------------------------------------
			  		WRITE RFID
			  		address from 0x4 to 0x14
			  		u8 TX_frame dim=10
			--------------------------------------------------------------------*/
		  sts=Time_Write();
		  if((EEPROMoffset+TX_frame_lenght)<=20)
			  for(i=0;i<(TX_frame_lenght>>1);i++)
				  sts=WriteBlock ((4+i+EEPROMoffset), &TX_frame[i<<1], &RX_frame[0]);		// FUNCTION TO WRITE  BLOCK TAG
		  else sts=1;
		  return sts;
}

/*******************************************************************************
* Function Name  : GetSR176_64UID
* Description    : Get SR176 64 bit unique ID
* Input          : 
*					RFID_UID	- buffer pointer to SR176 unique id								   
* Output         : None.
* Return         : Status.
*					1			- error  														   
*					0			- no error														   
*******************************************************************************/
u8 GetSR176_64UID(u32 *RFID_UID){
	u8 sts=0;
	UCHAR RX_frame[3];
			/*------------------------------------------------------------------
			  		READ RFID UID 64 bit
			--------------------------------------------------------------------*/
		  sts=Time_Read();
		  sts=ReadBlock (0x0, &RX_frame[0]);						// FUNCTION TO READ BLOCK TAG
		  RFID_UID[0]=(RX_frame[1])|(RX_frame[2]<<8);
		  sts=ReadBlock (0x1, &RX_frame[0]);						// FUNCTION TO READ BLOCK TAG
		  RFID_UID[0]|=(RX_frame[1]<<16)|(RX_frame[2]<<32);
		  sts=ReadBlock (0x2, &RX_frame[0]);						// FUNCTION TO READ BLOCK TAG
		  RFID_UID[1]=(RX_frame[1])|(RX_frame[2]<<8);
		  sts=ReadBlock (0x3, &RX_frame[0]);						// FUNCTION TO READ BLOCK TAG
		  RFID_UID[1]|=(RX_frame[1]<<16)|(RX_frame[2]<<32);
return sts;
}



/*******************************************************************************
* Function Name  : Write_RFID
* Description    : Write CRX14 Registers
* Input          : 
*					addr 		- CRX14 register address										   
*					data2write	- buffer pointer (buffer contains data to write in RFID)		   
*					lenght		- number of byte to send										   
* Output         : None.
* Return         : Status.
*					1			- error  														   
*					0			- no error														   
*******************************************************************************/
u8 Write_RFID(CRX14_Register addr, u8 *data2write, u8 lenght){
u8 status=0;
u8 state=0;
u8 i=0;

	while(state<=7){
		switch(state){
			/*-------------------------------------------------------------
				Host Send CRX14 register address
			--------------------------------------------------------------*/
			case 0:
			 	if(!(status=I2C_ByteSend_with_status (I2C0, addr))) state++;			//send address of byte
			 	else return status=1;
			 	break;
			/*-------------------------------------------------------------
				Host check acknowledge
			--------------------------------------------------------------*/
			case 1:
			    if(!((I2C0->SR2 & 0x10)>>4)) state++;									//check ack
			 	else return status=1;
			 	break;
			/*-------------------------------------------------------------
				Depending on CRX14 Register address Host decide what to do
			--------------------------------------------------------------*/
			case 2:
			    switch(addr){
					/*-------------------------------------------------------------
						I2C Write to Parameter Register
					--------------------------------------------------------------*/
			    	case CRX14_PR:
			    		state+=3;
			    		break;
					/*-------------------------------------------------------------
						I2C Write to I/O Frame Register
					--------------------------------------------------------------*/
			    	case CRX14_IOFR:
			    		state++;
			    		break;
					/*-------------------------------------------------------------
						I2C Write to Slot Marker Register
					--------------------------------------------------------------*/
			    	case CRX14_SMR:
						state=7;
						break;
					default:
					 	state=7;
					 	status=1;			    		
			 	 }
			 	break;
			/*-------------------------------------------------------------
				Host Write I/O Frame register: host send Number of byte 
			--------------------------------------------------------------*/
			case 3:
			    if(!(	status=I2C_ByteSend_with_status (I2C0, lenght))) state++;	//send Number of byte
			 	else return status=1;
			 	break;
			/*-------------------------------------------------------------
				Host check acknowledge
			--------------------------------------------------------------*/
			case 4:
			    if(!((I2C0->SR2 & 0x10)>>4)) state++;									//chec ack
			 	else return status=1;
			 	break;
			/*-------------------------------------------------------------
				Host send byte
			--------------------------------------------------------------*/
			case 5:
			    if(!(	status=I2C_ByteSend_with_status (I2C0, data2write[i++]))) state++;	//send byte
			 	else return status=1;
			 	break;
			/*-------------------------------------------------------------
				Host check acknowledge and:
					- loop to state 5 if (byte sent)<(byte to send)
					- step to state 7 if (byte sent)=(byte to send)
			--------------------------------------------------------------*/
			case 6:
			    if(!((I2C0->SR2 & 0x10)>>4)){
			             if (i<lenght) state=5;
			    		 else state++;														//check ack
				}
			 	else return status=1;
			 	break;
			/*-------------------------------------------------------------
				Generate I2C Stop
			--------------------------------------------------------------*/
			case 7:
				I2C_STOPGenerate (I2C0, ENABLE);										//Stop the communication
				state++;
			 	break;
		
			default: status=1;
		
		}
	}
	return status;
}


/*******************************************************************************
* Function Name  : Read_RFID
* Description    : Read CRX14 Registers
* Input          : 
*					addr 		- CRX14 register address										   
*					data2read	- buffer pointer												   
*					lenght		- buffer lenght													   
* Output         : None.
* Return         : Status.
*					1			- error  														   
*					0			- no error														   
*******************************************************************************/
u8 Read_RFID(CRX14_Register addr, u8 *data2read, u8 *lenght){
u8 status=0;
u8 state=0;
u8 i=0;
u8 none=0;


	while(state<=7){
		switch(state){
			/*-------------------------------------------------------------
				Host Send CRX14 register address
			--------------------------------------------------------------*/
			case 0:
			 	if(!(status=I2C_ByteSend_with_status (I2C0, addr))) state++;					//send address of byte
			 	else return status=1;
			 	break;
			/*-------------------------------------------------------------
				Host check acknowledge
			--------------------------------------------------------------*/
			case 1:
			    if(!((I2C0->SR2 & 0x10)>>4)) state++;											//check ack
			 	else return status=1;
			 	break;
			/*-------------------------------------------------------------
				Host use Polling to:
				  - regenerate I2C Start
				  - send CRX14 address + Read Command
				  - check acknowledge
			--------------------------------------------------------------*/
			case 2:
			    if((!Polling (0xA1))) state++;													//regenerate start and check ack
			 	else return status=1;
				delay(1000);
			 	break;
			/*-------------------------------------------------------------
				Depending on CRX14 Register address Host decide what to do
			--------------------------------------------------------------*/
			case 3:
				switch(addr){
					/*-------------------------------------------------------------
						I2C Read from Parameter Register
					--------------------------------------------------------------*/
					case CRX14_PR:
						state+=3;
				 	    I2C_AcknowledgeConfig (I2C0, DISABLE);
					 	(*lenght)=0;
					 	break;
					/*-------------------------------------------------------------
						I2C Read from I/O Parameter Register
					--------------------------------------------------------------*/
					case CRX14_IOFR:
						state++;
						break;
					default: state++;
			 	} 
			 	break;
			/*-------------------------------------------------------------
				Host Read from I/O Frame register: CRX14 send Number of byte 
			--------------------------------------------------------------*/
			case 4:
			    if(!(	status=I2C_ByteReceive_with_status (I2C0, lenght))) state++;	//recive Number of byte
			 	else return status=1;
			 	break;
			/*-------------------------------------------------------------
				Check aknowledge 
			--------------------------------------------------------------*/
			case 5:
			    if(!((I2C0->SR2 & 0x10)>>4)) state++;											//check ack
			 	else return status=1;
			 	break;
			/*-------------------------------------------------------------
				Host Read from CRX14 
			--------------------------------------------------------------*/
			case 6:
			 	if((!(*lenght))||((*lenght)==0xFF)||(i==((*lenght)-1))){
			 		 I2C_AcknowledgeConfig (I2C0, DISABLE);				//if Number of byte =0 host send NO acknowledge
			 		 state++;
			 	}
		    	if(I2C_ByteReceive_with_status (I2C0, &data2read[i++])) return 1; 	//receive data
			 	break;
			/*-------------------------------------------------------------
				Generate I2C Stop
			--------------------------------------------------------------*/
			case 7:
				I2C_STOPGenerate (I2C0, ENABLE);												//Stop the communication
			    if(!(status = I2C_ByteReceive_with_status (I2C0, &none))) 						//generate stop
			    	 I2C_AcknowledgeConfig (I2C0, ENABLE);										//enable ack
			 	else return status=1;
				state++;
			 	break;
			default: status=1;
		
		}
	}
	return status;
}



/*******************************************************************************
* Function Name  : init_RFID
* Description    : Initialize RFID communication
* Input          : 
*					RFID_CHIPID	- tag ID														   
*					SR176State	- tag State														   
* Output         : None.
* Return         : Status.
*					1			- error  														   
*					0			- no error														   
*******************************************************************************/
u8 init_RFID( UCHAR *RFID_CHIPID, SR176_State *SR176State){
  u8 sts=0;
  UCHAR RX_frame[3];
  
   

	switch(*SR176State){
	
		case SR176_POWEROFF:
					/*------------------------------------------------------------------
					  		Enable Magnetic Field Generation
					--------------------------------------------------------------------*/
					  if(!RF_ON()) (*SR176State)=SR176_READY;
					  else return sts=1;
					  break;
		case SR176_READY:
				/*------------------------------------------------------------------
				  		SEARCH DEVICE CLOSE TO THE ANTENNA
				  		
					RX[0]=1			Number of receive byte											   
					RX[1]=CHIP_ID	Chip ID of the Tag												   
																									   
					Rem: - Only one time for SR176 if the tag stays in field						   
																									   
					Trame receive from the CRX14 if there's no tag in field							  
					RX[0]=0			Number of receive byte											   
				  		
				--------------------------------------------------------------------*/
				  		if(!Initiate (&RX_frame[0]))
				  			if((!RX_frame[0])||(RX_frame[0]==0xFF))
				  								(*SR176State)=SR176_READY;		//no tags in the field
				  			else				(*SR176State)=SR176_ACTIVE;	//tags in the field
					  else return sts=1;
					  break;
		case SR176_ACTIVE:
				/*------------------------------------------------------------------
				  		DETECT TAG ID
				--------------------------------------------------------------------*/
				  	  (*RFID_CHIPID)=RX_frame[1];


				/*------------------------------------------------------------------
				  		ACTIVATE TAG WITH THE SELECTED ID

					Trame receive from the CRX14 if there's tag in field and good CHIP_ID transmitted  
					RX[0]=1			Number of receive byte											   
					RX[1]=CHIP_ID	Chip ID of the Tag												   
																									   
																									   
					Trame receive from the CRX14 if there's no tag in field or not initiate			   
					or false CHIP_ID																   
					RX[0]=0			Number of receive byte											   
				--------------------------------------------------------------------*/
				  	 if(!Select (RFID_CHIPID,&RX_frame[0]))					// RX_frame return the same CHIPID if it is recognized
					  	if((*RFID_CHIPID)==RX_frame[1])	(*SR176State)=SR176_SELECTED;
					  	else							(*SR176State)=SR176_ACTIVE;
					 else return sts=1;
					 break;
		case SR176_SELECTED:
					break;
		default:	(*SR176State)=SR176_POWEROFF;
		}

return sts;
}




/*******************************************************************************
* Function Name  : init_CRX14_I2C
* Description    : Configure and init I2C in order to communicates with CRX14
* Input          : None.
* Output         : None.
* Return         : Status.
*******************************************************************************/
void init_CRX14_I2C(void){
u32 f_RFID=I2C_frequency;
	
	/*-------------------------------------------------------------------------------------
	Configure the SDA And the SCL pin of the I2C0 and I2C1 to alternate function Open Drain
	--------------------------------------------------------------------------------------*/
	GPIO_Config (GPIO1,I2C0_SCL,GPIO_AF_OD);
	GPIO_Config (GPIO1,I2C0_SDA,GPIO_AF_OD);

	/*-------------------------------------------------------------------------------------
	Configure I2C0 module
	--------------------------------------------------------------------------------------*/
	I2C_Init (I2C0);
	I2C_FCLKConfig (I2C0);
	I2C_OnOffConfig (I2C0, ENABLE);					//Enable the I2C0
	I2C_SpeedConfig (I2C0, f_RFID);					//Configure the I2C0 speed
	I2C_AddressConfig (I2C0,0xA2, I2C_Mode7);		//Configure the I2C0 address (to be used specially when this cell has to be adressed)
	I2C_AcknowledgeConfig (I2C0, ENABLE);			//set str7 to send ack when receive data
}


/*******************************************************************************************/
/*																						   */
/*					C function for using SR176											   */
/*																						   */
/*******************************************************************************************/


/*******************************************************************************
* Function Name  : Initiate
* Description    : Initiate Tag
*					If there's Tag in the field this function returns the Chip ID					   
*					Trame transmit to the CRX14														   
* Input          : None.
* Output         : None.
*					Trame receive from the CRX14 if there's tag in field							   
*					RX[0]=1			Number of receive byte											   
*					RX[1]=CHIP_ID	Chip ID of the Tag												   
*																						   
*					Rem: - Only one time for SR176 if the tag stays in field						   
*			 		- At each time for SRIX4k if the tag stays in field						    
*																						   
*					Trame receive from the CRX14 if there's no tag in field							   
*					RX[0]=0			Number of receive byte											   
* Return         : Status.
* Note			 : 
*					TX[0]=2 		Number of transmit byte											   
*					TX[1]=06h		first byte of command											   
*					TX[2]=00h		Second byte of command											   
*******************************************************************************/
int Initiate (UCHAR *RX)

	{
	UCHAR TX[40];
	int RX_Length=3;

	TX[0]= 2;
	TX[1]=0x06;
	TX[2]=0x00;
	return Send_Receive_Data( TX, RX,RX_Length);
	}


/*******************************************************************************
* Function Name  : Select
* Description    : Select Tag
*					If there's Tag in the field this function returns the Chip ID					   
*					Trame transmit to the CRX14														   
* Input          : None.
* Output         : None.
*					Trame receive from the CRX14 if there's tag in field and good CHIP_ID transmitted  
*					RX[0]=1			Number of receive byte											   
*					RX[1]=CHIP_ID	Chip ID of the Tag												   
*																						   
*					Trame receive from the CRX14 if there's no tag in field or not initiate			   
*					or false CHIP_ID																   
*					RX[0]=0			Number of receive byte											   
* Return         : Status.
* Note			 : 
*		TX[0]=2 		Number of transmit byte											   
*		TX[1]=0Eh		first byte of command											   
*		TX[2]=CHIP_ID	Second byte of command											   
*******************************************************************************/
int Select (UCHAR *CHIP_ID,UCHAR *RX)

	{
	UCHAR TX[40];
	int RX_Length=3;

	TX[0]= 2;
	TX[1]=0x0E;
	TX[2]=CHIP_ID[0];
	return Send_Receive_Data( TX, RX,RX_Length);
	}


/*******************************************************************************
* Function Name  : ReadBlock
* Description    : Read a block Tag
*					If there's selected Tag in the field this function returns the content			   
*					of the block specified by the adress.											   
* Input          : 
*					Addr: adress of the block														   
* Output         : 
*					RX[1]... RX[n]: Content of the block											   
*
*					Trame received from the CRX14 if there's tag in field and good CHIP_ID transmitted  
*					RX[0]= nn		Number of receive byte											   
*					RX[1]= xx		DATA 1															   
*					RX[2]= xx		DATA 2															   
*		   				|				|															   
*					RX[nn]= xx		DATA nn															   
*																						   
*					Trame receive from the CRX14 if there's no tag in field or not selected			   
*					RX[0]=0			Number of receive byte											   
*
*					Rem: nn is number of bytes for each block										   
*						 nn =2 for SR176															   
*			 			 nn= 4 for SRIX4k															   
*
*
* Return         : Status.
* Note			 : 
*		Trame transmit to the CRX14														   
*		TX[0]=2 		Number of transmit byte											   
*		TX[1]=08h		first byte of command											   
*		TX[2]=Addr		Second byte of command											   
******************************************************************************/
int ReadBlock (UCHAR Addr, UCHAR *RX)
	
	{
	UCHAR TX[40];
	int RX_Length=6;

	TX[0]=2;
	TX[1]=0x08;
	TX[2]=Addr;
	return Send_Receive_Data( TX, RX,RX_Length);
	}


/*******************************************************************************
* Function Name  : WriteBlock
* Description    : Write a block Tag
*					If there's selected Tag in the field this function writes array of yy data		   
*					in Addr block and return no answers.											   
* Input          : 
*					Addr: adress of the block														   
*					DATA_WR: array of data to write in Addr block									   
* Output         : 
*					RX[1]... RX[n]: Content of the block											   
*
*					Trame received from the CRX14 if there's tag in field and good CHIP_ID transmitted  
*					RX[0]= nn		Number of receive byte											   
*					RX[1]= xx		DATA 1															   
*					RX[2]= xx		DATA 2															   
*		   				|				|															   
*					RX[nn]= xx		DATA nn															   
*																						   
*					Trame receive from the CRX14 if there's no tag in field or not selected			   
*					RX[0]=0			Number of receive byte											   
*
*					Rem: nn is number of bytes for each block										   
*						 nn =2 for SR176															   
*			 			 nn= 4 for SRIX4k															   
*
*
* Return         : Status.
* Note			 : 
*		Trame transmit to the CRX14														   
*		TX[0]= nn		Number of transmit byte											   
*		TX[1]= 09h		first byte of command											   
*		TX[2]= Addr		Second byte of command											   
*		TX[3]= xx		DATA_WR 1														   
*		TX[4]= xx		DATA_WR 2														   
*		   |				|															   
*		TX[nn]= xx		DATA_WR yy														   
*
*
*		Rem: nn is number of bytes for each block										   
*			 nn =2 for SR176															   
*			 nn= 4 for SRIX4k															   
******************************************************************************/
int WriteBlock (UCHAR Addr, UCHAR *DATA_WR, UCHAR *RX)

	{
	UCHAR TX[40];
	int RX_Length=3;

    TX[0]=4;
    TX[1]=0x09;
    TX[2]=Addr;
	TX[3]=DATA_WR[0];
	TX[4]=DATA_WR[1];
	return Send_Receive_Data ( TX, RX,RX_Length);
	}


/*******************************************************************************
* Function Name  : Completion
* Description    : Send completion command to the tag
*					If there's selected Tag in the field this function send completion command		   
* Input          : 
* Output         : 
*					Trame receive from the CRX14 													   
*					RX[0]=0			Number of receive byte											   
*
*
* Return         : Status.
* Note			 : 
*					Trame transmit to the CRX14														   
*					TX[0]=1 		Number of transmit byte											   
*					TX[1]=0Fh		first byte of command											   
*
******************************************************************************/
int Completion (UCHAR *RX)
	
	{
	UCHAR TX[40];
	int RX_Length=3;


    TX[0]= 1;             
    TX[1]=0x0F;
	return Send_Receive_Data ( TX, RX,RX_Length);
	}


/*******************************************************************************
* Function Name  : Get_Protect
* Description    : Send get protect command to the tag
*					If there's Tag in the field this function send get protect command				   
* Input          : 
* Output         : 
*					Trame receive from the CRX14 													   
*					RX[0]= 2		Number of receive byte											   
*					RX[1]= xx		CHIP_ID															   
*					RX[2]= xx		LOCK_REG														   
*																						   
*					Trame receive from the CRX14 if there's no tag in field or not selected			   
*					RX[0]=0			Number of receive byte											   
*
*
* Return         : Status.
* Note			 : 
*					Trame transmit to the CRX14														   
*					TX[0]=2 		Number of transmit byte											   
*					TX[1]=08h		first byte of command											   
*					TX[2]=0Fh		Second byte of command											   
*
******************************************************************************/
int Get_Protect (UCHAR *RX)
	
	{
	UCHAR TX[40];
	int RX_Length=3;

    TX[0]= 2;             
    TX[1]=0x08;
    TX[2]=0x0F;
	return Send_Receive_Data ( TX, RX,RX_Length);
	}


/*******************************************************************************
* Function Name  : Protect_Block
* Description    : Send protect block command to the tag
*					If there's Tag in the field this function send get protect command				   
* Input          : 
*					LOCK_REG: Protection register													   
* Output         : 
*					Trame receive from the CRX14
*					RX[0]=0			Number of receive byte											   
*
*
* Return         : Status.
* Note			 : 
*					Trame transmit to the CRX14														   
*					TX[0]= nn		Number of transmit byte											   
*					TX[1]= 09h		first byte of command											   
*					TX[2]= 0Fh		Second byte of command											   
*					TX[3]= 00h		Third byte of command											   
*					TX[4]= LOCK_REG	Request parameter												   
*
******************************************************************************/
int Protect_Block (UCHAR *Lock_Reg, UCHAR *RX) 

	{
	UCHAR TX[40];
	int RX_Length=3;

    TX[0]= 4;             
    TX[1]=0x09;
	TX[2]=0x0F;
	TX[3]=0x00;
	TX[4]=*Lock_Reg;
	return Send_Receive_Data ( TX, RX,RX_Length);
	}



/*******************************************************************************************/
/*																						   */
/*					C function for using CRX14											   */
/*																						   */
/*******************************************************************************************/
extern int timeout;

/*******************************************************************************
* Function Name  : Polling
* Description    : Wait device until it is ready
* Input          : i2c address.
* Output         : None.
*
* Return         : Status.
*					1 - No error
*					0 - error
*
******************************************************************************/
int Polling(int addr)
{

u8 status=0;
u8 state=0;
u8 timeout=0xff;

	while(state<=4){
		switch(state){
			case 0:
				I2C_STARTGenerate (I2C0, ENABLE);										//i2c_start();	/* Send addr to I2C */	
				while ((I2C_FlagStatus (I2C0,DIRECT,I2C_SB )==RESET)&&(--timeout>0));	//Wait til the SB flag is set to confirm the START generation
			 	if(timeout) state++;													//send address of byte
			 	else return status=1;
			 	break;
			case 1:
			  	if(!(I2C_ByteSend_with_status (I2C0, addr))) state++;					//i=i2c_send_byte(addr);	/* get ACK */
			 	else return status=1;
			 	break;
			case 2:
			    if(!((I2C0->SR2 & 0x10)>>4)) state++;									//checl ack
			 	else return status=1;
			 	break;
			case 3:
				while ((I2C_FlagStatus (I2C0,DIRECT,I2C_ENDAD )==RESET)&&(--timeout>0)); //Test the end of address transmission
			 	if(timeout) state++;													//send address of byte
			 	else return status=1;
			 	break;
			case 4:
				I2C_FlagClear (I2C0, I2C_ENDAD);
			 	state++;
			 	break;
			default: status=1;
		
		}
	}
	return status;
}



/*******************************************************************************
* Function Name  : Write_CRX14_IOFR
* Description    : Write IO Frame Register CRX14
* Input          : data to send.
* Output         : None.
*
* Return         : Status.
*					1 - No error
*					0 - error
******************************************************************************/
int Write_CRX14_IOFR(UCHAR *DATA){

int i=0;
	/*------------------------------------------------------------------
	  		Write Data @ addr
	--------------------------------------------------------------------*/
	  if(!Polling(0xA0)) i=Write_RFID(CRX14_IOFR,&DATA[1],DATA[0]);
	  else i=1;
	  return i;

}


/*******************************************************************************
* Function Name  : Write_CRX14_PR
* Description    : Write Parameter Register CRX14
* Input          : data to send.
* Output         : None.
*
* Return         : Status.
*					1 - No error
*					0 - error
******************************************************************************/
int Write_CRX14_PR(UCHAR *DATA)
{

int i=0;

	/*------------------------------------------------------------------
	  		Write Data @ addr
	--------------------------------------------------------------------*/
	  if(!Polling(0xA0)) i=Write_RFID(CRX14_PR,&DATA[1],1);
	  else i=1;


	  return i;

}


/*******************************************************************************
* Function Name  : Write_Authenticate
* Description    : Write Authenticate Register CRX14
* Input          : addr ,DATA.
* Output         : None.
*
* Return         : Status.
*					1 - No error
*					0 - error
******************************************************************************/
int Write_Authenticate(int addr ,UCHAR *DATA)
{

int i;

			if(!Polling(addr)) return 0 ;											/* Send addr to I2C			*/
			I2C_ByteSend (I2C0, 0x2);//if(!i2c_send_byte(0x02)) return 0;			/* Register 02h Authenticate*/
			for(i=0; i<15; i++)
				I2C_ByteSend (I2C0, DATA[i]);//if(!i2c_send_byte(DATA[i])) return 0;/* Send Frame to CRX14		*/
			I2C_STOPGenerate (I2C0, ENABLE);//i2c_stop();							/* run authenticate sequence */	


	return 1;												
}

/*******************************************************************************
* Function Name  : Write_Slot
* Description    : Write Slot Marker Register CRX14
* Input          : addr.
* Output         : None.
*
* Return         : Status.
*					1 - No error
*					0 - error
******************************************************************************/
int Write_Slot(int addr)
{

			if(!Polling(addr)) return 0;										/* Send addr to I2C */
			I2C_ByteSend (I2C0, 0x03);//if(!i2c_send_byte(0x03)) return 0;		/* Register 03h run */
			I2C_STOPGenerate (I2C0, ENABLE);//i2c_stop();						/* anti-collision sequence */
	return 1;													
}


/*******************************************************************************
* Function Name  : Read_CRX14_IOFR
* Description    : Read IO Frame Register CRX14
* Input          : None.
* Output         : DATA.
*
* Return         : Status.
*					1 - No error
*					0 - error
******************************************************************************/
int Read_CRX14_IOFR(UCHAR *DATA)
{
int i=0;
	/*------------------------------------------------------------------
	  		Read Data @ CRX14_IOFR
	--------------------------------------------------------------------*/
	  if(!Polling(0xA0)) i=Read_RFID(CRX14_IOFR,&DATA[1],&DATA[0]);
	  else i=1;
	  
	  return i;

}



/*******************************************************************************
* Function Name  : Read_CRX14_PR
* Description    : Read Parameter Register CRX14
* Input          : None.
* Output         : Data.
*
* Return         : Status.
*					1 - No error
*					0 - error
******************************************************************************/
int Read_CRX14_PR(UCHAR *DATA)
{
int i=0;


	/*------------------------------------------------------------------
	  		Read Data @ CRX14_IOFR
	--------------------------------------------------------------------*/
	  if(!Polling(0xA0)) i=Read_RFID(CRX14_PR,&DATA[1],&DATA[0]);
	  else i=1;
	  

	  
	  return i;

}


/*******************************************************************************
* Function Name  : Read_Sign
* Description    : Read Sign Register CRX14
* Input          : addr.
* Output         : Data.
*
* Return         : Status.
*					1 - No error
*					0 - error
******************************************************************************/
int Read_Sign(int addr ,UCHAR *DATA)
{


			if(!Polling(addr)) return 0;													/* Send addr to I2C wait ACK*/
			I2C_ByteSend (I2C0, 0x2);				//if(!i2c_send_byte(0x02)) return 0;	/* Register 01h				*/
			I2C_STARTGenerate (I2C0, ENABLE);		//i2c_start();							/* Restart					*/
			I2C_ByteSend (I2C0, addr+1);			//if(!i2c_send_byte(addr +1)) return 0;	/* Read mode				*/
			DATA[0]=I2C_ByteReceive (I2C0);			//i2c_read_byte(1);						/* Read data				*/
			DATA[1]=I2C_ByteReceive (I2C0);		//i2c_read_byte(0);	
			I2C_STOPGenerate (I2C0, ENABLE);		//i2c_stop();

	return 1;												
}


/*******************************************************************************
* Function Name  : Read_slot
* Description    : Read Slot Register CRX14
* Input          : addr.
* Output         : Data.
*
* Return         : Status.
*					1 - No error
*					0 - error
******************************************************************************/
int Read_slot(int addr ,UCHAR *DATA)
{

int i;
			if(!Polling(addr)) return 0;																/* Send addr to I2C wait ACK*/
			I2C_ByteSend (I2C0, 0x1);							//if(!i2c_send_byte(0x01)) return 0;	/* Register 01h I/O			*/
			I2C_STARTGenerate (I2C0, ENABLE);					//i2c_start();							/* Restart					*/
			I2C_ByteSend (I2C0, addr+1);						//if(!i2c_send_byte(addr +1)) return 0;	/* Read mode				*/
				for(i=0;i<17;i++) DATA[i]=I2C_ByteReceive (I2C0);//i2c_read_byte(1); 
			i++;
			DATA[i]=I2C_ByteReceive (I2C0);						//i2c_read_byte(0);						/* Read last byte NO_ACK*/
			I2C_STOPGenerate (I2C0, ENABLE);					//i2c_stop();

	return 1;													/* no error */
}


/*******************************************************************************************/
/********************* FUNCTION TO SEND PARAMETER TO THE CRX14 *****************************/
/*******************************************************************************************/
/*																						   */
/*																						   */
/* 		Function to send Parameter to CRX14 with low level function Write_IO:			   */
/*			TX[0]= xx   	Value of new parameter register								   */
/*																						   */
/*******************************************************************************************/

/*******************************************************************************
* Function Name  : Time_Read
* Description    : 
* Input          : None.
* Output         : None.
*
* Return         : Status.
*					1 - error
*					0 - No error
******************************************************************************/
int Time_Read (void)
{

UCHAR TRAME[128];
	if(Read_CRX14_PR(TRAME)) return 1;
	
	TRAME[1]&=0x9F;									/* reset bit 5 & bit 6 of parameter register */
	
	if(Write_CRX14_PR(TRAME)) return 1;
	if(Read_CRX14_PR(TRAME)) return 1;

	return 0;
}


/*******************************************************************************
* Function Name  : Time_Authenticate
* Description    : 
* Input          : None.
* Output         : None.
*
* Return         : Status.
*					1 - error
*					0 - No error
******************************************************************************/
int Time_Authenticate (void)
{

UCHAR TRAME[128];

	if(Read_CRX14_PR(TRAME)) return 1;
	TRAME[1]&=0xDF;									/* reset bit 5 of parameter register */
	TRAME[1]|=0x40;									/* set bit 6 of parameter register   */
	if(Write_CRX14_PR(TRAME)) return 1;
	if(Read_CRX14_PR(TRAME)) return 1;

	return 0;
}

/*******************************************************************************
* Function Name  : Time_Write
* Description    : 
* Input          : None.
* Output         : None.
*
* Return         : Status.
*					1 - error
*					0 - No error
******************************************************************************/
int Time_Write (void)
{

UCHAR TRAME[128];

	if(Read_CRX14_PR(TRAME)) return 1;
	TRAME[1]&=0xBF;									/* reset bit 6 of parameter register */
	TRAME[1]|=0x20;									/* set bit 5 of parameter register   */
	if(Write_CRX14_PR(TRAME)) return 1;
	if(Read_CRX14_PR(TRAME)) return 1;

	return 0;
}


/*******************************************************************************
* Function Name  : RF_OFF
* Description    : Cut off RF carrier
* Input          : None.
* Output         : None.
*
* Return         : Status.
*					1 - error
*					0 - No error
******************************************************************************/
int RF_OFF (void)
{

UCHAR TRAME[2];

	if(Read_CRX14_PR(TRAME)) return 1;
	TRAME[1]&=0xEF;						// RF OFF /* reset bit 4 of parameter register */
	if(Write_CRX14_PR(TRAME)) return 1;
	if(Read_CRX14_PR(TRAME)) return 1;

	return 0;
}

/*******************************************************************************
* Function Name  : RF_ON
* Description    : Switch on RF carrier
* Input          : None.
* Output         : None.
*
* Return         : Status.
*					1 - error
*					0 - No error
******************************************************************************/
int RF_ON (void)
{

UCHAR TRAME[2];

	if(Read_CRX14_PR(TRAME))	return 1;
	TRAME[1]|=0x10;						// RF ON  /* set bit 4 of parameter register */
	if(Write_CRX14_PR(TRAME))	return 1;
	if(Read_CRX14_PR(TRAME))	return 1;
	if (!(TRAME[1]&0x10))		return 1;
	

	return 0;
}


/*******************************************************************************
* Function Name  : Send_Receive_Data
* Description    : 
* 					This function send Trame (TX) to CRX14 using the Write_IO function and				   
*					Receive Trame from CRX14 using the Read_IO_data function.						   
* Input          : 
*					TX[0]= nn   	Number of byte of the command									   
*					TX[1]= xx   	First byte of command											   
*					TX[2]= xx		Second byte of command											   
*		  				 |					|														   
*					TX[nn]=xx		nn byte of command												   
* Output         : 
*					RX[0]= nn		Number of receive byte											   
*					RX[1]= xx		DATA 1															   
*					RX[2]= xx		DATA 2															   
*		  				 |				|															   
*					RX[nn]= xx		DATA nn															   
*
* Return         : Status.
*					1 - error
*					0 - No error
******************************************************************************/
int Send_Receive_Data(UCHAR *TX, UCHAR *RX, int Nb_RX )

	{
	if (!Write_CRX14_IOFR(TX))  
	return Read_CRX14_IOFR(RX);
	else return 1;
	}

/*******************************************************************************
* Function Name  : Send_Receive_Authenticate
* Description    : 
* 					This function send Trame (TX) to CRX14 using the Write_Authenticate function and				   
*					Receive Trame from CRX14 using the Read_Sign function.						   
* Input          : 
*					TX[0]= nn   	Number byte of the command										   
*					TX[1]= xx   	First byte of command											   
*					TX[2]= xx		Second byte of command											   
*		  				 |					|														   
*					TX[nn]=xx		nn byte of command												   
* Output         : 
*					RX[0]= nn		Number of receive byte											   
*					RX[1]= xx		DATA 1															   
*					RX[2]= xx		DATA 2															   
*		   				|				|															   
*					RX[nn]= xx		DATA nn															   
*
* Return         : Status.
*					1 - error
*					0 - No error
******************************************************************************/
int Send_Receive_Authenticate(UCHAR *TX, UCHAR *RX, int Nb_RX )

	{
	if (Write_Authenticate(ADDR ,TX))
	return Read_Sign(ADDR ,RX);
	else return 0;
	}


/*******************************************************************************
* Function Name  : Send_Receive_Inventory
* Description    : 
* 					This function start an automated anti-collision sequence
*					If there's selected Tag in the field this function sends Anti-collision sequence   
*					and returns the status and potential CHIP-ID in each time slots.				   
* Input          : None.
* Output         : 
*					RX[0]=18		Number of receive bytes											   
*					RX[1]=xx		Status Slot bit0-bit7											   
*					RX[2]=xx		Status Slot bit8-bit15											   
*					RX[3]=xx		Slot Register 0													   
*						|																			   
*					RX[nn]=xx		Slot Register 15												   
*
* Return         : Status.
*					1 - error
*					0 - No error
* Note			 :  IMPORTANT - FUNCTION NOT IMPLEMENTED
******************************************************************************/
int Send_Receive_Inventory (UCHAR *RX, int Nb_RX )

	{
	
	if(Write_Slot(ADDR)) return 1;//Read_IO_data(ADDR,RX);
	else return 0;
	}


/******************* (C) COPYRIGHT 2006 STMicroelectronics *****END OF FILE****/





