/******************** (C) COPYRIGHT 2008 STMicroelectronics ********************
* File Name          : iap.c
* Author             : MCD Application Team
* Version            : V2.0.0
* Date               : 08/08/2008
* Description        : This file provides all the software needed to implement the 
*                      dofferent protocol commands.
********************************************************************************
* THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
* WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME.
* AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT,
* INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE
* CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING
* INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
*******************************************************************************/
/* Includes ------------------------------------------------------------------*/
#include "iap.h"
/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
void Disable_Write_Protection(void);
void IAP_Init(void);
u8 Check_Address(u32 Address);
u32 Get_Address(void);
u8 Get_Byte(void);
void Send_Byte(u8 Byte);
u8 Verify_Checksum(u32 address,u8 checksum);
void SCU_Configuration(void);

/* Private functions ---------------------------------------------------------*/
/*******************************************************************************
* Function Name  : Send_Byte
* Description    : Send a byte
* Input          : The byte to be sent
* Output         : None
* Return         : None
*******************************************************************************/
void Send_Byte(u8 Byte)
{
    UART_SendData(UART0, Byte);
    UART_WaitForSend();
}

/*******************************************************************************
* Function Name  : Get_Byte
* Description    : Get the command
* Input          : None
* Output         : None
* Return         : The Command code
*******************************************************************************/
u8 Get_Byte(void)
{
    UART_WaitForByte() ;
    return (UART_ReceiveData(UART0));
}

/*******************************************************************************
* Function Name  : Get_Address
* Description    : Get the address from the user
* Input          : None
* Output         : None
* Return         : The address value
*******************************************************************************/
u32 Get_Address(void)
{
    u32 address = 0;
    /* Receive the start address */
    UART_WaitForByte();
    address |= ((int)UART0->DR)<<24;
    UART_WaitForByte();
    address |= ((int)UART0->DR)<<16;
    UART_WaitForByte();
    address |= ((int)UART0->DR)<<8;
    UART_WaitForByte();
    address |= ((int)UART0->DR);
    return (address);
}

/*******************************************************************************
* Function Name  : Check_Address
* Description    : Check if the address is valid
* Input          : The received address
* Output         : None
* Return         : The address area
*******************************************************************************/
u8 Check_Address(u32 address)
{

#ifdef  Flash_512KB_256KB
    if ((address >= 0x80000)&&(address < 0xFFFFF))
#endif

#ifdef  Flash_2MB_1MB
        if ((address >= 0x200000)&&(address < 0xFFFFFF))
#endif
        {
            return Flash_Area;
        }
        else
        {
            return Area_Error;
        }
}

/*******************************************************************************
* Function Name  : Verify_Checksum
* Description    : Check if the address checksum is ok or no
* Input          :  - address
*                   - checksum
* Output         : None
* Return         : 1 if checksum is ok & 0 if no
*******************************************************************************/

u8 Verify_Checksum(u32 address,u8 checksum)
{
    if (checksum == (((int)address>>24)^(((int)address & 0x00FFFFFF)>>16)^
                     (((int)address & 0x0000FFFF )>>8)^((int) address & 0x000000FF)))


        return 1;
    else return 0;

}
/*******************************************************************************
* Function Name  : Disable_Write_Protection
* Description    : Disable all Flash Bank0 write protection.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/

void Disable_Write_Protection(void )
{
    FMI_WriteProtectionCmd(FMI_B0S0, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S1, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S2, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S3, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S4, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S5, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S6, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S7, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S8, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S9, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S10, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S11, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S12, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S13, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S14, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S15, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S16, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S17, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S18, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S19, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S20, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S21, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S22, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S23, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S24, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S25, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S26, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S27, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S28, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S29, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S30, DISABLE);
    FMI_WriteProtectionCmd(FMI_B0S31, DISABLE);

}

/*******************************************************************************
* Function Name  : SCU_Configuration
* Description    : Configure System Clocks
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/

void SCU_Configuration(void)
{

    SCU_MCLKSourceConfig(SCU_MCLK_OSC);	/* Default configuration */
    FMI_Config(FMI_READ_WAIT_STATE_2, FMI_WRITE_WAIT_STATE_0, FMI_PWD_ENABLE,\
               FMI_LVD_ENABLE, FMI_FREQ_HIGH);/*Insert 2 Wait States for read*/

    SCU_PLLFactorsConfig(192, 25, 2); /* PLL factors Configuration based on*/
    /* a OSC/Crystal value = 25Mhz*/
    SCU_PLLCmd(ENABLE);  /* PLL Enable and wait for Locking*/
    SCU_RCLKDivisorConfig(SCU_RCLK_Div1); /* RCLK @96Mhz */
    SCU_HCLKDivisorConfig(SCU_HCLK_Div1); /* AHB @96Mhz */
    SCU_FMICLKDivisorConfig(SCU_FMICLK_Div1);/* FMI @96Mhz */
    SCU_PCLKDivisorConfig(SCU_PCLK_Div2); /* APB @48Mhz */
    SCU_MCLKSourceConfig(SCU_MCLK_PLL);  /* MCLK @96Mhz */

    /* Enable the FMI Clock */
    SCU_AHBPeriphClockConfig(__FMI, ENABLE);
    SCU_AHBPeriphReset(__FMI, DISABLE);
    /* Enable the UART0 Clock */
    SCU_APBPeriphClockConfig(__UART0, ENABLE);
    /* Enable the GPIO3 Clock */
    SCU_APBPeriphClockConfig(__GPIO3, ENABLE);
    /* Enable the GPIO5 Clock */
    SCU_APBPeriphClockConfig(__GPIO5, ENABLE);

}
/*******************************************************************************
* Function Name  : IAP_Init
* Description    : Initialize 
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/

void IAP_Init(void)
{

    GPIO_InitTypeDef GPIO_InitStructure;
    UART_InitTypeDef UART_InitStructure;

    SCU_Configuration();
    /* Configure UART0_Rx pin GPIO5.1 */
    GPIO_InitStructure.GPIO_Direction = GPIO_PinInput;
    GPIO_InitStructure.GPIO_Pin = GPIO_Pin_1;
    GPIO_InitStructure.GPIO_Type = GPIO_Type_PushPull ;
    GPIO_InitStructure.GPIO_IPInputConnected = GPIO_IPInputConnected_Enable;
    GPIO_InitStructure.GPIO_Alternate = GPIO_InputAlt1  ;
    GPIO_Init(GPIO5, &GPIO_InitStructure);

    /*Gonfigure UART0_Tx pin GPIO3.4*/
    GPIO_InitStructure.GPIO_Direction = GPIO_PinOutput;
    GPIO_InitStructure.GPIO_Pin = GPIO_Pin_4;
    GPIO_InitStructure.GPIO_Type = GPIO_Type_PushPull ;
    GPIO_InitStructure.GPIO_Alternate = GPIO_OutputAlt3  ;
    GPIO_Init(GPIO3, &GPIO_InitStructure);

    /* UART0 configured as follow:
          - Word Length = 8 Bits
          - One Stop Bit
          - Even parity
          - BaudRate = 115200 baud
          - Hardware flow control Disabled
          - Receive and transmit enabled
          - Receive and transmit FIFOs are Disabled
    */
    UART_InitStructure.UART_WordLength = UART_WordLength_8D;
    UART_InitStructure.UART_StopBits = UART_StopBits_1;
    UART_InitStructure.UART_Parity = UART_Parity_Even ;
    UART_InitStructure.UART_BaudRate = 115200;
    UART_InitStructure. UART_HardwareFlowControl = UART_HardwareFlowControl_None;
    UART_InitStructure.UART_Mode = UART_Mode_Tx_Rx;
    UART_InitStructure.UART_FIFO = UART_FIFO_Disable;
    UART_DeInit(UART0);

    UART_Init(UART0, &UART_InitStructure);
    UART_Cmd(UART0, ENABLE);
    /* Wait for 0x7F from the host */
    while ( UART_GetFlagStatus(UART0, UART_FLAG_RxFIFOEmpty) != RESET);
    /* After receiving 0x7F from the host, the STR9 sends an acknowledge byte
       to the host to say that it is ready to receive commands */
    Send_Byte(ACK_BYTE);

}

/*******************************************************************************
* Function Name  : Select_Command
* Description    : Select which command to apply (write, read etc...)
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void Select_Command(void)
{
    u16 Counter = 0;
    u16 Nb = 0;
    u32 Address, address = 0;
    u32 RAMAddress = 0;
    u8 Current_Command = 0;
    u8 Checksum =0;
    u8 checksum = 0;
    u8 Status =0;
    u8 x;

    IAP_Init();

    while (1)
    {
        /* Get the user command from the host */
        Current_Command = Get_Byte();

        switch  (Current_Command)
        {


            /* Get command */
        case (GC_Command):


        {
            /* Wait to receive the GET command complement */
            if ( Get_Byte()== 0xFF ^ GC_Command)

            {
                /* Send Acknowledge byte */
                Send_Byte(ACK_BYTE);
                /* Send the number of bytes to be sent - 1  (Version + Commands) */
                Send_Byte(0x7); /* 8 - 1 = 7 */
                /* Send the version */
                Send_Byte(Version_Number);
                /* Send the Supported Commands */
                Send_Byte(0x00); /* Get Command */
                Send_Byte(0x02); /* Get device ID */
                Send_Byte(0x11); /* Read Memory */
                Send_Byte(0x21); /* Go */
                Send_Byte(0x31); /* Write */
                Send_Byte(0x43); /* Erase */
                Send_Byte(0x71); /* Write unprotect */
                /* Send the Achnowledge byte */
                Send_Byte(ACK_BYTE);
            }
            else /* The device didn't receive the GET command complement */
            {
                Send_Byte(NACK_BYTE);
            }
            break;
        } /* end GC_Command */

        /* Get ID command */
        case  (GID_Command):
        {
            /* Wait for the GID command complement from the host */
            if (Get_Byte()== 0xFF ^ GID_Command)
            {
                /* Send the ACK byte */
                Send_Byte(ACK_BYTE);
                Send_Byte(0x3);
                /* Send the PID */
                Send_Byte(0x25);
                Send_Byte(0x96);
                Send_Byte(0x60);
                Send_Byte(0x41);
                /* Send the ACK byte */
                Send_Byte(ACK_BYTE);

            }
            else

            {
                Send_Byte(NACK_BYTE);

            }
            break;
        } /* End GID_Command */


        /* Go command */
        case  (GO_Command):
        {
            /* Wait for the GO command complement from the host */
            if (Get_Byte()== 0xFF ^ GO_Command)
            {
                /* Send the ACK byte */
                Send_Byte(ACK_BYTE);
                /* Receive the start address */
                Address = Get_Address();
                /* Receive the start address Checksum from the host */
                Checksum = Get_Byte();
                /* Verify if Checksum is ok or no: 0 if problem and 1 if ok */
                Status = Verify_Checksum(Address, Checksum);
                /* If Address is not valid OR the checksum is not ok */
                if ((Check_Address(Address) == Area_Error)||(Status == 0))
                {
                    /* Send the NACK byte */
                    Send_Byte(NACK_BYTE);
                }

                else /* Jump to the new program */
                {
                    /* Send the ACK byte */
                    Send_Byte(ACK_BYTE);
                    Execute_STR9Application();
                }
            }

            else

            {
                /* Send the NACK byte */
                Send_Byte(NACK_BYTE);

            }

            break;
        } /* end GO_Command */

        /* Read Memory Command */
        case ( RM_Command):

        {
              Disable_Write_Protection();
            /* Wait to receive the Read Command complement*/
            if (Get_Byte()== 0xFF ^ RM_Command)

            {
                /* Send the ACK byte */
                Send_Byte(ACK_BYTE);
                /* Receive the start address from the host */
                address =  Get_Address();
                /* Receive the start address checksum from the host */
                Checksum = Get_Byte();
                /* Verify if the address checksum is ok or no */
                Status = Verify_Checksum(address, Checksum);
                /* If address is not valid OR checksum is not ok  */
                if ((Check_Address(address) == Area_Error)|| (Status == 0))
                {
                    /* Send the NACK byte */
                    Send_Byte(NACK_BYTE);
                }
                /* If address is valid and Checksum is ok */
                else
                {
                    /* Send the Acknowledge byte to the host */
                    Send_Byte(ACK_BYTE);
                    /* Receive the number of bytes to be read - 1 */
                    Nb = Get_Byte();
                    /* Receive the checksum of the (number of bytes - 1) */
                    Checksum = Get_Byte();
                    /* If Checksum is ok */
                    if ((0xFF ^ Nb) == Checksum )
                    {
                        /* Send the Acknowledge */
                        Send_Byte(ACK_BYTE);
                        /* Send the read data */
                        for (Counter = Nb + 1 ; Counter !=0 ; Counter--)
                        {
                            Send_Byte(*(u8*)(address++));
                        }
                    }

                }

            }

            else
            {
                /* Send No Acknowledge */
                Send_Byte(NACK_BYTE);
            }
            break;
        } /* End RM_Command */

        /* Write memory command */
        case (WM_Command):
        {
            Disable_Write_Protection();
            if (Get_Byte()== 0xFF ^ WM_Command)
            {

                /* Send the ACK byte */
                Send_Byte(ACK_BYTE);
                /* Receive the start address */
                Address = Get_Address();
                /* Receive the start address Checksum from the host*/
                Checksum = Get_Byte();
                /* Verify if the address Checksum is ok or no */
                Status = Verify_Checksum(Address, Checksum);
                /* If address is not valid or the Checksum is not ok */
                if ((Check_Address(Address) == Area_Error)||(Status == 0))
                {
                    /* Send the NACK byte */
                    Send_Byte(NACK_BYTE);
                }

                else
                {

                    /* Send the ACK byte */
                    Send_Byte(ACK_BYTE);
                    /* Number of bytes to be written */
                    Nb = Get_Byte() + 0x1 ;
                    /* Checksum is initialized to the (number of bytes - 1) */
                    checksum =   Nb - 1;
                    /* RAMAddress Initialization */
                    RAMAddress = RAM_Buffer;
                    /* Write received data to RAM Buffer */
                    for (Counter =  Nb ; Counter !=0 ; Counter--)
                    {

                        *(u8*)(RAMAddress) =  Get_Byte();
                        checksum = checksum ^ (*(u8*)(RAMAddress));
                        RAMAddress = RAMAddress + 1;
                    }
                    /* Receive the checksum of the received bytes from the host */
                    Checksum = Get_Byte();
                    /* Compare checksum sent by the host and checksum calculated by the MCU */
                    if (checksum != Checksum)
                    {
                        Send_Byte(NACK_BYTE);
                    }
                    else
                    {


                        /* Update the code size and the RAM address to be half word aligned */
                        if ( Nb & 0x2)
                        {

                            Nb = ( Nb & 0xFE) + 2;
                            RAMAddress = (RAMAddress & 0xFFFFFFFE) + 2;
                        }
#ifdef Flash_2MB_1MB
                        SCU_FMICLKDivisorConfig(SCU_FMICLK_Div2);/* FMI @48Mhz */ 
#endif                     

                        /* Program Internal Flash */
                        for (Counter = Nb; Counter != 0; Counter -= 0x2)
                        {
                            FMI_WriteHalfWord((Address + Nb - Counter), *(u16 *)(RAMAddress - Counter));
                            FMI_WaitForLastOperation(FMI_BANK_0);
                        }
#ifdef Flash_2MB_1MB
                        SCU_FMICLKDivisorConfig(SCU_FMICLK_Div1);/* FMI @96Mhz */ 
#endif                          


                        if (FMI_GetFlagStatus(FMI_FLAG_PS, FMI_BANK_0))  Send_Byte(NACK_BYTE);
                        else Send_Byte(ACK_BYTE);

                    }
                }
            }


            else
            {

                Send_Byte(NACK_BYTE);
            }


            break;
        } /* End WM_Command */

        /* Erase Memory command */

        case ( ER_Command):
        {
            Disable_Write_Protection();
            if (Get_Byte()== 0xFF ^ ER_Command)
            {

                /* Send the ACK byte */
                Send_Byte(ACK_BYTE);
                x = Get_Byte();
                /* Wait for 0xFF */
                if (x == 0xFF)
                {
                    /* Total erase */
                    /* Wait for 0x00 */
                    if (Get_Byte()== 0x00)
                    {
                        /* Erase ALL Bank0 */
                        FMI_EraseBank(FMI_BANK_0);
                        FMI_WaitForLastOperation(FMI_BANK_0);

                    }

                }

                else
                {

                    /* The number of sectors - 1 to be erased  */
                    Nb = x;
                    /* RAMAddress Initialization */
                    RAMAddress = RAM_Buffer;
                    /* The checksum is initialized to the (number of sectors - 1) */
                    checksum = Nb;
                    /* UART receives the sectors' codes to RAM Buffer */
                    for (Counter = Nb + 1 ; Counter !=0 ; Counter--)
                    {

                        *(u8*)(RAMAddress)= Get_Byte();
                        checksum = checksum ^ (*(u8*)(RAMAddress));
                        RAMAddress = RAMAddress + 1 ;
                    }
                    /* Receive the checksum of the sectors' number - 1 + the received
                       sectors' codes from the host */
                    Checksum = Get_Byte();
                    if (Checksum != checksum)

                    {
                        Send_Byte(NACK_BYTE);

                    }
                    else
                    {
#ifdef Flash_2MB_1MB
                        SCU_FMICLKDivisorConfig(SCU_FMICLK_Div2);/* FMI @48Mhz */ 
#endif                     
                      /* Erase the needed sectors */
                        for (Counter = Nb + 1; Counter !=0 ; Counter--)
                        {
                            FMI_EraseSector(((*(u8 *)(RAMAddress-Counter))<<16) + FMI_BANK_0);
                            FMI_WaitForLastOperation(FMI_BANK_0);

                        }
#ifdef Flash_2MB_1MB                        
                        SCU_FMICLKDivisorConfig(SCU_FMICLK_Div1);/* FMI @96Mhz */
#endif                       

                    }



                }
                if (FMI_GetFlagStatus(FMI_FLAG_ES, FMI_BANK_0))  Send_Byte(NACK_BYTE);
                else  Send_Byte(ACK_BYTE);
            }
            else
            {
                Send_Byte(NACK_BYTE);
            }

            break;
        } /* End ER_Command */

        /* Write Unprotect Command */
        case (WUP_Command ):
        {
            if (Get_Byte()== 0xFF ^ WUP_Command)

            {
                /* Send the ACK byte */
                Send_Byte(ACK_BYTE);
                /* Disable the write protection for all the Flash sectors */
                Disable_Write_Protection();
                /* Send Acknowledge */
                Send_Byte(ACK_BYTE);
            }

            else
            {
                /* Send the NACK byte */
                Send_Byte(NACK_BYTE);
            }
            break;
        } /* End WUP_Command */


        default:
            break;


        } /* End switch*/


    }/* End while */


} /* End Select_Command */




/******************* (C) COPYRIGHT 2008 STMicroelectronics *****END OF FILE****/


