
/****************** (c) 2006  STMicroelectronics, Inc. ***********************/

/**************************************************************************
     PROGRAM  : SENSORLESS CONTROL OF 3 PHASE BLDC MOTOR
     COMPILER : COSMIC
     FILENAME : MOTCON.C 
     RELEASE  : 1.0
     DATE     : Feb 2007
     AUTHOR   : Power Systems Application Lab
                STMicroelectronics, Inc.
                Schaumburg, IL - 60173
***************************************************************************

MODIFICATIONS :

/******************************************************************************
... Test status: .............................................................*
******************************************************************************/

#define _MOTCON_C

#include "include.inc" 

#define  MCPU     (*(static u16 *) (& MCPUH))
#if TEST
  #define  FLAGDR   PDDR
  #define  FLAGPIN  BIT(7)
#endif

/*****************************************************************************/
/***********************  GLOBAL VARIABLES DEFINITIONS  **********************/
/*****************************************************************************/
u8    phasestep,    /* phase step pointer from 0 to 5 */
      Status_Step,  /* Status register for step behaviour during start up */
      
      /* General Flag Variables
      **************************/
      runCmd,             // on / off command
      rampEnable;         // duty ramp flag, to catch with analog duty input 
      
u16   duty,               // dutycycle data
      starting_failed;    // s/w watchdog flag tracking motor running condition

/*****************************************************************************/
/*************************  CONSTANTS  - ST72MC  *****************************/
/*****************************************************************************/
/* MPHST register values from step 1 to step 6 for bldc sensorLESS motor */
/* MPHST  ===>>>  IS1 IS0 T5 T4 T3 T2 T1 T0   */
const u8 phreg_SL[] = {
	0x89,  /* 10001001	;T1-T4 - Valid Comparator C  */
	0x61,  /* 01100001	;T1-T6 - Valid Comparator B  */
	0x24,  /* 00100100	;T3-T6 - Valid Comparator A  */
	0x86,  /* 10000110	;T3-T2 - Valid Comparator C  */
	0x52,  /* 01010010	;T5-T2 - Valid Comparator B  */
	0x18   /* 00011000	;T5-T4 - Valid Comparator A  */
};

/* MCRB reg values from steps 1 to 6 for bldc sensorLESS motor */
/* MCRB ===>>> -0-  CPB  HDM  SDM  OCV  OS2  OS1  OS0 */
const u8 volt_crbreg_SL[] = {
  0x10,  /* 00010000 ;CPB=0 Simulated demagn  */
  0x50,  /* 01010000 ;CPB=1 Simulated demagn  */
  0x10,  /* 00010000 ;CPB=0 Simulated demagn  */
  0x50,  /* 01010000 ;CPB=1 Simulated demagn  */
  0x10,  /* 00010000 ;CPB=0 Simulated demagn  */
  0x50   /* 01010000 ;CPB=1 Simulated demagn  */
};

/****************************************************************************/
/*****************************  SUBROUTINES  ********************************/
/****************************************************************************/

/*********************************************************************
Initialise 72MC for BLDC sensorless control
**********************************************************************/
void MCinit_bldc(void)  {

  // Initialize registers in page 1 first
  // *************************************
  bitSet(MCFR, RPGS);
  
  // MDTG  ===>>>   PCN  DTE  DTG[5:4:3:2:1:0]
  // RESET STATE -   1    1      [1 1 1 1 1 1]
  MDTG = 0;                 // Clr PCN bit first to modify DTE
  MDTG = SIX_STEP_MODE + 
         NO_DEAD_TIME;
  
  // MPOL  ===>>>   ZVD  REO  OP[5:4:3:2:1:0]
  // RESET STATE -   0    0     [1 1 1 1 1 1]
  MPOL = 0x3f;               // Switch driver input polarity (L6386D)
                             // ZVD=0; Z and D have opposite edge
                             // REO=0; read Z on PWM off of even channel
  
  // MPWME  ===>>>   DG  PWMW  PWMV  PWMU  OT[3:2:1:0]
  // RESET STATE  -  0    0     0     0      [0 0 0 0]
  MPWME = DEBUG_ON + 
          DISCONNECT_PWMU + 
          DISCONNECT_PWMV + 
          DISCONNECT_PWMW ;
  
  // MCONF  ===>>>   DS[3:2:1:0]  SOI  SOM  XT16  XT8
  // RESET STATE  -    [0 0 0 0]   0    0    1     0
  MCONF = Fmtc_by_4;
  
  // MPAR  ===>>>  TES1  TES0  OE5  OE4  OE3  OE2  OE1  OE0
  // RESET STATE  -  0     0     0    0    0    0    0    0
  MPAR = OE5 + OE3 + OE1;       // T5 T3 T1 == ODD
                                // T4 T2 T0 == EVEN
  
  // MZFR  ===>>>  ZEF[3:2:1:0]  ZWF[3:2:1:0]
  // RESET STATE  -    0 0 0 0       1 1 1 1
  MZFR = Z_sample(1) + 
         Z_blank_20us ;
  
  // MSCR  ===>>>  ZSV  -  -  -  SCF1  SCF0  ECM  DISS
  // RESET STATE  - 0   0  0  0    0     0    0     0
  MSCR = Sampling_250KHz +      // D and Z counter sampling clk = 250KHz
         Sampling_ON + 
         Float_unused_MCIx;     // Unused MCIx pins are HiZ
  
  // Initialize registers in page 0
  // ********************************
  bitClr(MCFR, RPGS);	
  
  // MCRC  ===>>>  SEI  HZ  SZ  SC  SPLG  VR2  VR1  VR0
  // RESET STATE  - 0   0   0    0    0     0   0    0
  //MCRC = MCRC_low_duty_cycle;
  
  // MDFR  ===>>>  DEF[3:2:1:0]  DWF[3:2:1:0]
  // RESET STATE  -    0 0 0 0       1 1 1 1
  MDFR = D_sample(4) + 
         D_blank_20us ;
  
  // MCFR  ===>>>  RPGS  RST  CFF2  CFF1  CFF0  CFW2  CFW1  CFW0
  // RESET STATE  -  0    0     0     0     0     0     0     0
  MCFR = Current_blank_3_0us +
         OC_sample(5);
  
  // MREF  ===>>>  HST  CL  CFAV  HFE1  HFE0  HFRQ2  HFRQ1  HFRQ0
  // RESET STATE  - 0   0    0     0     0      0      0      0
  MREF = OPAMP_in_OAZ +
         CHOPPING_OFF ;
  
  /* PWM carrier initialisation
  ******************************/
  // MPCR  ===>>>  PMS  OFLU  OFLV  OFLW  CMS  PCP2  PCP1  PCP0
  // RESET STATE  - 0    0     0     0     0    0     0     0
  MPCR  = Fmtc_div_1 +
          EDGE_ALIGNED +
          bit16;
          
  MCP0L = Carrier_mag % 256;	   // this value decides both ....
  MCP0H = Carrier_mag / 256;     //    ... carrier mag and freq
  MREP  = 16 - 1;                // Tupdate = 16 * Carrier_mag * Tclk
    
  return;
}

/********************************************************
Align the rotor to a known intitial position (sector 0)
*********************************************************/
void align_rotor (void)	{
u8  count_50ms;

  MCsoft_reset();          // reset of all the peripheral registers
  MCinit_bldc();
  
  /* bootstrap charging
  ***********************/
  MCRA  = DAC;       // direct access
  MPHST = 0;         // no output "ON"
  MCRA  = MOE | DAC; // output enabled
  wait_1ms(1);
  MPHST = 0x2a;      // on-off of low side
  wait_1ms(1);
  MPHST = 0;	       // put "OFF" all power switch 
  wait_1ms(1);       // all drivers off during 1mS to avoid short ckt
  
  /* register initialisation
  ***************************/
  MPRSR = 8;           // load MPRSR with first starting value
  MZREG = 100;
  MZPRV = 100;       // initialise with its reset value
  //MCOMP = 100;       // leaving MCOMP = 0 keeps MTIM = 0 until SWA = 1
  
  /* alignment initialisation 
  *****************************/
  #if TEST
    bitClr (FLAGDR, FLAGPIN);
  #endif
  
  rampEnable = 0;
  phasestep  = 0;
  MPHST = phreg_SL[phasestep];           //output alignment configuration T1-T4
  MCRB  = volt_crbreg_SL[phasestep];     //demag type and z edge settings
  
  duty  = start_duty;      //starting dutycycle
  MCPU  = (u16)duty << 4;
  MCRA  = MOE | CKE;       //enable output; start clk;  sensorless; volt mode
                           //DAC=0;--> MPHST xfer to MCOx on C
                           //MZPRV for computation
  
  /* aligment current ramp up
  *****************************/
  for (count_50ms = 30;   count_50ms-- ; ) {
    wait_1ms(50);
    duty += duty_ramp;
    MCPU  = (u16)duty << 3;
  }
  wait_1ms(200);
  
  return;
}

/*****************************************************************************/
void enable_MC_interrupts(void) {
  #if TEST
    bitSet (FLAGDR, FLAGPIN);
  #endif
  
  MWGHT = 100;                  // reset value of MWGHT
  MCRC  = MCRC_low_duty_cycle;  // first Z detection always by s/w
  if ( ++phasestep > 5 )
    phasestep = 0;
  MCRB  = volt_crbreg_SL[phasestep];   
  MPHST = phreg_SL[phasestep];
  MCRA |= SWA;                // enable autoswitched mode
  MIMR  = CIM;    // IT on C
  rampEnable = 1;
  return;
}

/*****************************************************************************/
/****************************************************************************
           Commutation & Demagnetisation Interrupt Routine
                    Event C or Event D interrupt
                    *** Exec time for C
****************************************************************************/
@interrupt  
void  mtcCD_ISR(void)  {
  
  // C EVENT
  if (bitTest_TRUE(MISR, CI) )   {  
    #if TEST
      bitToggle (FLAGDR, FLAGPIN);
    #endif
    
    starting_failed = 1000;   // activate and restart watchdog variable
    
    // preload for next commutation Cn+1    
    // get the preload values for next step C event (Cn+2)
    // MCRB ==>> -0-  CPB  HDM  SDM  OCV  OS2  OS1  OS0
    if ( ++phasestep > 5 )
      phasestep = 0;
    MCRB  = volt_crbreg_SL[phasestep];   
    MPHST = phreg_SL[phasestep];
    MDREG = ((u16)(MZPRV * (70*0xff/100))) >> 8;
    
    MISR = 0xff - CI;           //reset flag of C IT
  }
  
  // D EVENT
  if (bitTest_TRUE(MISR, DI))   {  
    MISR = 0xff - DI;   // reset flag of D IT
  }
  
  return;
}

/***********************************************************************
            Ratio & Zero crossing  Interrupt Routine
                  Event U or Event Z interrupt
***********************************************************************/
@interrupt  
void  mtcRZ_ISR(void)  {

  // BEMF Z EVENT
  if (bitTest_TRUE(MISR, ZI)) {
    MISR = 0xff - ZI;            // clear ZI IT
  }
  
  // Ratio Plus event
  if (bitTest_TRUE (MISR, RPI) )
    MISR = 0xff - RPI;  //ratio ITs not used, clear RPI, RMI IT
  
  // Ratio Minus event
  if (bitTest_TRUE (MISR, RMI) )
    MISR = 0xff - RMI;  //ratio ITs not used, clear RPI, RMI IT
  
  return;
}
/***********************************************************************
          Motor control - Event U or crnt loop or samp out interrupt
************************************************************************/
@interrupt @nosvf  
void mtcU_CL_SO_ISR(void)  {     

  // U EVENT
  if (bitTest_TRUE(MISR, PUI) )	
    MISR = 0xff - PUI;            //reset IT flag
  
  // CL EVENT
  if (bitTest_TRUE(MISR, CLI) )	
    MISR = 0xff - CLI;            //reset IT flag
  
  return;
}

/***********************************************************************
         Motor control Emergency Stop /
         Speed Error /
         FLASH start programming Interrupt
************************************************************************/
@interrupt  
void tli_ISR(void)  {      

  // TLI interrupt
  if (bitTest_TRUE(MISR, EI) )  {  //if E is active EMERGENCY STOP
    MCsoft_reset();        // reset MC
    MISR = 0xff - EI;      //reset IT flag
    while (1)
      ;   //infinite loop till reset
  }    
  return;
}

/************************************************************************/
/* Main Clock Control/ Real time clock interrupt *
            ***   Exec time is --  ~25uS   ***
/************************************************************************/
@interrupt @nosvf  
void mccRtc_ISR(void)	{
static u8   timebaseCount, key1count;
static u16  ref;
static s16  temp;

  if (starting_failed)
    --starting_failed;   // decrement watchdog variable
  
  /* ON/OFF switch - 100ms debounce time
  ***************************************/
  if (!(ON_OFF_PORT & ON_OFF_PIN))   {
    if (key1count != DEBOUNCE_TIME_COUNT)
      if (++key1count == DEBOUNCE_TIME_COUNT)
        runCmd ^= 0xff;
  } else  
    key1count = 0;
  /**/
  
  #define MAG_ERR_LIMIT  2
  
  /* Duty cycle softstart
  ************************/
  if (rampEnable) {
    getADC_10bit (ref, REF_CHANNEL); 
    if (ref > 90*Carrier_mag/100)
      ref = 90*Carrier_mag/100;
      
    if (++ timebaseCount == 5 ) {      // 20
      timebaseCount = 0;
      
    temp = (ref - duty);
    if (temp > MAG_ERR_LIMIT)
      duty += MAG_ERR_LIMIT;
    else if (temp < -MAG_ERR_LIMIT)
      duty -= MAG_ERR_LIMIT;
    else 
      duty = ref;
      
    MCPU  = (u16)duty << 3;
    }
  }
   
  MCCSR;	/* read MCCSR to clear interrupt  */
  return;
}          

/************************************************************************
  Code ends here
************************************************************************/
