/*****************************************************************************
COPYRIGHT 2005 STMicroelectronics
Source File Name : PMBus.c 
Group            : MPA Systems Lab.
Author           : Telecom Team
Date First Issued: 01/09/2006           
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

    THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. STMicroelectronics 
    SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL 
    DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-
********************************Documentation**********************************
General Purpose - Contains source code for following PMBus commands.
							- OPERATION
							- ON_OFF_CONFIG
							- CLEAR_FAULTS
							- VOUT_OV_FAULT_LIMIT
							- VOUT_OV_FAULT_RESPONSE
							- VOUT_OV_WARN_LIMIT
							- VOUT_UV_WARN_LIMIT
							- VOUT_UV_FAULT_LIMIT
							- VOUT_UV_FAULT_RESPONSE 
							- IOUT_OC_FAULT_LIMIT
							- IOUT_OC_FAULT_RESPONSE
							- IOUT_OC_WARN_LIMIT
							- OT_FAULT_RESPONSE
							- OT_FAULT_LIMIT
							- OT_WARN_LIMIT
							- VIN_OV_FAULT_LIMIT
							- VIN_OV_FAULT_RESPONSE 
							- VIN_UV_FAULT_LIMIT
							- VIN_UV_FAULT_RESPONSE
							- VIN_ON
							- VIN_OFF
							- VOUT_MODE
							- VOUT_COMMAND
							- VOUT_TRIM
							- VOUT_CAL
							- VOUT_MAX
							- VOUT_MARGIN_HIGH
							- VOUT_MARGIN_LOW
							- VOUT_TRANSITION_RATE
							- VOLTAGE_SCALE_LOOP
							- VOUT_DROOP
							- TON_DELAY
							- TON_RISE
							- TOFF_DELAY
							- TOFF_FALL
							- STATUS_BYTE
							- STATUS_WORD
							- STATUS_VOUT
							- STATUS_IOUT
							- STATUS_TEMPERATURE
							- STATUS_INPUT
							- STATUS_CML
							- STATUS_OTHER
							- STATUS_MFR_SPECIFIC
							- READ_VOUT
							- READ_VIN
							- READ_DUTY_CYCLE
							- READ_FREQUENCY
							- READ_TEMPERATURE
							- COEFFICIENTS
							- STORE_USER_ALL
							- RESTORE_USER_ALL
							- MAX_DUTY
							- FREQUENCY_SWITCH
							- IOUT_CAL_OFFSET
							- PMBUS_REVISION
							- MFR_ID
							- MFR_MODEL
							- MFR_REVISION
							- MFR_LOCATION
							- MFR_DATE
							- MFR_SERIAL
							- MFR_SPECIFIC_00
							- MFR_SPECIFIC_01
				This file also consists of some additional functions that are used in
				the above commands. 

********************************Revision History*******************************
_______________________________________________________________________________
Date :01/09/2006                  Release:1.0 
******************************************************************************/
#include "PMBus.h"
#include "ST7_hr.h"

#pragma space extern [] @tiny

@tiny volatile unsigned char SMB_Byte_Count ;   /* Variable to show data length */
extern volatile unsigned char SMB_Byte_Count1 ;   /* Variable to show data length */
@tiny volatile unsigned char SMBus_Mode ;/* Variable to show SMBus protocol */
@tiny volatile unsigned char Command_Code;/* Variable to store command code */
extern volatile unsigned int SMB_Ttimeout;                /* Time count variable */
extern volatile unsigned int SMB_TimeCount;                /* Time count variable */

extern volatile unsigned char SMB_Err_Status ;                    /* Error status */
extern volatile unsigned char PMBus_SlaveAdd ;
extern volatile unsigned char * SMB_TxAdd ;         /* Address of Tx buffer */
volatile unsigned char SMBus_PEC ;
volatile unsigned char SMB_PEC_Status ;                     /* CRC-8 result */
extern volatile unsigned char SMBm_PEC_Data ; 
extern unsigned char SMBm_Add ;                          /* Global address variable */
extern const unsigned char ComCode_DataLength_Buff [37];
extern const unsigned char Msg_Offset_Buff [39];
volatile unsigned char Data_Buff [32] ;       /* Buffer to store data */

/*-----------------------------------------------------------------------------
ROUTINE NAME : PMBus_Init
INPUT        : Adddress selection, Slave Adddresses, Commmunicaion Speed
OUTPUT       : None.
DESCRIPTION  : Initialization of all I2C registers. Acknowledge and interrupt 
               are enabled. Slave address of SMBus selected.
COMMENTS     : Must be called before starting any SMBus operation
-----------------------------------------------------------------------------*/
void PMBus_Init (void)
{
	#asm                             /* Loads reset value in all I2C registers */
	clr		_I2CCR
	#endasm                           /* Configuration of I2C control register */
     
	I2COAR1 = SLAVEADD ;             /* Configures I2C slave address */
	I2COAR2 = 0x40 ;

	Enable_I2C ();

	I2CCCR = I2C_SLOW_SPEED ;  /* Configures communication speed at 100kHz*/
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : PMBus_Init
INPUT        : Adddress selection, Slave Adddresses, Commmunicaion Speed
OUTPUT       : None.
DESCRIPTION  : Initialization of all I2C registers. Acknowledge and interrupt 
               are enabled. Slave address of SMBus selected.
COMMENTS     : Must be called before starting any SMBus operation
-----------------------------------------------------------------------------*/
void Enable_I2C (void)
{
	#asm                             /* Loads reset value in all I2C registers */
	bset	_I2CCR, #5
	#endasm                           /* Configuration of I2C control register */
  I2CCR = (PE | ACK) ;                             /* Acknowledge enabled */
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : PMBus_CommandRead
INPUT        : Command Code
OUTPUT       : PMBus communication status 
DESCRIPTION  : Reads data from PMBus module using PMBus commands. PMBus 
              commands use be using SMBus protocols.
COMMENTS     : None
-----------------------------------------------------------------------------*/
PMB_ErrCode_t PMBus_CommandRead (unsigned char * Buff)
{
	PMB_ErrCode_t temp ;
	
	Enable_I2C ();
	PMBus_DataLengthCalc ();
	SMBus_Mode |= (unsigned char) SMB_READ ;   
	
	temp = PMBus_DataComm (Buff); /* Calculates Data length and SMBus mode */
		
	return (temp); 
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : PMBus_CommandWrite
INPUT        : Command Code
OUTPUT       : PMBus communication status 
DESCRIPTION  : Writes data from PC using PMBus commands. PMBus commands use 
              SMBus protocols.
COMMENTS     : None
-----------------------------------------------------------------------------*/
PMB_ErrCode_t PMBus_CommandWrite (unsigned char * Buff)
{
	PMB_ErrCode_t temp ;
	
	Enable_I2C ();
	
	if (SMBus_Mode != SMB_GC)
	{
		SMBus_Mode = SMB_NO_PROTOCOL ;                   /* SMBus_Mode initialised */
	}
	PMBus_DataLengthCalc ();
	
	SMBus_Mode &= (unsigned char) ~SMB_READ ; 
	
	temp = PMBus_DataComm (Buff);         /* Calculates Data length and SMBus mode */	

	return (temp); 
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : PMBus_DataLengthCalc
INPUT        : None
OUTPUT       : None
DESCRIPTION  : Calculates data length of PMBus communication and SMBus bus 
						protocol depending on the command code. 
COMMENTS     : None
-----------------------------------------------------------------------------*/
PMB_ErrCode_t PMBus_DataComm (unsigned char * Buff)
{
	PMB_ErrCode_t temp ;
	SMB_Ttimeout = 0 ;                                /* Initialise Ttimeout */
	SMB_TimeCount = 0 ;
	SMB_Err_Status = 0 ;
	SMB_PEC_Status = 0 ; 
	if (SMBus_Mode & 0x08)
	{
		if ((SMBus_Mode & 0x02) && (SMBus_Mode & 0x01))
		{                            /* Check for Block mode to communicae 6 bytes */
			if (SMBus_Mode & SMB_READ)                                 /* Check for master mode */
			{
				temp = SMBm_ReadBlock (Buff);
			}
			else
			{
				temp = SMBm_WriteBlock (Buff, SMB_Byte_Count);
			}
		}
		else if (SMBus_Mode & 0x01)
		{                                       /* Word mode to communicae 2 bytes */
			if (SMBus_Mode & SMB_READ)                                 /* Check for master mode */
			{
				temp = SMBm_ReadWord (Buff);
			}
			else
			{
				temp = SMBm_WriteWord (Buff);
			}
		}	
		else
		{                              /* Check for Byte mode to communicae 1 byte */
			if (SMBus_Mode & SMB_READ)                                 /* Check for master mode */
			{
				temp = SMBm_ReadByte (Buff);
			}
			else
			{
				temp = SMBm_WriteByte ((unsigned char)*Buff);
			}
		}
	}
	else 
	{          /* Check for Byte mode to communicae 1 byte without command code*/
		temp = SMBm_SendByte (Command_Code);
	}
	SMB_Byte_Count = 0 ;
	return (temp); 
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : PMBus_DataLengthCalc
INPUT        : None
OUTPUT       : None
DESCRIPTION  : Calculates data length of PMBus communication and SMBus bus 
						protocol depending on the command code. 
COMMENTS     : None
-----------------------------------------------------------------------------*/
void PMBus_DataLengthCalc (void)
{
	unsigned char temp; 
	
	for (temp = 0; temp <39; temp ++)
	{
		if (Command_Code == Msg_Offset_Buff [temp])
		{
			SMB_Byte_Count = ComCode_DataLength_Buff [temp];
		}
	}
	
	if ((Command_Code == 0x03) || (Command_Code == 0x12)|| (Command_Code == 0x15)||(Command_Code == 0x16))
	{          /* Check for Byte mode to communicae 1 byte without command code*/
		SMB_Byte_Count = 1;               	/* Number of bytes updated in Byte_Count */
		SMBus_Mode |=  (unsigned char) SMB_BYTE_NC ; /* SMBus variable updated for Send byte mode*/
	}
	else if (SMB_Byte_Count == 1)
	{
		SMBus_Mode |= (unsigned char) (SMB_BYTE) ;      
	}
	else if (SMB_Byte_Count >= 6)
	{
		SMBus_Mode |= (unsigned char) (SMB_BLOCK)  ;    
		if ((Command_Code == 0x9B)||(Command_Code == 0x9C))
		{                            /* Check for Block mode to communicae 6 bytes */
			SMB_Byte_Count = 2;               	/* Number of bytes updated in Byte_Count */
		}
	}
	else if (SMB_Byte_Count == 2)
	{
		SMBus_Mode |= SMB_WORD ;          /* SMBus variable updated for word mode */
	}
	SMB_Byte_Count1 = SMB_Byte_Count ;
}


/*-----------------------------------------------------------------------------
ROUTINE NAME    : I2C_ISR
INPUT           : None
OUTPUT          : None
DESCRIPTION     : This function manages I2C communication when any I2C event 
								occurs.
COMMENTS        : None
-----------------------------------------------------------------------------*/
#ifdef _COSMIC_                                  /* Test for Cosmic Compiler */
@interrupt                                      /* Cosmic interrupt handling */
#else
#error"Unsupported Compiler!"                 /* Compiler Defines not found! */
#endif  

void  I2C_ISR (void)
{               
	SMB_Byte_Count -- ;
	
	SMB_Ttimeout = (SMB_Ttimeout + SMB_TimeCount);
	SMB_TimeCount = 0 ;        

	SMB_Err_Status |= I2CSR2 ;   
	
	if (SMB_Err_Status & 0x10)
	{
		SMB_Err_Status |= I2CSR2 ;   
		_asm ("BRES	_I2CCR,#0");               /* Interrupt disabled */
		I2CSR1 ;
		if (!(SMBus_Mode & SMB_GC))
		{
			I2CCR |= STOP ;
		}
	}
	else if (!(I2CSR2))
	{
	if (I2CSR1 & SB)
	{
		I2CDR = SMBm_Add ;                   /* Slave address transmission */
		if (SMBus_PEC){ 
			SMBm_PEC_Data = SMBm_Add ;   
			SMB_CRC8 () ;                                               
		}
	}                                                              
	else if ((I2CSR1 & BTF) && (!(SMB_Err_Status & 0x10)))
	{
		if (I2CSR1 & TRA)
		{
			if (SMBus_Mode & 0x08)
			{
				I2CDR = Command_Code ;              /* command code transmission */
				if (SMBus_PEC){ 
					SMBm_PEC_Data = Command_Code;   
					SMB_CRC8 () ;                                               
				}          
								/* Indication that command code transmitted */
				SMBus_Mode &= 0xF7 ;    
			}                   
			else if (SMBus_Mode & SMB_READ)
			{                                                       
				SMBus_Mode &= (unsigned char) ~(SMB_READ) ; 
				SMBm_Add = (unsigned char)(SMBm_Add | 0x01) ;
				I2CCR |= START ;        /* Generates repeated start condition */
			}   
			else if (SMBus_Mode & 0x02)
			{                                  /* Number of bytes to be transmitted */
				I2CDR = (unsigned char) (SMB_Byte_Count - 1); 
				if (SMBus_PEC){ 
					SMBm_PEC_Data = (unsigned char)(SMB_Byte_Count-1);   
					SMB_CRC8 () ;                                               
				}  
				SMBus_Mode &=  0xFD ; 
			}                                                   
			else if ( (!(SMBus_Mode & 0x08)) && (SMB_Byte_Count))
			{
				I2CDR = *(SMB_TxAdd);                   /* data transmission */
				if (SMBus_PEC){ 
					SMBm_PEC_Data = *SMB_TxAdd;   
					SMB_CRC8 () ;                                               
				}
				SMB_TxAdd ++ ;
			}                                                       
			else if (!(SMB_Byte_Count))
			{   
				if (SMBus_PEC) { 
					I2CDR = SMB_PEC_Status ; 
					SMB_PEC_Status = 0 ; 
				}                                                          
				I2CCR &= (unsigned char) ~ITE ;         /* Interrupt disabled */
				if (!(SMBus_Mode & SMB_GC))
				{
					I2CCR |= STOP ;                /* STOP condition generation */
				}
			}                                                       
		}
		else 
		{
			if (SMBus_Mode & 0x02)
			{   
				SMB_Byte_Count = I2CDR;             /* Number of bytes to be received*/
				if (!(SMB_Byte_Count))
				{
					I2CCR &= (unsigned char) ~ITE ;         /* Interrupt disabled */
					/* Peripheral disabled to release lines */
					I2CCR &= (unsigned char) ~PE ; 
				}
				else if (SMBus_PEC){ 
					SMBm_PEC_Data = SMB_Byte_Count;   
					SMB_Byte_Count ++ ; 
					SMB_CRC8 () ;                                               
				}
				else if (SMB_Byte_Count == 1){ 
					I2CCR &= (unsigned char) ~ACK ;         /* Acknowledge disabled */
				}
				SMBus_Mode &=  0xFD ; 
			}                                                    
			else if (((SMB_Byte_Count == 1)))
			{
				I2CCR &= (unsigned char) ~ACK;/* Acknowledge disabled before last byte */
				*(SMB_TxAdd) = I2CDR;                               /* Data reception */
				if (SMBus_PEC){ 
					SMBm_PEC_Data = *SMB_TxAdd;   
					SMB_CRC8 () ;                                               
				}
				SMB_TxAdd ++ ;
			}                                              
			else if ((SMB_Byte_Count))
			{
				*(SMB_TxAdd) = I2CDR;                               /* Data reception */
				if (SMBus_PEC){ 
					SMBm_PEC_Data = *SMB_TxAdd;   
					SMB_CRC8 () ;                                               
				}  
				SMB_TxAdd ++ ;
			}                                                       
			else if (!(SMB_Byte_Count))
			{   
				I2CCR &= (unsigned char) ~ITE ;         /* Interrupt disabled */
				I2CCR |= STOP ;                  /* STOP condition generation */
				*(SMB_TxAdd) = I2CDR;                      /* Data reception */	
				if (SMBus_PEC){ 
				
					if ((*(SMB_TxAdd)) != SMB_PEC_Status)
					{
						SMB_Err_Status = 0xf0 ; /* Indication for PEC failure */
					}
				}  
			} 		
		}
	}         
	else 
	{
		if ((!(SMBus_Mode & 0x10))&& (!(SMBus_Mode & 0x01))&& (SMBus_PEC != PEC_ENABLE))
		{
			I2CCR &= (unsigned char) ~ACK;/* Acknowledge disabled before last byte */
		}
		else
		{
			_asm ("BSET	_I2CCR,#5");            /* Clearing sequence for EV6 */
		}
	}
	}
}


/*-----------------------------------------------------------------------------
ROUTINE NAME    : ARA_ISR
INPUT           : None
OUTPUT          : None
DESCRIPTION     : This function manages Alert Signals coming from slave devices communication when any I2C event 
COMMENTS        : None
-----------------------------------------------------------------------------*/
#ifdef _COSMIC_                                  /* Test for Cosmic Compiler */
@interrupt                                      /* Cosmic interrupt handling */
#else
#error"Unsupported Compiler!"                 /* Compiler Defines not found! */
#endif  

void  ARA_ISR (void)
{               
	if (!(PADR & 0x08))
	{
		PAOR &= 0xF7 ; /* To disable Alert interrupt so that master can process */
		SMBus_Mode = SMB_ARA ; 
	}
}

/*** (c) 2005  ST Microelectronics ****************** END OF FILE ************/