/*****************************************************************************
COPYRIGHT 2005 STMicroelectronics
Source File Name : SMBus_Master.c 
Group            : MPA Systems Lab.
Author           : Telecom Team
Date First Issued: 01/09/2006           
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

    THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. STMicroelectronics 
    SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL 
    DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-
********************************Documentation**********************************
General Purpose - Contains source code for following SMBus master bus protocols.
								1. Send Byte
								2. Read Byte
								3. Write Byte
								4. Read Word
								5. Write Word
								6. Block Read 
								7. Block Write
								8. Process Call 
								This file also consists of some additional functions to develop 
							the SMBus bus protocols.

********************************Revision History*******************************
_______________________________________________________________________________
Date :01/09/2006                  Release:1.0 
******************************************************************************/
#include "ST7_hr.h"                           /* Declaration of HW registers */
#include "SMBus_Master.h"  /* Prototype definitions of SMBus slave functions */

/*---------------------------------------------------------------------------*/
                                          /* Declaration of Global Variables */
#pragma space extern [] @tiny

extern volatile unsigned char SMB_PEC_Status ;                     /* CRC-8 result */
volatile unsigned char SMBm_PEC_Data ; 
extern volatile unsigned char PMBus_SlaveAdd ;
extern volatile unsigned char SMBus_Mode ;
volatile unsigned char * SMB_TxAdd ;         /* Address of Tx buffer */
unsigned char SMBm_Add ;                          /* Global address variable */
@tiny volatile unsigned int SMB_Ttimeout;/* Time count variable for Ttimeout */
                  /* Time count variable for Cumulative clock low extend time*/
@tiny volatile unsigned int SMB_TimeCount;                
extern volatile unsigned char SMB_Byte_Count;/* Transmitter and receiver buffer count */
@tiny volatile unsigned char SMB_Err_Status ;                    /* Error status */
extern volatile unsigned char SMBus_PEC ;

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMB_TxRx
INPUT        : None
OUTPUT       : Byte transmission status.
DESCRIPTION  : Sends a byte.
COMMENTS     : None.
-----------------------------------------------------------------------------*/
PMB_ErrCode_t SMB_TxRx (void)
{
  I2CCR |= (unsigned char) START ;              /* Generates START condition */	
	I2CCR |= (unsigned char) ITE ;                   /* Enable I2C Interrupt */
                                 /* Waiting for ReadByte process to complete */
	while (((SMB_Byte_Count)&& (!(SMB_ClkLow_Delay ())))&&(SMB_Ttimeout < 1775) &&(!(SMB_Err_Status & 0x10)) );
	if ((!(SMB_Err_Status))&&((SMB_Byte_Count)|| (SMB_Ttimeout > 1775)))
	{
		I2CCR &= (unsigned char)(~PE) ;                   /* Disables peripheral */
		SMB_Err_Status |= 0x18 ;                      /* Indication for Timeout */
	}
	else if (SMB_Err_Status & 0x10)
	{
		SMB_Err_Status |= I2CSR2 ;
		I2CCR &= (unsigned char)(~PE) ;                   /* Disables peripheral */
	}
	
	return (SMB_Err_Status) ;                     /* Returns reception status */
}


/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBm_SendByte
INPUT        : Input Parameter1 - Slave_Add
               Input Parameter2 - Tx_Byte
OUTPUT       : Byte transmission status.
DESCRIPTION  : Sends a byte.
COMMENTS     : None.
-----------------------------------------------------------------------------*/
PMB_ErrCode_t SMBm_SendByte (unsigned char Tx_Byte)
{
                          /* All user parameters copied to a global variable */
	SMBm_Add = PMBus_SlaveAdd ;
	SMB_TxAdd = &Tx_Byte ;          
                                      /* Number of times Interrupt processed */
	SMB_Byte_Count = (unsigned char)(SMB_Byte_Count + 3) ;  
	
	SMB_Err_Status = (@tiny unsigned char) SMB_TxRx ();
	
	if (SMB_Err_Status == 0xE0)
	{
		SMB_Err_Status = 0 ;
	}
	return (SMB_Err_Status) ;                     /* Returns reception status */
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBm_ReceiveByte
INPUT        : Input Parameter1 - Slave_Add
               Input Parameter3 - *Rx_Buffer
OUTPUT       : Recepti on status
DESCRIPTION  : Sends ARA address and receives device address which alerted the 
               master.
COMMENTS     : None.
-----------------------------------------------------------------------------*/
PMB_ErrCode_t SMBm_ReceiveByte (unsigned char Slave_Add, unsigned char * Rx_Buffer)
{
	SMB_Err_Status = 0 ;
                          /* All user parameters copied to a global variable */
	SMBm_Add = (unsigned char)(Slave_Add | 0x01) ;
	SMB_TxAdd = Rx_Buffer ; 

	SMB_Byte_Count = 3 ; /* Number of times Interrupts processed */
	if (SMBus_PEC == PEC_ENABLE){
  SMB_Byte_Count ++ ;
	}
	
	SMB_PEC_Status = 0 ; 
	
	SMB_TimeCount = 0 ;
	SMB_Ttimeout = 0 ;
	
	SMB_Err_Status = (@tiny unsigned char) SMB_TxRx ();
	
	SMB_Byte_Count = 0 ;
	return (SMB_Err_Status) ;                     /* Returns reception status */
}


/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBm_WriteByte
INPUT        : Input Parameter1 - Slave_Add
               Input Parameter2 - Com_Code
               Input Parameter3 - Tx_Byte
OUTPUT       : Byte transmission status.
DESCRIPTION  : Sends command code and a byte.
COMMENTS     : None.
-----------------------------------------------------------------------------*/
PMB_ErrCode_t SMBm_WriteByte ( unsigned char Tx_Byte)
{
                          /* All user parameters copied to a global variable */
	SMBm_Add = PMBus_SlaveAdd ;
	
	SMB_TxAdd = &Tx_Byte ;
		                                  /* Number of times Interrupt processed */
	SMB_Byte_Count = (unsigned char)(SMB_Byte_Count + 4) ;  
	
	SMB_Err_Status = (@tiny unsigned char) SMB_TxRx ();
	
	if (SMB_Err_Status == 0xE0)
	{
		SMB_Err_Status = 0 ;
	}
	
	return (SMB_Err_Status) ;                     /* Returns reception status */
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBm_ReadByte
INPUT        : Input Parameter1 - Slave_Add
               Input Parameter2 - Com_Code
               Input Parameter3 - *Rx_Buffer
OUTPUT       : Word reception status.
DESCRIPTION  : Sends command code and receives a byte.
COMMENTS     : None.
-----------------------------------------------------------------------------*/
PMB_ErrCode_t SMBm_ReadByte (unsigned char * Rx_Buffer)
{
                          /* All user parameters copied to a global variable */
	SMBm_Add = PMBus_SlaveAdd ;

	SMB_TxAdd = Rx_Buffer ; 

	SMB_Byte_Count = (unsigned char)(SMB_Byte_Count + 6) ; /* Number of times Interrupts processed */
	if (SMBus_PEC == PEC_ENABLE){
  SMB_Byte_Count ++ ;
	SMB_PEC_Status = 0 ; 
	}
	
	SMB_Err_Status = (@tiny unsigned char) SMB_TxRx ();

	return (SMB_Err_Status) ;                     /* Returns reception status */
}


/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBm_WriteWord
INPUT        : Input Parameter1 - Slave_Add
               Input Parameter2 - Com_Code
               Input Parameter3 - *Tx_Buffer
OUTPUT       : Word transmission status.
DESCRIPTION  : Sends command code and a word.
COMMENTS     : None.
-----------------------------------------------------------------------------*/
PMB_ErrCode_t SMBm_WriteWord (unsigned char * Tx_Buffer)
{
	unsigned char temp ;
                          /* All user parameters copied to a global variable */
	SMBm_Add = PMBus_SlaveAdd ;

	SMB_TxAdd = Tx_Buffer ; 

	SMB_Byte_Count = (unsigned char)(SMB_Byte_Count + 4);   /* Number of times Interrupt processed */
	
	temp = *(SMB_TxAdd) ;
	*(SMB_TxAdd) = *(SMB_TxAdd + 1) ;                
	*(SMB_TxAdd + 1) = temp;     
	
	SMB_Err_Status = (@tiny unsigned char) SMB_TxRx ();
	
	if (SMB_Err_Status == 0xE0)
	{
		SMB_Err_Status = 0 ;
	}

	return (SMB_Err_Status) ;               /* Returns reception status */
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBm_ReadWord
INPUT        : Input Parameter1 - Slave_Add
               Input Parameter2 - Com_Code
               Input Parameter3 - *Rx_Buffer
OUTPUT       : Word reception status.
DESCRIPTION  : Sends command code and receives a word.
COMMENTS     : None.
-----------------------------------------------------------------------------*/
PMB_ErrCode_t SMBm_ReadWord (unsigned char * Rx_Buffer) 
{
	unsigned char temp ;
                          /* All user parameters copied to a global variable */
	SMBm_Add = PMBus_SlaveAdd ;

	SMB_TxAdd = Rx_Buffer ; 

	SMB_Byte_Count = (unsigned char)(SMB_Byte_Count + 6) ; /* Number of times Interrupts processed */

	SMB_PEC_Status = 0 ; 
  if (SMBus_PEC == PEC_ENABLE){
  SMB_Byte_Count ++ ;
	}

	SMB_Err_Status = (@tiny unsigned char) SMB_TxRx ();
	
	if (!(SMB_Err_Status))
	{												/* Swapping LSB and MSB */
		temp = *(Rx_Buffer) ;
		*(Rx_Buffer) = *(Rx_Buffer + 1) ;                
		*(Rx_Buffer + 1) = temp;     
	}

	return (SMB_Err_Status) ;                     /* Returns reception status */
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBm_WriteBlock
INPUT        : Input Parameter1 - Slave_Add
               Input Parameter2 - Com_Code
               Input Parameter3 - *Tx_Buffer
               Input Parameter4 -Tx_ByteCount-Number of bytes to be transmitted
OUTPUT       : Block reception status.
DESCRIPTION  : Sends command code and transmits a block of data. 
COMMENTS     : None.
-----------------------------------------------------------------------------*/
PMB_ErrCode_t SMBm_WriteBlock(unsigned char * Tx_Buffer, unsigned char Tx_ByteCount)
{
                          /* All user parameters copied to a global variable */
	SMBm_Add = PMBus_SlaveAdd ;

	SMB_TxAdd = Tx_Buffer ; 

	SMB_Byte_Count = (unsigned char)(SMB_Byte_Count + 5) ; /* Number of times Interrupts processed */
	
	SMB_Err_Status = (@tiny unsigned char) SMB_TxRx ();
	
	if (SMB_Err_Status == 0xE0)
	{
		SMB_Err_Status = 0 ;
	}

	return (SMB_Err_Status) ;                 /* Returns reception status */
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBm_ReadBlock
INPUT        : Input Parameter1 - Slave_Add
               Input Parameter2 - Com_Code
               Input Parameter3 - *Rx_Buffer
               Input Parameter4 - *Rx_ByteCount - Address that stores Bytecount
OUTPUT       : Block reception status.
DESCRIPTION  : Sends command code and receives a block of data. 
COMMENTS     : None.
-----------------------------------------------------------------------------*/
PMB_ErrCode_t SMBm_ReadBlock (unsigned char * Rx_Buffer)
{
                          /* All user parameters copied to a global variable */
	SMBm_Add = PMBus_SlaveAdd ;

	SMB_TxAdd = Rx_Buffer ; 
	
	SMB_PEC_Status = 0 ; 

	SMB_Byte_Count = 7 ;            /* Number of times Interrupts processed */

	SMB_Err_Status = (@tiny unsigned char) SMB_TxRx ();
	
	return (SMB_Err_Status) ;                 /* Returns reception status */
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMB_ClkLow_Delay
INPUT        : None.
OUTPUT       : Delay status.
DESCRIPTION  : Returns TRUE if 10ms time-out has not elapsed and FALSE if 
               10ms time-out has elapsed.
COMMENTS     : None
-----------------------------------------------------------------------------*/
BOOL SMB_ClkLow_Delay (void)
{
  SMB_TimeCount++ ;
  if (SMB_TimeCount <= 705)         /* Fcpu/ 2160 for time-out */
    return (FALSE)  ;                     /* 10ms time-out not elapsed */
  else
    return (TRUE)  ;                          /* 10ms time-out elapsed */
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMB_CRC8
INPUT        : None
OUTPUT       : None
DESCRIPTION  : Calculates PEC using CRC-8 polynomial
COMMENTS     : This function is used in all communication functions to 
               calculate PEC.  
-----------------------------------------------------------------------------*/
void SMB_CRC8 (void)
{
	unsigned char temp;     
  unsigned int CRC_Byte ;       
	
	temp = (unsigned char)(SMBm_PEC_Data ^ SMB_PEC_Status) ;   

	CRC_Byte = (unsigned int)(temp << 8)  ; 
                  /* u8 data -> u16 data -> left shift 8 times -> Add 8 '0's */
	CRC_Byte &= CRC_U16 ;    

	for (temp = CRC_COUNT; temp != 0; temp --)  
	{
		if (CRC_Byte & CRC_MSB)              /* Checking for bit '1' as MSB */
		{
			CRC_Byte ^= CRC_POLY ;                 /* XOR with 1 0000 0111 */
		}                        
		CRC_Byte = CRC_Byte << 1 ;              /* Checks for next '1' bit */
	}
	SMB_PEC_Status =((unsigned char)(CRC_Byte >>8));/* u16 data -> u8 data */
}  

 
/*** (c) 2005  ST Microelectronics ****************** END OF FILE ************/