/******************************************************************************
COPYRIGHT 2005 STMicroelectronics
Source File Name : SMBus_Slave.c 
Group            : MPA Systems Lab.
Author           : Telecom Team
Date First Issued: 01/09/2006   
-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-

    THE SOFTWARE INCLUDED IN THIS FILE IS FOR GUIDANCE ONLY. STMicroelectronics 
    SHALL NOT BE HELD LIABLE FOR ANY DIRECT, INDIRECT OR CONSEQUENTIAL 
    DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM USE OF THIS SOFTWARE.

-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-
********************************Documentation**********************************
General Purpose - Contains source code for following SMBus slave bus protocols.
								1. Send Byte
								2. Read Byte
								3. Write Byte
								4. Read Word
								5. Write Word
								6. Block Read 
								7. Block Write
								8. Process Call
							This file also consists of some additional functions to develop 
							the SMBus bus protocols.
********************************Revision History*******************************
_______________________________________________________________________________
Date :01/09/2006                  Release:1.0 
******************************************************************************/

#include "ST7_hr.h"                       /* Declaration of I2C HW registers */
#include "SMBus_Slave.h"   /* Prototype definitions of SMBus slave functions */
#include "SMBus_Master.h"  /* Prototype definitions of SMBus slave functions */

/*---------------------------------------------------------------------------*/
                                          /* Declaration of Global Variables */
#pragma space extern [] @tiny
#ifdef SMB_PEC
@tiny volatile unsigned char SMB_PEC_Status ;                     /* CRC-8 result */
static volatile unsigned char SMBs_PEC_Count;       /* PEC calculation count */
#endif

extern unsigned char * SMB_TxAdd ;         /* Address of Tx buffer */
extern volatile unsigned char SMBm_PEC_Data ; 
static volatile unsigned char SMBs_BusReq ;           /* Indication variable */
extern volatile unsigned char SMB_Err_Status ;            /* Error variable */
volatile unsigned char * SMB_RxAdd ;         /* Address of Rx buffer */
extern volatile unsigned char SMB_Byte_Count ;       /* Byte count variable */
extern volatile unsigned int SMB_TimeCount;                /* Time count variable */
static volatile unsigned char SMBs_PC_Count ;    /* Block Process call count */

/*---------------------------------------------------------------------------*/
/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBs_Init
INPUT        : Add_Param, Slave_Add
OUTPUT       : None.
DESCRIPTION  : Initialization of all I2C registers. Acknowledge and interrupt 
               are enabled. Slave address of SMBus selected.
COMMENTS     : Must be called before starting any SMBus operation
-----------------------------------------------------------------------------*/
void SMBs_Init (SMBs_Address_t Add_Param,unsigned char Slave_Add1,
                unsigned char Slave_Add2)
{
		 #asm                             /* Loads reset value in all I2C registers */
		clr		_I2CCR
		clr		_I2CCCR
		clr		_I2COAR1
		clr		_I2COAR2
		#endasm                           /* Configuration of I2C control register */
     
     if (Add_Param == SMBs_MISC)
     {
          I2COAR1 = Slave_Add1 ;             /* Configures I2C slave address */
          I2COAR2 = Slave_Add2 ;
     } 
     else
          I2COAR1 = (unsigned char) Add_Param ;

		#asm                             /* Loads reset value in all I2C registers */
		bset	_I2CCR, #5
		#endasm                           /* Configuration of I2C control register */
    I2CCR = (PE | ACK) ;                             /* Acknowledge enabled */
     if (Add_Param == SMBs_GCAL)
          I2CCR |= 0x10 ;                                /* Setting GCAL bit */
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBs_SendByte
INPUT        : Address from where data to be transmitted
OUTPUT       : Transmission status.
DESCRIPTION  : Master sends address with request for one byte. After address
               received from Master, transmits bytes of data.
COMMENTS     : 
-----------------------------------------------------------------------------*/
SMBs_ErrCode_t SMBs_SendByte (unsigned char *Tx_Buffer)
{
     SMB_TxAdd = Tx_Buffer ;
     SMBs_BusReq = 0x03 ;                         /* Indication for SendByte */
     SMB_Byte_Count = 2 ;
     #ifdef SMB_PEC
     SMB_PEC_Status = 0 ; 
     SMBs_PEC_Count = 0 ;         
     #endif
     SMB_Err_Status = 0 ;
     I2CCR |= ITE ;                                     /* Interrupt enabled */
     while ((SMB_Byte_Count == 2)&& (!(SMB_Err_Status)));                 
                                              /* Waits for address reception */
     if (SMB_Err_Status & ERR)              /* Checking for error condition */
		 {
        #asm                             /* Clear error flags if set */
				tnz		_I2CSR2
				#endasm
		}
     SMBs_Tx () ;                                            /* Transmission */
		 
     return (SMB_Err_Status) ;               /* Returns Transmission status */
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBs_Rx_CmdCode
INPUT        : Parameter1: Address where received command code to be stored
               Parameter2: Address where Mode of communication to be stored.
               if the value is FFh, then slave will enter into transmitter mode
               or if the value is 00h, then slave will enter into receiver mode
OUTPUT       : Reception status
DESCRIPTION  : Receives slave address and command code. Stores command code and 
               mode of communication (transmission/reception) in the user 
               defined address.
COMMENTS     : The R/W bit(LSB) and the command code can be used to instruct to 
               either store the forthcoming data to a register specified by the 
               SMBus command code or output the data from the specified 
               register.
-----------------------------------------------------------------------------*/
SMBs_ErrCode_t SMBs_Rx_CmdCode(unsigned char* Rx_Buffer,
                               unsigned char * Mode_TxRx)
{
     SMB_RxAdd = Rx_Buffer ; 
     SMB_Err_Status = 0 ;     
     #ifdef SMB_PEC
     SMB_PEC_Status = 0 ;     
     SMBs_PEC_Count = 0 ;
     #endif
     SMBs_BusReq = 0x02 ;           /* Indication for command code reception */
     SMB_Byte_Count = 3 ;
     I2CCR |= ITE ;                                     /* Interrupt enabled */
     while ((SMB_Byte_Count == 3)&& (!(SMB_Err_Status)));                 
                                              /* Waits for address reception */
     while ((SMB_Byte_Count) && (!(SMB_ClkLow_Delay ()))); 
                                                   /* Waits for command code */
     if (SMB_Byte_Count)
     {
          I2CCR &= (unsigned char)(~PE) ;             /* Disables peripheral */
          SMB_Err_Status |= 0x18 ;                /* Indication for Timeout */
     }                                    
     if ((SMB_Err_Status & ERR)&& (I2CSR2 & ERR))  /* Checking error status */
     {
          #asm                             /* Clear error flags if set */
					tnz		_I2CSR2
					#endasm
          I2CCR &= (unsigned char)(~ACK) ;          /* Indication for master */
     }
     if (SMBs_BusReq == 3)
          (* Mode_TxRx) = 0xff ;                   /* Indication for Tx mode */
     else
          (* Mode_TxRx) = 0x00 ;                   /* Indication for Rx mode */
	
     return (SMB_Err_Status) ;                  /* Returns Reception status */
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBs_WriteByte
INPUT        : Buffer address from where data to be transmitted 
OUTPUT       : Transmission status.
DESCRIPTION  : Sends data byte.
COMMENTS     : None.
-----------------------------------------------------------------------------*/
SMBs_ErrCode_t SMBs_WriteByte (unsigned char * Tx_Buffer)
{            
     SMB_TxAdd = Tx_Buffer ;     
     SMB_Err_Status = 0 ;     
     SMBs_BusReq = 0x03 ;                        /* Indication for WriteByte */
     SMB_Byte_Count = 2 ;
     I2CCR |= ITE ;                                     /* Interrupt enabled */
     SMBs_Tx () ;                                            /* Transmission */
     return (SMB_Err_Status) ;               /* Returns Transmission status */
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBs_ReadByte
INPUT        : Buffer address where received data to be stroed
OUTPUT       : Reception status and received data.
DESCRIPTION  : Receives data byte.
COMMENTS     : None.
-----------------------------------------------------------------------------*/
SMBs_ErrCode_t SMBs_ReadByte (unsigned char * Rx_Buffer)
{            
     * Rx_Buffer = SMBs_PC_Count ;
     SMB_RxAdd = (++ Rx_Buffer) ;
     SMB_Err_Status = 0 ;     
     SMBs_BusReq = 0x03 ;                         /* Indication for SendByte */
     SMB_Byte_Count = 1 ;       
     I2CCR |= ITE ;                                     /* Interrupt enabled */
     SMBs_Rx () ;                                               /* Reception */
     return (SMB_Err_Status) ;                  /* Returns Reception status */
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBs_WriteWord
INPUT        : Buffer address from where word to be transmitted
OUTPUT       : Word transmission status.
DESCRIPTION  : Sends a word.
COMMENTS     : None.
-----------------------------------------------------------------------------*/
SMBs_ErrCode_t SMBs_WriteWord (unsigned char * Tx_Buffer)
{
     unsigned char temp ;         
     temp = *Tx_Buffer ;
     *(Tx_Buffer) = *(Tx_Buffer + 1) ;               /* Swapping LSB and MSB */
     *(Tx_Buffer + 1) = temp;     
     SMB_TxAdd = Tx_Buffer ;
     SMB_Err_Status = 0 ;     
     SMBs_BusReq = 0x03 ;                        /* Indication for WriteWord */
     SMB_Byte_Count = 3 ;
     I2CCR |= ITE ;                                     /* Interrupt enabled */
     SMBs_Tx () ;                                            /* Transmission */
     return (SMB_Err_Status) ;               /* Returns Transmission status */
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBs_ReadWord
INPUT        : Buffer address where received data to be stroed
OUTPUT       : Word reception status.
DESCRIPTION  : Receives a word.
COMMENTS     : None.
-----------------------------------------------------------------------------*/
SMBs_ErrCode_t SMBs_ReadWord (unsigned char * Rx_Buffer)                          
{
     unsigned char temp ;         
     * Rx_Buffer = SMBs_PC_Count ;          
     SMB_RxAdd = (++Rx_Buffer) ;
     SMB_Err_Status = 0 ;     
     SMBs_BusReq = 0x03 ;                         /* Indication for SendByte */
     SMB_Byte_Count = 2 ;
     I2CCR |= ITE ;                                     /* Interrupt enabled */
     SMBs_Rx () ;                                               /* Reception */
     temp = *(--Rx_Buffer) ;
     *(Rx_Buffer) = *(Rx_Buffer + 1) ;               /* Swapping LSB and MSB */
     *(Rx_Buffer + 1) = temp;     
     return (SMB_Err_Status) ;                  /* Returns Reception status */
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBs_WriteBlock
INPUT        : Input Parameter1 - *Tx_Buffer
               Input Parameter2 - NbOfBytes                              
OUTPUT       : Block transmission status.
DESCRIPTION  : Sends a block of data. 
COMMENTS     : None.
-----------------------------------------------------------------------------*/
SMBs_ErrCode_t SMBs_WriteBlock(unsigned char *Tx_Buffer,
                               unsigned char NbOfBytes)
{  
     SMB_TxAdd = Tx_Buffer ;
     SMB_Err_Status = 0 ;     
     SMBs_BusReq = 0x04 ;                       /* Indication for WriteBlock */
     SMB_Byte_Count = (unsigned char)(NbOfBytes + 0x02) ;
     I2CCR |= ITE ;                                     /* Interrupt enabled */
     SMBs_Tx () ;                                            /* Transmission */
     return (SMB_Err_Status) ;               /* Returns Transmission status */
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBs_ReadBlock
INPUT        : Buffer address where received data to be stroed
OUTPUT       : Block reception status.
DESCRIPTION  : Receives a block of data. 
COMMENTS     : None.
-----------------------------------------------------------------------------*/
SMBs_ErrCode_t SMBs_ReadBlock (unsigned char * Rx_Buffer)
{
     SMB_Byte_Count = (unsigned char)(SMBs_PC_Count + 1) ;
     SMB_RxAdd = Rx_Buffer ;
     SMB_Err_Status = 0 ;     
     SMBs_BusReq = 0x03 ;                            /* Indication for Block */
     I2CCR |= ITE ;                                     /* Interrupt enabled */
     SMBs_Rx () ;                                               /* Reception */
     return (SMB_Err_Status) ;                  /* Returns Reception status */
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBs_ProcessCall_Block
INPUT        : Input Parameter1 - *Tx_Buffer
               Input Parameter2 - NbOfBytes 
               Input Parameter3 - *Rx_Buffer
OUTPUT       : Block transmission and reception status.
DESCRIPTION  : Reads block of data and writes block of data. 
COMMENTS     : None
-----------------------------------------------------------------------------*/
SMBs_ErrCode_t SMBs_ProcessCall_Block(unsigned char *Tx_Buffer,
                             unsigned char NbOfBytes,unsigned char * Rx_Buffer)
{
     SMB_Err_Status = 0 ;      
     SMB_Byte_Count = (unsigned char)(SMBs_PC_Count + 1) ;       
     SMBs_PC_Count = NbOfBytes ;
     SMB_TxAdd = Tx_Buffer ;
     SMB_RxAdd = Rx_Buffer ;
     SMBs_BusReq = 0x04 ;                            /* Indication for Block */
     SMB_Byte_Count = 1 ;    
     I2CCR |= ITE ;                                     /* Interrupt enabled */
     SMBs_Tx () ;      
     return (SMB_Err_Status) ;               /* Returns Transmission status */     
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBs_Tx
INPUT        : None
OUTPUT       : None
DESCRIPTION  : Data transmission takes place in this function
COMMENTS     : None
-----------------------------------------------------------------------------*/
void SMBs_Tx (void)
{
     unsigned char temp ;
     while ((SMB_Byte_Count) && (!(SMB_ClkLow_Delay ()))); 
                                              /* Waits for data transmission */
     if (SMB_Byte_Count)
     {
          I2CCR &= (unsigned char)(~PE) ;             /* Disables peripheral */
          SMB_Err_Status |= 0x18 ;                /* Indication for Timeout */
     }                   
    #asm                             /* Clear error flags if set */
		tnz		_I2CSR1
		#endasm
		
     I2CDR = 0xff ;
     while ((!(I2CSR2 & STOPF))&& (!(SMB_ClkLow_Delay())));/* Wait for STOP */
     if ((SMB_Err_Status & ERR)&& (I2CSR2 & ERR))  /* Checking error status */
     {
          #asm                             /* Clear error flags if set */
					tnz		_I2CSR2
					#endasm
          I2CCR &= (unsigned char)(~ACK) ;          /* Indication for master */
     }                       
     if (I2CSR1 & BUSY)                                                        
     {
          I2CCR &= (unsigned char)(~PE) ;             /* Disables peripheral */
          SMB_Err_Status |= 0x1A ;               /* Indication for bus busy */
     }
}

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBs_Rx
INPUT        : None
OUTPUT       : None
DESCRIPTION  : Data Reception takes place in this function
COMMENTS     : None
-----------------------------------------------------------------------------*/
static void SMBs_Rx (void)
{
     while ((SMB_Byte_Count) && (!(SMB_ClkLow_Delay ()))); 
                           /* Waits for address and command code from master */
     if ((SMB_Err_Status & ERR)&& (I2CSR2 & ERR))  /* Checking error status */
     {
          #asm                             /* Clear error flags if set */
					tnz		_I2CSR2
					#endasm
          I2CCR &= (unsigned char)(~ACK) ;          /* Indication for master */      
     }
     if (SMB_Byte_Count)
     {
          I2CCR &= (unsigned char)(~PE) ;             /* Disables peripheral */
          SMB_Err_Status |= 0x18 ;                /* Indication for Timeout */
     }
     else
     {                              
          while ((!(I2CSR2 & STOPF))&& (!(SMB_ClkLow_Delay ())));    
                                                  /* Wait for STOP condition */
          if (I2CSR1 & BUSY)                                                        
          {
               I2CCR &= (unsigned char)(~PE) ;        /* Disables peripheral */
               SMB_Err_Status |= 0x1A ;          /* Indication for bus busy */
          }
     }
}

#ifdef SMB_PEC
/*-----------------------------------------------------------------------------
ROUTINE NAME : SMB_CRC8
INPUT        : None
OUTPUT       : None
DESCRIPTION  : Calculates PEC using CRC-8 polynomial
COMMENTS     : This function is used in all communication functions to 
               calculate PEC.  
-----------------------------------------------------------------------------*/
void SMB_CRC8 (void)
{
	unsigned char temp;     
  unsigned int CRC_Byte ;       
	
  if (I2CSR1 & MSL)
	{
		temp = (unsigned char)(SMBm_PEC_Data ^ SMB_PEC_Status) ;   
	}
	else
	{
		if (SMBs_PEC_Count < 2)
			temp = (unsigned char)(I2CDR ^ SMB_PEC_Status) ;    
		else if (SMBs_PEC_Count == 2)                                   
			temp = (unsigned char)(SMB_PEC_Status ^ (*(SMB_TxAdd -1 ))); 
		else
			temp = (unsigned char)(SMB_PEC_Status ^ (SMB_Byte_Count-1)) ;
	}
	CRC_Byte = (unsigned int)(temp << 8)  ; 
                  /* u8 data -> u16 data -> left shift 8 times -> Add 8 '0's */
	CRC_Byte &= CRC_U16 ;    

	for (temp = CRC_COUNT; temp != 0; temp --)  
	{
		if (CRC_Byte & CRC_MSB)              /* Checking for bit '1' as MSB */
		{
			CRC_Byte ^= CRC_POLY ;                 /* XOR with 1 0000 0111 */
		}                        
		CRC_Byte = CRC_Byte << 1 ;              /* Checks for next '1' bit */
	}
	SMB_PEC_Status =((unsigned char)(CRC_Byte >>8));/* u16 data -> u8 data */
}       
#endif

/*-----------------------------------------------------------------------------
ROUTINE NAME : SMBs_IT_Function
INPUT        : None
OUTPUT       : None
DESCRIPTION  : Communication takes place in interrupt subroutine
COMMENTS     : None
-----------------------------------------------------------------------------*/
void SMBs_IT_Function (void)
{  		
	unsigned char SMBs_SR1 ;
	SMBs_SR1 = I2CSR1;                 /* Stores content of status register */

	#ifdef SMB_PEC          
	if (!(SMBs_PEC_Count))
	{
		while ((!(I2CSR1 & ADSL)) && (!(SMB_ClkLow_Delay ()))); 
		SMB_TimeCount = 0 ;
		SMB_CRC8 ();                               /* CRC-8 PEC calculation */
	}
	#endif
	
	if (SMBs_BusReq == 0x02)
	{
		#ifdef SMB_PEC
		SMBs_PEC_Count = 1 ;
		#endif         
				
		if (SMB_Byte_Count == 1)
		{
			while ((!(I2CSR1 & BTF))&& (!(SMB_ClkLow_Delay ()))); 
			SMB_TimeCount = 0 ;
			*(SMB_RxAdd) = I2CDR ;             /* Command code recpetion */
			#ifdef SMB_PEC
			SMB_CRC8 ();                        /* CRC-8 PEC calculation */
			#endif                        
		}                        
		if (!(SMB_Byte_Count))
		{
			if (!(SMBs_SR1 & BTF))
				SMBs_BusReq = 0x03 ;   /* Indication for write operation */
			else
				SMBs_PC_Count = I2CDR ;                /* data recpetion */

			#ifdef SMB_PEC
			SMB_CRC8 ();                        /* CRC-8 PEC calculation */
			#endif                             
			I2CCR &= (unsigned char)(~ITE) ;        /* Interrupt disabled */
		}                             
	}     
	else if (SMBs_BusReq == 0x03)                    /* Byte/word protocols */
	{           
		while ((!(I2CSR1 & BTF))&& (!(SMB_ClkLow_Delay ()))); 
		SMB_TimeCount = 0 ;
		if (I2CSR1 & TRA)           
		{                                           /* Checking TRA status */
			#ifdef SMB_PEC
			SMBs_PEC_Count = 0x02 ;
			#endif                
			if (SMB_Byte_Count)
			{    
				I2CDR = *(SMB_TxAdd ++); 
				#ifdef SMB_PEC
				SMB_CRC8 ();                    /* CRC-8 PEC calculation */
				#endif 
			}
			else
			{
				#ifdef SMB_PEC
				I2CDR = SMB_PEC_Status ;           /* To Send PEC status */
				#endif                                                      
				I2CCR &=(unsigned char)~(ITE);      /* Interrupt disabled */
			}                                         
		}
		else
		{               
			if (I2CSR1 & BTF)              /* Checking for Data reception */
			{                
				*(SMB_RxAdd ++) = I2CDR ;       /* Received data stored */
				if (SMB_Byte_Count)
				{    
					#ifdef SMB_PEC
					SMBs_PEC_Count = 1 ;
					SMB_CRC8 ();              /* CRC-8 PEC calculation */
					#else
					if (SMB_Byte_Count == 1)
					SMB_Byte_Count = 0 ;
					#endif
				}
				else
				{                   
					if (!(SMB_Byte_Count))
						I2CCR &=(unsigned char)~(ITE);/* Interrupt disabled */
				}
			}             
		}
	}                    
	else                                                 /* Other protocols */
	{
		if (I2CSR1 & 0x20)                          /* Checking TRA status */
		{
			if (SMBs_BusReq > 3)
			{
				#ifdef SMB_PEC
				SMBs_PEC_Count = 3 ;     
				#endif                                            
				I2CDR = (unsigned char)(SMB_Byte_Count-1) ; 
                                                  /* No of bytes transmitted */
			}                                                               
			else
			{
				#ifdef SMB_PEC
				SMBs_PEC_Count = 2 ;     
				#endif         
				I2CDR = *(SMB_TxAdd++) ;           /* Transmitting data */
			}
			SMBs_BusReq-- ;     
			#ifdef SMB_PEC
			SMB_CRC8 ();                        /* CRC-8 PEC calculation */
			#endif                        
		}
		else
		{
			if (SMBs_SR1 & BTF)
			{                              /* Checking for Data reception */
				(*(SMB_RxAdd ++) ) = I2CDR ;                           
				#ifdef SMB_PEC
				SMBs_PEC_Count = 1 ;
				SMB_CRC8 ();                   /* CRC-8 PEC calculation */
				#endif                      
			}              
			if (!(SMB_Byte_Count)) 
			{                                            
				SMBs_BusReq = 4 ;
				SMB_Byte_Count = (unsigned char)(SMBs_PC_Count + 3) ; 
                                         /* Byte count transmitted to master */
			}                             
		}
	}              
	if (SMB_Err_Status & ERR)                     /* Checking error status */
		I2CCR &= (unsigned char)(~ACK) ; 
}

/*** (c) 2005  ST Microelectronics ****************** END OF FILE ************/