/**
  ******************************************************************************
  * @file    main.c 
  * @author  IMMCC Prague FA - MH
  * @version V1.0.0
  * @date    25-May-2011
  * @brief   Main program body.
  ******************************************************************************
  * @copy
  *
  * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
  * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
  * TIME. AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY
  * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
  * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
  * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
  *
  * <h2><center>&copy; COPYRIGHT 2011 STMicroelectronics</center></h2>
  */ 


/* Includes ------------------------------------------------------------------*/
#include "main.h"
#include "dspin.h"
#include "clock.h"


/** @addtogroup Examples
  * @{
  */

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define DELAY_COUNT    0x3FFFF

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
  RCC_ClocksTypeDef RCC_ClockFreq;
  uint32_t dSPIN_rx_data = 0;
  dSPIN_RegsStruct_TypeDef dSPIN_RegsStruct;
  uint8_t temp;

/* Private function prototypes -----------------------------------------------*/
  void Delay(__IO uint32_t nCount);

/* Private functions ---------------------------------------------------------*/

/**
  * @brief  Main program.
  * @param  None
  * @retval None
  */
int main(void)
{
	/*!< At this stage the microcontroller clock setting is already configured, 
	   this is done through SystemInit() function which is called from startup
	   file (startup_stm32f10x_xx.s) before to branch to application main.
	   To reconfigure the default setting of SystemInit() function, refer to
	   system_stm32f10x.c file
	 */     
	   
	/* Configure the System clock frequency, HCLK, PCLK2 and PCLK1 prescalers */
	SetSysClock();
	
	/* This function fills the RCC_ClockFreq structure with the current
	 frequencies of different on chip clocks (for debug purpose) */
	RCC_GetClocksFreq(&RCC_ClockFreq);
	
	/* Initialize peripherals used by dSPIN */
	dSPIN_Peripherals_Init();
	  
	/* Structure initialization by default values, in order to avoid blank records */
	dSPIN_Regs_Struct_Reset(&dSPIN_RegsStruct);
	
	/* Acceleration rate settings to 466 steps/s2, range 14.55 to 59590 steps/s2 */
	dSPIN_RegsStruct.ACC 		= AccDec_Steps_to_Par(466);
	/* Deceleration rate settings to 466 steps/s2, range 14.55 to 59590 steps/s2 */
	dSPIN_RegsStruct.DEC 		= AccDec_Steps_to_Par(466); 
	/* Maximum speed settings to 488 steps/s, range 15.25 to 15610 steps/s */
	dSPIN_RegsStruct.MAX_SPEED 	= MaxSpd_Steps_to_Par(488);
	/* Minimum speed settings to 0 steps/s, range 0 to 976.3 steps/s */
	dSPIN_RegsStruct.MIN_SPEED	= MinSpd_Steps_to_Par(0);
	/* Full step speed settings 252 steps/s, range 7.63 to 15625 steps/s */
	dSPIN_RegsStruct.FS_SPD 	= FSSpd_Steps_to_Par(252);
	/* Hold duty cycle (torque) settings to 10%, range 0 to 99.6% */
	dSPIN_RegsStruct.KVAL_HOLD 	= Kval_Perc_to_Par(10);
	/* Run duty cycle (torque) settings to 10%, range 0 to 99.6% */
	dSPIN_RegsStruct.KVAL_RUN 	= Kval_Perc_to_Par(10);
	/* Acceleration duty cycle (torque) settings to 10%, range 0 to 99.6% */
	dSPIN_RegsStruct.KVAL_ACC 	= Kval_Perc_to_Par(10);
	/* Deceleration duty cycle (torque) settings to 10%, range 0 to 99.6% */
	dSPIN_RegsStruct.KVAL_DEC 	= Kval_Perc_to_Par(10);		
	/* Intersect speed settings for BEMF compensation to 200 steps/s, range 0 to 3906 steps/s */
	dSPIN_RegsStruct.INT_SPD 	= IntSpd_Steps_to_Par(200);
	/* BEMF start slope settings for BEMF compensation to 0.038% step/s, range 0 to 0.4% s/step */
	dSPIN_RegsStruct.ST_SLP 	= BEMF_Slope_Perc_to_Par(0.038);
	/* BEMF final acc slope settings for BEMF compensation to 0.063% step/s, range 0 to 0.4% s/step */
	dSPIN_RegsStruct.FN_SLP_ACC = BEMF_Slope_Perc_to_Par(0.063);
	/* BEMF final dec slope settings for BEMF compensation to 0.063% step/s, range 0 to 0.4% s/step */
	dSPIN_RegsStruct.FN_SLP_DEC = BEMF_Slope_Perc_to_Par(0.063);
	/* Thermal compensation param settings to 1, range 1 to 1.46875 */
	dSPIN_RegsStruct.K_THERM 	= KTherm_to_Par(1);
	/* Overcurrent threshold settings to 1500mA */
	dSPIN_RegsStruct.OCD_TH 	= dSPIN_OCD_TH_1500mA;
	/* Stall threshold settings to 1000mA, range 31.25 to 4000mA */
	dSPIN_RegsStruct.STALL_TH 	= StallTh_to_Par(1000);
	/* Step mode settings to 128 microsteps */
	dSPIN_RegsStruct.STEP_MODE 	= dSPIN_STEP_SEL_1_128;
	/* Alarm settings - all alarms enabled */
	dSPIN_RegsStruct.ALARM_EN 	= dSPIN_ALARM_EN_OVERCURRENT | dSPIN_ALARM_EN_THERMAL_SHUTDOWN
		| dSPIN_ALARM_EN_THERMAL_WARNING | dSPIN_ALARM_EN_UNDER_VOLTAGE | dSPIN_ALARM_EN_STALL_DET_A
		| dSPIN_ALARM_EN_STALL_DET_B | dSPIN_ALARM_EN_SW_TURN_ON | dSPIN_ALARM_EN_WRONG_NPERF_CMD;
	/* Internal oscillator, 2MHz OSCOUT clock, supply voltage compensation disabled, *
	 * overcurrent shutdown enabled, slew-rate = 290 V/us, PWM frequency = 15.6kHz   */
	dSPIN_RegsStruct.CONFIG 	= dSPIN_CONFIG_INT_16MHZ_OSCOUT_2MHZ | dSPIN_CONFIG_SW_HARD_STOP
		| dSPIN_CONFIG_VS_COMP_DISABLE | dSPIN_CONFIG_OC_SD_ENABLE | dSPIN_CONFIG_SR_290V_us
		| dSPIN_CONFIG_PWM_DIV_2 | dSPIN_CONFIG_PWM_MUL_1;	

	/* Program all dSPIN registers */
	dSPIN_Registers_Set(&dSPIN_RegsStruct);

	/* Move by 60,000 steps rorward, range 0 to 4,194,303 */
	dSPIN_Move(FWD, (uint32_t)(60000));
	
	/* Wait untill not busy - busy pin test */
	while(dSPIN_Busy_HW());
	
	/* Send dSPIN command change hold duty cycle to 0.5% */
	dSPIN_Set_Param(dSPIN_KVAL_HOLD, Kval_Perc_to_Par(0.5));
	
	/* Send dSPIN command change run duty cycle to 5% */
	dSPIN_Set_Param(dSPIN_KVAL_RUN, Kval_Perc_to_Par(5));
	
	/* Run constant speed of 50 steps/s reverse direction */
	dSPIN_Run(REV, Speed_Steps_to_Par(50));

	/* Wait few seconds - motor turns */
	Delay(0x00FFFFFF);
	
	/* Perform SoftStop commmand */
	dSPIN_Soft_Stop();
	
	/* Wait untill not busy - busy status check in Status register */
	while(dSPIN_Busy_SW());
	
	/* Move by 100,000 steps forward, range 0 to 4,194,303 */
	dSPIN_Move(FWD, (uint32_t)(100000));
	
	/* Wait untill not busy */
	while(dSPIN_Busy_SW());
	
	/* Test of the Flag pin by polling, wait in endless cycle if problem is detected */
	if(dSPIN_Flag()) while(1);
	
	/* Issue dSPIN Go Home command */
	dSPIN_Go_Home();
	/* Wait untill not busy - busy pin test */
	while(dSPIN_Busy_HW());
	
	/* Issue dSPIN Go To command */
	dSPIN_Go_To(0x0000FFFF);
	/* Wait untill not busy - busy pin test */
	while(dSPIN_Busy_HW());
	
	/* Issue dSPIN Go To command */
	dSPIN_Go_To_Dir(FWD, 0x0001FFFF);
	/* Wait untill not busy - busy pin test */
	while(dSPIN_Busy_HW());
	
	/* Read run duty cycle (dSPIN_KVAL_RUN) parameter from dSPIN */
	dSPIN_rx_data = dSPIN_Get_Param(dSPIN_KVAL_RUN);
	
	/* Read intersect speed (dSPIN_INT_SPD) parameter from dSPIN */
	dSPIN_rx_data = dSPIN_Get_Param(dSPIN_INT_SPD);

	/* Read Status register content */
	dSPIN_rx_data = dSPIN_Get_Status();
	
	/* Read absolute position (dSPIN_ABS_POS) parameter from dSPIN */
	dSPIN_rx_data = dSPIN_Get_Param(dSPIN_ABS_POS);

	/* Reset position counter */
	dSPIN_Reset_Pos();

	/* Read absolute position (dSPIN_ABS_POS) parameter from dSPIN */
	dSPIN_rx_data = dSPIN_Get_Param(dSPIN_ABS_POS);

	/* Issue dSPIN Hard HiZ command - disable power stage (High Impedance) */
	dSPIN_Hard_HiZ();
	
	while(1);
}

/**
  * @brief  Inserts a delay time.
  * @param  nCount: specifies the delay time length.
  * @retval None
  */
void Delay(__IO uint32_t nCount)
{
  for(; nCount!= 0;nCount--);
}

#ifdef  USE_FULL_ASSERT
/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t* file, uint32_t line)
{ 
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

  /* Infinite loop */
  while (1)
  {
  }
}
#endif

/**
  * @}
  */ 

/**
  * @}
  */ 

/******************* (C) COPYRIGHT 2011 STMicroelectronics *****END OF FILE****/
