/******************** (C) COPYRIGHT 2009 STMicroelectronics ********************
* File Name          : main.c
* Author             : MCD Application Team
* Version            : V1.0.0
* Date               : 02/19/2009
* Description        : Main program body.
********************************************************************************
* THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
* WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME.
* AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT,
* INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE
* CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING
* INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
*******************************************************************************/

/* Includes ------------------------------------------------------------------*/
#include "main.h"

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
#define HEADER_SIZE 100

/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
#ifdef USE_LCD
/* Information blocks */
#ifndef _ADPCM
u8 DemoTitle[20] = "STM32 I2S  PCM  Demo";
#else
u8 DemoTitle[20] = "STM32 I2S ADPCM Demo";
#endif
u8 CmdTitle0[20] = "  Control Buttons:  ";
u8 CmdTitle1Playing[20] = "KEY>Pause  UP  >Vol+";
u8 CmdTitle2Playing[20] = "SEL>Stop   DOWN>Vol-";
u8 CmdTitle1Paused[20] =  "KEY>Play   UP  >Spkr";
u8 CmdTitle2Paused[20] =  "SEL>Stop   DOWN>Head";
u8 CmdTitle1Stopped[20] = "    UP > Speaker    ";
u8 CmdTitle2Stopped[20] = "  DOWN > Headphone  ";
u8 StatusTitleStopped[20] = "      Stopped       ";
u8 StatusTitlePlaying[20] = "      Playing       ";
u8 StatusTitlePaused[20] = "       Paused       ";
/* Error messages */ 
u8 i2cerr[20] =  "ERROR:I2C com. ->RST";
u8 memerr[20] =  "ERROR: Memory  ->RST";
u8 fileerr[20] = "ERROR: File    ->RST";
/* Variable to store the previous progress status */
static u8 previoustmp = 50;
#endif /* USE_LCD */

#ifdef USE_LCD
static void I2S_CODEC_LCDConfig(void);
static void SysTick_Config(void);
#endif /* USE_LCD */

ErrorStatus HSEStartUpStatus;

/* Private function prototypes -----------------------------------------------*/
void RCC_Configuration(void);
void GPIO_Configuration(void);
void NVIC_Configuration(void);
void Delay(vu32 nCount);
static void VECT_TAB_Config(void);
static void RCC_Config(void);
static void Joystick_Config(void);

/* Private functions ---------------------------------------------------------*/
/*******************************************************************************
* Function Name  : main
* Description    : Main program.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
int main(void)
{
#ifdef DEBUG
  debug();
#endif
      
  /* System Clocks Configuration */
  RCC_Config();

  /* Set Vector Table base address */
  VECT_TAB_Config();

  /* Choose number of repetitions: 0 => infinite repetitions */
  I2S_CODEC_ReplayConfig(0);

  /* Initialize the application environment */
  I2S_CODEC_Init(OutputDevice_SPEAKER, AUDIO_FILE_ADDRESS);

  /* Configure the Joystick related functionalities (EXTI, NVIC, GPIO) */
  Joystick_Config();

#ifdef USE_LCD
  /* Display the welcome screen and the commands */
  LCD_Update(ALL);

  /* Configure the SysTick for the Progress LCD information update */
  SysTick_Config();
#endif /* USE_LCD */
  
  /* Infinite loop */
  while (1)
  {}
}


/*******************************************************************************
* Function Name  : Joystick_Config
* Description    : Configure the Joystick GPIOs and EXTI, NVIC channels.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void Joystick_Config(void)
{
  EXTI_InitTypeDef EXTI_InitStructure;
  GPIO_InitTypeDef GPIO_InitStructure;
  NVIC_InitTypeDef NVIC_InitStructure;

  /* Enable GPIOB, GPIOC and AFIO clock */
  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIOG | RCC_APB2Periph_GPIOD | RCC_APB2Periph_AFIO, ENABLE);

  EXTI_DeInit();

  /* Configure PG.07, PG.08, PG.13, PG.14 and PG.15 as input floating */
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_7 | GPIO_Pin_8 | GPIO_Pin_13 | GPIO_Pin_14 | GPIO_Pin_15;
  GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN_FLOATING;
  GPIO_Init(GPIOG, &GPIO_InitStructure);

  /* Configure PD.03 as input floating */
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_3;
  GPIO_Init(GPIOD, &GPIO_InitStructure);

  GPIO_EXTILineConfig(GPIO_PortSourceGPIOG, GPIO_PinSource7);
  GPIO_EXTILineConfig(GPIO_PortSourceGPIOG, GPIO_PinSource8);
  GPIO_EXTILineConfig(GPIO_PortSourceGPIOG, GPIO_PinSource13);
  GPIO_EXTILineConfig(GPIO_PortSourceGPIOG, GPIO_PinSource14);
  GPIO_EXTILineConfig(GPIO_PortSourceGPIOG, GPIO_PinSource15);
  GPIO_EXTILineConfig(GPIO_PortSourceGPIOD, GPIO_PinSource3);

  EXTI_InitStructure.EXTI_Line = EXTI_Line3 | EXTI_Line7 | EXTI_Line8 | EXTI_Line13 | EXTI_Line14 | EXTI_Line15;
  EXTI_InitStructure.EXTI_Mode = EXTI_Mode_Interrupt;
  EXTI_InitStructure.EXTI_Trigger = EXTI_Trigger_Falling;
  EXTI_InitStructure.EXTI_LineCmd = ENABLE;
  EXTI_Init(&EXTI_InitStructure);

  /* Enable the EXTI3 Interrupt */
  NVIC_InitStructure.NVIC_IRQChannel = EXTI3_IRQChannel;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 3;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);

  /* Enable the EXTI7 and  EXTI8 Interrupt */
  NVIC_InitStructure.NVIC_IRQChannel = EXTI9_5_IRQChannel;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 2;
  NVIC_Init(&NVIC_InitStructure);

  /* Enable the EXTI13, EXTI14 and EXTI15 Interrupt */
  NVIC_InitStructure.NVIC_IRQChannel = EXTI15_10_IRQChannel;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 1;
  NVIC_Init(&NVIC_InitStructure);
}

#ifdef USE_LCD
/*******************************************************************************
* Function Name  : I2S_CODEC_LCDConfig
* Description    : Initialize the LCD device and display the welcom screen.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void I2S_CODEC_LCDConfig(void)
{
  /* Initialize the LCD component */
  STM3210E_LCD_Init();

  Delay(0xFFFF);

  /* Set the text and the background color */
  LCD_SetBackColor(Blue);
  LCD_SetTextColor(Black);
  LCD_Clear(White);

  Delay(0xFFFF);

  /* Display the Titles */
  LCD_DisplayStringLine(Line0, DemoTitle);
  LCD_DisplayStringLine(Line2, CmdTitle0);
}
#endif /* USE_LCD */

/*******************************************************************************
* Function Name  : LCD_Update
* Description    : Updates the display information.
* Input          : Status: new status to be updated, could be: PLAY, PAUSE, STOP,
*                   PROGRESS, VOL or ALL.
* Output         : None
* Return         : None
*******************************************************************************/
void LCD_Update(u32 Status)
{
#ifdef USE_LCD
  u8 tmp = 0;
  u32 counter = 0;

  /* Enable the FSMC that share a pin w/ I2C1 (LBAR) */
  RCC_AHBPeriphClockCmd(RCC_AHBPeriph_FSMC, ENABLE);

  switch (Status)
  {
    case PROGRESS:
      tmp = (u8) ((u32)((GetVar_AudioDataIndex()) * 100) / GetVar_AudioDataLength());
      if (GetVar_AudioDataIndex() < (HEADER_SIZE*2))
      {
        LCD_SetTextColor(Magenta);
        LCD_ClearLine(Line8);
        LCD_DrawRect(Line8, 310, 16, 300);
      }
      else
      {
        LCD_SetTextColor(Magenta);
        LCD_DrawLine(Line8, 310 - (tmp * 3), 16, Vertical);
      }
      break;
    case FRWRWD:
      tmp = (u8) ((u32)((GetVar_AudioDataIndex()) * 100) / GetVar_AudioDataLength());

      LCD_SetTextColor(Magenta);
      LCD_ClearLine(Line8);
      LCD_DrawRect(Line8, 310, 16, 300);
      LCD_SetTextColor(Magenta);

      for (counter = 0; counter <= tmp; counter++)
      {
        LCD_DrawLine(Line8, 310 - (counter * 3), 16, Vertical);
      }
      break;
    case STOP:
      /* Display the stopped status menu */
      LCD_SetTextColor(White);
      LCD_DisplayStringLine(Line3, CmdTitle1Stopped);
      LCD_DisplayStringLine(Line4, CmdTitle2Stopped);
      LCD_SetTextColor(Red);
      LCD_DisplayStringLine(Line6, StatusTitleStopped);
      LCD_ClearLine(Line9);
      LCD_SetTextColor(Black);
      LCD_DisplayChar(Line9, 250, 'v');
      LCD_DisplayChar(Line9, 235, 'o');
      LCD_DisplayChar(Line9, 220, 'l');
      LCD_DisplayChar(Line9, 200, '-');
      LCD_DisplayChar(Line9, 85, '+');
      LCD_DrawRect(Line9 + 8, 185, 10, 100);
      break;
    case PAUSE:
      /* Display the paused status menu */
      LCD_SetTextColor(White);
      LCD_DisplayStringLine(Line3, CmdTitle1Paused);
      LCD_DisplayStringLine(Line4, CmdTitle2Paused);
      LCD_SetTextColor(Red);
      LCD_DisplayStringLine(Line6, StatusTitlePaused);
      break;
    case PLAY:
      /* Display the Titles */
      LCD_SetTextColor(Black);
      LCD_DisplayStringLine(Line0, DemoTitle);
      LCD_DisplayStringLine(Line2, CmdTitle0);

      /* Display the Playing status menu */
      LCD_SetTextColor(White);
      LCD_DisplayStringLine(Line3, CmdTitle1Playing);
      LCD_DisplayStringLine(Line4, CmdTitle2Playing);
      LCD_SetTextColor(Red);
      LCD_DisplayStringLine(Line6, StatusTitlePlaying);
      LCD_ClearLine(Line9);
      LCD_SetTextColor(Black);
      LCD_DisplayChar(Line9, 250, 'v');
      LCD_DisplayChar(Line9, 235, 'o');
      LCD_DisplayChar(Line9, 220, 'l');
      LCD_DisplayChar(Line9, 200, '-');
      LCD_DisplayChar(Line9, 85, '+');
      LCD_DrawRect(Line9 + 8, 185, 10, 100);
      break;
    case ALL:
      I2S_CODEC_LCDConfig();
      /* Display the stopped status menu */
      LCD_SetTextColor(White);
      LCD_DisplayStringLine(Line3, CmdTitle1Stopped);
      LCD_DisplayStringLine(Line4, CmdTitle2Stopped);
      LCD_SetTextColor(Red);
      LCD_DisplayStringLine(Line6, StatusTitleStopped);
      LCD_ClearLine(Line9);
      LCD_SetTextColor(Black);
      LCD_DisplayChar(Line9, 250, 'v');
      LCD_DisplayChar(Line9, 235, 'o');
      LCD_DisplayChar(Line9, 220, 'l');
      LCD_DisplayChar(Line9, 200, '-');
      LCD_DisplayChar(Line9, 85, '+');
      LCD_DrawRect(Line9 + 8, 185, 10, 100);
      break;
  }
  /* Update the volume bar in all cases except when progress bar is to be apdated */
  if (Status != PROGRESS)
  {
    /* Compute the current volume percentage */
    tmp = (u8) ((u16)((0xFF - GetVar_CurrentVolume()) * 100) / 0xFF) ;

    /* Clear the previuos volume bar */
    LCD_SetTextColor(Blue);
    LCD_DrawLine(Line9 + 10, 185 - previoustmp , 8, Vertical);
    LCD_DrawLine(Line9 + 10, 185 - previoustmp + 1 , 8, Vertical);

    /* Draw the new volume bar */
    LCD_SetTextColor(Red);
    LCD_DrawLine(Line9 + 10, 185 - tmp, 8, Vertical);
    LCD_DrawLine(Line9 + 10, 185 - tmp + 1, 8, Vertical);

    /* save the current position */
    previoustmp = tmp;
  }
  /* Disable the FSMC that share a pin w/ I2C1 (LBAR) */
  RCC_AHBPeriphClockCmd(RCC_AHBPeriph_FSMC, DISABLE);
#endif /* USE_LCD */
}

/*******************************************************************************
* Function Name  : SysTick_Config
* Description    : Configure the SysTick to generate an interrupt every ~1% of the
*                   audio file period.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void SysTick_Config(void)
{
  u32 reloadvalue = 0, err = 0;

  /* Initialize the audio file to determine the audio frequency
    and the Data Length in order to set SysTick reload value */
  err = AudioFile_Init();

  /* Check if the initialization passed */
  if ((err == 1) || (err == 2))
  {
    LCD_DisplayError(err);
  }
  /* Systick IT time = ~1% of the total audio stream duration:
     reloadvalue = ((((WaveFileLength/(2 * Fs)) / 100) * HCLK_Div8) / 2)
                 = (WaveFileLength / Fs) * 22500 */
  reloadvalue = (u32)((GetVar_AudioDataLength() / (GetVar_i2saudiofreq())) * 20000);

  /* Check if the reload value overflows */
  if (reloadvalue >= 0xFFFFFF)
  {
    reloadvalue = 0xFFFFFF;
  }

  /* Configure and enable the SysTick peripheral */
  SysTick_CLKSourceConfig(SysTick_CLKSource_HCLK_Div8);
  SysTick_SetReload(reloadvalue);
  SysTick_ITConfig(ENABLE);
  NVIC_SystemHandlerPriorityConfig(SystemHandler_SysTick, 0, 0);
  SysTick_CounterCmd(SysTick_Counter_Enable);
}

/*******************************************************************************
* Function Name  : LCD_DisplayError
* Description    : Displays error message on the LCD screen and prompt user to
*                   reset the application.
* Input          : err: Error code.
* Output         : None
* Return         : None
*******************************************************************************/
void LCD_DisplayError(u32 err)
{
#ifdef USE_LCD
  I2S_CODEC_LCDConfig();

  LCD_SetTextColor(Red);
  LCD_ClearLine(Line7);

  /* The memory initialazation failed */
  if (err == 1)
  {
    LCD_DisplayStringLine(Line7, memerr);
  }

  /* The audio file initialization failed (wrong audio format or wrong file) */
  if (err == 2)
  {
    LCD_DisplayStringLine(Line7, fileerr);
  }

  /* I2C communication failure occured */
  if (err == 3)
  {
    LCD_DisplayStringLine(Line7, i2cerr);
  }
#endif /* USE_LCD */

  /* Enter infinite loop waiting user to reset system */
  while (1)
  {}
}

/*******************************************************************************
* Function Name  : Delay
* Description    : Inserts a delay time.
* Input          : nCount: Number of cycles to be counted to make a delay.
* Output         : None
* Return         : None
*******************************************************************************/
void Delay(vu32 nCount)
{
  for (; nCount != 0; nCount--);
}

/*******************************************************************************
* Function Name  : RCC_Config
* Description    : Configures the different system clocks.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void RCC_Config(void)
{
  /* RCC system reset(for debug purpose) */
  RCC_DeInit();

  /* Enable HSE */
  RCC_HSEConfig(RCC_HSE_ON);

  /* Wait till HSE is ready */
  HSEStartUpStatus = RCC_WaitForHSEStartUp();

  if (HSEStartUpStatus == SUCCESS)
  {
    /* Enable Prefetch Buffer */
    FLASH_PrefetchBufferCmd(FLASH_PrefetchBuffer_Enable);

    /* Flash 2 wait state */
    FLASH_SetLatency(FLASH_Latency_2);

    /* HCLK = SYSCLK */
    RCC_HCLKConfig(RCC_SYSCLK_Div1);

    /* PCLK2 = HCLK */
    RCC_PCLK2Config(RCC_HCLK_Div1);

    /* PCLK1 = HCLK/2 */
    RCC_PCLK1Config(RCC_HCLK_Div2);

    /* PLLCLK = 8MHz * 9 = 72 MHz */
    RCC_PLLConfig(RCC_PLLSource_HSE_Div1, RCC_PLLMul_9);

    /* Enable PLL */
    RCC_PLLCmd(ENABLE);

    /* Wait till PLL is ready */
    while (RCC_GetFlagStatus(RCC_FLAG_PLLRDY) == RESET)
    {}

    /* Select PLL as system clock source */
    RCC_SYSCLKConfig(RCC_SYSCLKSource_PLLCLK);

    /* Wait till PLL is used as system clock source */
    while (RCC_GetSYSCLKSource() != 0x08)
    {}
  }
  }

/*******************************************************************************
* Function Name  : VECT_TAB_Config
* Description    : Configures Vector Table base location. 
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void  VECT_TAB_Config(void)
{
#ifdef  VECT_TAB_RAM
  /* Set the Vector Table base location at 0x20000000 */
  NVIC_SetVectorTable(NVIC_VectTab_RAM, 0x0);
#else  /* VECT_TAB_FLASH  */
  /* Set the Vector Table base location at 0x08000000 */
  NVIC_SetVectorTable(NVIC_VectTab_FLASH, 0x0);
#endif

  /* Set the Interrupts priority group */
  NVIC_PriorityGroupConfig(NVIC_PriorityGroup_2);
}



/*******************************************************************************
* Function Name  : GPIO_Configuration
* Description    : Configures the different GPIO ports.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void GPIO_Configuration(void)
{
  GPIO_InitTypeDef GPIO_InitStructure;

  /* Once the DAC channel is enabled, the corresponding GPIO pin is automatically 
     connected to the DAC converter. In order to avoid parasitic consumption, 
     the GPIO pin should be configured in analog */
  GPIO_InitStructure.GPIO_Pin =  GPIO_Pin_4 | GPIO_Pin_5;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AIN;
  GPIO_Init(GPIOA, &GPIO_InitStructure);
}

#ifdef  DEBUG
/*******************************************************************************
* Function Name  : assert_failed
* Description    : Executes an infinite loop when wrong parameter is entered to
*                   a lib function.
* Input          : file: pointer to the source file name
*                  line: assert_param error line source number
* Output         : None
* Return         : None
*******************************************************************************/
void assert_failed(u8* file, u32 line)
{
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

  /* Infinite loop */
  while (1)
  {}
}
#endif /* DEBUG */

/******************* (C) COPYRIGHT 2009 STMicroelectronics *****END OF FILE****/
