/******************** (C) COPYRIGHT 2009 STMicroelectronics ********************
* File Name          : main.c
* Author             : MCD Application Team
* Version            : V1.0.0
* Date               : 02/06/2009
* Description        : Main program body
********************************************************************************
* THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
* WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE TIME.
* AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY DIRECT,
* INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING FROM THE
* CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE CODING
* INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
*******************************************************************************/

/* Includes ------------------------------------------------------------------*/
#include "stm32f10x_cec.h"
#include "lcd.h"

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
extern u8 ReceiveBuffer[10];
u8 ReceivedFrame=0;
u8 SendFrame=0;
u32 ReceiveInfos = 0;
extern ErrorStatus SendStatus;
u8 *BufferPointer;
u8 ByteNumber=0;
u8 SendTo = 0;

/* Private function prototypes -----------------------------------------------*/
void RCC_Configuration(void);
void NVIC_Configuration(void);
void EXTI_Configuration(void);
void GPIO_Configuration(void);
void DisplayDataByte (u8 Data, u8 Line, u8 Column);
void DisplayHexToChar (u8 Data, u16 Line, u16 Column);
u8 HexToChar(u8 Data);
void DisplayBuffer(u8 *DataBuffer, u8 BufferSize, u16 Line);

/* Private functions ---------------------------------------------------------*/
/*******************************************************************************
* Function Name  : main
* Description    : Main program.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
int main(void)
{
  
#ifdef DEBUG
  debug();
#endif

  /* RCC configuration */
  RCC_Configuration();
  
  /* NVIC configuration */
  NVIC_Configuration();

  /* CEC initialization */
  CEC_Init();
    
  /* GPIO configuration */
  GPIO_Configuration();
        
  /* EXTI configuration */
  EXTI_Configuration();
    
  /* LCD initialization */
  STM3210B_LCD_Init();

  /* Clear the LCD */
  LCD_Clear(Blue);

  /* Display initialization */
  LCD_SetTextColor(White);
  LCD_SetBackColor(Black);
  LCD_DisplayString(Line0, Column0, "Receive:");
  LCD_DisplayString(Line6, Column0, "Send to:");
  LCD_SetBackColor(Blue);
  LCD_DisplayString(Line5, Column0, "____________________");

  /* Infinite loop */
  while (1)
  {
    /* If a frame has been received */
    if(ReceivedFrame)
    {
      LCD_SetBackColor(Blue);
      
      /* Check if the frame has been received correctly */
      if(ReceiveInfos & ReceiveFrameStatusMask)
      {
        /* Check if the frame has been sent to me */
        if(ReceiveInfos & FrameSendToMeMask)
        {
          LCD_SetBackColor(Green);
          LCD_DisplayString(Line0, Column10, "Succeeded");  
        }
        else /* The receive was successful, but I'm not the concerned follower */
        {
          LCD_SetBackColor(Magenta);
          LCD_DisplayString(Line0, Column10, "Detected ");  
        }
      }
      else /* The receive was failed */
      {
         LCD_SetBackColor(Red);
         LCD_DisplayString(Line0, Column10, " Failed  ");  
      }
      /* Display the receive status, the initiator address, the received data 
      frame and  the number of received bytes  */
      LCD_SetBackColor(Blue);
      LCD_DisplayString(Line1, Column0, "Sender Address = ");
      DisplayHexToChar ((u8)(ReceiveInfos & InitiatorAddressMask), Line1,Column17);
      LCD_DisplayString(Line2, Column0, "Number of bytes:");
      DisplayHexToChar ((u8)(ReceiveInfos>>8), Line2,Column17);
      DisplayBuffer(ReceiveBuffer,ReceiveInfos>>8, Line3);
      ReceivedFrame=0;
    }
    if (SendFrame)
    {
      /* Check if the frame has been sent successfully */
      if (SendStatus != ERROR ) /* The send succeeded */
      {
        LCD_SetBackColor(Green);
    	LCD_DisplayString(Line7, Column0, "Send Status: Succeed"); 
      } 
      else  /* The send failed */
      {
        LCD_SetBackColor(Red); 
        LCD_DisplayString(Line7, Column0, "Send Status: Fail   "); 
      }
      
      /* Display the data frame to send and the address of the follower */
      LCD_SetBackColor(Blue);
      DisplayBuffer(BufferPointer,ByteNumber , Line8);
      DisplayHexToChar (SendTo, Line6, Column8);
      SendFrame=0;
    }
  }
}

/*******************************************************************************
* Function Name  : RCC_Configuration
* Description    : Configures the different system clocks.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void RCC_Configuration(void)
{   
  ErrorStatus HSEStartUpStatus;
  
  /* RCC system reset(for debug purpose) */
  RCC_DeInit();

  /* Enable HSE */
  RCC_HSEConfig(RCC_HSE_ON);

  /* Wait till HSE is ready */
  HSEStartUpStatus = RCC_WaitForHSEStartUp();

  if(HSEStartUpStatus == SUCCESS)
  {
    /* Enable Prefetch Buffer */
    FLASH_PrefetchBufferCmd(FLASH_PrefetchBuffer_Enable);

    /* Flash 2 wait state */
    FLASH_SetLatency(FLASH_Latency_2);
 
    /* HCLK = SYSCLK */
    RCC_HCLKConfig(RCC_SYSCLK_Div1); 
  
    /* PCLK2 = HCLK */
    RCC_PCLK2Config(RCC_HCLK_Div1); 

    /* PCLK1 = HCLK/2 */
    RCC_PCLK1Config(RCC_HCLK_Div2);

    /* PLLCLK = 8MHz * 9 = 72 MHz */
    RCC_PLLConfig(RCC_PLLSource_HSE_Div1, RCC_PLLMul_9);

    /* Enable PLL */ 
    RCC_PLLCmd(ENABLE);

    /* Wait till PLL is ready */
    while(RCC_GetFlagStatus(RCC_FLAG_PLLRDY) == RESET)
    {
    }

    /* Select PLL as system clock source */
    RCC_SYSCLKConfig(RCC_SYSCLKSource_PLLCLK);

    /* Wait till PLL is used as system clock source */
    while(RCC_GetSYSCLKSource() != 0x08)
    {
    }
  }
  
  /* Enable GPIOB (for key button) and GPIOD (for SPI communication of the LCD) clocks */
  RCC_APB2PeriphClockCmd(RCC_APB2Periph_GPIOB | RCC_APB2Periph_GPIOD , ENABLE);
}

/*******************************************************************************
* Function Name  : NVIC_Configuration
* Description    : Configures Vector Table base location and NVIC global 
*                  interrupts.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void NVIC_Configuration(void)
{
  NVIC_InitTypeDef NVIC_InitStructure;
  
  #ifdef  VECT_TAB_RAM  
  /* Set the Vector Table base location at 0x20000000 */ 
  NVIC_SetVectorTable(NVIC_VectTab_RAM, 0x0); 
#else  /* VECT_TAB_FLASH  */
  /* Set the Vector Table base location at 0x08000000 */ 
  NVIC_SetVectorTable(NVIC_VectTab_FLASH, 0x0);   
#endif
  
  /* Configure two bits for preemption priority */
  NVIC_PriorityGroupConfig(NVIC_PriorityGroup_2);
  
  /* Enable the EXTI9_5 Interrupt */
  NVIC_InitStructure.NVIC_IRQChannel = EXTI9_5_IRQChannel;
  NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 2;
  NVIC_InitStructure.NVIC_IRQChannelSubPriority = 3;
  NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
  NVIC_Init(&NVIC_InitStructure);
}

/*******************************************************************************
* Function Name  : GPIO_Configuration
* Description    : Configures the different GPIO ports.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void GPIO_Configuration(void)
{
  GPIO_InitTypeDef GPIO_InitStructure;
    
  /* Configure GPIOB Pin 9 as input floating (For key button) */
  GPIO_InitStructure.GPIO_Pin = GPIO_Pin_9;
  GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN_FLOATING;
  GPIO_Init(GPIOB, &GPIO_InitStructure);
  
  /* Connect EXTI Line9 to GPIOB Pin 9 */
  GPIO_EXTILineConfig(GPIO_PortSourceGPIOB, GPIO_PinSource9);
}

/*******************************************************************************
* Function Name  : EXTI_Configuration
* Description    : Configures the different GPIO ports.
* Input          : None
* Output         : None
* Return         : None
*******************************************************************************/
void EXTI_Configuration(void)
{
  EXTI_InitTypeDef EXTI_InitStructure;

  /* Configure EXTI Line9 to generate an interrupt on falling edge */  
  EXTI_InitStructure.EXTI_Line = EXTI_Line9;
  EXTI_InitStructure.EXTI_Mode = EXTI_Mode_Interrupt;
  EXTI_InitStructure.EXTI_Trigger = EXTI_Trigger_Falling;
  EXTI_InitStructure.EXTI_LineCmd = ENABLE;
  EXTI_Init(&EXTI_InitStructure);
}

/*******************************************************************************
* Function Name  : DisplayBuffer
* Description    : Displays the content of a buffer on the LCD in Hex format.
* Input          : - DataBuffer: a pointer to the buffer to be displayed.
*                  - BufferSize: the size of the buffer in byte.
*                  - Line: the line where the data will be displayed.
* Output         : None
* Return         : None
*******************************************************************************/
void DisplayBuffer(u8 *DataBuffer, u8 BufferSize, u16 Line)
{
  u8 i;
  u16 column = Column5;
  u16 line = Line;
 
  LCD_DisplayString(Line, Column0, "                    "); 
  LCD_DisplayString(Line+24, Column0, "                    "); 
  LCD_DisplayString(Line, Column0, "Data:"); 

  for (i=0;i<BufferSize;i++)
  { 
    DisplayHexToChar (*(DataBuffer++), line, column);
    column -=32;
    LCD_DisplayChar (line, column, ',');
   
    if (i==4)
    {
      line += 24;
      column = Column0;
    }
    
    column -=16;
  }
}

/*******************************************************************************
* Function Name  : DisplayHexToChar
* Description    : Displays a data byte on the LCD in Hex format.
* Input          : - Data: the data to be displayed.
*                  - Line: the line where the data will be displayed.
*                  - Column: the column where the data will be displayed.
* Output         : None
* Return         : None
*******************************************************************************/
void DisplayHexToChar (u8 Data, u16 Line, u16 Column)
{
  LCD_DisplayChar(Line,Column-16,HexToChar(Data));
  LCD_DisplayChar(Line,Column,HexToChar(Data>>4));
}


/*******************************************************************************
* Function Name  : HexToChar
* Description    : Converts a data (four bits) to a character to be displayed 
*                  in Hex format.
* Input          : - Data: byte to be converted.
* Output         : - A character to be displayed in the LCD.
* Return         : None
*******************************************************************************/
u8 HexToChar(u8 Data)
{
  Data &= 0xF;
  
  switch(Data)
  {
    case 0xA: return 0x41;
    case 0xB: return 0x42;
    case 0xC: return 0x43;
    case 0xD: return 0x44;
    case 0xE: return 0x45;
    case 0xF: return 0x46;
    default: return (Data+0x30);
  }
}

#ifdef  DEBUG
/*******************************************************************************
* Function Name  : assert_failed
* Description    : Reports the name of the source file and the source line number
*                  where the assert_param error has occurred.
* Input          : - file: pointer to the source file name
*                  - line: assert_param error line source number
* Output         : None
* Return         : None
*******************************************************************************/
void assert_failed(u8* file, u32 line)
{ 
  /* User can add his own implementation to report the file name and line number,
     ex: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */

  /* Infinite loop */
  while (1)
  {
  }
}
#endif

/******************* (C) COPYRIGHT 2009 STMicroelectronics *****END OF FILE****/
