/**
 ******************************************************************************
 * @file    easyspin.c
 * @author  EMEA AMS/IPD Marketing & Application - VE
 * @version V1.0.1
 * @date    June-2012
 * @brief   easySPIN (L6474) product related routines
 ******************************************************************************
 * @copy
 *
 * THE PRESENT FIRMWARE WHICH IS FOR GUIDANCE ONLY AIMS AT PROVIDING CUSTOMERS
 * WITH CODING INFORMATION REGARDING THEIR PRODUCTS IN ORDER FOR THEM TO SAVE
 * TIME. AS A RESULT, STMICROELECTRONICS SHALL NOT BE HELD LIABLE FOR ANY
 * DIRECT, INDIRECT OR CONSEQUENTIAL DAMAGES WITH RESPECT TO ANY CLAIMS ARISING
 * FROM THE CONTENT OF SUCH FIRMWARE AND/OR THE USE MADE BY CUSTOMERS OF THE
 * CODING INFORMATION CONTAINED HEREIN IN CONNECTION WITH THEIR PRODUCTS.
 *
 * <h2><center>&copy; COPYRIGHT 2012 STMicroelectronics</center></h2>
 */

/* Includes ------------------------------------------------------------------*/
#include "easyspin.h"
#include "stm32f10x.h"

/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
uint8_t easySPIN_Write_Byte(uint8_t byte);

/* Private functions ---------------------------------------------------------*/

/**
 * @brief  Transmits/Receives one byte to/from easySPIN over SPI.
 * @param  Transmited byte
 * @retval Received byte
 */
uint8_t easySPIN_Write_Byte(uint8_t byte) {

	/* nSS signal activation - low */
	GPIO_ResetBits(easySPIN_nSS_Port, easySPIN_nSS_Pin);

	SPI_I2S_GetFlagStatus(easySPIN_SPI, SPI_I2S_FLAG_RXNE);

	/* SPI byte send */
	SPI_I2S_SendData(easySPIN_SPI, byte);

	/* Wait for SPIx Busy flag */
	while (SPI_I2S_GetFlagStatus(easySPIN_SPI, SPI_I2S_FLAG_BSY) != RESET)
		;

	/* nSS signal deactivation - high */
	GPIO_SetBits(easySPIN_nSS_Port, easySPIN_nSS_Pin);

	SPI_I2S_GetFlagStatus(easySPIN_SPI, SPI_I2S_FLAG_RXNE);

	return (uint8_t) (SPI_I2S_ReceiveData(easySPIN_SPI));
}

/**
 * @brief  Initializes uC peripherals, GPIOs, clocks used by easySPIN.
 * @param  None
 * @retval None
 */
void easySPIN_Peripherals_Init(void) {
	GPIO_InitTypeDef GPIO_InitStructure;
	SPI_InitTypeDef SPI_InitStructure;

	/* Used peripherals clock enable -------------------------------------------*/
	RCC_APB1PeriphClockCmd(easySPIN_PERIPHERAL_CLKs_APB1, ENABLE);
	RCC_APB2PeriphClockCmd(easySPIN_PERIPHERAL_CLKs_APB2, ENABLE);

	/* Configure SPI pin: SCK --------------------------------------------------*/
	GPIO_InitStructure.GPIO_Pin = easySPIN_SCK_Pin;
	GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
	GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF_PP;
	GPIO_Init(easySPIN_SCK_Port, &GPIO_InitStructure);

	/* Configure SPI pin: MOSI -------------------------------------------------*/
	GPIO_InitStructure.GPIO_Pin = easySPIN_MOSI_Pin;
	GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
	GPIO_InitStructure.GPIO_Mode = GPIO_Mode_AF_PP;
	GPIO_Init(easySPIN_MOSI_Port, &GPIO_InitStructure);

	/* Configure SPI pin: nSS --------------------------------------------------*/
	GPIO_InitStructure.GPIO_Pin = easySPIN_nSS_Pin;
	GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
	GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
	GPIO_Init(easySPIN_nSS_Port, &GPIO_InitStructure);

	/* Configure easySPIN - DIR pin --------------------------------------------*/
	GPIO_InitStructure.GPIO_Pin = easySPIN_DIR_Pin;
	GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
	GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
	GPIO_Init(easySPIN_DIR_Port, &GPIO_InitStructure);

	/* Configure easySPIN - Flag pin -------------------------------------------*/
	GPIO_InitStructure.GPIO_Pin = easySPIN_FLAG_Pin;
	GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
	GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IPU;
	GPIO_Init(easySPIN_FLAG_Port, &GPIO_InitStructure);

	/* Configure easySPIN - STCK pin -------------------------------------------*/
	GPIO_InitStructure.GPIO_Pin = easySPIN_STCK_Pin;
	GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
	GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
	GPIO_Init(easySPIN_STCK_Port, &GPIO_InitStructure);

	/* Configure easySPIN - STBY/RESET pin -------------------------------------*/
	GPIO_InitStructure.GPIO_Pin = easySPIN_STBY_RESET_Pin;
	GPIO_InitStructure.GPIO_Speed = GPIO_Speed_50MHz;
	GPIO_InitStructure.GPIO_Mode = GPIO_Mode_Out_PP;
	GPIO_Init(easySPIN_STBY_RESET_Port, &GPIO_InitStructure);

	/* SPI configuration -------------------------------------------------------*/
	SPI_InitStructure.SPI_Direction = SPI_Direction_2Lines_FullDuplex;
	SPI_InitStructure.SPI_Mode = SPI_Mode_Master;
	SPI_InitStructure.SPI_DataSize = SPI_DataSize_8b;
	SPI_InitStructure.SPI_CPOL = SPI_CPOL_High;
	SPI_InitStructure.SPI_CPHA = SPI_CPHA_2Edge;
	SPI_InitStructure.SPI_NSS = SPI_NSS_Soft;
	SPI_InitStructure.SPI_BaudRatePrescaler = SPI_BaudRatePrescaler_16;
	SPI_InitStructure.SPI_FirstBit = SPI_FirstBit_MSB;
	SPI_InitStructure.SPI_CRCPolynomial = 7;
	SPI_Init(easySPIN_SPI, &SPI_InitStructure);

	/* Enable SPI */
	SPI_Cmd(easySPIN_SPI, ENABLE);
}

/**
 * @brief  Initializes easySPIN related peripherals, disables powerstage and releases reset pin
 * @param  None
 * @retval None
 */
void easySPIN_Init(void) {
	/* Initialize peripherals used by easySPIN */
	easySPIN_Peripherals_Init();

	/* Disable easySPIN powerstage */
	easySPIN_Disable();

	/* Standby-reset deactivation */
	easySPIN_ReleaseReset();
}

/**
 * @brief  Fills-in easySPIN configuration structure with default values.
 * @param  Structure address (pointer to struct)
 * @retval None
 */
void easySPIN_Regs_Struct_Reset(
		easySPIN_RegsStruct_TypeDef* easySPIN_RegsStruct) {
	easySPIN_RegsStruct->ABS_POS = 0x00;
	easySPIN_RegsStruct->EL_POS = 0x00;
	easySPIN_RegsStruct->MARK = 0x00;

	easySPIN_RegsStruct->TVAL = 0x00;
	easySPIN_RegsStruct->T_FAST = easySPIN_TOFF_FAST_8_0_us
			| easySPIN_FAST_STEP_8_0_us;
	easySPIN_RegsStruct->TON_MIN = 0x00;
	easySPIN_RegsStruct->TOFF_MIN = 0x01;

	/* OCD_TH register setup */
	easySPIN_RegsStruct->OCD_TH = easySPIN_OCD_TH_375mA;

	/* STEP_MODE register */
	easySPIN_RegsStruct->STEP_MODE = easySPIN_STEP_SEL_1
			| easySPIN_SYNC_SEL_1_2;

	/* ALARM_EN register setup */
	easySPIN_RegsStruct->ALARM_EN = easySPIN_ALARM_EN_OVERCURRENT
			| easySPIN_ALARM_EN_THERMAL_SHUTDOWN
			| easySPIN_ALARM_EN_THERMAL_WARNING
			| easySPIN_ALARM_EN_UNDERVOLTAGE | easySPIN_ALARM_EN_SW_TURN_ON
			| easySPIN_ALARM_EN_WRONG_NPERF_CMD;

	/* CONFIG register setup */
	easySPIN_RegsStruct->CONFIG = easySPIN_CONFIG_INT_16MHZ
			| easySPIN_CONFIG_EN_TQREG_INT_REG | easySPIN_CONFIG_OC_SD_ENABLE
			| easySPIN_CONFIG_SR_180V_us | easySPIN_CONFIG_TSW_8_us;
}

/**
 * @brief  Configures easySPIN internal registers with values in the config structure.
 * @param  Configuration structure address (pointer to configuration structure)
 * @retval None
 */
void easySPIN_Registers_Set(easySPIN_RegsStruct_TypeDef* easySPIN_RegsStruct) {
	easySPIN_SetParam(easySPIN_ABS_POS, easySPIN_RegsStruct->ABS_POS);
	easySPIN_SetParam(easySPIN_EL_POS, easySPIN_RegsStruct->EL_POS);
	easySPIN_SetParam(easySPIN_MARK, easySPIN_RegsStruct->MARK);
	easySPIN_SetParam(easySPIN_TVAL, easySPIN_RegsStruct->TVAL);
	easySPIN_SetParam(easySPIN_T_FAST, easySPIN_RegsStruct->T_FAST);
	easySPIN_SetParam(easySPIN_TON_MIN, easySPIN_RegsStruct->TON_MIN);
	easySPIN_SetParam(easySPIN_TOFF_MIN, easySPIN_RegsStruct->TOFF_MIN);
	easySPIN_SetParam(easySPIN_OCD_TH, easySPIN_RegsStruct->OCD_TH);
	easySPIN_SetParam(easySPIN_STEP_MODE, easySPIN_RegsStruct->STEP_MODE);
	easySPIN_SetParam(easySPIN_ALARM_EN, easySPIN_RegsStruct->ALARM_EN);
	easySPIN_SetParam(easySPIN_CONFIG, easySPIN_RegsStruct->CONFIG);
}

/* Application Commands implementation ----------------------------------------*/

/**
 * @brief  Issues easySPIN NOP command.
 * @param  None
 * @retval None
 */
void easySPIN_Nop(void) {
	/* Send NOP operation code to easySPIN */
	easySPIN_Write_Byte(easySPIN_NOP);
}

/**
 * @brief  Issues easySPIN SetParam command.
 * @param  easySPIN register address, value to be set
 * @retval None
 */
void easySPIN_SetParam(easySPIN_Registers_TypeDef param, uint32_t value) {
	/* Send SetParam operation code to easySPIN */
	easySPIN_Write_Byte(easySPIN_SET_PARAM | param);
	switch (param) {
	case easySPIN_ABS_POS:
		;
	case easySPIN_MARK:
		/* Send parameter - byte 2 to easySPIN */
		easySPIN_Write_Byte((uint8_t) (value >> 16));
	case easySPIN_EL_POS:
		;
	case easySPIN_CONFIG:
		;
	case easySPIN_STATUS:
		/* Send parameter - byte 1 to easySPIN */
		easySPIN_Write_Byte((uint8_t) (value >> 8));
	default:
		/* Send parameter - byte 0 to easySPIN */
		easySPIN_Write_Byte((uint8_t) (value));
	}
}

/**
 * @brief  Issues easySPIN GetParam command.
 * @param  easySPIN register address
 * @retval Register value - 1 to 3 bytes (depends on register)
 */
uint32_t easySPIN_GetParam(easySPIN_Registers_TypeDef param) {
	uint32_t temp = 0;
	uint32_t rx = 0;

	/* Send GetParam operation code to easySPIN */
	temp = easySPIN_Write_Byte(easySPIN_GET_PARAM | param);
	/* MSB which should be 0 */
	temp = temp << 24;
	rx |= temp;
	switch (param) {
	case easySPIN_ABS_POS:
		;
	case easySPIN_MARK:
		temp = easySPIN_Write_Byte((uint8_t) (0x00));
		temp = temp << 16;
		rx |= temp;
	case easySPIN_EL_POS:
		;
	case easySPIN_CONFIG:
		;
	case easySPIN_STATUS:
		temp = easySPIN_Write_Byte((uint8_t) (0x00));
		temp = temp << 8;
		rx |= temp;
	default:
		temp = easySPIN_Write_Byte((uint8_t) (0x00));
		rx |= temp;
	}
	return rx;
}

/**
 * @brief Issues easySPIN Enable command.
 * @param  None
 * @retval None
 */
void easySPIN_Enable(void) {
	/* Send Enable operation code to easySPIN */
	easySPIN_Write_Byte(easySPIN_ENABLE);
}

/**
 * @brief Issues easySPIN Disable command.
 * @param  None
 * @retval None
 */
void easySPIN_Disable(void) {
	/* Send Disable operation code to easySPIN */
	easySPIN_Write_Byte(easySPIN_DISABLE);
}

/**
 * @brief  Issues easySPIN GetStatus command.
 * @param  None
 * @retval Status Register content
 */
uint16_t easySPIN_Get_Status(void) {
	uint16_t temp = 0;
	uint16_t rx = 0;

	/* Send GetStatus operation code to easySPIN */
	easySPIN_Write_Byte(easySPIN_GET_STATUS);
	/* Send zero byte / receive MSByte from easySPIN */
	temp = easySPIN_Write_Byte((uint8_t) (0x00));
	temp <<= 8;
	rx |= temp;
	/* Send zero byte / receive LSByte from easySPIN */
	temp = easySPIN_Write_Byte((uint8_t) (0x00));
	rx |= temp;
	return rx;
}

/**
 * @brief  Checks easySPIN Flag signal.
 * @param  None
 * @retval one if Flag signal is active, otherwise zero
 */
uint8_t easySPIN_Flag(void) {
	if (!(GPIO_ReadInputDataBit(easySPIN_FLAG_Port, easySPIN_FLAG_Pin)))
		return 0x01;
	else
		return 0x00;
}

/* Additional Application Commands implementation -----------------------------*/

/**
 *	@brief  Resets easySPIN by STBY/RESET signal activation
 * @param  None
 *	@retval None
 */
void easySPIN_Reset(void) {
	/* Standby-reset signal activation - low */
	GPIO_ResetBits(easySPIN_STBY_RESET_Port, easySPIN_STBY_RESET_Pin);
}

/**
 * @brief  Release STBY/RESET signal on easySPIN
 *	@param  None
 * @retval None
 */
void easySPIN_ReleaseReset(void) {
	/* Standby-reset signal de-activation - high */
	GPIO_SetBits(easySPIN_STBY_RESET_Port, easySPIN_STBY_RESET_Pin);
}

/**
 * @brief	easySPIN movement direction setup
 * @param	direction: specifies the direction of movement
 *   This parameter can be either DIR_Forward or DIR_Reverse
 * @retval None
 */
void easySPIN_DirectionSetup(easySPIN_Direction_TypeDef direction) {
	if (direction == DIR_Forward) {
		GPIO_WriteBit(easySPIN_DIR_Port, easySPIN_DIR_Pin, Bit_SET);
	} else {
		GPIO_WriteBit(easySPIN_DIR_Port, easySPIN_DIR_Pin, Bit_RESET);
	}
}

/******************* (C) COPYRIGHT 2012 STMicroelectronics *****END OF FILE****/
